/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_RADIOBUTTON
#define __GLIB_RADIOBUTTON

#include "glib/gui/GWindow.h"

/**
 * This is the low level window class that implements a radio button control.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.12
 */
class GRadioButton : public GWindow
{
   private:

      /**
       * The class representing the system provided radio button control.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      class Peer : public GWindow
      {
         public:

            Peer ( GRadioButton& parentWin, bool noFocusPaint );
            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }
      };

      bool useHotKey;

      mutable Peer peer;

   public:

      GRadioButton ( const GString& name,
                     const GString& constraints,
                     GWindow& parentWin,
                     long winStyle = WS_VISIBLE,
                     long winStyle2 = WS2_DEFAULTPAINT,
                     bool noFocusPaint = false,
                     bool useHotKey = true );

      virtual ~GRadioButton ();

   private:

      /** Disable the copy constructor. */
      GRadioButton ( const GRadioButton& src ) : peer(*this, false) {}

      /** Disable the assignment operator. */
      GRadioButton& operator= ( const GRadioButton& ) { return *this; }

   private:

      void setToggleState_ ( bool state, bool noRadio );
      void updateRadioGroupAfterClick ();

   protected:

      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );
      virtual bool onKeyDown ( const GKeyMessage& key );

   public:

      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual bool isEmpty () const;
      virtual GString queryValue () const;

      /**
       * This method is called when the user has pressed the mnemonic
       * keyboard code (Alt+X, where X is the hot mnemonic character).
       *
       * We will perform just as if the radio button has been clicked by
       * the mouse, and toggle the state of the button.
       */
      virtual void hotKeyAction ();

      /**
       * Get a reference to the peer window.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      GWindow& getPeer ();

      /**
       * Get the current toggle state.
       */
      bool getToggleState () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void grabFocus ( bool force );

      /**
       * We override this method in order to send 
       * {@link GDialogMessageHandler#GM_CTRLCHANGED} only if
       * the radio button value is true. That is, it will send 
       * {@link GDialogMessageHandler#GM_CTRLCHANGED} messages only 
       * when it is toggled on.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.28
       */
      virtual void sendCtrlChanged ();

      /**
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void setEnabled ( bool flag, bool repaint );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.07
       */
      virtual void setOily ( bool flag );

      /**
       * Set the toggle state on (true) or off (false).
       */
      void setToggleState ( bool state );

      /**
       * Set the text of the radio button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void setText ( const GString& text );
};

#endif
