/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GRadioButton.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/layout/GBorderLayout.h"

GRadioButton::Peer::Peer ( GRadioButton& parentWin, 
                           bool noFocusPaint )
                   :GWindow("Peer",
                            GBorderLayout::CENTER,
                            &parentWin,
                            &parentWin,
                            WS_VISIBLE | BS_RADIOBUTTON | (noFocusPaint ? BS_NOCURSORSELECT : 0),
                            WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                            GWindowClass::BUTTON)
{
   setFocusable(true);
}

GRadioButton::Peer::~Peer ()
{
}

GRadioButton::GRadioButton ( const GString& name,
                             const GString& constraints,
                             GWindow& parentWin,
                             long winStyle,
                             long winStyle2,
                             bool noFocusPaint,
                             bool useHotKey )
             :GWindow(name, 
                      constraints, 
                      &parentWin, 
                      &parentWin,
                      winStyle,
                      winStyle2 | WS2_OS2Y | WS2_NOT_STATIC),
              useHotKey(useHotKey),
              peer(*this, noFocusPaint)
{
   setLayoutManager(new GBorderLayout(), true);
}

GRadioButton::~GRadioButton ()
{
}

GWindow& GRadioButton::getPeer ()
{
   return peer;
}

void GRadioButton::setText ( const GString& text )
{
   GString prevText = getText();
   if (text == prevText)
      return;

   GWindow::setText(text);
   GString str = text;
   if (useHotKey)
   {
      int pos = autoAssocAHotKey(str);
      if (pos >= 0)
         str.insert(GPushButton::MNEMONIC_MARKER_CHAR, pos);
   }
   peer.setText(str);
}

void GRadioButton::grabFocus ( bool force )
{
   peer.grabFocus(force);
}

void GRadioButton::setEnabled ( bool flag, bool repaint )
{
   GWindow::setEnabled(flag, repaint);
   peer.setEnabled(flag, repaint);
}

void GRadioButton::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

void GRadioButton::setToggleState_ ( bool state, bool noRadio )
{
   bool prev = getToggleState();
   if (state != prev)
   {
      peer.sendMessage(BM_SETCHECK, GWindowMessage::Param1(state));
      if (!noRadio)
         updateRadioGroupAfterClick();
   }
}

void GRadioButton::setToggleState ( bool state )
{
   setToggleState_(state, false);
}

bool GRadioButton::getToggleState () const
{
   return bool(peer.sendMessage(BM_QUERYCHECK));
}

void GRadioButton::hotKeyAction ()
{
   grabFocus(false);
   if (!getToggleState())
      setToggleState(true);
}

bool GRadioButton::onKeyDown ( const GKeyMessage& key )
{
   if (key.getCharacter() != '\0' && 
       key.getCode() == GKey::KEY_SPACE &&
      !key.isAltDown() && 
      !key.isCtrlDown() && 
      !key.isShiftDown())
   {
      if (!getToggleState())
         setToggleState(true);
      return true;
   }
   else
   {
      return GWindow::onKeyDown(key);
   }
}

bool GRadioButton::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case BN_CLICKED:
      case BN_DBLCLICKED:
      {
         setToggleState(true);
         break;
      }
   }

   return true;
}

GString GRadioButton::queryValue () const
{
   bool state = getToggleState();
   return GInteger::ToString(state);
}

void GRadioButton::changeValue ( const GString& newValue, bool notify )
{
   bool newval = (newValue == "1" || 
                  newValue.equalsIgnoreCase("true") || 
                  newValue.equalsIgnoreCase("yes") || 
                  newValue.equalsIgnoreCase("on"));

   enterCtrlChangedFilter();
   setToggleState(newval);
   exitCtrlChangedFilter();

   if (notify)
      fireValueChangeListeners();
}

bool GRadioButton::isEmpty () const 
{ 
   return false; 
}

void GRadioButton::sendCtrlChanged ()
{
   bool on = getToggleState();
   if (!on)
      return;
   GWindow::sendCtrlChanged();
}

void GRadioButton::updateRadioGroupAfterClick ()
{
   GWindow* owner = getOwnerDialogPanel();
   if (owner == null)
      owner = getOwnerWindow();

   if (owner == null)
      return; // No owner means that we are "alone".

   // Don't send GM_CTRLCHANGED until we have done.
   enterCtrlChangedFilter();

   // Collect all radio buttons in the group, except our self.
   GArray<GRadioButton> group(10);
   GString myname = getName();
   const int ourOwnIndex = owner->getComponentIndex(myname);
   // Collect backward...
   for (int bidx=ourOwnIndex-1; bidx >= 0; bidx--)
   {
      GRadioButton* butt = dynamic_cast<GRadioButton*>(&owner->getComponentByIndex(bidx));
      if (butt == null)
         break;
      group.add(butt, false);
   }
   // Collect foreward...
   const int numCtrls = owner->getComponentCount();
   for (int fidx=ourOwnIndex+1; fidx<numCtrls; fidx++)
   {
      GRadioButton* butt = dynamic_cast<GRadioButton*>(&owner->getComponentByIndex(fidx));
      if (butt == null)
         break;
      group.add(butt, false);
   }

   // Uncheck all other buttons in the group (except our self).
   const int groupNum = group.getCount();
   for (int i=0; i<groupNum; i++)
   {
      GRadioButton& butt = group[i];
      butt.setToggleState_(false, true);
   }

   // Re-enable the GM_CTRLCHANGED message.
   exitCtrlChangedFilter();

   // Finally, send the GM_CTRLCHANGED message.
   fireValueChangeListeners();
}
