/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GMultiLineEditor.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GWindowMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GLog.h"
#include "glib/vfs/GFile.h"

GMultiLineEditor::Peer::Peer ( GMultiLineEditor& theEntry )
                       :GWindow(GWindowClass::MLE, GString::Empty, true),
                        theEntry(theEntry)
{
}

GMultiLineEditor::Peer::~Peer ()
{
}

bool GMultiLineEditor::Peer::onKeyDown ( const GKeyMessage& key )
{
   GKey::KeyCode code = key.getCode();
   switch (code)
   {
      case GKey::KEY_RIGHT:
      case GKey::KEY_LEFT:
         if (dynamic_cast<GDialogFrame*>(&getTopLevelWindow()) == null)
            return GWindow::onKeyDown(key);
         // Give this keyboard event directly to the system.
         return callDefaultKeyHandler(key, true, true);

      case GKey::KEY_UP:
      case GKey::KEY_DOWN:
      case GKey::KEY_ENTER:
      case GKey::KEY_TABULATOR:
         if (dynamic_cast<GDialogFrame*>(&getTopLevelWindow()) == null)
            return GWindow::onKeyDown(key);
         if (theEntry.useAsSingleLine)
            return GWindow::onKeyDown(key);
         if (theEntry.ignoreTab && code == GKey::KEY_TABULATOR)
            return GWindow::onKeyDown(key);
         // Give this keyboard event directly to the system.
         return callDefaultKeyHandler(key, true, true);

      default:
         return GWindow::onKeyDown(key);
   }
}

GMultiLineEditor::PeerParent::PeerParent ( GMultiLineEditor& theEntry )
                 :GWindow("PeerParent", 
                          GBorderLayout::CENTER, 
                          &theEntry, 
                          &theEntry, 
                          WS_VISIBLE, 
                          WS2_OS2Y | WS2_AUTO_PAINT_BACKGROUND),
                  theEntry(theEntry)
{
   setBackgroundColor(defaultBackgroundColor = GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYBCK));
   setForegroundColor(defaultForegroundColor = GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYTXT));
   setLayoutManager(new GBorderLayout(), true);
}

GMultiLineEditor::PeerParent::~PeerParent ()
{
}

GMultiLineEditor::GMultiLineEditor ( const GString& name,
                                     const GString& constraints,
                                     GWindow& parentWin,
                                     long winStyle,
                                     long winStyle2,
                                     int maxTextLength,
                                     bool wordWrap,
                                     bool sysBorder,
                                     bool readOnly,
                                     bool ignoreTab,
                                     bool useAsSingleLine )
                 :GAbstractTextField(parentWin, name, 
                                     constraints, 
                                     winStyle,
                                     winStyle2 | WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG),
                  peerParent(*this),
                  peer(*this),
                  ignoreTab(ignoreTab),
                  useAsSingleLine(useAsSingleLine)
{
   setBackgroundColor(defaultBackgroundColor = GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYBCK));
   setForegroundColor(defaultForegroundColor = GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYTXT));

   int f = (WS_VISIBLE | WS_SYNCPAINT | MLS_LIMITVSCROLL |
           (useAsSingleLine ? 0 : MLS_VSCROLL) |
           (sysBorder ? MLS_BORDER : 0) |
           (wordWrap ? MLS_WORDWRAP : (useAsSingleLine?0:MLS_HSCROLL)) |
           (readOnly ? MLS_READONLY : 0) |
           (ignoreTab ? MLS_IGNORETAB : 0));
   peer.init("Peer", &peerParent, &peerParent, f, WS2_DEFAULTPAINT | WS2_IGNORE_COLORS_AND_FONT_PROFILE | WS2_NOT_STATIC | WS2_OS2Y | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT);
   if (useAsSingleLine && !sysBorder)
      peerParent.setInsets(new GInsets(-4, -3, -4, 0), true);

   peer.setFocusable(true);
   setLayoutManager(new GBorderLayout(), true);
   setMaxTextLength(maxTextLength <= 0 ? -1 : maxTextLength);
}


GMultiLineEditor::~GMultiLineEditor ()
{
}

GWindow& GMultiLineEditor::getPeer () const
{
   return peer;
}

bool GMultiLineEditor::onBackgroundColorChanged ( const GColor& color )
{
   peerParent.setBackgroundColor(color);
   GAbstractTextField::onBackgroundColorChanged(color);
   return true;
}

bool GMultiLineEditor::onFontNameSizeChanged ( const GString& fontNameSize )
{
   peerParent.setFontNameSize(fontNameSize);
   GAbstractTextField::onFontNameSizeChanged(fontNameSize);
   return true;
}

bool GMultiLineEditor::onForegroundColorChanged ( const GColor& color )
{
   peerParent.setForegroundColor(color);
   GAbstractTextField::onForegroundColorChanged(color);
   return true;
}

bool GMultiLineEditor::PeerParent::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != theEntry.peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case MLN_CHANGE: 
         // The Multiline Entry Field Text has been changed.
         theEntry.fireValueChangeListeners();
         break;
   }

   return true;
}

void GMultiLineEditor::copy () const
{
   peer.sendMessage(MLM_COPY);
}

void GMultiLineEditor::cut ()
{
   peer.sendMessage(MLM_CUT);
}

void GMultiLineEditor::paste ()
{
   peer.sendMessage(MLM_PASTE);
}

void GMultiLineEditor::setSelection ( int selStart, int selEnd )
{
   peer.sendMessage(MLM_SETSEL, GWindowMessage::Param1(selStart), GWindowMessage::Param2(selEnd));
}

int GMultiLineEditor::getSelectionEnd () const
{
   return int(peer.sendMessage(MLM_QUERYSEL, GWindowMessage::Param1(MLFQS_MAXSEL)));
}

int GMultiLineEditor::getSelectionStart () const
{
   return int(peer.sendMessage(MLM_QUERYSEL, GWindowMessage::Param1(MLFQS_MINSEL)));
}

int GMultiLineEditor::getTextLength () const
{
   return int(peer.sendMessage(MLM_QUERYTEXTLENGTH));
}

int GMultiLineEditor::getLineNum () const
{
   return int(peer.sendMessage(MLM_QUERYLINECOUNT));
}

GString GMultiLineEditor::getLineText ( int lineIndex ) const
{
   int lineStartPos = int(peer.sendMessage(MLM_CHARFROMLINE, GWindowMessage::Param1(lineIndex)));
   int lineLen = int(peer.sendMessage(MLM_QUERYLINELENGTH, GWindowMessage::Param1(lineStartPos)));
   if (lineLen <= 0)
      return GString::Empty;
   int buffSize = lineLen + 1;
   char* buff = new char[buffSize];
   peer.sendMessage(MLM_SETIMPORTEXPORT, GWindowMessage::Param1(buff), GWindowMessage::Param2(buffSize));
   peer.sendMessage(MLM_EXPORT, GWindowMessage::Param1(&lineStartPos), GWindowMessage::Param2(&lineLen));
   buff[buffSize - 1] = '\0';
   GString ret = buff;
   delete [] buff;
   return ret;
}

int GMultiLineEditor::getCaretPosition () const
{
   return IPT(peer.sendMessage(MLM_QUERYSEL, MPFROMSHORT(MLFQS_CURSORSEL)));
}

int GMultiLineEditor::getCaretLineIndex () const
{
   int pos = getCaretPosition();
   return int(peer.sendMessage(MLM_LINEFROMCHAR, GWindowMessage::Param1(pos)));
}

int GMultiLineEditor::getCaretColumnPos () const
{
   int line = getCaretLineIndex();
   int lineStartPos = int(peer.sendMessage(MLM_CHARFROMLINE, GWindowMessage::Param1(line)));
   int curPos = getCaretPosition();
   return curPos - lineStartPos;
}

GString GMultiLineEditor::getPartOfWordBeforeCaret () const
{
   int lineIndex = getCaretLineIndex();
   if (lineIndex <= -1)
      return GString::Empty;

   int endIndex = getCaretColumnPos();
   if (endIndex <= 0)
      return GString::Empty;

   int startIndex = endIndex;
   GString text = getLineText(lineIndex);
   if (startIndex > text.length()) // Shall never happen, but in case
      return GString::Empty;

   while (startIndex > 0)
   {
      char chr = text.charAt(startIndex - 1);
      if (isspace(chr) || GFile::IsSlash(chr) || chr == '"')
         break;
      else
         startIndex--;
   }

   return text.substring(startIndex, endIndex);
}

void GMultiLineEditor::setMaxTextLength ( int maxLength )
{
   peer.sendMessage(MLM_SETTEXTLIMIT, GWindowMessage::Param1(maxLength));
}

void GMultiLineEditor::appendText ( const GString& text )
{
   int len = getTextLength();
   peer.sendMessage(MLM_SETSEL, GWindowMessage::Param1(len), GWindowMessage::Param2(len));
   replaceSelection(text);
   peer.sendMessage(MLM_SETFIRSTCHAR); // Just to force the caret to be visible
}

void GMultiLineEditor::setText ( const GString& text )
{
   // Do it in two steps so that the caret gets moved behind the inserted text.
   long len = long(peer.sendMessage(MLM_QUERYTEXTLENGTH));
   peer.sendMessage(MLM_SETSEL, 0, GWindowMessage::Param2(len));
   replaceSelection(text);
   peer.sendMessage(MLM_SETFIRSTCHAR); // Just to force the caret to be visible.
}

void GMultiLineEditor::replaceSelection ( const GString& text )
{
   peer.sendMessage(MLM_INSERT, GWindowMessage::Param1(text.cstring()));
}

bool GMultiLineEditor::isIgnoreTab () const
{
   return ignoreTab;
}

int GMultiLineEditor::getPreferredHeight () const
{
   return getHeightOfString("X") + 12;
}

int GMultiLineEditor::getPreferredWidth () const
{
   return (32 * getWidthOfString("x")) + 8;
}
