/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include "glib/gui/GMenuPopup.h"
#include "glib/gui/GMenu.h"
#include "glib/util/GMath.h"
#include "glib/sys/GSystem.h"
#include "glib/resource/GAccelItem.h"

int GMenuPopup::MenuIDCounter = 0;

GMenuPopup::TheWindow::TheWindow ( GMenuPopup& menuObj,
                                   const GString& name,
                                   GWindow& ownerWin,
                                   const GString& constraints,
                                   GMenu& topLevelMenu,
                                   bool isMenuBar,
                                   int level )
                      :GWindow(GWindowClass::MENU, constraints, true),
                       menuObj(menuObj)
{
   // Initiate the window object, but don't create the HWND yet.
   int flags2 = WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED |
                WS2_IGNORE_COLORS_PROFILE |
                WS2_DONT_CREATE_HWND |
                WS2_NO_RECURSIVE_KEY_HANDLING |
                WS2_NOT_STATIC |
                WS2_OS2Y;
   if (level > 0)
      flags2 |= WS2_IGNORE_FONT_PROFILE;
   if (isMenuBar)
      flags2 |= WS2_DEFAULTPAINT; // Let the system handle the WM_PAINT of the menu-bar it self.
   if (!topLevelMenu.useFancyMenues)
      flags2 |= WS2_DEFAULTPAINT; // Let the system handle the WM_PAINT if "fancy menues" is disabled.
   GWindow::init(name, &ownerWin, &ownerWin, WS_VISIBLE, flags2);

   // Create an empty menu window.
   HWND hwnd = ::WinCreateMenu(ownerWin.getHWND(), null);

   // Since we gave true to the "postponeWinCrea" parameter of GWindow
   // we must assign the HWND now.
   setHWND(hwnd);

   // It is very important that _all_ menu windows that are created and
   // managed by G-Lib have the window id FID_MENU. This is in order for
   // the handler of WM_DRAWITEM to be able to identify menu controls from
   // other types of controls.
   ::WinSetWindowUShort(hwnd, QWS_ID, FID_MENU);

   // We must subclass the window so that our {@link #handleWindowMessage}
   // gets called the same way as for other window classes.
   setDefaultMsgProc(GWindow::SubclassWindow(hwnd, GWindow::GenericWindowProc));
   ::WinSetWindowPtr(hwnd, QWL_USER, (void*) getWindowEntry()); // Because QWL_USER was not set upon WM_CREATE.
}

GMenuPopup::TheWindow::~TheWindow ()
{
}

GMenuPopup::GMenuPopup ( const GString& name,
                         const GString& constraints,
                         GMenu& topLevelMenu,
                         GWindow& parentWin,
                         GMenuPopupParams* popup,
                         bool mnemonicsOn,
                         int level,
                         bool dontInit,
                         bool isMenuBar )
           :level(level),
            topLevelMenu(topLevelMenu),
            items(18, 8),
            parentWin(parentWin),
            iconMenuItemTlgMark(null),
            iconMenuItemSubMenu(null),
            theWin(*this, name, parentWin, constraints, topLevelMenu, isMenuBar, level)
{
   if (!dontInit) // If called by top level GMenu.
      init_(parentWin, popup, mnemonicsOn);
}

GMenuPopup::~GMenuPopup ()
{
}

void GMenuPopup::init_ ( GWindow& ownerWin,
                         GMenuPopupParams* popup,
                         bool mnemonicsOn )
{
   iconMenuItemTlgMark = GIcon::GetIcon("%STDICON_MENUITEMTOGGLEMARK_ON");
   iconMenuItemSubMenu = GIcon::GetIcon("%STDICON_MENUITEM_SUBMENU");

   // To automatically assign a mnemonic character to each menu item/popup
   // we need a bag to keep track of which mnemonic characters are occupied.
   const GString validMnemonics("ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789");
   GString mnemonics;
   GString mitext(64);

   // Add all the items into the menu window
   for (int i=0, num=popup->items.getCount(); i<num; i++)
   {
      int id = ++MenuIDCounter;
      GMenuItemParams& itemParams = popup->items.get(i);
      GMenuPopupItem* item = new GMenuPopupItem(*this, itemParams, id);
      topLevelMenu.commandMap.addCommand(id, item);
      items.add(item);

      // Insert the OS specific mnemonic tag, if we can find a character in
      // the menu item text that is not already used as a mnemonic character.
      item->mnemonicChar = -1;
      if (mnemonicsOn && item->iType != GMenuItemParams::SEPARATOR)
      {
         for (int i=0, len=item->text.length(); i<len; i++)
         {
            char chr = GCharacter::ToUpperCase(item->text[i]);
            if (chr == '\t')
               break; // We have reached the shortcut key description!
            if (chr == ' ')
               continue;
            if (validMnemonics.indexOf(chr) <= -1)
               continue;
            if (mnemonics.contains(chr))
               continue;
            mnemonics += chr;
            item->mnemonicChar = i;
            break;
         }
      }

      switch (item->iType)
      {
         case GMenuItemParams::ITEM:
         {
            MENUITEM mi;
            memset(&mi, 0, sizeof(mi));
            mi.iPosition = MIT_END;
            if (!isMenubar() && topLevelMenu.useFancyMenues)
            {
               mi.afStyle |= MIS_OWNERDRAW;
               mi.hItem = (ULONG) item; // Pointer to the menu item object it self.
            }
            else
            {
               mitext = item->getSystemText();
               mi.afStyle |= MIS_TEXT;
               mi.hItem = (ULONG) mitext.cstring();
            }
            mi.id = (USHORT) id;
            theWin.sendMessage(MM_INSERTITEM, GWindowMessage::Param1(&mi), GWindowMessage::Param2(mi.hItem));
            break;
         }

         case GMenuItemParams::POPUP:
         {
            // The below created instance of <i>GMenuPopup</i> will
            // be automatically destroyed upon receiving
            // the <i>WM_DESTROY</i> message. 
            // TODO: Make sure the above statement is true!
            GMenuPopup* popWin = new GMenuPopup(GString::Empty, GString::Empty, topLevelMenu, ownerWin, (GMenuPopupParams *) &itemParams, true, level + 1, false, false);
            item->subPopup = popWin;
            MENUITEM mi;
            memset(&mi, 0, sizeof(mi));
            mi.iPosition = MIT_END;
            mi.afStyle = MIS_SUBMENU;
            if (item->getIDString() == "HELPMENU")
               mi.afStyle |= MIS_BUTTONSEPARATOR;
            if (!isMenubar() && topLevelMenu.useFancyMenues)
            {
               mi.afStyle |= MIS_OWNERDRAW;
               mi.hItem = (ULONG) item; // Pointer to the menu item object it self
            }
            else
            {
               mitext = item->getSystemText();
               mi.afStyle |= MIS_TEXT;
               mi.hItem = (ULONG) mitext.cstring();
            }
            mi.id = (USHORT) id;
            mi.hwndSubMenu = popWin->theWin.getHWND();
            theWin.sendMessage(MM_INSERTITEM, GWindowMessage::Param1(&mi), GWindowMessage::Param2(mi.hItem));
            break;
         }

         case GMenuItemParams::SEPARATOR:
         {
            MENUITEM mi;
            memset(&mi, 0, sizeof(mi));
            mi.iPosition = MIT_END;
            mi.afStyle = MIS_SEPARATOR;
            mi.id = (USHORT) id;
            mi.hItem = (ULONG) item; // Pointer to the menu item object it self.
            theWin.sendMessage(MM_INSERTITEM, GWindowMessage::Param1(&mi), 0);
            break;
         }
      }
   }
}

GMenuPopup::TheWindow& GMenuPopup::getTheWindow () 
{ 
   return theWin; 
}

GMenuPopupItem& GMenuPopup::getIndexedItem ( int idx ) 
{ 
   return items.get(idx); 
}

int GMenuPopup::getItemCount () const 
{ 
   return items.getCount(); 
}

GMenu& GMenuPopup::getTopLevelMenu () 
{ 
   return topLevelMenu; 
}

bool GMenuPopup::isMenubar () const 
{ 
   return false; 
}

void GMenuPopup::setUseFancyMenues ( bool fancy )
{
   bool menubar = isMenubar();
   if (!menubar)
      theWin.setDefaultPaint(!fancy);
   GString mitext(64);
   for (int i=0, num=getItemCount(); i<num; i++)
   {
      MENUITEM mi;
      memset(&mi, sizeof(mi), 0);
      GMenuPopupItem& item = getIndexedItem(i);
      int itemSysID = item.getSysID();
      if (theWin.sendMessage(MM_QUERYITEM, MPFROM2SHORT(itemSysID, false), MPFROMP(&mi)))
      {
         if (mi.afStyle & MIS_SEPARATOR)
            continue;
         if (!menubar)
         {
            if (fancy)
            {
               mi.afStyle |= MIS_OWNERDRAW;
               mi.afStyle &= ~MIS_TEXT;
               mi.hItem = (ULONG) &item; // Pointer to the menu item object it self.
            }
            else
            {
               mitext = item.getSystemText();
               mi.afStyle |= MIS_TEXT;
               mi.afStyle &= ~MIS_OWNERDRAW;
               mi.hItem = (ULONG) mitext.cstring();
            }
            theWin.sendMessage(MM_SETITEM, MPFROM2SHORT(0, false), MPFROMP(&mi));
            // Due to a bug in OS/2 Warp 4.0 (?) we have to explicitly update
            // the menu item text. This is not updated upon MM_SETITEM,
            // as it should.
            if (!fancy)
               theWin.sendMessage(MM_SETITEMTEXT, MPFROMSHORT(itemSysID), MPFROMP(mitext.cstring()));
         }
         if (mi.afStyle & MIS_SUBMENU)
         {
            GMenuPopup* subPopup = item.subPopup;
            if (subPopup != null)
               subPopup->setUseFancyMenues(fancy);
         }
      }
   }
}

void GMenuPopup::setVisible ( bool flag )
{
   theWin.setVisible(flag);
}

bool GMenuPopup::isVisible () const
{
   return theWin.isVisible();
}

GWindow& GMenuPopup::getParentWindow () 
{ 
   return parentWin; 
}

bool GMenuPopup::TheWindow::onPaint ( GGraphics& g, const GRectangle& rect )
{
   // Draw the submenu window face area
   GColor bckColor = GSystem::GetSystemColor(GSystem::SCID_MENUBCK);
   g.drawFilledRectangle(rect, bckColor);
   
   GRectangle winRect = getWindowRect();
   winRect.height -= 1;
   g.setColor(GColor::DGRAY);
   g.setPosition(0, winRect.y);
   g.drawLineTo(0, winRect.y + winRect.height - 1);
   g.drawLineTo(winRect.x + winRect.width - 1, winRect.y + winRect.height - 1);
   winRect.height -= 1;
   winRect.x += 1;
   winRect.width -= 2;

   int left = winRect.x;
   int bottom = winRect.y;
   int right = left + winRect.width - 1;
   int top = bottom + winRect.height - 1;

   const bool down = false;
   g.setColor(down ? GColor::BLACK : GColor::WHITE);
   g.setPosition(left, bottom);
   g.drawLineTo(left, top);
   g.drawLineTo(right - 1, top);
   g.setColor(down ? GColor::WHITE : GColor::BLACK);
   g.setPosition(right, top);
   g.drawLineTo(right, bottom);
   g.drawLineTo(left, bottom);

   left += 1;
   bottom += 1;
   right -= 1;
   top -= 1;

   g.setColor(down ? GColor::DGRAY : GColor::GRAY);
   g.setPosition(left, bottom);
   g.drawLineTo(left, top);
   g.drawLineTo(right - 1, top);
   g.setColor(down ? GColor::GRAY : GColor::DGRAY);
   g.setPosition(right, top);
   g.drawLineTo(right, bottom);
   g.drawLineTo(left, bottom);

   // Draw the separators explicitly, since we doesn't get WM_DRAWITEM
   // messages for such menu items.
   int num = menuObj.items.getCount();
   for (int i=0; i<num; i++)
   {
      GMenuPopupItem& item = menuObj.items.get(i);
      int itemSysID = item.getSysID();

      OWNERITEM oi;
      memset(&oi, 0, sizeof(oi));
      oi.hwnd = getHWND();
      oi.hps = g.getHandle();
      oi.fsState = 0;
      oi.fsAttribute = 0;
      sendMessage(MM_QUERYITEMRECT, MPFROM2SHORT(itemSysID, false), MPFROMP(&oi.rclItem));

      MENUITEM mi;
      sendMessage(MM_QUERYITEM, MPFROM2SHORT(itemSysID, false), MPFROMP(&mi));
      oi.fsAttribute = mi.afAttribute;
      oi.fsState = mi.afStyle;

      // Due to a bug (?) in OS/2 PM we need to adjust the area of the
      // last item slighly. (We have to emulate that "bug", so that the
      // item gets painted at the same location as when it is painted
      // again due to a system generated WM_DRAWITEM.
      if (i == num - 1) // If this is the last item
      {
         oi.rclItem.yTop += 1;
         oi.rclItem.yBottom += 2;
      }

      if (item.iType == GMenuItemParams::POPUP)
      {
         oi.fsState |= MIS_SUBMENU;
      }
      else
      if (item.iType == GMenuItemParams::SEPARATOR)
      {
         oi.fsState |= MIS_SEPARATOR;
         int ypos = oi.rclItem.yTop - ((oi.rclItem.yTop - oi.rclItem.yBottom) / 2);
         g.setColor(GColor::DGRAY);
         g.setPosition(2, ypos);
         g.drawLineTo(oi.rclItem.xRight - 2, ypos);
         g.setColor(GColor::WHITE);
         g.setPosition(2, ypos-1);
         g.drawLineTo(oi.rclItem.xRight - 2, ypos-1);
         continue;
      }

      oi.idItem = itemSysID;
      oi.hItem = (ULONG) &item;

      bool itemDisabled = ((oi.fsAttribute & MIA_DISABLED) != 0);
      bool itemHilited = ((oi.fsAttribute & MIA_HILITED) != 0);
      bool itemChecked = ((oi.fsAttribute & MIA_CHECKED) != 0);
      GRectangle rect;
      rect.x = oi.rclItem.xLeft;
      rect.y = sysY(oi.rclItem.yBottom);
      rect.width = oi.rclItem.xRight - oi.rclItem.xLeft + 1;
      rect.height = oi.rclItem.yTop - oi.rclItem.yBottom + 1;
      rect.inflateRect(-2, 0);

      menuObj.drawMenuItem(g, item, rect, itemDisabled, itemHilited, itemChecked);
   }

   return true;
}

bool GMenuPopup::TheWindow::onKeyDown ( const GKeyMessage& key )
{
   if (!key.isPureCharacter())
      return false;

   char chr = GCharacter::ToUpperCase(key.getCharacter());
   int num = menuObj.items.getCount();
   for (int i=0; i<num; i++)
   {
      GMenuPopupItem& mi = menuObj.items.get(i);
      if (mi.mnemonicChar >= 0)
      {
         char mnemonicChar = mi.text.charAt(mi.mnemonicChar);
         if (chr == GCharacter::ToUpperCase(mnemonicChar))
         {
            // Select the menu item that owns the mnemonic character.
            // This will post a corresponding WM_COMMAND message as well.
            bool dismiss = (mi.iType == GMenuItemParams::ITEM);
            int miSysID = mi.getSysID();
            sendMessage(MM_SELECTITEM, MPFROM2SHORT(miSysID, false), MPFROM2SHORT(0, dismiss));
            return true;
         }
      }
   }

   return GWindow::onKeyDown(key);
}

void GMenuPopup::drawMenuItem ( GGraphics& g,
                                GMenuPopupItem& item,
                                const GRectangle& rect,
                                bool itemDisabled,
                                bool itemHilited,
                                bool itemChecked )
{
   if (item.iType == GMenuItemParams::SEPARATOR)
   {
      g.setLineType(GGraphics::LTSolid);
      int ypos = rect.y + (rect.height / 2);
      g.setColor(GColor::DGRAY);
      g.setPosition(2, ypos);
      g.drawLineTo(rect.x + rect.width - 1, ypos);
      g.setColor(GColor::WHITE);
      g.setPosition(2, ypos-1);
      g.drawLineTo(rect.x + rect.width - 1, ypos-1);
      return;
   }

   GColor bckColor;
   GColor frgColor;
   if (itemDisabled)
   {
      frgColor = GSystem::GetSystemColor(GSystem::SCID_MENUDISABLEDTXT);
      if (itemHilited)
      {
         bckColor = GSystem::GetSystemColor(GSystem::SCID_MENUHILITEBCK);
         int menuBckColor = GSystem::GetSystemColor(GSystem::SCID_MENUBCK);
         if (menuBckColor == bckColor)
         {
            bckColor = GColor::DBLUE;
            frgColor = GColor::DGRAY;
         }
      }
      else
      {
         bckColor = GSystem::GetSystemColor(GSystem::SCID_MENUBCK);
      }
   }
   else
   if (itemHilited)
   {
      bckColor = GSystem::GetSystemColor(GSystem::SCID_MENUHILITEBCK);
      frgColor = GSystem::GetSystemColor(GSystem::SCID_MENUHILITETXT);
      GColor menuBckColor = GSystem::GetSystemColor(GSystem::SCID_MENUBCK);
      if (menuBckColor == bckColor)
      {
         bckColor = GColor::DBLUE;
         frgColor = GColor::WHITE;
      }
   }
   else
   {
      bckColor = GSystem::GetSystemColor(GSystem::SCID_MENUBCK);
      frgColor = GSystem::GetSystemColor(GSystem::SCID_MENUTXT);
   }

   g.drawFilledRectangle(rect, bckColor);

   int xpos = rect.x;

   // Draw the icon used to visualize that the item is checked.

   int widthOfToggleIcon = 16; // Default width, in case icon isn't found

   if (itemChecked)
   {
      if (iconMenuItemTlgMark != null)
      {
         widthOfToggleIcon = iconMenuItemTlgMark->getWidth();
         int ypos = rect.y + (rect.height / 2) - (iconMenuItemTlgMark->getHeight() / 2);
         g.drawIcon(xpos, ypos, *iconMenuItemTlgMark);
      }
   }

   xpos += widthOfToggleIcon + 2;

   // Draw the icon, if any.

   int widthOfItemIcon = 20; // Default, in case icon isn't found
   if (item.icon != null)
   {
      widthOfItemIcon = item.icon->getWidth();
      int ypos = rect.y + (rect.height / 2) - (item.icon->getHeight() / 2);
      g.drawIcon(xpos, ypos, *item.icon);
   }

   xpos += widthOfItemIcon + 2;

   // Draw the item text.
   GRectangle clipRect = rect;
   clipRect.x = xpos + 4;
   clipRect.width -= (xpos - rect.x + 1);
   if (item.mnemonicChar >= 0)
   {
      g.setColor(frgColor);
      int fonth = g.getFontHeight();
      int ypos = clipRect.y + clipRect.height/2 - fonth/2;
      g.drawTextMnemonic(clipRect.x, ypos, item.text, item.mnemonicChar);
   }
   else
   {
      g.drawText(item.text, clipRect, frgColor);
   }

   // Draw the item shortcut key text, if any.
   clipRect.width -=  8;
   GAccelItem* ai = parentWin.getAccelItemByCommandID(item.getIDString());
   if (ai != null)
   {
      const GString& keystr = ai->getKeyNameForHumans();
      if (keystr != "")
         g.drawText(keystr, clipRect, /* TODO: frgColor */ GColor::DGRAY, GGraphics::DUMMY_COLOR, GGraphics::RIGHT);
   }
   else
   if (item.accelKeyID != "")
   {
      ai = parentWin.getAccelItemByCommandID(item.accelKeyID);
      if (ai != null)
      {
         const GString& keystr = ai->getKeyNameForHumans();
         if (keystr != "")
            g.drawText(keystr, clipRect, /* TODO: frgColor */ GColor::DGRAY, GGraphics::DUMMY_COLOR, GGraphics::RIGHT);
      }
   }

   // Draw the popup menu arrow, if item is a door to a sub-menu.

   if (item.subPopup != null)
   {
      if (iconMenuItemSubMenu != null)
      {
         int iconWidth = iconMenuItemSubMenu->getWidth();
         int ypos = rect.y + (rect.height / 2) - (iconWidth / 2);
         xpos = rect.x + rect.width - iconWidth - 6;
         g.drawIcon(xpos, ypos, *iconMenuItemSubMenu);
      }
   }

   if (itemDisabled)
   {
      // Visualize to the user that the menu item is disabled.
      g.setPattern(PATSYM_HALFTONE);
      g.setColor(bckColor);
      g.drawFilledRectangle(rect);
      g.setPattern(PATSYM_SOLID);
   }
}

GDimension GMenuPopup::measureMenuItem ( GGraphics& g, GMenuPopupItem& item )
{
   GDimension dim;

   // Respect the space required by the icon used to visualize checked items.
   if (iconMenuItemTlgMark != null)
   {
      dim.width += iconMenuItemTlgMark->getWidth() + 2;
      dim.height = GMath::Max(dim.height, iconMenuItemTlgMark->getHeight());
   }

   // Respect the space required by the item icon, if any.
   if (item.icon != null)
   {
      dim.width += item.icon->getWidth() + 2;
      dim.height = GMath::Max(dim.height, item.icon->getHeight());
   }

   // Respect the space required by the item text.
   GDimension itemDim = g.getTextDim(item.text);
   dim.width += itemDim.width + 12;
   dim.height = GMath::Max(dim.height, itemDim.height);

   // Respect the space required by the text of the item shortcut key.
   const GString& id = item.getIDString();
   GAccelItem* ai = parentWin.getAccelItemByCommandID(id);
   if (ai != null)
   {
      const GString& keystr = ai->getKeyNameForHumans();
      itemDim = g.getTextDim(keystr);
      dim.width += itemDim.width + 8;
   }
   else
   if (item.accelKeyID != "")
   {
      ai = parentWin.getAccelItemByCommandID(item.accelKeyID);
      if (ai != null)
      {
         const GString& keystr = ai->getKeyNameForHumans();
         itemDim = g.getTextDim(keystr);
         dim.width += itemDim.width + 8;
      }
   }

   // If the item is a door to a sub-menu, add space for the submenu-icon (an arrow).
   if (item.iType == GMenuItemParams::POPUP)
      dim.width += iconMenuItemSubMenu->getWidth();

   dim.height += 2;
   dim.width += 4;
   return dim;
}
