/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GIcon.h"
#include "glib/gui/GBitmap.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GWindow.h"
#include "glib/util/GLog.h"
#include "glib/GProgram.h"

GKeyBag<GIcon> GIcon::IconPoolNormal(32);
GKeyBag<GIcon> GIcon::IconPoolDisabled(32);

GIcon::A::A () 
         :bmImage(null), 
          bmMask(null) 
{
}

GIcon::A::~A () 
{ 
   delete bmImage; 
   delete bmMask; 
}

GIcon::B::B () 
         :hicon(null),
          wasLoadedFromFile(false)
{
}

GIcon::B::~B () 
{ 
   if (hicon != null)
   {
      if (wasLoadedFromFile)
         ::WinFreeFileIcon(hicon);
      else
         ::WinDestroyPointer(hicon);
   }
}

// Creates an icon that is initially in the A-form.
GIcon::GIcon ( const GIconResource& ires, bool grayed )
      :a(null),
       b(null),
       width(ires.getWidth()),
       height(ires.getHeight())
{
   // We need a graphics for creating the icon bitmaps.
   // Use the main window of the application for that.
   GProgram& prg = GProgram::GetProgram();
   GWindow& win = prg.getMainWindow();
   GGraphics g(win);
   const bool os2y = g.isOS2Y();

   // Create the two icon bitmaps, being initially "blank".
   a = new A();
   a->bmImage = new GBitmap(g, width, height);
   a->bmMask = new GBitmap(g, width, height);

   // If we are to paint a grayed icon, use some special colors.
   const int darkerGray = 0x5F;
   const int rgbDarkGray = (grayed ? GColor::DGRAY.getRGB() : 0);
   const int rgbDarkerGray = (grayed ? GColor(darkerGray, darkerGray, darkerGray).getRGB() : 0);

   // Draw the icon on the two icon bitmaps in memory.
   GGraphics::Handle gh = g.getHandle();
   GGraphics psImage(*a->bmImage, win, gh);
   GGraphics psMask(*a->bmMask, win, gh);
   GRectangle r(0, 0, width, height);
   psImage.drawFilledRectangle(r, GColor::BLACK);
   psMask.drawFilledRectangle(r, GColor::WHITE);
   psMask.setColor(GColor::BLACK);
   const GVector<int>& pixels = ires.getPixels();
   if (ires.getColorBits() == 0)
   {
      // Paint with true RGB colors.
      int prevRgb = -1;
      int yp = os2y ? height-1 : 0;
      for (int y=0, pixidx=0; y<height; y++, yp += (os2y ? -1 : 1))
      {
         for (int x=0; x<width; x++, pixidx++)
         {
            int rgb = pixels[pixidx];
            if (rgb == -1) // If pixel is to be transparent.
               continue; // Don't paint it.

            if (grayed) // If icon is to be grayed.
               rgb = GIcon::GetGrayedRgb(rgb, rgbDarkGray, rgbDarkerGray);

            if (rgb != prevRgb)
            {
               psImage.setColor(rgb);
               prevRgb = rgb;
            }

            psImage.drawPixel(x, yp);
            psMask.drawPixel(x, yp);
         }
      }
   }
   else // if (colorBits == 4 || colorBits == 8)
   {
      // Paint with 16 or 256 indexed colors.
      int prevIdx = -1;
      const GVector<int>& palette = ires.getPalette();
      int yp = os2y ? height-1 : 0;
      for (int y=0, pixidx=0; y<height; y++, yp += (os2y ? -1 : 1))
      {
         for (int x=0; x<width; x++, pixidx++)
         {
            int colidx = pixels[pixidx];
            if (colidx == -1) // If pixel is to be transparent.
               continue; // Don't paint it.

            if (colidx != prevIdx)
            {
               int rgb = palette[colidx];
               if (grayed) // If icon is to be grayed.
                  rgb = GIcon::GetGrayedRgb(rgb, rgbDarkGray, rgbDarkerGray);
               psImage.setColor(rgb);
               prevIdx = colidx;
            }

            psImage.drawPixel(x, yp);
            psMask.drawPixel(x, yp);
         }
      }
   }
}

int GIcon::GetGrayedRgb ( int rgb, int darkGray, int darkerGray )
{
   if (rgb == 0) // If black.
      return darkerGray;
   GColor col(rgb);
   int r = col.getRed();
   int g = col.getGreen();
   int b = col.getBlue();
   if (r <= 0x7F && g <= 0x7F && b <= 0x7F) // If very dark.
      return darkerGray;
   return darkGray;
}

// Creates an icon that is initially in the B-form.
GIcon::GIcon ( const GString& path, bool smallIcon )
      :a(null),
       b(null),
       width(0),
       height(0)
{
   b = new B();   
   b->hicon = ::WinLoadFileIcon(path, false);
   b->wasLoadedFromFile = true;
   if (b->hicon == null && GLog::Filter(GLog::TEST))
      GLog::Log(this, "Error loading icon: %s", GVArgs(path));
}

GIcon::~GIcon ()
{
   delete a;
   delete b;
}

int GIcon::getWidth () const
{
   return width;
}

int GIcon::getHeight () const
{
   return height;
}

// Static method for the factory pattern.
const GIcon* GIcon::GetIcon ( const GString& iconName, bool grayed )
{
   if (iconName == "")
      return null;

   // Lookup the icon from the icon bitmap pool, which is the pool 
   // of icon which bitmap has already been painted in memory.
   GKeyBag<GIcon>& pool = (grayed ? IconPoolDisabled : IconPoolNormal);
   GIcon* icon = pool.get(iconName);
   if (icon != null)
      return icon;

   // The icon is not in the cache, so we must create it
   // and put it into the icon cache now. This also draws the 
   // icon pixels on its contained memory bitmaps.
   GProgram& prg = GProgram::GetProgram();
   GResourceTable& res = prg.getResourceTable();
   GIconResource* ires = res.getIconResource(iconName);
   if (ires == null)
      return null;

   icon = new GIcon(*ires, grayed);
   pool.put(iconName, icon);
   return icon;
}

int GIcon::getSystemHandle () const
{
   if (b == null)
   {
      POINTERINFO pi;
      memset(&pi, 0, sizeof(pi));
      pi.fPointer = false; // Create an icon, not a pointer.
      pi.hbmColor = a->bmImage->getHBM();
      pi.hbmPointer = a->bmMask->getHBM(); // TODO: Documentation says that this bitmap should be twice the height as that of "hbmColor". This is not the case by now. Does this work anyway?
      b = new B();
      b->wasLoadedFromFile = false;
      b->hicon = ::WinCreatePointerIndirect(HWND_DESKTOP, &pi);
   }
   return int(b->hicon);
}
