/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GWallPaperParams.h"
#include "glib/gui/GStatusbar.h"
#include "glib/gui/GCommandPalette.h"
#include "glib/gui/GFocusManager.h"
#include "glib/gui/GMenu.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GToggleButton.h"
#include "glib/gui/GRadioButton.h"
#include "glib/gui/GMultiLineEditor.h"
#include "glib/gui/GGroupBox.h"
#include "glib/gui/GSpinner.h"
#include "glib/gui/GDropList.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/GStaticIcon.h"
#include "glib/gui/GProgressBar.h"
#include "glib/gui/GTabbedPanel.h"
#include "glib/gui/GLinePlotter.h"
#include "glib/gui/GTreeView.h"
#include "glib/gui/GAddComponentException.h"
#include "glib/gui/GNoSuchComponentException.h"
#include "glib/gui/border/GEmptyBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/layout/GGridLayout.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/util/GMath.h"
#include "glib/util/GTokenizer.h"
#include "glib/resource/GRcException.h"
#include "glib/primitives/GLong.h"
#include "glib/primitives/GULong.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/GProgram.h"

const int GDialogPanel::COMPONENT_DISTANCE_Y = 2;
const int GDialogPanel::COMPONENT_DISTANCE_X = 2;
const int GDialogPanel::COMPONENT_EXTRAFONTH = 6;

GDialogPanel::HotKeyItem::HotKeyItem ( char chr, int key )
                         :comp(null),
                          hotChar(chr),
                          hotKey(key)
{
}

GDialogPanel::HotKeyItem::~HotKeyItem ()
{
}

GDialogPanel::GDialogPanel ( GDialogFrame& ownerFrame,
                             const GString& name,
                             const GString& constraints,
                             GDialogMessageHandler* msgProc,
                             const GDialogResource& params )
             :GWindow(name,
                      constraints,
                      &ownerFrame,
                      &ownerFrame,
                      WS_VISIBLE | WS_CLIPCHILDREN,
                      WS2_DEFAULTPAINT | WS2_OS2Y),
              focusRoot(true),
              ownerDlg(ownerFrame),
              ownerDlgPane(null)
{
   ownerFrame.topLevelPanel = this;
   init(params, msgProc);
}

GDialogPanel::GDialogPanel ( GDialogPanel& ownerPanel,
                             GWindow& parentWin,
                             const GString& name,
                             const GString& constraints,
                             GDialogMessageHandler* msgProc,
                             const GDialogResource& params )
             :GWindow(name,
                      constraints,
                      &parentWin,
                      &ownerPanel,
                      WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN, 
                      WS2_OS2Y),
              focusRoot(true),
              ownerDlg(ownerPanel.ownerDlg),
              ownerDlgPane(&ownerPanel)
{
   init(params, msgProc);
}

void GDialogPanel::init ( const GDialogResource& params, GDialogMessageHandler* msgProc )
{
   setDialogMessageHandler(msgProc);
   GBorderLayout* layout = new GBorderLayout();
   setLayoutManager(layout, true);

   destructorIsCalled = false;
   cmdPalette = null;
   statusbar = null;
   defPushButton = null;
   escPushButton = null;
   icon = null;
   dlgWallPaper = null;
   hotKeys = null;
   clientArea = new GWindow("ClientArea", 
                            GBorderLayout::CENTER, 
                            this, 
                            this, 
                            WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN, 
                            WS2_AUTO_PAINT_BACKGROUND | WS2_OS2Y | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT);

   // Initially update <i>iDlgFontW</i> and <i>iDlgFontH</i>.
   GDimension fontDim = getStringDimension(GWindow::FontDimTemplateStr);
   iDlgFontH = fontDim.height + 3;
   iDlgFontW = fontDim.width / GWindow::FontDimTemplateStr.length();

   // OS2's WarpSans-font is so narrow that it needs some extra space 
   // per character.
   GString fontName = getFontNameSize();
   fontName.toUpperCase();
   if (fontName.indexOf("WARPSANS") >= 0)
      iDlgFontW += 1;

   // ---
   GResourceTable& res = getResourceTable();

   GAccelResource* accel = res.getAcceleratorResource(params.getAccelID());
   if (accel != null)
      setAccelTable(accel);

   GMenuResource* men = res.getMenuResource(params.getMenuID());
   if (men != null)
   {
      GProgram& prg = GProgram::GetProgram();
      setMenubar(*men, prg.isUseFancyMenubars());
   }

   GToolbarResource* tb = res.getToolbarResource(params.getToolbarID());
   if (tb != null)
      setToolbar(*tb);

   GIconResource* icon = res.getIconResource(params.getIconID());
   if (icon != null)
      setIcon(*icon);

   setText(params.getTextID());
   if (ownerDlgPane == null)
      ownerDlg.setText(params.getTextID());

   // Activate the identified layout manager (if any).
   GString layoutID = params.getLayoutManagerID();
   if (layoutID.equalsIgnoreCase("") ||
         layoutID.equalsIgnoreCase("none") ||
         layoutID.equalsIgnoreCase("null"))
   {
      // The layout of the Components are to be managed by this class it
      // self, using the size and position of Components as are given by
      // user program.
      clientArea->setLayoutManager(null, true);
   }
   else
   if (layoutID.equalsIgnoreCase("border"))
      clientArea->setLayoutManager(new GBorderLayout(), true);
   else
   if (layoutID.equalsIgnoreCase ("grid1"))
      clientArea->setLayoutManager(new GGridLayout(0, 1), true); // Grid, width 1 column of N rows
   else
   if (layoutID.equalsIgnoreCase ("grid2"))
      clientArea->setLayoutManager(new GGridLayout(0, 2), true); // Grid, width 2 column of N rows
   else
   if (layoutID.equalsIgnoreCase ("grid3"))
      clientArea->setLayoutManager(new GGridLayout(0, 3), true); // Grid, width 3 column of N rows
   else
   if (layoutID.equalsIgnoreCase ("grid4"))
      clientArea->setLayoutManager(new GGridLayout(0, 4), true); // Grid, width 4 column of N rows
   else
   if (layoutID.equalsIgnoreCase ("grid5"))
      clientArea->setLayoutManager(new GGridLayout(0, 5), true); // Grid, width 5 column of N rows
   else
      clientArea->setLayoutManager(null, true); // TODO: Unknown layout manager, so set to default

   // Set the initial size of the dialog.
   const GComponentSize& dlgSize = params.getSize();
   setDialogSize(dlgSize);

   // Add all Components of the dialog resource into this dialog panel.
   const GArray<GComponentParams>& ctrls = params.getComponents();
   int num = ctrls.getCount();
   for (int i=0; i<num; i++)
   {
      const GComponentParams& cp = ctrls[i];
      try {
         addComponent(cp);
      } catch (std::exception& e) {
         GString msg("Error adding Component #%d in dialog '%s'. Component ID is '%s'. Original message is: ", GVArgs(i+1).add(getName()).add(cp.getIDString()).add(e.what()));
         gthrow_(GAddComponentException(msg));
      }
   }
}

GDialogPanel::~GDialogPanel ()
{
   destructorIsCalled = true;
   setDialogMessageHandler(null);
   setLayoutManager(null, false);

   // Remove and destroy all components of ours.
   for (int i=getComponentCount()-1; i>=0; i--)
      removeComponent(i);

   // ---
   if (hotKeys != null)
      delete hotKeys;
   hotKeys = null;
   if (dlgWallPaper != null) 
      delete dlgWallPaper;
   dlgWallPaper = null;
   if (clientArea != null)
      delete clientArea;
   clientArea = null;
   if (cmdPalette != null)
      delete cmdPalette;
   cmdPalette = null;
   if (statusbar != null)
      delete statusbar;
   statusbar = null;
}

GDialogFrame& GDialogPanel::getOwnerFrame () 
{ 
   return ownerDlg; 
}

GDialogPanel* GDialogPanel::getOwnerPanel () 
{ 
   return ownerDlgPane; 
}

int GDialogPanel::getFontWidth () const 
{ 
   return iDlgFontW; 
}

int GDialogPanel::getFontHeight () const 
{ 
   return iDlgFontH + COMPONENT_EXTRAFONTH; 
}

void GDialogPanel::ensureCommandPalette ()
{
   if (cmdPalette == null)
      cmdPalette = new GCommandPalette(*this, "CmdPalette", GBorderLayout::NORTH);
}

void GDialogPanel::setIcon ( GIconResource& icon )
{
   this->icon = &icon;
   /* TODO: What about icons on dialog frames?
   if (ownerDlgPane == null) // If this is the Top Level Panel
      ownerDlg.setIcon(icon);
      */
}

void GDialogPanel::setIcon ( const GString& iconName )
{
   GResourceTable& res = getResourceTable();
   GIconResource* icon = res.getIconResource(iconName);
   if (icon != null)
      setIcon(*icon);
}

void GDialogPanel::setStatbarText ( const GString& newText )
{
   if (newText == "")
   {
      if (statusbar != null)
         statusbar->setText(newText);
   }
   else
   {
      if (statusbar == null)
      {
         statusbar = new GStatusbar("Statusbar", GBorderLayout::SOUTH, *this);
         layout();
      }
      statusbar->setText(newText);
   }
}

GWallPaperParams& GDialogPanel::getWallPaperParams ()
{
   if (dlgWallPaper == null)
      dlgWallPaper = new GWallPaperParams("", false, GWallPaperParams::DWP_TILEIMAGE);
   return *dlgWallPaper;
}

void GDialogPanel::addAvailableHotKey ( char chr, int key )
{
   GCommandPalette* cp = getCommandPalette();
   if (cp != null)
   {
      GMenu* mb = cp->getMenubar();
      if (mb != null)
      {
         int num = mb->getItemCount();
         for (int i=0; i<num; i++)
         {
            GMenuPopupItem& m = mb->getIndexedItem(i);
            char menuMnemonic = m.getMnemonicChar();
            if (menuMnemonic == chr)
               return;
         }
      }
   }

   GString keystr(1);
   keystr += chr;
   HotKeyItem* hk = new HotKeyItem(chr, key);
   hotKeys->put(keystr, hk);
}

void GDialogPanel::initBagOfAvailableHotKeys ()
{
   if (hotKeys == null)
      hotKeys = new GKeyBag<HotKeyItem>(40);
   else
      hotKeys->removeAll();

   addAvailableHotKey('0', GKey::KEY_ALT_0);
   addAvailableHotKey('1', GKey::KEY_ALT_1);
   addAvailableHotKey('2', GKey::KEY_ALT_2);
   addAvailableHotKey('3', GKey::KEY_ALT_3);
   addAvailableHotKey('4', GKey::KEY_ALT_4);
   addAvailableHotKey('5', GKey::KEY_ALT_5);
   addAvailableHotKey('6', GKey::KEY_ALT_6);
   addAvailableHotKey('7', GKey::KEY_ALT_7);
   addAvailableHotKey('8', GKey::KEY_ALT_8);
   addAvailableHotKey('9', GKey::KEY_ALT_9);
   addAvailableHotKey('A', GKey::KEY_ALT_A);
   addAvailableHotKey('B', GKey::KEY_ALT_B);
   addAvailableHotKey('C', GKey::KEY_ALT_C);
   addAvailableHotKey('D', GKey::KEY_ALT_D);
   addAvailableHotKey('E', GKey::KEY_ALT_E);
   addAvailableHotKey('F', GKey::KEY_ALT_F);
   addAvailableHotKey('G', GKey::KEY_ALT_G);
   addAvailableHotKey('H', GKey::KEY_ALT_H);
   addAvailableHotKey('I', GKey::KEY_ALT_I);
   addAvailableHotKey('J', GKey::KEY_ALT_J);
   addAvailableHotKey('K', GKey::KEY_ALT_K);
   addAvailableHotKey('L', GKey::KEY_ALT_L);
   addAvailableHotKey('M', GKey::KEY_ALT_M);
   addAvailableHotKey('N', GKey::KEY_ALT_N);
   addAvailableHotKey('O', GKey::KEY_ALT_O);
   addAvailableHotKey('P', GKey::KEY_ALT_P);
   addAvailableHotKey('Q', GKey::KEY_ALT_Q);
   addAvailableHotKey('R', GKey::KEY_ALT_R);
   addAvailableHotKey('S', GKey::KEY_ALT_S);
   addAvailableHotKey('T', GKey::KEY_ALT_T);
   addAvailableHotKey('U', GKey::KEY_ALT_U);
   addAvailableHotKey('V', GKey::KEY_ALT_V);
   addAvailableHotKey('W', GKey::KEY_ALT_W);
   addAvailableHotKey('X', GKey::KEY_ALT_X);
   addAvailableHotKey('Y', GKey::KEY_ALT_Y);
   addAvailableHotKey('Z', GKey::KEY_ALT_Z);
}

void GDialogPanel::setFontNameSize ( const GString& fontNameSize )
{
   GWindow::setFontNameSize(fontNameSize);
   if (clientArea != null)
      clientArea->setFontNameSize(fontNameSize);
}

void GDialogPanel::setMenubar ( GMenuResource& newMenu, bool useFancy, GResourceTable* res, bool topMnemonicsOn )
{
   ensureCommandPalette();
   cmdPalette->setMenubar(newMenu, useFancy, res, topMnemonicsOn);
   layout();
}

void GDialogPanel::setToolbar ( GToolbarResource& tb, GResourceTable* res )
{
   ensureCommandPalette();
   cmdPalette->setToolbar(tb, res);
   layout();
}

GString GDialogPanel::getComponentValue ( const GString& id, const GString& defVal ) const
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return defVal;
   GDialogPanel* self = const_cast<GDialogPanel*>(this);
   const GWindow& ctrl = self->getComponentByIndex(idx);
   return ctrl.queryValue();
}

int GDialogPanel::getComponentIntValue ( const GString& id, int defaultVal ) const
{
   try {
      GString str = getComponentValue(id, GString::Empty);
      if (str == GString::Empty)
         return defaultVal;
      int ret = GInteger::ParseInt(str);
      return ret;
   } catch (GException& /*e*/) {
      return defaultVal;
   }
}

longlong GDialogPanel::getComponentLongValue ( const GString& id, longlong defaultVal ) const
{
   try {
      GString str = getComponentValue(id, GString::Empty);
      if (str == GString::Empty)
         return defaultVal;
      longlong ret = GLong::ParseLong(str);
      return ret;
   } catch (GException& /*e*/) {
      return defaultVal;
   }
}

bool GDialogPanel::getComponentBoolValue ( const GString& id, bool defVal ) const
{
   GString str = getComponentValue(id);
   if (str == "1" || str.equalsIgnoreCase("true") || str.equalsIgnoreCase("on") || str.equalsIgnoreCase("yes"))
      return true;
   else
   if (str == "0" || str.equalsIgnoreCase("false") || str.equalsIgnoreCase("off") || str.equalsIgnoreCase("no"))
      return false;
   else
      return defVal;
}

void GDialogPanel::queryProfile ( const GString& sectName )
{
   queryProfileWallPaper(sectName, "Wall");
   GWindow::queryProfile(sectName);
}

void GDialogPanel::writeProfile ( const GString& sectName, bool force )
{
   GWindow::writeProfile(sectName, force);
}

void GDialogPanel::setWallPaper ( const GWallPaperParams& wp )
{
   getWallPaperParams(); // Just to make sure that "dlgWallPaper" is not null
   *dlgWallPaper = wp;
   invalidateAll(false);
}

void GDialogPanel::queryProfileWallPaper ( const GString& section, const GString& item )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   GString str = ini.getString(section, item, "");
   if (str == "")
      return;

   GTokenizer strTok(str, ", ", "", false);
   GString fileName = strTok.getNextToken()->toString();
   GString on = strTok.getNextToken()->toString();
   GString flag = strTok.getNextToken()->toString();
   if (fileName == "" || on == "" || flag == "")
      return;

   bool bOn;
   if (on.equalsIgnoreCase("on"))
      bOn = true;
   else
   if (on.equalsIgnoreCase("off"))
      bOn = false;
   else
      return;

   GWallPaperParams::DWP_FLAG iFlag;
   if (flag.equalsIgnoreCase("tile"))
      iFlag = GWallPaperParams::DWP_TILEIMAGE;
   else
   if (flag.equalsIgnoreCase("stretch"))
      iFlag = GWallPaperParams::DWP_STRETCHIMAGE;
   else
   if (flag.equalsIgnoreCase("center"))
      iFlag = GWallPaperParams::DWP_CENTERIMAGE;
   else
      return;

   GWallPaperParams& wp = getWallPaperParams();
   wp = GWallPaperParams(fileName, bOn, iFlag);
}

void GDialogPanel::setTitleText ( const GString& text )
{
   if (ownerDlgPane == null)
      ownerDlg.setText(text);
}

GWindow& GDialogPanel::addComponent ( const GComponentParams& params, 
                                      const GString& constraints )
{
   GString compID = params.getIDString();
   const GString& compTextID = params.getTextID();
   const GString& compConstraints = params.getConstraints();
   const GComponentPos& compPos = params.getPos();
   const GComponentSize& compSize = params.getSize();

   // Test if the ID of the Component is already in use by another
   // component in this dialog box. In that case, throw an exception.
   if (compID != "" && clientArea->childrenByName.containsKey(compID))
      gthrow_(GAddComponentException(GString("Two or more Components share the same ID: '%s'", GVArgs(compID))));

   // ---
   GString cname = params.getTypeName();
   cname.toLowerCase();
   GWindow* comp;
   double defaultWidth;
   double defaultHeight;
   if (cname == "statictext")
   {
      defaultWidth = 23.0;
      defaultHeight = 1.0;

      bool leader = params.extraArgs.contains("LEADER");
      bool hotKey = !params.extraArgs.contains("NOHOTKEY");
      bool isleft = params.extraArgs.contains("LEFT");
      bool isright = params.extraArgs.contains("RIGHT");
      bool istop = params.extraArgs.contains("TOP");
      bool isbottom = params.extraArgs.contains("BOTTOM");
      bool iscenter = params.extraArgs.contains("CENTER");
      bool wordWrap = params.extraArgs.contains("WORDWRAP");
      GStaticText::ADJUST adjust = isright ? GStaticText::RIGHT : (iscenter ? GStaticText::CENTER : (isleft ? GStaticText::LEFT : GStaticText::LEFT));
      GStaticText::VADJUST vadjust = istop ? GStaticText::TOP : (isbottom ? GStaticText::BOTTOM : GStaticText::VCENTER);
      GWindow::LeaderStyle style = GWindow::LeaderStyle_DEFAULT;

      comp = new GStaticText(compID, compConstraints, *clientArea, 0, 0,
                             adjust, leader, hotKey, style, wordWrap, vadjust);
   }
   else
   if (cname == "textentry")
   {
      defaultWidth = 33.0;
      defaultHeight = 1.0;

      int maxText = params.extraArgs.getInteger("MAXTEXT", int(compSize.width) + 1);
      bool noborder = params.extraArgs.contains("NOBORDER");
      bool password = params.extraArgs.contains("PASSWORD");
      bool readOnly = params.extraArgs.contains("READONLY");
      GTextEntry::ADJUST adjust;
      if (params.extraArgs.contains("RIGHT"))
         adjust = GTextEntry::RIGHT;
      else
      if (params.extraArgs.contains("CENTER"))
         adjust = GTextEntry::CENTER;
      else
         adjust = GTextEntry::LEFT;

      comp = new GTextEntry(compID, compConstraints, *clientArea, 0, 0,
                            maxText, adjust, !noborder, password, readOnly);
   }
   else
   if (cname == "pushbutton")
   {
      defaultWidth = 14.0;
      defaultHeight = 1.4;

      GString iconID = params.extraArgs.getString("ICON");
      bool isDefault = params.extraArgs.contains("DEFAULT");
      bool isEscape = params.extraArgs.contains("ESCAPE");
      bool noFocusPaint = params.extraArgs.contains("NOFOCUSPAINT");
      GPushButton::ICONPOS iconPos = GPushButton::IP_LEFT;
      bool noBoldBorder = params.extraArgs.contains("NOBOLDBORDER");
      bool useHotKey = !params.extraArgs.contains("NOHOTKEY");

      GPushButton* c;
      comp = c = new GPushButton(compID, compConstraints, *clientArea, 0, 0,
                                 iconID, isDefault, noFocusPaint, iconPos, 
                                 noBoldBorder, useHotKey);
      if (isDefault)
         setDefaultPushButton(c);
      if (isEscape)
         setEscapePushButton(c);
   }
   else
   if (cname == "togglebutton")
   {
      defaultWidth = 17.0;
      defaultHeight = 1.0;

      bool useHotKey = !params.extraArgs.contains("NOHOTKEY");
      bool is3State = params.extraArgs.contains("IS3STATE");
      bool noFocusPaint = params.extraArgs.contains("NOFOCUSPAINT");

      comp = new GToggleButton(compID, compConstraints, *clientArea, 0, 0,
                               is3State, noFocusPaint, useHotKey);
   }
   else
   if (cname == "radiobutton")
   {
      defaultWidth = 17.0;
      defaultHeight = 1.0;

      bool useHotKey = !params.extraArgs.contains("NOHOTKEY");
      bool noFocusPaint = params.extraArgs.contains("NOFOCUSPAINT");

      comp = new GRadioButton(compID, compConstraints, *clientArea, 0, 0, 
                              noFocusPaint, useHotKey);
   }
   else
   if (cname == "groupbox")
   {
      defaultWidth = 23.0;
      defaultHeight = 5.0;

      bool hotKey = !params.extraArgs.contains("NOHOTKEY");
      bool noLeft = params.extraArgs.contains("NOLEFT");
      bool noRight = params.extraArgs.contains("NORIGHT");
      bool noTop = params.extraArgs.contains("NOTOP");
      bool noBottom = params.extraArgs.contains("NOBOTTOM");

      comp = new GGroupBox(compID, compConstraints, *clientArea, 0, 0, 
                           hotKey, noLeft, noRight, noTop, noBottom);
   }
   else
   if (cname == "spinner")
   {
      defaultWidth = 12.0;
      defaultHeight = 1.0;

      int minValue = params.extraArgs.getInteger("MIN", 1);
      int maxValue = params.extraArgs.getInteger("MAX", 100);
      int stepValue = params.extraArgs.getInteger("STEP", 1);
      GSpinner::ADJUST adj;
      if (params.extraArgs.contains("CENTER"))
         adj = GSpinner::CENTER;
      else
      if (params.extraArgs.contains("LEFT"))
         adj = GSpinner::LEFT;
      else
         adj = GSpinner::RIGHT;
      GArray<GString> strings(maxValue - minValue + 1);
      if (params.extraArgs.contains(GInteger::ToString(minValue)))
      {
         for (int i=minValue; i<=maxValue; i++)
         {
            GString str = params.extraArgs.getString(GInteger::ToString(i), "");
            if (str == "")
            {
               // There wasn't given a String for each possible value, so
               // don't use any of the strings then!
               strings.removeAll();
               break;
            }
            strings.add(new GString(str));
         }
      }

      GSpinner* c;
      comp = c = new GSpinner(compID, compConstraints, *clientArea, 0, 0, 
                              adj, minValue, maxValue, stepValue, 
                              strings.getCount() > 0 ? &strings : null);

      // Make sure the initial value is within the legal range.
      if (compTextID == "")
      {
         c->setValue(minValue);
      }
      else
      try {
         int val = GInteger::ParseInt(compTextID);
         if (!c->valueIsLegal(val))
            val = minValue;
         c->setValue(val);
      } catch (GNumberFormatException& /*e*/) {
         c->setValue(minValue);
      }
   }
   else
   if (cname == "droplist")
   {
      defaultWidth = 25.0;
      defaultHeight = 1.0;

      GDropList* c;
      comp = c = new GDropList(compID, compConstraints, *clientArea, 0);

      // Add all items specified in the resource script.
      for (int i=1;; i++)
      {
         GString val = params.extraArgs.getString(GInteger::ToString(i), "");
         if (val == "")
            break;
         GString strval = GProgram::LoadText(val);
         GString userDataStr = params.extraArgs.getString("U" + GInteger::ToString(i));
         GObject* userData = null;
         if (userDataStr != "")
            userData = new GString(userDataStr);
         c->addItem(strval, GString::Empty, userData, userData == null ? false : true);
      }

      // Activate the default item.
      GString txt = params.getTextID();
      if (txt != "") try {
         int idx = GInteger::ParseInt(txt);
         if (idx >= 1)
            c->setSelectedIndex(idx - 1);
      } catch (GNumberFormatException& /*e*/) {
         // The text is not a number. This is not critical, so just ignore.
      }
   }
   else
   if (cname == "listbox")
   {
      defaultWidth = 15.0;
      defaultHeight = 5.0;

      bool multisel = params.extraArgs.contains("MULTISEL");

      comp = new GListBox(compID, compConstraints, *clientArea, multisel, 0);
   }
   else
   if (cname == "multilineedit")
   {
      defaultWidth = 33.0;
      defaultHeight = 6.0;

      bool ignoreTab = true;
      bool wordWrap = params.extraArgs.contains("WORDWRAP");
      bool noborder = params.extraArgs.contains("NOBORDER");
      bool readOnly = params.extraArgs.contains("READONLY");
      int maxText = params.extraArgs.getInteger("MAXTEXT", 1024);
      GString fontn = params.extraArgs.getString("FONT", "");

      comp = new GMultiLineEditor(compID, compConstraints, *clientArea, 0,  0,
                                  maxText, wordWrap, !noborder, 
                                  readOnly, ignoreTab);

      if (fontn.equalsIgnoreCase("HELV"))
         comp->setFontNameSize(GWindow::SysDefFontNameSize);
      else
      if (fontn.equalsIgnoreCase("COURIER"))
         comp->setFontNameSize(GWindow::SysDefFixedFontNameSize);
      else
         comp->setFontNameSize(getFontNameSize());
   }
   else
   if (cname == "staticicon")
   {
      defaultWidth = 5.0;
      defaultHeight = 2.0;

      GString iconName = params.extraArgs.getString("ICON");

      comp = new GStaticIcon(compID, compConstraints, *clientArea, iconName, 0);
   }
   else
   if (cname == "progressbar")
   {
      defaultWidth = 28.0;
      defaultHeight = 1.0;

      int minValue = params.extraArgs.getInteger("MINVALUE", 0);
      int maxValue = params.extraArgs.getInteger("MAXVALUE", 100);
      bool vertical = params.extraArgs.contains("VERTICAL");

      GProgressBar* c;
      comp = c = new GProgressBar(compID, compConstraints, *clientArea, 0, 0, 
                                  minValue, maxValue, !vertical);

      if (compTextID != "")
      {
         try {
            int cur = GInteger::ParseInt(compTextID);
            c->setCurrentValue(cur);
         } catch (GNumberFormatException& /*e*/) {
         }
      }
   }
   else
   if (cname == "tabbedpanel")
   {
      defaultWidth = 32.0;
      defaultHeight = 8.0;

      GTabbedPanel::TabsPos tabsPos;
      if (params.extraArgs.contains("BOTTOM"))
         tabsPos = GTabbedPanel::BOTTOM;
      else
         tabsPos = GTabbedPanel::TOP;

      GTabbedPanel* c;
      comp = c = new GTabbedPanel(compID, compConstraints, *clientArea, 0, tabsPos);

      for (int i=1;; i++)
      {
         GString dlgIDParamName("TAB%d", GVArgs(i));
         GString dlgID = params.extraArgs.getString(dlgIDParamName, "");
         if (dlgID == "")
            break;

         GString tabIDParamName("ID%d", GVArgs(i));
         GString tabID = params.extraArgs.getString(tabIDParamName, "");

         GString tabTextParamName("TEXT%d", GVArgs(i));
         GString tabText = params.extraArgs.getString(tabTextParamName, "");

         GString tabHintParamName("HINT%d", GVArgs(i));
         GString tabHint = params.extraArgs.getString(tabHintParamName, "");

         // Add a new tab using the specified dialog resource as template.
         GDialogMessageHandler* msgProc = null; // Must be set programatically.
         c->addDialogTab(dlgID, msgProc, tabID, tabText, tabHint);
      }

      bool visibleTabs = !params.extraArgs.contains("INVISIBLETABS");
      c->setTabsVisible(visibleTabs);
   }
   else
   if (cname == "tree")
   {
      defaultWidth = 25.0;
      defaultHeight = 5.0;

      bool border = params.extraArgs.getBoolean("BORDER", true);
      bool lines = params.extraArgs.getBoolean("LINES", true);
      bool linesAtRoot = params.extraArgs.getBoolean("LINESATROOT", false);
      bool buttons = params.extraArgs.getBoolean("BUTTONS", true);
      bool showRoot = params.extraArgs.getBoolean("SHOWROOT", false);

      comp = new GTreeView(compID, // name
                           compConstraints, // constraints
                           *clientArea, // parentWin
                           border, // border
                           lines, // hasLines
                           linesAtRoot, // linesAtRoot
                           buttons, // hasButtons
                           showRoot, // showRoot
                           true); // autoGrabFocus
   }
   else
   if (cname == "lineplotter")
   {
      defaultWidth = 25.0;
      defaultHeight = 5.0;

      comp = new GLinePlotter(compID, compConstraints, *clientArea, 0);
   }
   else
   {
      gthrow_(GAddComponentException(GString("Unknown Component Type: '%s'", GVArgs(cname))));
   }

   if (ownerDlg.dlgDismissWasCalled)
   {
      delete comp;
      gthrow_(GAddComponentException(GString("Something went wrong when adding a Component: '%s'", GVArgs(cname))));
   }

   // Everything is OK, so add our new component to the bag.
   compID = comp->getName(); // In case original ID was empty.

   // Set the initial common component properties.
   double compWidth = (compSize.width == 0 ? defaultWidth : compSize.width);
   double compHeight = (compSize.height == 0 ? defaultHeight : compSize.height);
   setComponentSize(compID, compWidth, compHeight);
   comp->setText(GProgram::LoadText(compTextID, GResourceTable::LT_PREFER_NONE));
   setComponentPos(compID, compPos.x, compPos.y);
   comp->setOily(params.isOily());
   comp->setEnabled(!params.isDisabled());
   comp->setVisible(!params.isHidden());

   // Initially layout the component.
   comp->layout();

   // If this Dialog Panel has already received the DM_INITDIALOG message
   // (that is if it has been executed but not neccessarily visible) then
   // we must let the new Component initialize it self using the same
   // methods as when the Dialog Panel was executed.
   if (isInitialized())
   {
      bool dummyBreakSem = false;
      comp->prepareForExecution(null, &dummyBreakSem);
      clientArea->layout();
   }

   return *comp;
}

GWindow& GDialogPanel::addComponent ( const GString& script, 
                                      const GString& constraints )
{
   try {
      GComponentParams params(script);
      return addComponent(params, constraints);
   } catch (GRcException& e) {
      gthrow_(GAddComponentException(GString("Syntax error in component script. Message is: %s", GVArgs(e.toString()))));
   }
}

void GDialogPanel::removeComponent ( int index )
{
   GWindow* comp = &getComponentByIndex(index);

   if (comp == defPushButton)
      defPushButton = null;

   if (comp == escPushButton)
      escPushButton = null;

   // Remove the corresponding hot-key element, if any.
   if (hotKeys != null)
   {
      int num = hotKeys->getCount();
      for (int i=0; i<num; i++)
      {
         HotKeyItem& item = hotKeys->getIndexedItem(i);
         if (item.comp == comp)
         {
            hotKeys->removeIndexedItem(i);
            break;
         }
      }
   }

   // Remove the component from the bag.
   clientArea->removeChildWindow(*comp);
   delete comp;

   // ---
   GDialogFrame& ownerFrame = getOwnerFrame();
   if (!ownerFrame.isAlive())
      if (!destructorIsCalled) // Don't layout upon destruction!
         layout();
}

void GDialogPanel::removeComponent ( const GString& ctrlName )
{
   removeComponent(getComponentIndex(ctrlName));
}

int GDialogPanel::autoAssocCtrlToAHotKey ( GWindow& comp, const GString& text )
{
   if (comp.getOwnerDialogPanel() != this)
      return -1;

   // Create the IKeyBag of all possible Hot Keys that can be associated
   // with a Component within the Dialog Panel, if it hasn't been created
   // alredady.
   if (hotKeys == null)
      initBagOfAvailableHotKeys();

   // In case the specified Component is already associated to a Hot Key
   // then unregister it before giving the same Component a new Hot Key.
   char oldChar = comp.getHotKeyChar();
   if (oldChar != 0)
   {
      char keystr[2] = { oldChar, 0 };
      HotKeyItem* hk = hotKeys->get(keystr);
      if (hk != null && hk->comp != null)
         hk->comp = null; // Mark this key to not be in use
   }

   // Try to find a free Hot Key and associate the Component to that free
   // Hot Key.
   int strLen = text.length();
   for (int pos=0; pos<strLen; pos++)
   {
      // Find out if this hotkey is already in use by another Component
      char ch = (char) toupper(text[pos]);
      char keystr[2] = { ch, 0 };
      HotKeyItem* hk = hotKeys->get(keystr);
      if (hk == null || hk->comp != null)
         continue;

      hk->comp = &comp;
      return pos;
   }

   return -1;
}

GComponentPos GDialogPanel::getComponentPos ( const GString& id )
{
   GWindow& comp = getComponentByID(id);
   GPoint pt = comp.getWindowPos();
   return convertToCharPosition(pt.x, pt.y);
}

void GDialogPanel::setComponentPos ( const GString& id, double xPos, double yPos )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;

   GWindow& comp = getComponentByIndex(idx);
   GPoint pt = convertToPixelPosition(xPos, yPos);
   if (isOS2Y())
   {
      // We must always convert to Window'ish Y-coordinates because 
      // dialog components are always positioned with such coordinates.
      GDimension compDim = comp.getWindowSize();
      GDimension dlgDim = clientArea->getWindowSize();
      pt.y = dlgDim.height - compDim.height - pt.y;
   }

   comp.setWindowPos(pt.x, pt.y);
}

void GDialogPanel::setComponentPos ( const GString& id, const GComponentPos& pt )
{
   setComponentPos(id, pt.x, pt.y);
}

void GDialogPanel::setComponentSize ( const GString& id, double width, double height )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;
   GWindow& comp = getComponentByIndex(idx);
   GDimension dim = convertToPixelDimension(width, height);
   comp.setWindowSize(dim.width, dim.height);
}

void GDialogPanel::setComponentSize ( const GString& id, const GComponentSize& newSize )
{
   setComponentSize(id, newSize.width, newSize.height);
}

GComponentSize GDialogPanel::getComponentSize ( const GString& id )
{
   GWindow& comp = getComponentByID(id);
   GDimension dim = comp.getWindowSize();
   return convertToCharDimension(dim.width, dim.height);
}

void GDialogPanel::setDefaultPushButton ( GPushButton* ctrl )
{
   if (ctrl == defPushButton)
      return;
   if (defPushButton != null)
      defPushButton->setDefault(false);
   defPushButton = ctrl;
   if (defPushButton != null)
      defPushButton->setDefault(true);
}

void GDialogPanel::setEscapePushButton ( GPushButton* ctrl )
{
   escPushButton = ctrl;
}

void GDialogPanel::setComponentVisible ( const GString& id, bool show )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;
   GWindow& ctrl = getComponentByIndex(idx);
   ctrl.setVisible(show); // This might cause recursive calls!
   if (!show)
      if (ownerDlg.currentFocus != null && ownerDlg.currentFocus->isDescendant(ctrl))
         activateNextEditableComponent(false);
}

GPoint GDialogPanel::convertToPixelPosition ( double xPos, double yPos )
{
   int pixX = int(iDlgFontW * (xPos - 1));
   int pixY = int((iDlgFontH + COMPONENT_EXTRAFONTH) * (yPos - 1));
   return GPoint(pixX, pixY);
}

GPoint GDialogPanel::convertToPixelPosition ( const GComponentPos& pos )
{
   return convertToPixelPosition(pos.x, pos.y);
}

GComponentPos GDialogPanel::convertToCharPosition ( int xpos, int ypos )
{
   double charX = convertToCharXPosition(xpos);
   double charY = convertToCharYPosition(ypos);
   return GComponentPos(charX, charY);
}

double GDialogPanel::convertToCharXPosition ( int xpos )
{
   if (iDlgFontW == 0)
      return 0;
   else
      return ((double) xpos / iDlgFontW) + 1;
}

double GDialogPanel::convertToCharYPosition ( int ypos )
{
   int h = iDlgFontH + COMPONENT_EXTRAFONTH;
   if (h == 0)
      return 0;
   else
      return ((double) ypos / h) + 1;
}

GDimension GDialogPanel::convertToPixelDimension ( const GComponentSize& dim )
{
   return convertToPixelDimension(dim.width, dim.height);
}

GDimension GDialogPanel::convertToPixelDimension ( double width, double height )
{
   int pixW = convertToPixelWidth(width);
   int pixH = convertToPixelHeight(height);
   return GDimension(pixW, pixH);
}

int GDialogPanel::convertToPixelWidth ( double width )
{
   return (int) ((iDlgFontW * width) - COMPONENT_DISTANCE_X);
}

int GDialogPanel::convertToPixelHeight ( double height )
{
   return (int) (((iDlgFontH + COMPONENT_EXTRAFONTH) * height) - COMPONENT_DISTANCE_Y);
}

GComponentSize GDialogPanel::convertToCharDimension ( double width, double height )
{
   int w = iDlgFontW;
   int h = iDlgFontH + COMPONENT_EXTRAFONTH;
   double charW = (w == 0 ? 0 : (width + COMPONENT_DISTANCE_X) / w);
   double charH = (h == 0 ? 0 : (height + COMPONENT_DISTANCE_Y) / h);
   return GComponentSize(charW, charH);
}

void GDialogPanel::setDialogSize ( double width, double height )
{
   if (ownerDlgPane == null)
   {
      ownerDlg.setDialogSize(width, height);
   }
   else
   {
      GDimension dm = convertToPixelDimension(width, height);
      setWindowSize(dm.width, dm.height);
   }
}

void GDialogPanel::setDialogSize ( const GComponentSize& newSize ) 
{ 
   setDialogSize(newSize.width, newSize.height); 
}

GComponentSize GDialogPanel::getDialogSize ()
{
   if (!ownerDlg.isIn_DoExecute || (isIn_DmInitDialog && ownerDlgPane == null))
   {
      return ownerDlg.size;
   }
   else
   {
      GDimension dim = getWindowSize();
      return convertToCharDimension(dim.width, dim.height);
   }
}

void GDialogPanel::grabFocus ( bool force )
{
   if (recentFocus != null)
   {
      recentFocus->grabFocus(force);
      return;
   }

   // Find the first activeable Component within the Dialog Panel and
   // traverse the focus request to that Component.
   int num = getComponentCount();
   for (int i=0; i<num; i++)
   {
      try {
         GWindow& ctrl = getComponentByIndex(i);
         if (ctrl.isCtrlActivable())
         {
            ctrl.grabFocus(force);
            return;
         }
      } catch (GNoSuchComponentException& /*e*/) {
         continue;
      }
   }

   GWindow::grabFocus(force);
}

GWindow* GDialogPanel::getFocusComponent ()
{
   if (ownerDlg.currentFocus == null || !ownerDlg.currentFocus->isDescendant(*this))
      return null;

   // Find the Component of ours that contains the current Component.
   GDialogPanel* pane = ownerDlg.currentFocus->getOwnerDialogPanel();
   while (pane != null && pane->ownerDlgPane != this)
      pane = pane->ownerDlgPane;
   return pane;
}

GString GDialogPanel::getFocusComponentID ( const GString& defaultID )
{
   GWindow* win = getFocusComponent();
   if (win == null)
      return defaultID;
   if (!ownerDlg.currentFocus->isDescendant(*this))
      return defaultID;
   return win->getName();
}

GString GDialogPanel::getDefaultPushButtonID ()
{
   if (defPushButton == null)
      return "";
   else
      return defPushButton->getName();
}

GPushButton* GDialogPanel::getDefaultPushButton ( bool searchUpwards )
{
   if (defPushButton != null)
      return defPushButton;
   else
   if (searchUpwards && ownerDlgPane != null)
      return ownerDlgPane->getDefaultPushButton(searchUpwards);
   else
      return null;
}

GPushButton* GDialogPanel::getEscapePushButton ( bool searchUpwards )
{
   if (escPushButton != null)
      return escPushButton;
   else
   if (searchUpwards && ownerDlgPane != null)
      return ownerDlgPane->getEscapePushButton(searchUpwards);
   else
      return null;
}

GString GDialogPanel::getEscapePushButtonID ()
{
   if (escPushButton == null)
      return "";
   else
      return escPushButton->getName();
}

bool GDialogPanel::isFocusCycleRoot () const
{
   return ownerDlgPane == null ? true : focusRoot;
}

int GDialogPanel::getComponentCount () const
{
   return clientArea->getChildWindowCount();
}

bool GDialogPanel::isContainingComponent ( const GString& id ) const
{
   return clientArea->childrenByName.containsKey(id);
}

GWindow& GDialogPanel::getComponentByIndex ( int index )
{
   return clientArea->getChildWindow(index);
}

GWindow& GDialogPanel::getComponentByID ( const GString& id )
{
   return clientArea->getChildWindow(id);
}

GString GDialogPanel::getComponentID ( int index ) const
{
   GDialogPanel* self = (GDialogPanel*) this; // Throw away const.
   GWindow& ctrl = self->getComponentByIndex(index);
   return ctrl.getName();
}

bool GDialogPanel::isComponentDisabled ( const GString& id ) const
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
   {
      // Component doesn't exist.
      // This is probably (?) to be treated as if it is disabled.
      return true; 
   }

   GDialogPanel* self = const_cast<GDialogPanel*>(this);
   const GWindow& ctrl = self->getComponentByIndex(idx);
   return !ctrl.isEnabled();
}

bool GDialogPanel::isComponentEnabled ( const GString& id ) const
{ 
   return !isComponentDisabled(id); 
}

bool GDialogPanel::isComponentEmpty ( const GString& id ) const
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return false;

   GDialogPanel* self = const_cast<GDialogPanel*>(this);
   const GWindow& ctrl = self->getComponentByIndex(idx);
   return ctrl.isEmpty();
}

void GDialogPanel::setComponentEnabled ( const GString& id, bool enable )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;
   GWindow& ctrl = getComponentByIndex(idx);
   ctrl.setEnabled(enable);
   if (!enable)
      if (ownerDlg.currentFocus != null && ownerDlg.currentFocus->isDescendant(ctrl))
         activateNextEditableComponent(false);
}

void GDialogPanel::setComponentDisabled ( const GString& id, bool flag ) 
{ 
   setComponentEnabled(id, !flag); 
}

void GDialogPanel::clearComponentValue ( const GString& id, bool notify )
{
   setComponentValue(id, GString::Empty, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, const GString& newVal, bool notify )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;
   GWindow& ctrl = getComponentByIndex(idx);
   ctrl.changeValue(newVal, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, const char* newVal, bool notify )
{
   GString str(newVal);
   setComponentValue(id, str, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, int newVal, bool notify )
{
   GString str = GInteger::ToString(newVal);
   setComponentValue(id, str, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, longlong newVal, bool notify )
{
   GString str = GLong::ToString(newVal);
   setComponentValue(id, str, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, unsigned long newVal, bool notify )
{
   GString str = GULong::ToString(newVal);
   setComponentValue(id, str, notify);
}

void GDialogPanel::setComponentValue ( const GString& id, bool newVal, bool notify )
{
   GString str = GInteger::ToString(newVal ? 1 : 0);
   setComponentValue(id, str, notify);
}

void GDialogPanel::setComponentText ( const GString& id, const GString& newText )
{
   int idx = getComponentIndex(id);
   if (idx <= -1)
      return;
   GWindow& ctrl = getComponentByIndex(idx);
   ctrl.setText(newText);
}

const GString& GDialogPanel::getDismissArgument ()
{
   return ownerDlg.getDismissArgument();
}

bool GDialogPanel::activateNextEditableComponent ( bool skipOily )
{
   GWindow* c = ownerDlg.currentFocus;
   GWindow* start = (c == null ? this : c);
   ownerDlg.focusMngr->focusNextComponent(start, skipOily);
   return c != ownerDlg.currentFocus;
}

bool GDialogPanel::activatePrevEditableComponent ( bool skipOily )
{
   GWindow* c = ownerDlg.currentFocus;
   if (ownerDlg.focusMngr != null)
      ownerDlg.focusMngr->focusPreviousComponent(ownerDlg.currentFocus, skipOily);
   return c != ownerDlg.currentFocus;
}

bool GDialogPanel::onKeyDown ( const GKeyMessage& key )
{

   GKey::KeyCode code = key.getCode();
   switch (code)
   {
      case GKey::KEY_TABULATOR:
         activateNextEditableComponent(false);
         return true;

      case GKey::KEY_SHIFT_TABULATOR:
         activatePrevEditableComponent(false);
         return true;

      case GKey::KEY_DOWN:
      case GKey::KEY_RIGHT:
         activateNextEditableComponent(true);
         return true;

      case GKey::KEY_UP:
      case GKey::KEY_LEFT:
         activatePrevEditableComponent(true);
         return true;

      case GKey::KEY_ESCAPE:
         if (ownerDlg.onClose())
            return true;
         return GWindow::onKeyDown(key);

      case GKey::KEY_ENTER: 
      {
         GPushButton* defbutt = getDefaultPushButton(true);
         if (defbutt != null && defbutt->getOwnerDialogPanel() != null)
         {
            // Emulate that the default push-button has been pressed
            defbutt->doClick();
            return true;
         }

         // By default, handle <enter> as almost <tab>, except oily
         // Controls should be skipped.
         activateNextEditableComponent(true);
         return true; 
      }

      default:
      {
         // Test if the key is any of the possible Hot Keys. If so then
         // call <i>hotKeyAction()</i> on the associated Component.
         if (hotKeys == null)
            return GWindow::onKeyDown(key);
         char ch = '\0';
         if (code >= GKey::KEY_ALT_A && code <= GKey::KEY_ALT_Z)
            ch = char(code - GKey::KEY_ALT_A + 'A');
         else
         if (code >= GKey::KEY_ALT_0 && code <= GKey::KEY_ALT_9)
            ch = char(code - GKey::KEY_ALT_0 + '0');
         if (ch == '\0')
            return GWindow::onKeyDown(key);
         char chstr[2] = { ch, 0 };
         HotKeyItem* hk = hotKeys->get(chstr);
         if (hk == null || hk->comp == null)
            return GWindow::onKeyDown(key);
         if (hk->comp->isVisible() && hk->comp->isEnabled())
            hk->comp->hotKeyAction();
         return true;
      }
   }
}

bool GDialogPanel::onTimer ( const GString& timerID, GObject* userData )
{
   if (GWindow::onTimer(timerID, userData))
      return true;
   GDialogMessage msg(*this, GDialogMessageHandler::GM_TIMER, timerID, userData);
   return onDialogMessage(msg);
}

bool GDialogPanel::isAlive () const
{
   return ownerDlg.isAlive();
}

GWindow& GDialogPanel::getClientArea () 
{ 
   return *clientArea; 
}

GCommandPalette* GDialogPanel::getCommandPalette () 
{ 
   return cmdPalette; 
}

bool GDialogPanel::executeAbstractCommand ( GAbstractCommand* cmd )
{
   if (cmd == null)
      return false;

   const GString& cmdID = cmd->getIDString();
   if (sendDialogMessage(GDialogMessageHandler::GM_COMMAND, cmdID))
      return true; // Yes, the user code did handle the message

   return GWindow::executeAbstractCommand(cmd);
}
