/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DIALOGFRAME
#define __GLIB_DIALOGFRAME

#include "glib/gui/GFrameWindow.h"
#include "glib/gui/GComponentPos.h"
#include "glib/gui/GComponentSize.h"

/**
 * Main class to provide dialog boxes.
 *
 * Most programs shouldn't create new instances of this class directly.
 * They are probably better using the method
 * {@link GProgram#executeDialog} for modal dialogs or
 * {@link GProgram#makeDialog} for modeless or dynamic dialogs.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.21
 * @see     GProgram#executeDialog
 * @see     GProgram#makeDialog
 */
class GDialogFrame : public GFrameWindow
{
   friend class GWindow;
   friend class GDialogPanel;
   friend class ModalMessageLoopFinaliser;

   public:

      /** The default font for dialog box windows. */
      static const GString DefaultFont;
      static const GString DefaultMonoFont;

   private:

      /**
       * The window that was actually specified by the caller to be
       * owner window of the dialog. This is likely to be a window that
       * is not the same as the owner window from the system point of view,
       * because (at least OS/2) can possibly decide to use another window
       * as the physical owner, for some technical reasons.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.03
       */
      class GWindow* logicalOwnerWin;

      /**
       * The top level dialog panel of the dialog box.
       */
      class GDialogPanel* topLevelPanel;

      /**
       * A pointer to the low level component window (within the Dialog
       * Box) that currently has the keyboard focus.
       */
      class GWindow* currentFocus;

      /**
       * A pointer to the low level component window (within the Dialog
       * Box) that was the component that used to have the focus before the
       * currently focused component.
       */
      class GWindow* recentFocus;

      /**
       * Variable used to control if focus is changed during execution
       * of focus listener code.
       */
      int focusGained_counter;

      /**
       * A copy of the string as has been specified with the most recent
       * call to {@link #dismiss}.
       */
      GString dismissArg;

      /** True if {@link #dismiss} has been called. */
      bool dlgDismissWasCalled;

      /** True if the dialog was active when it was dismissed. */
      bool wasActiveUponDismiss;

      /** True if the dialog is currently being executed as a modal dialog. */
      bool modal;

      /**
       * This flag will be true when the Dialog Box is about being executed.
       * This is all the time between DM_INITDIALOG and DM_DISMISSDIALOG.
       * Only when this flag is true the program defined message handler of any
       * of the child Dialog Panels should be called.
       */
      bool isIn_DoExecute;

      /**
       * Position of the dialog, given in characters.
       */
      GComponentPos pos;

      /**
       * Size of the dialog (excl. frame), given in characters.
       */
      GComponentSize size;

      /**
       * True if dialog has been dismissed. False if dialog is currently living.
       * That is between DM_INITDIALOG and DM_DISMISSDIALOG.
       */
      bool breakMsgLoop;

      /**
       * False until the dialog box has been dismissed.
       */
      bool hasBeenDismissed;

      /**
       * @see #focusMngr
       */
      bool autoDeleteFocusMngr;

      /**
       * A pointer to the focus manager of the Dialog Box. We will delete
       * this object automatically if <i>autoDeleteFocusMngr</i> is true.
       * @see #setFocusManager
       */
      class GFocusManager* focusMngr;

      /**
       * @see #setStayInvisible
       */
      bool stayInvisible;

   public:

      /**
       * Create a new and empty invisible dialog box window.
       *
       * Most programs should not instantiate dialog boxes directly
       * using this constructor. They should, however, use the method
       * {@link GProgram#executeDialog} or {@link GProgram#makeDialog}.
       *
       * @param   msgProc    A reference to an object of where to call the
       *                     method <i>handleDialogMessage</i> to handle
       *                     events on the dialog window. Can be null.
       * @param   params     The dialog resource parameters as read from a
       *                     dialog resource script. If this parameter is
       *                     null then the dialog will be initially empty
       *                     without any title text, position, size,
       *                     components, etc.
       * @see     #executeModal
       * @see     #executeModeless
       * @see     GProgram#executeDialog
       * @see     GProgram#makeDialog
       * @throws  GAddComponentException in case of any error constructing
       *                                 the new dialog box or panel window.
       */
      GDialogFrame ( GDialogMessageHandler* msgProc,
                     const class GDialogResource& params );

      virtual ~GDialogFrame ();

   private:

      /** Disable the copy constructor. */
      GDialogFrame ( const GDialogFrame& src ) {}

      /** Disable the assignment operator. */
      GDialogFrame& operator= ( const GDialogFrame& ) { return *this; }

   private:

      /**
       * A modal message loop and dispatcher method to be used
       * by <i>execute()</i> only.
       *
       * This methos will wait for, fetch and dispatch the next message from
       * the message queue of the current calling GUI thread. Dispatching a
       * message means that the message handler of the target component will
       * be called. The message loop will not continue (and thus, the UI will
       * freeze) until the target component has finished handling the message
       * and returned from its message handler method (or until the
       * message handler enters another modal message loop).
       */
      void modalMessageLoop ();

      /**
       * This method is called each time some of the contained descendant 
       * windows receives input focus. It will take care of keeping track 
       * of which component currently has the focus as well as sending 
       * {@link GDialogMessageHandler#GM_FOCUSSET} and 
       * {@link GDialogMessageHandler#GM_FOCUSLOSS} to every 
       * affected component(s) in a recursive manner.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.30
       */
      void trackContainedComponentFocusGain ( class GWindow& win );

      /**
       * Calculate physical screen position of this new window.
       */
      void adjustWindowPosOnScreen ( int xpos, int ypos );

      /**
       * Get the name of the component which is a child component of the 
       * specified owner dialog and which is a parent of (or the same as-) 
       * the specified child window.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.30
       */
      GString GetChildComponentNameFor ( class GDialogPanel& owner, 
                                         class GWindow* child );

   public:

      /**
       * Callback procedure of the IDD_GENERIC_DIALOG dialogbox, which is
       * used as a dialog template with this default message handler.
       *
       * We use the dialog template only in order to be able to use the
       * system defined dialog API's in a simple and standard way, while our
       * extended functionality is put "above" those low level API's.
       */
      static GWindowMessage::Answer EXPENTRY DefaultDlgProc ( HWND hDlg, 
                                                              GWindowMessage::IDType msg, 
                                                              GWindowMessage::Param1 mp1, 
                                                              GWindowMessage::Param2 mp2 );

      /**
       * Return true if and only if the dialog has not yet been dismissed.
       * True can be returned even if the dialog is not currently 
       * being executed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.24
       * @see     #isExecutingAsDialog
       */
      virtual bool isAlive () const;

      /**
       * Return true if the dialog is currently being executed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.03
       * @see     #isAlive
       */
      virtual bool isExecutingAsDialog () const;

      /**
       * Return true if and only if the dialog frame is currently 
       * being executed as a modal dialog box.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      bool isModal () const;

      /**
       * Get a reference to the argument that was most recently given
       * to {@link #dismiss}. This method should not be called before
       * {@link #dismiss} has been called to terminate the Dialog.
       * If you do, then we will return an empty string.
       */
      const GString& getDismissArgument ();

      /**
       * Get a pointer to the currently focused component window,
       * or null if there is no component of ours that currently has the
       * input focus.
       *
       * Please note that this method will return a legal pointer even
       * if the dialog frame is not currently the active window. In that
       * case the returned window is the window that was the last one to
       * have the keyboard focus before the dialog frame was deactivated.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.02
       */
      virtual GWindow* getFocusedWindow () const;

      /**
       * Set the focus manager of which to be used by the Dialog Box
       * and all of its sub-dialog panels. This methid is typically not to
       * be used by user code, because the default GLib implementation
       * sets a default focus manager automatically.
       *
       * @param fm         The new focus manager of which to activate.
       * @param autoDelete True if we shall delete the given focus manager
       *                   object when it is no longer needed, or else false.
       */
      void setFocusManager ( class GFocusManager* fm, bool autoDelete );

      /**
       * Enable or disable the close button of the frame window.
       */
      void setEnableCloseButton ( bool flag );

      /**
       * Set the size of the Dialog Box Panel. This method should be called with
       * great care. It is to be used on Dynamic Dialogs only, and it should be
       * called before the Dialog is executed. Normally it shouldn't be called
       * at all.
       *
       * @param   dlgWidth   New width of this dialog, in character cells.
       * @param   dlgHeight  New height of this dialog, in character cells.
       */
      void setDialogSize ( double dlgWidth, double dlgHeight );

      /**
       * We override this method just in order to get a crispy
       * dialog arrival, by initually and synchronosuly painting all 
       * parts of it when it is made visible.

       * @author  Leif Erik Larsen
       * @since   2004.10.06
       */
      virtual void setVisible ( bool flag = true );

      /**
       * Get the size of the Top Level Dialog Panel.
       * The returned size is measured in characters (not in pixels).
       *
       * @author  Leif Erik Larsen
       * @since   1999.07.28
       * @see     #setDialogSize
       */
      const GComponentSize& getDialogSize ();

      /**
       * Set the location of the dialog frame. The location is relative with
       * respect to the owner panel. This method should be used on the
       * dialog frame object before it is executed.
       *
       * @param  xpos  New X-position of the dialog, in character cells.
       * @param  ypos  New Y-position of the dialog, in character cells.
       */
      void setDialogPosition ( double xpos, double ypos );

      /**
       * By default all dialog boxes is automatically shown as soon 
       * as the top level dialog panel has been successfully initialized 
       * and set up for execution. This usually happens after the top
       * level dialog panel has received and handled the messages 
       * {@link GDialogMessageHandler#GM_INITDIALOG} and 
       * {@link GDialogMessageHandler#GM_QUERYPROFILE}.
       *
       * In some cases an application might want to prevent the dialog from 
       * being automatically shown. It such cases this method must be called 
       * with a true argument, preferably upon handling the 
       * {@link GDialogMessageHandler#GM_INITDIALOG} message. If this is 
       * done then the dialog is not made visible until the application 
       * code manually calls {@link GDialogFrame#setVisible} at a later time.
       *
       * An example of when this feature is needed is for the progress 
       * dialog used by Larsen Commander when realoading filnames into one 
       * of the file panels. For most directories the number of filenames 
       * are so small that the progress dialog will not make any sense 
       * at all, other than just disturbing the user with its flashing.
       * Thus, the progress dialog will not be visible until a certain 
       * amount of time has been used to load filename items.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.29
       */
      void setStayInvisible ( bool yes );

      /**
       * Get a reference to the Top Level Dialog Panel of the Dialog Box.
       */
      class GDialogPanel& getDialogPanel ();

      /**
       * Dismiss the Dialog Box, but don't destroy it.
       * This will cause DM_DISMISSDIALOG to be sent.
       * A dismissed dialog can be reexecuted at a later time.
       *
       * @param   arg  The object of which to be returned
       *               by <i>getDismissArg()</i>.
       * @see     GDialogPanel#dismiss
       */
      virtual void dismiss ( const GString& arg = GString::Empty );

      /**
       * Execute the Dialog Box.
       *
       * @param   ownerWin Pointer to which window is to be used as the
       *                   owner of the dialog box, or else null if it
       *                   should not have any owner window at all.
       * @param   modal    True of the Dialog Box is to be executed as a 
       *                   Modal Dialog Box, or else false to execute it as 
       *                   a Modeless Dialog Box. A Modal Dialog is a dialog 
       *                   which owner window can not be activated until 
       *                   the Dialog has been dismissed. If true is 
       *                   specified then we will not return from this
       *                   method until the dialog is dismissed. If false
       *                   is specified then we will return as soon as 
       *                   the dialog has been created.
       * @param   initArg  Object reference of which to give in the first 
       *                   Message Parameter upon DM_INITDIALOG. Can be null.
       * @throws  GExecuteDialogException if we fail to execute the dialog.
       *                                  We will throw such an exception also
       *                                  in case the dialog was dismissed by
       *                                  the program during handling of
       *                                  the <i>DM_INITDIALOG</i> message.
       * @see     #executeModal
       * @see     #executeModeless
       */
      void execute ( GWindow* ownerWin, bool modal, GObject* initArg = null );

      /**
       * Execute the Dialog as a Modal Dialog. A Modal Dialog is a dialog
       * which owner window can not be activated until the dialog has been
       * dismissed.
       *
       * Typically, programs are not using this method directly to execute
       * modal dialogs. They are usually
       * using {@link no.cba.ica.IGUIProgram#executeDialog} instead.
       *
       * This method will not return until the dialog has been dismissed.
       *
       * @param   ownerWin Pointer to which window is to be used as the
       *                   owner of the dialog box, or else null if it
       *                   should not have any owner window at all.
       * @param   initArg  Object reference of which to give in the first 
       *                   Message Parameter upon DM_INITDIALOG. Can be null.
       * @return  The argument that was given to the <i>dismiss()</i> method upon
       *          closing the dialog. This can be, and often is, a null value.
       * @see     #executeModeless
       * @throws  GExecuteDialogException if we fail to execute the dialog.
       */
      const GString& executeModal ( GWindow* ownerWin, GObject* initArg = null );

      /**
       * Execute the dialog as a modeless dialog. A modeless dialog is a dialog
       * which owner window can be activated while the dialog is still open.
       *
       * This method will return as soon as the dialog has been created and the
       * message <i>DM_INITDIALOG</i> has been handled.
       *
       * @param   ownerWin Pointer to which window is to be used as the
       *                   owner of the dialog box, or else null if it
       *                   should not have any owner window at all.
       * @param   initArg  Object reference of which to give in the first 
       *                   Message Parameter upon DM_INITDIALOG. Can be null.
       * @return  A reference to the top level dialog panel.
       * @see     #executeModal
       * @throws  GExecuteDialogException if we fail to execute the dialog.
       */
      class GDialogPanel& executeModeless ( GWindow* ownerWin, GObject* initArg = null );

   protected:

      virtual bool onClose ();
      virtual bool onUserMessage ( class GUserMessage& msg );

      /**
       * Need to override this method in order to make sure that
       * the system-dependent default message handler will never
       * receive the Escape-key, because we must have the full control
       * on how and when the dialog is dismissed.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.04
       */
      virtual bool onKeyDown ( const class GKeyMessage& key );
};

#endif
