/* --------------------------------------------------------------------------
 *
 * This file was part of the "More for C++" library.
 * Copyright (c) 1999-2003 by Thorsten Goertz.
 * The "More for C++" library is free software; you can
 * redistribute it and/or modify it under the terms of the license
 * that comes with http://www.morefor.org/.
 *
 * ------------------------------------------------------------------------ */

#ifndef MORE_POINTER_HPP
#define MORE_POINTER_HPP

#include "glib/exceptions/GNullPointerException.h"
#include "glib/exceptions/GIllegalCastException.h"
#include "glib/gc/gc.h"

//  The template class "p" will be used for declaring smart
//  pointers to objects of a specific type.

//  Only objects that have been created with the special "new" operator
//  (in header file "glib/gc/create.h" can be assigned to these pointers.

//  Several operators have been declared for handling the
//  "smart" pointers like ordinary pointers in C++.

template<typename Type> class p: public GC::Adapter
{
   public:

   p( Type* pType = 0 );
   p( const p<Type>& rpType );
   ~p( );
   const p<Type>& operator = ( Type* pType );
   const p<Type>& operator = ( const p<Type>& rpType );
   bool operator < ( const p<Type>& rpType ) const;
   bool operator == ( Type* pType ) const;
   bool operator != ( Type* pType ) const;
   Type* operator -> ( ) const;
   Type* in( ) const;
   Type* out( ) const;
   operator Type* ( ) const;

   private:
   Type** m_ppObject;
};

template<typename Type1, typename Type2>
p<Type1> /**/ cast /**/ ( const p<Type2>& pType2 )
{
   p<Type1> pResult;

   if (pType2 == null)
      gthrow_(GNullPointerException());

   pResult = dynamic_cast<Type1*>(pType2.in());
   if (pResult == null)
      gthrow_(GIllegalCastException());

   return pResult;
}

template<typename Type> inline p<Type>::p ( Type* pType )
{
  GObject* pFinalizable = ( pType == 0 ) ? 0 : dynamic_cast<GObject*>( pType );
  Adapter::InitPointer( ( void**& ) m_ppObject, pType, pFinalizable );
}

template<typename Type> inline p<Type>::p ( const p<Type>& rpType )
{
  Adapter::InitPointer( ( void**& ) m_ppObject, *rpType.m_ppObject, 0 );
}

template<typename Type> inline p<Type>::~p ()
{
  Adapter::ForgetObject( ( void**& ) m_ppObject );
}

template<typename Type> inline const p<Type>& p<Type>::operator = ( Type* pType )
{
  GObject* pFinalizable = (pType == 0) ? 0 : dynamic_cast<GObject*>(pType);
  Adapter::AssignObject((void**&) m_ppObject, pType, pFinalizable);
  return *this;
}

template<typename Type> inline const p<Type>& p<Type>::operator = ( const p<Type>& rpType )
{
  Adapter::AssignObject((void**&) m_ppObject, *rpType.m_ppObject, 0);
  return *this;
}

template<typename Type> inline bool p<Type>::operator < ( const p<Type>& rpType ) const
{
  return *m_ppObject < *rpType.m_ppObject;
}

template<typename Type> inline bool p<Type>::operator == ( Type* pType ) const
{
  return *m_ppObject == pType;
}

template<typename Type> inline bool p<Type>::operator != ( Type* pType ) const
{
  return *m_ppObject != pType;
}

template<typename Type> inline Type* p<Type>::operator -> ( ) const
{
  if (*m_ppObject == 0) 
     gthrow_(GNullPointerException(""));
  return *m_ppObject;
}

template<typename Type> inline Type* p<Type>::in( ) const
{
  return *m_ppObject;
}

template<typename Type> inline Type* p<Type>::out( ) const
{
  return *m_ppObject;
}

template<typename Type> inline p<Type>::operator Type* ( ) const
{
  return *m_ppObject;
}

#endif
