/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ENVIRONMENTVARS
#define __GLIB_ENVIRONMENTVARS

#include "glib/primitives/GString.h"
#include "glib/util/GKeyBag.h"

/**
 * This class is to represent the environment variables list of the
 * program.
 *
 * Usually just one instance of this class will be created
 * during the life of a program.
 *
 * @author  Leif Erik Larsen
 * @since   1998.12.12
 */
class GEnvironment : public GObject
{
   public:

      /**
       * Bag of case insensitive environment variables.
       */
      GKeyBag<GString> envBag;

   public:

      /**
       */
      GEnvironment ();
      virtual ~GEnvironment ();

   public:

      /**
       * Get a copy of the string value of the specified environment
       * variable. Specified invoronment variable name is not case
       * significant.
       *
       * If the specified key doesn't exist in the environment then we will
       * return the specified default value.
       */
      const char* getEnv ( const char* envName, const char* defaultVal = null ) const;

      const char* getEnv ( const GString& envName, const char* defaultVal = null ) const { return getEnv(envName.cstring(), defaultVal); }

      /**
       * Get a reference to the string value of the indexed environment variable.
       */
      const GString& getEnv ( int index ) const;

      /**
       * Set the content of the specified environment variable. The specified
       * string must be a "script" in the format "X=N", where X is the name
       * of the variable and N is the new value. X is not case significant.
       *
       * To remove an environment variable completely, just specufy "X=".
       *
       * We will parse the variable value dynamically so that any %Z% values
       * within Y will be expanded automatically. For instance, if the PATH
       * environment variable is already defined as "C:\;C:\OS2" and this
       * method is called with "PATH=%PATH%;C:\MYDIR" then the
       * new value of the PATH environment will be "C:\;C:\OS2;C:\MYDIR".
       */
      void setEnv ( const GString& envScript );

      /**
       * Get the current number of variables that are contained in the
       * environment.
       */
      int getVarCount () const;

      /**
       * Get a reference to the indexed environment variable name.
       */
      const GString& getKey ( int index ) const;

      /**
       * This procedure tries to locate an object (e.g. an executable) in
       * the path pointed to by an environment variable. This may be used if
       * the user only specified the executable's filename without a
       * extension and/or fully qualified path.
       *
       * If we return false then the caller can assume that we have not
       * tauched the <i>obj</i> parameter value.
       *
       * @param  searchPath  The semicolon-separated search path.
       * @param  objToFind   The object we are searching for in searchPath.
       * @param  obj         Buffer of where to return the fully quilified
       *                     path the found object.
       * @return True if the object was successfully located and returned in
       *         the <i>obj</i> parameter, or else false.
       */
      bool locateFileInEnv ( const GString& searchPath, 
                             const GString& objToFind, 
                             class GFile& obj ) const;

      /**
       * Get a string representation of the whole environment, in the
       * format "ENV1=Value1\nENV2=Value2\n\0\0".
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.13
       */
      GString makeString ( char eq = '=', char eol = '\n', char end1 = '\0', char end2 = '\0' ) const;

      /**
       * Return a linefeed separated list of the contained
       * environment variable names and values.
       *
       * @author  Leif Erik Larsen
       * @since   2003.01.13
       */
      virtual GString toString () const { return makeString('=', '\n'); }

      /**
       * Find the specified program file by searching through the PATH, first
       * for EXE-files, then for COM-files, CMD-files and BAT-files.
       *
       * If the program is not found then we will return an empty string.
       */
      GString which ( const GString& prog ) const;
};

#endif


