/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include "glib/GCmdLineParameters.h"
#include "glib/util/GTokenizer.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GDouble.h"
#include "glib/primitives/GInteger.h"
#include "glib/exceptions/GNumberFormatException.h"

GCmdLineParameters::GCmdLineParameters ( int argc, const char *args[] )
                   :params(10)
{
   GString paramsStr;
   for (int i=1; i<argc; i++)
   {
      GString* strp = new GString(args[i]);
      params.add(strp);
   }
}

GCmdLineParameters::GCmdLineParameters ( const GString& paramsStr )
                   :params(10)
{
   GTokenizer tokenizer(paramsStr, " \t\r\n", GString::Empty, true);
   for (;;)
   {
      const GToken* token = tokenizer.getNextToken();
      if (token->isEmpty())
         break; // No more tokens.
      GString tokenstr = token->toString();
      GString* strp = new GString(tokenstr);
      params.add(strp);
   }
}

GCmdLineParameters::~GCmdLineParameters ()
{
}

int GCmdLineParameters::getCount () const
{
   return params.getCount();
}

GString GCmdLineParameters::getString ( const GString& name, 
                                        const GString& defval ) const
{
   int num = getCount() - 1;
   for (int i=0; i<num; i++)
   {
      const GString& pname = getIndexedString(i);
      if (pname.equalsIgnoreCase(name))
      {
         const GString& pval = getIndexedString(++i);
         if (pval.length() > 0 && pval[0] == '-')
            return defval;
         else
            return pval;
      }
   }
   return defval;
}

int GCmdLineParameters::getInteger ( const GString& name, 
                                     int defval ) const
{
   GString str = getString(name, GString::Empty);
   if (str == GString::Empty)
      return defval;
   try {
      return GInteger::ParseInt(str);
   } catch (GNumberFormatException& /*e*/) {
      return defval;
   }
}

double GCmdLineParameters::getDouble ( const GString& name, 
                                       double defval ) const
{
   GString str = getString(name, GString::Empty);
   if (str == GString::Empty)
      return defval;
   try {
      return GDouble::ParseDouble(str);
   } catch (GNumberFormatException& /*e*/) {
      return defval;
   }
}

bool GCmdLineParameters::getBoolean ( const GString& name, 
                                      bool defval ) const
{
   GString str = getString(name, GString::Empty);
   return GBoolean::ParseBoolean(str, defval);
}

const GString& GCmdLineParameters::getIndexedString ( int index ) const
{
   return params.get(index);
}

bool GCmdLineParameters::isContainingParam ( const GString& name, 
                                             bool ignoreCase ) const
{
   int num = getCount();
   for (int i=0; i<num; i++)
   {
      const GString& p = getIndexedString(i);
      if (ignoreCase)
      {
         if (p.equalsIgnoreCase(name))
            return true;
      }
      else
      {
         if (p == name)
            return true;
      }
   }
   return false;
}

bool GCmdLineParameters::isContainingHelpRequest () const
{
   if (isContainingParam("/?") || isContainingParam("-?"))
      return true;
   else
      return false;
}

