/**********************************************************************
  Copyright (c) 2000-2002, Michael Dillon
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

       Redistributions of source code must retain the above copyright 
       notice, this list of conditions and the following disclaimer. 

       Redistributions in binary form must reproduce the above 
       copyright notice, this list of conditions and the following 
       disclaimer in the documentation and/or other materials provided 
       with the distribution. 

       Neither the name of Crystalline Realms, Midnight's Hour BBS,
       Elysium Software nor the names of their contributors may be 
       used to endorse or promote products derived from this software 
       without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
 *********************************************************************/
/*********************************************************************
 tprocess.cpp - All the functions required to handle and toss TIC
                files and their associated files.
 *********************************************************************/
#include "tinytic.h"
#include "parse.h"
#include "ticlog.h"
#include "filecopy.h"
#include "CDirect.h"
#include "tprocess.h"

#ifndef _MAX_PATH
# define _MAX_PATH 4096
#endif

string CurrentDirectory;

void SearchForTIC(void);
void ProcessTIC(const FileData *hFileData);
bool CheckArea(const char *area_name, string &Destination);
long FileSize(const char *src);
bool WriteEntry(const char *file, long size, string Desc[10], long DescPos, const char *area);

bool ProcessTICFiles(void)
{
        string Log;
        char szTmpBuffer[_MAX_PATH] = "\x0";

        // Remember this directory
        getcwd( szTmpBuffer, _MAX_PATH );
        CurrentDirectory = szTmpBuffer;

        // Attempt to move to the inbound directory
        if (chdir(Inbound.c_str()))
        {
                // The inbound directory doesn't exist.
                Log  = "Unable to find the directory \'";
                Log += Inbound.c_str();
                Log += "\' which was specified for the Inbound, aborting!";

                PutLog(Log.c_str(), FATAL);

                return (false);
        }

        SearchForTIC();

        return (true);
}

void SearchForTIC(void)
{
        CDirect cd("*.TIC");
        FileData hFileData;

        if (cd.Find() != true)
        {
                PutLog("Nothing to be done in the inbound.", INFO);
        }
        else
        {

                do {
                        hFileData = cd.GetInfo();

                        cout << "Processing \'" << hFileData.Name << "\' ("
                                << hFileData.Size << " bytes)..."
                                << endl;

                        // Process the TIC file, move the associated file
                        // to the destination directory
                        ProcessTIC(&hFileData);
                } while (cd.Find());
        }

        cout << "All files have been processed." << endl;

        return;
}

void ProcessTIC(const FileData *hFileData)
{
        // Area associated with the file
        string          Area;
        // Maximum of 10 lines of description
        string          Desc[30];
        // Actual File
        string          AssociatedFile;
        // Used for logging
        string          Log;
        // File stream for the file specified by file_src
        ifstream        FileIn(hFileData->Name, ios::in);
        // Temporary buffer
        char            szTmpBuffer[BUFFER_MAXLEN] = "\x0";
        // Description Position
        long            DescPos = 0;
        // File's size
        signed long     FSize = 0;
        // Temporary stream for verification
        fstream         AssocFileStream;
        // File's Destination
        string          Destination;
        // Used for internal error flagging
        bool            TICerror = false;
        // Secondary Buffer
        string          Buffer;
        // Iterator to use with the string types
        string::iterator        idx;

        if (!FileIn)
        {
                Log = "Unable to open \'";
                Log += hFileData->Name;
                Log += "\' for input.";
                PutLog(Log.c_str(), FATAL);

                return;
        }

        while (!FileIn.eof())
        {
                FileIn >> szTmpBuffer;
                if (stricmp(szTmpBuffer, "Area") == 0)
                {
                        // Get the area associated with the file
                        FileIn >> szTmpBuffer;
                        Area = szTmpBuffer;
                }
                else if (stricmp(szTmpBuffer, "Desc") == 0)
                {
                        FileIn.getline(szTmpBuffer, BUFFER_MAXLEN);
                        if (DescPos != 30)
                        {
                                Desc[DescPos] = szTmpBuffer;
                                DescPos++;
                        }
                }
                else if (stricmp(szTmpBuffer, "File") == 0)
                {
                        FileIn >> szTmpBuffer;
                        AssociatedFile = szTmpBuffer;
                }
                else if (stricmp( szTmpBuffer, "Size") == 0)
                {
                        FileIn >> FSize;
                }
                else
                {
                        // Ignore the line, misc.
                        FileIn.getline(szTmpBuffer, BUFFER_MAXLEN);
                }
        }

        FileIn.close();

        // Reduce the Description Position by 1
        DescPos--;

        // Verify the file specified exists
        if (AssociatedFile != "")
        {
                AssocFileStream.open(AssociatedFile.c_str(), ios::in);
                if (!AssocFileStream)
                {
                        Log  = "File specified in \'";
                        Log += hFileData->Name;
                        Log += "\' named \'";
                        Log += AssociatedFile.c_str();
                        Log += "\' does not exist in the inbound directory.";
                        PutLog(Log.c_str(), FATAL);

                        TICerror = true;
                }

                if (CheckArea(Area.c_str(), Destination) == true && !TICerror)
                {
                        // Verify the destination directory exists...
                        // if it doesn't, try to create it.
                        if (VerifyDirectory(Destination.c_str()) == false)
                        {
                                Log  = "Directory \'";
                                Log += Destination.c_str();
                                Log += "\' does not exist.";
                                PutLog(Log.c_str(), INFO);

                                cerr << "The directory \'" << Destination.c_str()
                                        << "\' does not exist..." << endl;
                                cerr << "Cannot move the file: "
                                        << AssociatedFile.c_str() << endl;

                                TICerror = true;
                        }

                        // Copy the file to the new destination
                        Buffer = Destination.c_str();
                        idx = Buffer.end() - 1;
#if defined(__TINYTIC_WINDOWS) || defined(__TINYTIC_OS2)
                        if (*idx != '\\')
                        {
                                Buffer += "\\";
                        }
#elif defined(__TINYTIC_POSIX)
                        if ( *idx != '/' )
                        {
                                Buffer += "/";
                        }
#endif
                        Buffer += AssociatedFile.c_str();
                        if (!FileCopy(
                                (VerifyFileUsable(AssociatedFile.c_str()) ? AssociatedFile.c_str() : NULL),
                                (VerifyFileUsable(Buffer.c_str()) ? Buffer.c_str() : NULL))
                                && !TICerror)
                        {
                                Log  = "Unable to copy \'";
                                Log += AssociatedFile.c_str();
                                Log += "\' to \'";
                                Log += Buffer.c_str();
                                Log += "\'!";

                                PutLog(Log.c_str(), FATAL);

                                TICerror = true;
                        }
                        else if (!TICerror)
                        {
                                Log  = "\'";
                                Log += AssociatedFile.c_str();
                                Log += "\' was copied to the directory for area \'";
                                Log += Area.c_str();
                                Log += "\'.";

                                PutLog(Log.c_str(), INFO);
                        }

                        // check sizes
                        if (FileSize(AssociatedFile.c_str()) != FSize && !TICerror)
                        {
                                Log  = "File size differs for \'";
                                Log += AssociatedFile.c_str();
                                Log += "\' specified in \'";
                                Log += hFileData->Name;
                                Log += "\'.";

                                PutLog(Log.c_str(), INFO);
                        }

                        if (FileSize(AssociatedFile.c_str()) != FileSize(Buffer.c_str()) && !TICerror )
                        {
                                cerr << "File \'" << AssociatedFile.c_str() << "\' will not be removed." << endl;

                                Log  = "File size difference between \'";
                                Log += AssociatedFile.c_str();
                                Log += "\' and \'";
                                Log += Buffer.c_str();
                                Log += "\'.";
                                PutLog(Log.c_str(), FATAL);

                                TICerror = true;
                        }

                        // Modify the date and time

                        // Unlink the old file
                        if (!TICerror)
                        {
                                if (FIsOpen(AssocFileStream) == true)
                                {
                                        AssocFileStream.close();
                                }
                                if (FIsOpen(FileIn) == true)
                                {
                                        FileIn.close();
                                }
                                if (remove(AssociatedFile.c_str()) != 0)
                                {
                                        Log  = "\'";
                                        Log += AssociatedFile.c_str();
                                        Log += "\' was not able to be removed.";
                                        PutLog(Log.c_str(), FATAL);
                                }
                                if (remove(hFileData->Name) != 0)
                                {
                                        Log  = "\'";
                                        Log += hFileData->Name;
                                        Log += "\' was not able to be removed.";
                                        PutLog(Log.c_str(), FATAL);
                                }
                        }

                        // Add Entry to the datafile
                        if (!TICerror)
                        {
                                if (WriteEntry(AssociatedFile.c_str(),
                                               FileSize(Buffer.c_str()),
                                               Desc, DescPos,
                                               Area.c_str()) == false)
                                {
                                        Log  = "Description entry for \'";
                                        Log += AssociatedFile.c_str();
                                        Log += "\' was not written to the DataFile.";
                                        PutLog(Log.c_str(), FATAL);
                                }
                        }
                }
                else if (!TICerror)
                {
                        Log  = "The area specified in TIC file \'";
                        Log += hFileData->Name;
                        Log += "\' as \'";
                        Log += Area.c_str();
                        Log += "\' is not defined in the config file.";
                        PutLog( Log.c_str(), FATAL );

                        TICerror = true;
                }
                               
                if (FIsOpen(AssocFileStream) == true)
                {
                        AssocFileStream.close();
                }
                if (FIsOpen(FileIn) == true)
                {
                        FileIn.close();
                }
        }
}

bool CheckArea(const char *area_name, string &Destination)
{
        fstream fConfig;
        char    szTmpBuffer[ BUFFER_MAXLEN ] = "\x0";
        string  Buffer;
        string  Log;
        bool    AreaExists = false;

        Buffer = szHomeDir.c_str();
        Buffer += szConfigFile.c_str();

        fConfig.open(Buffer.c_str(), ios::in);

        if (!fConfig)
        {
                Log  = "The configuration file \'";
                Log += szConfigFile.c_str();
                Log += " (";
                Log += Buffer.c_str();
                Log += ")\' was unable to be opened.";
                PutLog(Log.c_str(), FATAL);

                return (false);
        }

        while (!fConfig.eof())
        {
                fConfig >> szTmpBuffer;
                if (stricmp(szTmpBuffer, "AreaName") == 0)
                {
                        fConfig >> szTmpBuffer;
                        if (stricmp(szTmpBuffer, area_name) == 0)
                        {
                                AreaExists = true;
                                fConfig >> szTmpBuffer;
                                Destination = szTmpBuffer;
                        }

                        fConfig.ignore(BUFFER_MAXLEN, '\n');
                }
        }

        fConfig.close();

        return (AreaExists);
}

bool VerifyDirectory( const char *dir_name )
{
        string  CurDir;
        char    szTmpBuffer[ BUFFER_MAXLEN ] = "\x0";
        bool    DirExists = false;

        getcwd( szTmpBuffer, _MAX_PATH );
        CurDir = szTmpBuffer;

        if (chdir(dir_name) == 0)
        {
                DirExists = true;
        }
        else
        {
                DirExists = false;
        }

        chdir(CurDir.c_str());

        return (DirExists);
}

long FileSize(const char *src)
{
        fstream FileIn;
        long Size = 0;

        FileIn.open(src, ios::in | ios::binary);

        if (!FileIn)
        {
                // Unable to open the file for size checking
                return (-1);
        }

        FileIn.seekg(0L, ios::end);

        Size = FileIn.tellg();

        FileIn.close();

        return (Size);
}

bool WriteEntry(const char *file, long size, string Desc[10], long DescPos, const char *area)
{
        ofstream DFile;
        string   Log;
        long     idx;

        // Is a datafile specified at the command line ?
        if (szCmdDataFile != "")
        {
                // Use it, overrides any read or placed
                // from the config
                DFile.open(szCmdDataFile.c_str(), ios::app);
        }
        else
        {
                // No, use default or config file specified
                DFile.open(DataFile.c_str(), ios::app);
        }                        

        if (!DFile)
        {
                Log  = "Unable to open \'";
                Log += DataFile.c_str();
                Log += "\' for description writing.";
                PutLog(Log.c_str(), FATAL, 1);

                return (false);
        }

        for (idx = 0; idx <= DescPos; idx++)
        {
                if (idx == 0)
                {
                        DFile.setf(ios::left);
                        DFile << "Area : "
                              << setw(30) << area
                              << endl;
                        DFile.flush();
                        DFile << "-------------------------------------------------------------------------------" << endl;
                        DFile.flush();
                        DFile.setf(ios::left);
                        DFile << setw(18) << file
                              << setw(15) << size
                              << Desc[idx].c_str()
                              << endl;
                        DFile.flush();
                }
                else
                {
                        DFile.setf(ios::left);
                        DFile << setw(33) << " "
                              << setw(30) << Desc[idx].c_str()
                              << endl;
                        DFile.flush();
                }
        }

        DFile << endl;
        DFile.flush();

        DFile.close();

        return (true);
}
