package jp.co.sra.smalltalk.menu;

import java.awt.Component;
import java.awt.MenuComponent;
import java.awt.MenuContainer;
import java.awt.MenuItem;
import java.awt.MenuShortcut;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

/**
 * StMenuItemViewAwt class
 * 
 *  @author    nisinaka
 *  @created   2005/02/16 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StMenuItemViewAwt.java,v 8.9 2008/02/20 06:33:36 nisinaka Exp $
 */
public class StMenuItemViewAwt extends StObject implements StMenuItemView {

	protected StMenuItem menuItem;
	protected MenuComponent menuItemView;

	/**
	 * Create a new instance of StMenuItemViewAwt and initialize it.
	 *
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @category Instance creation
	 */
	public StMenuItemViewAwt(StMenuItem aMenuItem) {
		this.initialize();
		this.setMenuItem(aMenuItem);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		menuItem = null;
		menuItemView = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.setMenuItem(null);
		super.release();
	}

	/**
	 * Answer my current menu item.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuItem
	 * @see jp.co.sra.smalltalk.menu.StMenuItemView#menuItem()
	 * @category model accessing
	 */
	public StMenuItem getMenuItem() {
		return menuItem;
	}

	/**
	 * Set my new menu itme.
	 * 
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @category model accessing
	 */
	public void setMenuItem(StMenuItem aMenuItem) {
		if (menuItem != null) {
			menuItem.removeDependentListener(this);
		}
		menuItem = aMenuItem;
		if (menuItem != null) {
			menuItem.addDependentListener(this);
		}
	}

	/**
	 * Answer my window.
	 * If not created nor added yet, return null.
	 * 
	 * @return java.awt.Window
	 * @category accessing
	 */
	public Window getWindow() {
		if (menuItemView == null) {
			return null;
		}

		MenuContainer aMenuContainer = menuItemView.getParent();
		while (aMenuContainer != null) {
			if (aMenuContainer instanceof MenuComponent == false) {
				break;
			}
			aMenuContainer = ((MenuComponent) aMenuContainer).getParent();
		}

		Component aComponent = (aMenuContainer instanceof Component) ? (Component) aMenuContainer : null;
		while (aComponent != null) {
			if (aComponent instanceof Window) {
				break;
			}
			aComponent = aComponent.getParent();
		}

		return (Window) aComponent;
	}

	/**
	 * Answer the menu item view.
	 * 
	 * @return java.awt.MenuComponent
	 * @category user interface
	 */
	protected MenuComponent menuItemView() {
		if (menuItemView == null) {
			menuItemView = this.createMenuItemView();
			this.updateMenuItemView();
		}
		return menuItemView;
	}

	/**
	 * Answer the menu item view as MenuItem.
	 * 
	 * @return java.awt.MenuItem
	 * @category converting
	 */
	public MenuItem toMenuItem() {
		return (MenuItem) this.menuItemView();
	}

	/**
	 * Update for the change notice.
	 * 
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aspect = evt.getAspect();
		if (aspect == $("enabled")) {
			this.updateEnabled();
		} else if (aspect == $("label")) {
			this.updateLabel();
		}
	}

	/**
	 * Update the enabled of the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateEnabled() {
		this.toMenuItem().setEnabled(this.getMenuItem().isEnabled());
	}

	/**
	 * Update the label of the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateLabel() {
		this.toMenuItem().setLabel(this.getMenuItem().labelString());
	}

	/**
	 * Update the menu item view.
	 * 
	 * @category updating
	 */
	protected void updateMenuItemView() {
		this.updateEnabled();
		this.updateLabel();
	}

	/**
	 * Create a menu item view.
	 * 
	 * @return java.awt.MenuComponent
	 * @category private
	 */
	protected MenuComponent createMenuItemView() {
		MenuItem aMenuItem = new MenuItem();
		aMenuItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				StMenuItemViewAwt.this.performMenu($("action"), e);
			}
		});

		if (this.getMenuItem().shortcutKeyCharacter() > 0) {
			int keyCode = this.getMenuItem().shortcutKeyCharacter();
			int modifiers = this.getMenuItem().shortcutModifiers();
			if ((modifiers & InputEvent.CTRL_MASK) == 0) {
				System.err.println("must specify CTRL_MASK as a shortcut modifier for AWT MenuItem - " + KeyEvent.getKeyModifiersText(modifiers));
			}
			if ((modifiers & ~(InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK)) > 0) {
				System.err.println("can't specify a shortcut modifier besides CTRL_MASK and SHIFT_MASK for AWT MenuItem - " + KeyEvent.getKeyModifiersText(modifiers));
			}

			boolean useShiftModifier = (modifiers & InputEvent.SHIFT_MASK) > 0;
			aMenuItem.setShortcut(new MenuShortcut(keyCode, useShiftModifier));
		}

		return aMenuItem;
	}

	/**
	 * Perform for the menu action.
	 * 
	 * @param aspect jp.co.sra.smalltalk.StSymbol
	 * @param parameter java.lang.Object
	 * @category private
	 */
	protected void performMenu(StSymbol aspect, Object parameter) {
		MenuEvent aMenuEvent = new MenuEvent(this, aspect, parameter);
		this.getMenuItem().performMenu(aMenuEvent);
	}

	/**
	 * Create an appropriate view for the menu item.
	 * 
	 * @param aMenuItem jp.co.sra.smalltalk.menu.StMenuItem
	 * @return jp.co.sra.smalltalk.menu.StMenuItemViewAwt
	 * @category Utilities
	 */
	protected static StMenuItemViewAwt _New(StMenuItem aMenuItem) {
		if (aMenuItem == null) {
			return null;
		}

		Class aClass = null;
		try {
			aClass = Class.forName(aMenuItem.getClass().getName() + "ViewAwt");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}

		return (aClass == null) ? null : (StMenuItemViewAwt) StObject._New(aClass, aMenuItem);
	}

}
