package jp.co.sra.smalltalk;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager2;
import java.awt.Rectangle;
import java.util.Hashtable;

/**
 * WindowSpecLayout class
 * 
 *  @author    nisinaka
 *  @created   2003/05/08 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: WindowSpecLayout.java,v 8.11 2008/02/20 06:33:18 nisinaka Exp $
 */
public class WindowSpecLayout implements LayoutManager2 {

	public static class Position {
		float[] proportion = new float[4];
		int[] offset = new int[4];

		public Position(float leftProportion, int leftOffset, float topProportion, int topOffset, float rightProportion, int rightOffset, float bottomProportion, int bottomOffset) {
			proportion[0] = leftProportion;
			offset[0] = leftOffset;
			proportion[1] = topProportion;
			offset[1] = topOffset;
			proportion[2] = rightProportion;
			offset[2] = rightOffset;
			proportion[3] = bottomProportion;
			offset[3] = bottomOffset;
		}

		public float leftProportion() {
			return proportion[0];
		}

		public int leftOffset() {
			return offset[0];
		}

		public float topProportion() {
			return proportion[1];
		}

		public int topOffset() {
			return offset[1];
		}

		public float rightProportion() {
			return proportion[2];
		}

		public int rightOffset() {
			return offset[2];
		}

		public float bottomProportion() {
			return proportion[3];
		}

		public int bottomOffset() {
			return offset[3];
		}

		public Rectangle getBoundsFor(int width, int height) {
			int left = (int) (width * proportion[0] + offset[0]);
			int top = (int) (height * proportion[1] + offset[1]);
			if (Float.isNaN(proportion[2]) || Float.isNaN(proportion[3])) {
				return new Rectangle(left, top, Integer.MIN_VALUE, Integer.MIN_VALUE);
			} else {
				int right = (int) (width * proportion[2]) - 1 + offset[2];
				int bottom = (int) (height * proportion[3]) - 1 + offset[3];
				return new Rectangle(left, top, right - left + 1, bottom - top + 1);
			}
		}

		protected void setBoundsFor(Container parent, Component component) {
			Insets insets = parent.getInsets();
			int width = parent.getWidth() - insets.left - insets.right;
			int height = parent.getHeight() - insets.top - insets.bottom;
			Rectangle bounds = this.getBoundsFor(width, height);
			bounds.x += insets.left;
			bounds.y += insets.top;
			if (bounds.width == Integer.MIN_VALUE || bounds.height == Integer.MIN_VALUE) {
				Dimension d = component.getPreferredSize();
				bounds.width = d.width;
				bounds.height = d.height;
			}
			component.setBounds(bounds);
		}

		public boolean equals(Object anObject) {
			if (anObject instanceof Position == false) {
				return false;
			}
			Position position = (Position) anObject;
			for (int i = 0; i < 4; i++) {
				if (this.proportion[i] != position.proportion[i]) {
					return false;
				}
				if (this.offset[i] != position.offset[i]) {
					return false;
				}
			}
			return true;
		}

		public String toString() {
			StringBuffer stringBuffer = new StringBuffer();
			stringBuffer.append("LayoutPosition (");
			stringBuffer.append(this.leftProportion());
			stringBuffer.append(',');
			stringBuffer.append(this.leftOffset());
			stringBuffer.append(',');
			stringBuffer.append(this.topProportion());
			stringBuffer.append(',');
			stringBuffer.append(this.topOffset());
			stringBuffer.append(',');
			stringBuffer.append(this.rightProportion());
			stringBuffer.append(',');
			stringBuffer.append(this.rightOffset());
			stringBuffer.append(',');
			stringBuffer.append(this.bottomProportion());
			stringBuffer.append(',');
			stringBuffer.append(this.bottomOffset());
			stringBuffer.append(')');
			return stringBuffer.toString();
		}
	}

	protected Hashtable positions = new Hashtable();

	/**
	 * Create a new instance of <code>WindowSpecLayout</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public WindowSpecLayout() {
		super();
	}

	/**
	 * Create a new Position object which will be passed to the layout manager as a constraint.
	 *
	 * @return jp.co.sra.smalltalk.WindowSpecLayout
	 * @param leftProportion float
	 * @param leftOffset int
	 * @param topProportion float
	 * @param topOffset int
	 * @category Instance creation
	 */
	public static WindowSpecLayout.Position Position(float leftProportion, int leftOffset, float topProportion, int topOffset) {
		return Position(leftProportion, leftOffset, topProportion, topOffset, Float.NaN, 0, Float.NaN, 0);
	}

	/**
	 * Create a new Position object which will be passed to the layout manager as a constraint.
	 *
	 * @return jp.co.sra.smalltalk.WindowSpecLayout
	 * @param leftProportion float
	 * @param leftOffset int
	 * @param topProportion float
	 * @param topOffset int
	 * @param rightProportion float
	 * @param rightOffset int
	 * @param bottomProportion float
	 * @param bottomOffset int
	 * @category Instance creation
	 */
	public static WindowSpecLayout.Position Position(float leftProportion, int leftOffset, float topProportion, int topOffset, float rightProportion, int rightOffset, float bottomProportion, int bottomOffset) {
		return new WindowSpecLayout.Position(leftProportion, leftOffset, topProportion, topOffset, rightProportion, rightOffset, bottomProportion, bottomOffset);
	}

	/**
	 * Adds the specified component to the layout, using the specified constraint object.
	 *
	 * @param comp the component to be added
	 * @param constraints  where/how the component is added to the layout.
	 * @see java.awt.LayoutManager2#addLayoutComponent(java.awt.Component, java.lang.Object)
	 * @category layout accessing
	 */
	public void addLayoutComponent(Component comp, Object constraints) {
		if (constraints instanceof WindowSpecLayout.Position) {
			positions.put(comp, constraints);
		} else if (constraints != null) {
			throw new IllegalArgumentException("cannot add to layout: constraints must be a WindowSpecLayout.Position");
		}
	}

	/**
	 * Adds the specified component with the specified name to the layout.
	 * 
	 * @param name the component name
	 * @param comp the component to be added
	 * @see java.awt.LayoutManager#addLayoutComponent(java.lang.String, java.awt.Component)
	 * @category layout accessing
	 */
	public void addLayoutComponent(String name, Component comp) {
	}

	/**
	 * Returns the alignment along the x axis.  This specifies how
	 * the component would like to be aligned relative to other 
	 * components.  The value should be a number between 0 and 1
	 * where 0 represents alignment along the origin, 1 is aligned
	 * the furthest away from the origin, 0.5 is centered, etc.
	 * 
	 * @param target java.awt.Container
	 * @return float
	 * @see java.awt.LayoutManager2#getLayoutAlignmentX(java.awt.Container)
	 * @category layout accessing
	 */
	public float getLayoutAlignmentX(Container target) {
		return 0;
	}

	/**
	 * Returns the alignment along the y axis.  This specifies how
	 * the component would like to be aligned relative to other 
	 * components.  The value should be a number between 0 and 1
	 * where 0 represents alignment along the origin, 1 is aligned
	 * the furthest away from the origin, 0.5 is centered, etc.
	 * 
	 * @param target java.awt.Container
	 * @return float
	 * @see java.awt.LayoutManager2#getLayoutAlignmentY(java.awt.Container)
	 * @category layout accessing
	 */
	public float getLayoutAlignmentY(Container target) {
		return 0;
	}

	/**
	 * Invalidates the layout, indicating that if the layout manager
	 * has cached information it should be discarded.
	 * 
	 * @param target java.awt.Container
	 * @see java.awt.LayoutManager2#invalidateLayout(java.awt.Container)
	 * @category layout accessing
	 */
	public void invalidateLayout(java.awt.Container target) {
	}

	/** 
	 * Lays out the container in the specified panel.
	 *
	 * @param parent the component which needs to be laid out 
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 * @category layout accessing
	 */
	public void layoutContainer(Container parent) {
		Component[] components = parent.getComponents();
		for (int i = 0; i < components.length; i++) {
			Position position = (Position) positions.get(components[i]);
			if (position != null) {
				position.setBoundsFor(parent, components[i]);
			}
		}
	}

	/** 
	 * Returns the maximum size of this component.
	 * 
	 * @param target the component to be laid out
	 * @return java.awt.Dimension
	 * @see java.awt.LayoutManager2#maximumLayoutSize(java.awt.Container)
	 * @category layout accessing
	 */
	public Dimension maximumLayoutSize(Container target) {
		return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
	}

	/** 
	 * Calculates the minimum size dimensions for the specified 
	 * panel given the components in the specified parent container.
	 * 
	 * @param parent the component to be laid out
	 * @return java.awt.Dimension
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 * @category layout accessing
	 */
	public Dimension minimumLayoutSize(Container parent) {
		return new Dimension(10, 10);
	}

	/**
	 * Calculates the preferred size dimensions for the specified 
	 * panel given the components in the specified parent container.
	 *
	 * @param parent the component to be laid out
	 * @return java.awt.Dimension
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 * @category layout accessing
	 */
	public Dimension preferredLayoutSize(Container parent) {
		return parent.getSize();
	}

	/**
	 * Removes the specified component from the layout.
	 *
	 * @param comp the component to be removed
	 * @see java.awt.LayoutManager#removeLayoutComponent(java.awt.Component)
	 * @category layout accessing
	 */
	public void removeLayoutComponent(Component comp) {
		positions.remove(comp);
	}
}
