package jp.co.sra.smalltalk;

/**
 * StNumber class
 * 
 *  This is a utility class to parse a Number from a ReadStream.
 * 
 *  @author    nisinaka
 *  @created   2003/04/25 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StNumber.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StNumber extends StObject {

	public static final int UNKNOWN = 0;
	public static final int INTEGER = 1;
	public static final int FLOAT = 2;
	public static final int DOUBLE = 3;

	/**
	 * This is a utility class and never create an instance.
	 * 
	 * @category Private
	 */
	private StNumber() {
	}

	/**
	 * Negate the Number.
	 * 
	 * @return java.lang.Number
	 * @param aNumber java.lang.Number
	 * @category Arithmetic
	 */
	public static Number _Negate(Number aNumber) {
		if (aNumber instanceof Integer) {
			return new Integer(-aNumber.intValue());
		} else if (aNumber instanceof Long) {
			return new Long(-aNumber.longValue());
		} else if (aNumber instanceof Float) {
			return new Float(-aNumber.floatValue());
		} else if (aNumber instanceof Double) {
			return new Double(-aNumber.doubleValue());
		}
		return null;
	}

	/**
	 * Answer an instance of Number as described on the stream.
	 *
	 * @return java.lang.Number
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category Instance creation
	 */
	public static Number ReadFrom_(StReadStream aStream) {
		return ReadSmalltalkSyntaxFrom_(aStream);
	}

	/**
	 * Answer an instance of me as described on the stream.
	 * 
	 * @return long
	 * @param radix int
	 * @param value long
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @throws SmalltalkException
	 * @category Private - Compiler support
	 */
	protected static long CheckForOldSyntax_with_on_(int radix, long value, StReadStream aStream) {
		if (aStream.atEnd() == false && aStream.peekFor_('.')) {
			if (aStream.atEnd() == false && Character.digit(aStream.peek(), radix) < 0) {
				throw new SmalltalkException("Non-radix 10 numbers may not contain decimal points");
			} else {
				aStream.skip_(-1);
			}
		}

		if (aStream.atEnd() == false && Character.toLowerCase(aStream.peek()) == 'e') {
			throw new SmalltalkException("Non-radix 10 numbers may not contain decimal points");
		}

		return value;
	}

	/**
	 * Answer the number type specified with the exponent character.
	 * 
	 * @return int
	 * @param exponentChar int
	 * @category Private - Compiler support
	 */
	protected static int ChooseFloatRepresentationFor_(char exponentChar) {
		int nDataType;
		switch (exponentChar) {
			case 'e':
			case 's':
				nDataType = FLOAT;
				break;
			case 'd':
			case 'q':
				nDataType = DOUBLE;
				break;
			default:
				nDataType = UNKNOWN;
				break;
		}

		return nDataType;
	}

	/**
	 * Answer an int value read from the stream.
	 *
	 * @return int
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @param radix int
	 * @category Private - Compiler support
	 */
	protected static int ReadIntegerFrom_radix_(StReadStream aStream, int radix) {
		int value = 0;
		while (aStream.atEnd() == false) {
			int digit = Character.digit(aStream.next(), radix);
			if (digit < 0) {
				aStream.skip_(-1);
				return value;
			} else {
				value = value * radix + digit;
			}
		}
		return value;
	}

	/**
	 * Answer a long value read from the stream.
	 *
	 * @return long
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @param radix int
	 * @category Private - Compiler support
	 */
	protected static long ReadLongFrom_radix_(StReadStream aStream, int radix) {
		long value = 0;
		while (aStream.atEnd() == false) {
			int digit = Character.digit(aStream.next(), radix);
			if (digit < 0) {
				aStream.skip_(-1);
				return value;
			} else {
				value = value * radix + digit;
			}
		}
		return value;
	}

	/**
	 * Answer an instance of Number as described on the stream.
	 * 
	 * @return int
	 * @param integerPart int
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category Private - Compiler support
	 */
	protected static Number ReadSmalltalkFloat_from_(long integerPart, StReadStream aStream) {
		int coercionClass = INTEGER;
		double decimalPart = (double) 0;
		double decimalPlace = (double) 0.1;
		if (aStream.peekFor_('.')) {
			if (!aStream.atEnd() && Character.isDigit(aStream.peek())) {
				coercionClass = FLOAT;
				boolean atEnd;
				char digit;
				while (!(atEnd = aStream.atEnd()) && Character.isDigit((digit = aStream.next()))) {
					decimalPart = decimalPart + (decimalPlace * (digit - '0'));
					decimalPlace = decimalPlace / 10;
				}
				if (!atEnd) {
					aStream.skip_(-1);
				}
			} else {
				aStream.skip_(-1);
			}
		}

		int possibleCoercionClass;
		char exponentCharacter;
		if (aStream.peek() == 0) {
			possibleCoercionClass = 0;
			exponentCharacter = 0;
		} else {
			exponentCharacter = Character.toLowerCase(aStream.peek());
			possibleCoercionClass = ChooseFloatRepresentationFor_(exponentCharacter);
			if (possibleCoercionClass != 0) {
				aStream.next();
			}
		}

		int integerPart0 = 0;
		if (possibleCoercionClass != 0) {
			coercionClass = possibleCoercionClass;
			int integerParaStream = aStream.position();
			boolean neg = ('-' == aStream.peek());
			if (neg) {
				aStream.next();
			}
			if ((aStream.peek() != 0) && Character.isDigit(aStream.peek())) {
				integerPart0 = ReadIntegerFrom_radix_(aStream, 10);
				if (neg) {
					integerPart0 = -integerPart0;
				}
			} else {
				aStream.position_(integerParaStream);
			}
		}

		Number result;
		if (coercionClass == INTEGER) {
			if (Integer.MIN_VALUE <= integerPart && integerPart <= Integer.MAX_VALUE) {
				result = new Integer((int) integerPart);
			} else {
				result = new Long(integerPart);
			}
			return result;
		}

		double doubleValue = integerPart + decimalPart;
		if (integerPart0 != 0) {
			switch (exponentCharacter) {
				case 'd':
				case 'q':
					if (integerPart0 < 0) {
						for (int i = 0; i > integerPart0; i--) {
							doubleValue = doubleValue / 10;
						}
					} else {
						for (int i = 0; i < integerPart0; i++) {
							doubleValue = doubleValue * 10;
						}
					}
					result = new Double(doubleValue);
					break;
				case 'e':
					if (integerPart0 < 0) {
						for (int i = 0; i > integerPart0; i--) {
							doubleValue = doubleValue / 10;
						}
					} else {
						for (int i = 0; i < integerPart0; i++) {
							doubleValue = doubleValue * 10;
						}
					}
					result = new Float(doubleValue);
					break;
				case 's':
					result = new Float(doubleValue);
					break;
				default:
					result = new Float(doubleValue);
					break;
			}
		} else {
			if (coercionClass == FLOAT) {
				result = new Float((float) doubleValue);
			} else {
				result = new Double(doubleValue);
			}
		}

		return result;
	}

	/**
	 * Answer an instance of Number as described on the stream.
	 * 
	 * @return intdStream
	 * @param radix int
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category Private - Compiler support
	 */
	protected static int ReadSmalltalkRadix_from_(int radix, StReadStream aStream) {
		if (radix < 2) {
			throw SmalltalkException.Error("INVALID RADIX");
		}

		if (aStream.atEnd() || Character.digit(aStream.next(), radix) < 0) {
			aStream.skip_(-1);
			return radix;
		}

		return ReadIntegerFrom_radix_(aStream, radix);
	}

	/**
	 * Answer an instance of Number as described on the stream.
	 * 
	 * @return java.lang.Number
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category Private - Compiler support
	 */
	protected static Number ReadSmalltalkSyntaxFrom_(StReadStream aStream) {
		if (aStream.atEnd() || Character.isLetter(aStream.peek())) {
			return new Integer(0);
		}

		boolean neg = aStream.peekFor_('-');
		long value = ReadLongFrom_radix_(aStream, 10);
		if (aStream.peekFor_('r')) {
			int radix = (int) value;
			value = ReadSmalltalkRadix_from_(radix, aStream);
			value = CheckForOldSyntax_with_on_(radix, value, aStream);
			value *= (neg ? -1 : 1);
			if (Integer.MIN_VALUE <= value && value <= Integer.MAX_VALUE) {
				return new Integer((int) value);
			} else {
				return new Long(value);
			}
		} else {
			Number number = ReadSmalltalkFloat_from_(value, aStream);
			return (neg) ? _Negate(number) : number;
		}
	}
}
