package jp.co.sra.smalltalk;

import java.awt.Color;
import java.awt.image.ColorModel;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * StColorValue class
 * 
 *  @author    He Weijie
 *  @created   1998/10/10 (by xiaolai)
 *  @updated   2000/01/06 (by Mitsuhiro Asada)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StColorValue.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StColorValue extends StObject {
	//
	// Class variables
	//
	public static int _MaxDistanceSquared = -1;
	public static int _MaxValue = 0xFF;
	public static int _MaxLuminanceValue = _MaxValue * 1000;
	public static double _LuminanceScaleValue = 1.0f / _MaxLuminanceValue;
	//
	// Constant color values.
	//
	public static final Color Black = Color.black;
	public static final Color Blue = Color.blue;
	public static final Color Brown = new Color(127, 63, 63);
	public static final Color Chartreuse = new Color(127, 255, 0);
	public static final Color Cyan = new Color(0, 255, 255);
	public static final Color DarkCyan = new Color(0, 127, 127);
	public static final Color DarkGray = new Color(95, 95, 95);
	public static final Color DarkGreen = new Color(0, 127, 0);
	public static final Color DarkMagenta = new Color(127, 0, 127);
	public static final Color DarkRed = new Color(127, 0, 0);
	public static final Color Gray = new Color(127, 127, 127);
	public static final Color Green = new Color(0, 255, 0);
	public static final Color LightGray = new Color(170, 170, 170);
	public static final Color LightCyan = new Color(127, 255, 255);
	public static final Color LightYellow = new Color(255, 255, 127);
	public static final Color Magenta = new Color(255, 0, 255);
	public static final Color Navy = new Color(0, 0, 127);
	public static final Color Olive = new Color(127, 127, 0);
	public static final Color Orange = new Color(255, 127, 0);
	public static final Color Orchid = new Color(255, 127, 255);
	public static final Color PaleGreen = new Color(127, 255, 127);
	public static final Color Pink = new Color(255, 191, 191);
	public static final Color Purple = new Color(159, 31, 223);
	public static final Color Red = Color.red;
	public static final Color RoyalBlue = new Color(127, 127, 255);
	public static final Color Salmon = new Color(255, 127, 127);
	public static final Color SpringGreen = new Color(0, 255, 127);
	public static final Color VeryDarkGray = new Color(63, 63, 63);
	public static final Color VeryLightGray = new Color(212, 212, 212);
	public static final Color White = Color.white;
	public static final Color Yellow = new Color(255, 255, 0);

	/** Colors with its name as a key. */
	protected static Hashtable NamedColors = new Hashtable(31);

	static {
		NamedColors.put("black", Black);
		NamedColors.put("blue", Blue);
		NamedColors.put("brown", Brown);
		NamedColors.put("chartreuse", Chartreuse);
		NamedColors.put("cyan", Cyan);
		NamedColors.put("darkCyan", DarkCyan);
		NamedColors.put("darkGray", DarkGray);
		NamedColors.put("darkGreen", DarkGreen);
		NamedColors.put("darkMagenta", DarkMagenta);
		NamedColors.put("darkRed", DarkRed);
		NamedColors.put("gray", Gray);
		NamedColors.put("green", Green);
		NamedColors.put("lightGray", LightGray);
		NamedColors.put("lightCyan", LightCyan);
		NamedColors.put("lightYellow", LightYellow);
		NamedColors.put("magenta", Magenta);
		NamedColors.put("navy", Navy);
		NamedColors.put("olive", Olive);
		NamedColors.put("orange", Orange);
		NamedColors.put("orchid", Orchid);
		NamedColors.put("paleGreen", PaleGreen);
		NamedColors.put("pink", Pink);
		NamedColors.put("purple", Purple);
		NamedColors.put("red", Red);
		NamedColors.put("royalBlue", RoyalBlue);
		NamedColors.put("salmon", Salmon);
		NamedColors.put("springGreen", SpringGreen);
		NamedColors.put("veryDarkGray", VeryDarkGray);
		NamedColors.put("veryLightGray", VeryLightGray);
		NamedColors.put("white", White);
		NamedColors.put("yellow", Yellow);
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return boolean
	 * @param color1 java.awt.Color
	 * @param color2 java.awt.Color
	 * @category Accessing
	 */
	public static double _DistanceFrom(Color color1, Color color2) {
		return Math.sqrt(_DistanceSquaredFrom(color1, color2));
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return boolean
	 * @param color1 java.awt.Color
	 * @param color2 java.awt.Color
	 * @category Accessing
	 */
	public static int _DistanceSquaredFrom(Color color1, Color color2) {
		int rdiff = color1.getRed() - color2.getRed();
		int gdiff = color1.getGreen() - color2.getGreen();
		int bdiff = color1.getBlue() - color2.getBlue();
		return (rdiff * rdiff) + (gdiff * gdiff) + (bdiff * bdiff);
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.lang.Integer or null
	 * @param color1 java.awt.Color
	 * @param color2 java.awt.Color
	 * @param dist int
	 * @category Accessing
	 */
	public static Integer _DistanceSquaredFrom_ifLessThan_(Color color1, Color color2, int dist) {
		int diff;
		int rdiff;
		int gdiff;
		int bdiff;

		rdiff = color1.getRed() - color2.getRed();
		diff = rdiff * rdiff;
		if (diff >= dist) {
			return null;
		}

		gdiff = color1.getGreen() - color2.getGreen();
		diff = diff + (gdiff * gdiff);
		if (diff >= dist) {
			return null;
		}

		bdiff = color1.getBlue() - color2.getBlue();
		diff = diff + (bdiff * bdiff);
		if (diff >= dist) {
			return null;
		}

		return new Integer(diff);
	}

	/**
	 * Get the alpha component in the range 0-255.
	 * 
	 * @return int
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static int _GetAlpha(Color color) {
		ColorModel rgbColorModel = ColorModel.getRGBdefault();
		return rgbColorModel.getAlpha(color.getRGB());
	}

	/**
	 * Get the brightness component in the range 0f-1f.
	 * 
	 * @return float
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static float _GetBrightness(Color color) {
		float[] values = new float[3];
		Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(), values);
		return values[2];
	}

	/**
	 * Get the Coler by name.
	 * 
	 * @return java.awt.Color
	 * @param colorName java.lang.String
	 * @category Instance creation
	 */
	public static Color _GetColorByName_(String colorName) {
		Color color = (Color) NamedColors.get(colorName);
		if (color == null) {
			char[] chars = colorName.toCharArray();

			if (Character.isLowerCase(chars[0])) {
				chars[0] = Character.toUpperCase(chars[0]);
				color = (Color) NamedColors.get(new String(chars));
			}
		}
		return color;
	}

	/**
	 * Answer the hue component in the range 0f-1f.
	 * 
	 * @return float
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static float _GetHue(Color color) {
		float[] values = new float[3];
		Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(), values);
		return values[0];
	}

	/**
	 * Answer the luminance component in the range 0f-1f.
	 * 
	 * @return double
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static double _GetLuminance(Color color) {
		return _LuminanceScaleValue * _ScaledLuminance_(color);
	}

	/**
	 * Answer the saturation component in the range 0f-1f.
	 * 
	 * @return float
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static float _GetSaturation(Color color) {
		float[] values = new float[3];
		Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(), values);
		return values[1];
	}

	/**
	 * Answer true if color is gray, otherwise false.
	 * 
	 * @return boolean
	 * @param color java.awt.Color
	 * @category Testing
	 */
	public static boolean _IsGray(Color color) {
		return ((color.getRed() == color.getGreen()) && (color.getGreen() == color.getBlue()) && ((color.getRGB() >> 24 & 0xFF) != 0x00));
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return double
	 * @category Constants
	 */
	public static int _MaxDistanceSquared() {
		if (_MaxDistanceSquared == -1) {
			_MaxDistanceSquared = _DistanceSquaredFrom(Color.black, Color.white);
		}
		return _MaxDistanceSquared;
	}

	/**
	 * Create new color from red, green, blue and alpha.
	 * 
	 * @return java.awt.Color
	 * @param red float
	 * @param green float
	 * @param blue float
	 * @param alpha float
	 * @category Instance creation
	 */
	public static Color _NewColor(float red, float green, float blue, float alpha) {
		return _NewColor(red * 255, green * 255, blue * 255, alpha * 255);
	}

	/**
	 * Create new color from red, green, blue and alpha.
	 * 
	 * @return java.awt.Color
	 * @param red int
	 * @param green int
	 * @param blue int
	 * @param alpha int
	 * @category Instance creation
	 */
	public static Color _NewColor(int red, int green, int blue, int alpha) {
		int pixel = (alpha << 24) | (red << 16) | (green << 8) | (blue << 0);
		return new Color(pixel);
	}

	/**
	 * Answer the scaled luminance component in the range 0-255.
	 * 
	 * @return float
	 * @param color java.awt.Color
	 * @category Accessing
	 */
	public static int _ScaledLuminance_(Color color) {
		return (299 * color.getRed()) + (587 * color.getGreen()) + (114 * color.getBlue());
	}

	/**
	 * Answer the store string of the specified color.
	 * 
	 * @return java.lang.String
	 * @param aColor java.awt.Color
	 * @category Printing
	 */
	public static String _StoreStringOf(Color aColor) {
		int MaxValue = 0x1fff;
		float ScaleValue = (float) MaxValue / 0xff;
		StringBuffer buffer = new StringBuffer();

		buffer.append("(ColorValue scaledRed: ");
		buffer.append(Math.round(aColor.getRed() * ScaleValue));
		buffer.append(" scaledGreen: ");
		buffer.append(Math.round(aColor.getGreen() * ScaleValue));
		buffer.append(" scaledBlue: ");
		buffer.append(Math.round(aColor.getBlue() * ScaleValue));
		buffer.append(")");

		return buffer.toString();
	}

	/**
	 * Answer the print string of the specified color.
	 * 
	 * @return java.lang.String
	 * @param aColor java.awt.Color
	 * @category Printing
	 */
	public static String _PrintStringOf(Color aColor) {
		StringBuffer buffer = new StringBuffer();
		buffer.append("(");
		buffer.append("ColorValue");
		if (aColor.getRed() == aColor.getGreen() && aColor.getRed() == aColor.getBlue()) {
			buffer.append(" brightness: ");
			buffer.append(aColor.getBlue() / 255f);
		} else {
			buffer.append(" red: ");
			buffer.append(aColor.getRed() / 255f);
			buffer.append(" green: ");
			buffer.append(aColor.getGreen() / 255f);
			buffer.append(" blue: ");
			buffer.append(aColor.getBlue() / 255f);
		}
		buffer.append(")");
		return buffer.toString();
	}

	/**
	 * Answer a new Color which is 50% blend of the two colors.
	 * 
	 * @param aColor1 java.awt.Color
	 * @param aColor2 java.awt.Color
	 * @return java.awt.Color
	 * @category Converting
	 */
	public static Color Blend(Color aColor1, Color aColor2) {
		int r = (aColor1.getRed() + aColor2.getRed()) / 2;
		int g = (aColor1.getGreen() + aColor2.getGreen()) / 2;
		int b = (aColor1.getBlue() + aColor2.getBlue()) / 2;
		int a = (aColor1.getAlpha() + aColor2.getAlpha()) / 2;
		return new Color(r, g, b, a);
	}

	/**
	 * Create new color from brightness component in the range 0f-1f.
	 * 
	 * @return java.awt.Color
	 * @param b double
	 * @category Instance creation
	 */
	public static Color Brightness_(double b) {
		int sb = Math.round((float) (b * 255));
		if ((sb >= 0) && (sb <= 255)) {
			return new Color(sb, sb, sb);
		} else {
			return Color.black;
		}
	}

	/**
	 * Return the names of all the constant color.
	 * 
	 * @return java.util.Vector
	 * @category Constants
	 */
	public static Vector ConstantNames() {
		Vector colorNames = new Vector();
		Enumeration keys = NamedColors.keys();
		while (keys.hasMoreElements()) {
			colorNames.addElement($((String) keys.nextElement()));
		}
		return colorNames;
	}

	/**
	 * Create new color from cyan, magenta and yellow.
	 * 
	 * @return java.awt.Color
	 * @param cyan double
	 * @param magenta double
	 * @param yellow double
	 * @category Instance creation
	 */
	public static Color Cyan_Magenta_Yellow_(double cyan, double magenta, double yellow) {
		return new Color((float) (1 - cyan), (float) (1 - magenta), (float) (1 - yellow));
	}

	/**
	 * Create new color from scaled red, scaled green and scaled blue.
	 * 
	 * @return java.awt.Color
	 * @param r int
	 * @param g int
	 * @param b int
	 * @category Instance creation
	 */
	public static Color ScaledRed_ScaledGreen_ScaledBlue_(int r, int g, int b) {
		int scaledRed = Math.min(Math.max((r / 32), 0), 255);
		int scaledGreen = Math.min(Math.max((g / 32), 0), 255);
		int scaledBlue = Math.min(Math.max((b / 32), 0), 255);
		return new Color(scaledRed, scaledGreen, scaledBlue);
	}
}
