package jp.co.sra.smalltalk;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Window;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.border.Border;

import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuBarView;
import jp.co.sra.smalltalk.menu.StMenuBarViewSwing;

/**
 * StApplicationWindowSwing class
 * 
 *  @author    nisinaka
 *  @created   2005/02/22 (by nisinaka)
 *  @updated   2006/10/04 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StApplicationWindowSwing.java,v 8.9 2008/02/20 06:33:17 nisinaka Exp $
 */
public class StApplicationWindowSwing extends JFrame implements StApplicationWindow {

	protected StApplicationModel application;
	protected StView view;
	protected StMenuBarViewSwing menuBarView;

	/**
	 * Create a new instance of StApplicationWindowSwing and initialize it.
	 *
	 * @param anApplication jp.co.sra.smalltalk.StApplicationModel
	 * @category Instance creation
	 */
	public StApplicationWindowSwing(StApplicationModel anApplication, StView aView) {
		this.application_(anApplication);
		this.view_(aView);
		this.buildWindow();
	}

	/**
	 * Answer my current application.
	 * 
	 * @return jp.co.sra.smalltalk.StApplicationModel
	 * @see jp.co.sra.smalltalk.StApplicationWindow#application()
	 * @category accessing
	 */
	public StApplicationModel application() {
		return application;
	}

	/**
	 * Set my new application.
	 * 
	 * @param anApplication jp.co.sra.smalltalk.StApplication
	 * @see jp.co.sra.smalltalk.StApplicationWindow#application_(jp.co.sra.smalltalk.StApplicationModel)
	 * @category accessing
	 */
	public void application_(StApplicationModel anApplication) {
		application = anApplication;
	}

	/**
	 * Answer my current view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationWindow#view()
	 * @category accessing
	 */
	public StView view() {
		return view;
	}

	/**
	 * Set my new view.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationWindow#view_(jp.co.sra.smalltalk.StView)
	 * @category accessing
	 */
	public void view_(StView aView) {
		view = aView;
	}

	/**
	 * Answer my current menu bar view.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBarView
	 * @see jp.co.sra.smalltalk.StApplicationWindow#menuBarView()
	 * @category accessing
	 */
	public StMenuBarView menuBarView() {
		return menuBarView;
	}

	/**
	 * Set my new menu bar view.
	 * 
	 * @param aMenuBarView jp.co.sra.smalltalk.menu.StMenuBarViewSwing
	 * @category accessing
	 */
	protected void menuBarView_(StMenuBarViewSwing aMenuBarView) {
		if (this.view() instanceof StViewSwing) {
			aMenuBarView.setLightWeightPopupEnabled(((StViewSwing) this.view()).isLightWeightPopupEnabled());
		}
		this.setJMenuBar(aMenuBarView.toMenuBar());
		this.pack();

		menuBarView = aMenuBarView;
	}

	/**
	 * Convert to Window.
	 * 
	 * @return java.awt.Window
	 * @see jp.co.sra.smalltalk.StApplicationWindow#toWindow()
	 * @category converting
	 */
	public Window toWindow() {
		return (Window) this;
	}

	/**
	 * Convert to Frame.
	 * 
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationWindow#toFrame()
	 * @category converting
	 */
	public Frame toFrame() {
		return (Frame) this;
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a dependent.
	 * The argument anAspectSymbol is typically a Symbol that indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		// nothing to do
	}

	/**
	 * Update the window title.
	 * 
	 * @return boolean
	 * @see jp.co.sra.smalltalk.StApplicationWindow#updateWindowTitle()
	 * @category updating
	 */
	public boolean updateWindowTitle() {
		if (this.application() == null) {
			return false;
		}
		String windowTitle = this.application().windowTitle();
		if (windowTitle == null) {
			return false;
		}

		this.setTitle(windowTitle);
		return true;
	}

	/**
	 * Update the menu bar.
	 * 
	 * @return boolean
	 * @see jp.co.sra.smalltalk.StApplicationWindow#updateMenuBar()
	 * @category updating
	 */
	public boolean updateMenuBar() {
		if (this.application() == null) {
			return false;
		}
		StMenuBar menuBar = this.application()._menuBar();
		if (menuBar == null) {
			return false;
		}

		this.menuBarView_((StMenuBarViewSwing) this.view()._createMenuBarView(menuBar));
		return true;
	}

	/**
	 * Build the window.
	 * 
	 * @category private
	 */
	protected void buildWindow() {
		if (this.application() == null) {
			return;
		}
		if (this.view() == null) {
			return;
		}

		this.updateWindowTitle();
		this.updateMenuBar();

		Component aComponent = this.view().toComponent();
		if (this.application()._viewOnBorderedPanel()) {
			JPanel aPanel = _PanelWithBorder();
			aPanel.setLayout(new BorderLayout());
			aPanel.add(aComponent, BorderLayout.CENTER);
			aComponent = aPanel;
		}
		this.getContentPane().add(aComponent);
		this.pack();
		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

		// Set the window size.
		Insets insets = this.getInsets();
		Dimension size = this.view().toComponent().getPreferredSize();
		this.setSize(size.width + insets.left + insets.right, size.height + insets.top + insets.bottom);

		if (this.application().windowResizable() == false) {
			this.setResizable(false);
		}
	}

	/**
	 * Create a panel with border.
	 * 
	 * @return javax.swing.JPanel
	 * @category Utilities
	 */
	public static JPanel _PanelWithBorder() {
		JPanel borderedPanel = new JPanel() {
			public void paint(Graphics graphics) {
				this.paintChildren(graphics);
				this.paintBorder(graphics);
			}
		};
		borderedPanel.setBorder(new Border() {
			public Insets getBorderInsets(Component c) {
				return new Insets(2, 2, 2, 2);
			}

			public boolean isBorderOpaque() {
				return false;
			}

			public void paintBorder(Component c, Graphics g, int x, int y, int width, int height) {
				// Black lines
				g.setColor(Color.black);
				g.drawLine(1, 1, width - 3, 1);
				g.drawLine(1, 1, 1, height - 3);

				// Gray lines.
				g.setColor(Color.gray);
				g.drawLine(0, 0, width - 2, 0);
				g.drawLine(0, 0, 0, height - 2);

				// Light gray lines.
				g.setColor(Color.lightGray);
				g.drawLine(1, height - 2, width - 2, height - 2);
				g.drawLine(width - 2, 1, width - 2, height - 2);

				// White lines.
				g.setColor(Color.white);
				g.drawLine(0, height - 1, width - 1, height - 1);
				g.drawLine(width - 1, 0, width - 1, height - 1);
			}
		});

		return borderedPanel;
	}

}
