package jp.co.sra.qt4jun;

import java.awt.Rectangle;
import java.awt.Dimension;
import java.awt.Image;
import java.io.File;

/**
 * JunQTNativeMovie class
 * 
 *  @author    Hoshi Takanori
 *  @created   2005/01/19 (by Hoshi Takanori)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunQTNativeMovie.java,v 8.10 2008/02/20 06:33:17 nisinaka Exp $
 */
public class JunQTNativeMovie implements JunQTMovie {
	private static final boolean UseMovieController = true;

	protected int movie;

	/**
	 * get the movie interface.
	 *
	 * @return jp.co.sra.qt4jun.JunMovieInterface
	 * @category Private
	 */
	static JunQTInterface _interface() {
		return JunQTInterface.Interface();
	}

	/**
	 * answer true if getMovieFile() works.
	 *
	 * @return boolean
	 * @category Instance creation
	 */
	static boolean canGetMovieFile() {
		return _interface().canGetMovieFile();
	}

	/**
	 * open the standard file preview dialog and get the movie filename.
	 *
	 * @return java.lang.String
	 * @category Instance creation
	 */
	static String getMovieFile() {
		return _interface().getMovieFile();
	}

	/**
	 * open the movie.
	 *
	 * @param filename java.lang.String
	 * @return jp.co.sra.qt4jun.JunQTMovie
	 * @category Instance creation
	 */
	static JunQTMovie openMovie(String filename) {
		int movie = _interface().openMovie(filename);
		if (movie != 0) {
			if (UseMovieController) {
				return new JunQTNativeMovieWithMovieController(movie);
			} else {
				return new JunQTNativeMovie(movie);
			}
		} else {
			return null;
		}
	}

	/**
	 * constructor.
	 *
	 * @param movie int
	 * @category Instance creation
	 */
	public JunQTNativeMovie(int movie) {
		this.movie = movie;
	}

	/**
	 * finalize.
	 *
	 * @category initialize-release
	 */
	protected void finalize() {
		this.release();
	}

	/**
	 * release the movie.
	 *
	 * @category initialize-release
	 */
	public void release() {
		if (movie != 0) {
			this._interface().qtDisposeMovie(movie);
			movie = 0;
		}
	}

	/**
	 * set the movie to active/deactive.
	 *
	 * @param active boolean
	 * @category accessing
	 */
	public void setActive(boolean active) {
		this._interface().qtSetMovieActive(movie, active);
	}

	/**
	 * get the movie box.
	 *
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle getBox() {
		return this._interface().getMovieBox(movie);
	}

	/**
	 * set the movie box.
	 *
	 * @param box java.awt.Rectangle
	 * @category accessing
	 */
	public void setBox(Rectangle box) {
		this._interface().setMovieBox(movie, box);
	}

	/**
	 * get the duration.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getDuration() {
		return this._interface().qtGetMovieDuration(movie);
	}

	/**
	 * get the next interesting time.
	 *
	 * @param time int
	 * @param rate int
	 * @return int
	 * @category accessing
	 */
	public int getNextInterestingTime(int time, int rate) {
		return this._interface().nextInterestingTimeOfMovie(movie, time, rate);
	}

	/**
	 * select portions of the movie.
	 *
	 * @param start int
	 * @param duration int
	 * @category accessing
	 */
	public void setSelection(int start, int duration) {
		this._interface().qtSetMovieSelection(movie, start, duration);
	}

	/**
	 * get the time scale.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getTimeScale() {
		return this._interface().qtGetMovieTimeScale(movie);
	}

	/**
	 * get the time value.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getTimeValue() {
		return this._interface().qtGetMovieTimeValue(movie);
	}

	/**
	 * set the time value.
	 *
	 * @param time int
	 * @category accessing
	 */
	public void setTimeValue(int time) {
		this._interface().qtSetMovieTimeValue(movie, time);
	}

	/**
	 * get the volume.
	 *
	 * @return double
	 * @category accessing
	 */
	public double getVolume() {
		int volume = this._interface().qtGetMovieVolume(movie);
		return (double) volume / this._interface().kFullVolume;
	}

	/**
	 * set the volume.
	 *
	 * @param volume double
	 * @category accessing
	 */
	public void setVolume(double volume) {
		int newVolume = (int) (volume * this._interface().kFullVolume);
		this._interface().qtSetMovieVolume(movie, newVolume);
	}

	/**
	 * get the image at the given time value.
	 *
	 * @param time int
	 * @param extent java.awt.Dimension
	 * @return java.awt.Image
	 * @category converting
	 */
	public Image getImageAt(int time, Dimension extent) {
		return this._interface().getMovieImageAt(movie, time, extent);
	}

	/**
	 * get the sound data from the movie.
	 *
	 * @param start int
	 * @param duration int
	 * @return byte[]
	 * @category converting
	 */
	public byte[] getSoundData(int start, int duration) {
		return this._interface().convertMovieToSoundData(movie, start, duration);
	}

	/**
	 * copy the selected portions of the movie.
	 *
	 * @return jp.co.sra.qt4jun.JunQTMovie
	 * @category copying
	 */
	public JunQTMovie copySelection() {
		int newMovie = this._interface().qtCopyMovieSelection(movie);
		if (newMovie != 0) {
			if (UseMovieController) {
				return new JunQTNativeMovieWithMovieController(newMovie);
			} else {
				return new JunQTNativeMovie(newMovie);
			}
		} else {
			return null;
		}
	}

	/**
	 * flatten the movie to file.
	 *
	 * @param filename java.io.File
	 * @category writing
	 */
	public void flattenToFile(File filename) {
		this._interface().flattenMovie_filename_(movie, filename);
	}

	/**
	 * go to beginning of the movie.
	 *
	 * @category playing
	 */
	public void goToBeginning() {
		this._interface().qtGoToBeginningOfMovie(movie);
	}

	/**
	 * go to end of the movie.
	 *
	 * @category playing
	 */
	public void goToEnd() {
		this._interface().qtGoToEndOfMovie(movie);
	}

	/**
	 * is the movie done or not?
	 *
	 * @return boolean
	 * @category playing
	 */
	public boolean isDone() {
		return this._interface().isMovieDone(movie);
	}

	/**
	 * start the movie.
	 *
	 * @category playing
	 */
	public void start() {
		this._interface().qtStartMovie(movie);
	}

	/**
	 * stop the movie.
	 *
	 * @category playing
	 */
	public void stop() {
		this._interface().qtStopMovie(movie);
	}

	/**
	 * do movie task.
	 *
	 * @category playing
	 */
	public void task() {
		this._interface().qtMoviesTask(movie, 0);
	}

	/**
	 * update the movie.
	 *
	 * @category playing
	 */
	public void update() {
		this._interface().qtUpdateMovie(movie);
	}

	/**
	 * set the movie port.
	 *
	 * @param port jp.co.sra.qt4jun.JunQTPort
	 * @category private
	 */
	public void setPort(JunQTPort port) {
		int grafPtr = ((JunQTNativePort) port).getGrafPtr();
		this._interface().qtSetMovieGWorld(movie, grafPtr, 0);
	}
}
