package jp.co.sra.qt4jun;

import java.awt.*;
import java.awt.image.*;
import java.io.File;

/**
 * JunQTInterface class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2001/01/11 (by NISHIHARA Satoshi)
 *  @updated   2001/12/19 (by Hoshi Takanori)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunQTInterface.java,v 8.11 2008/02/20 06:33:17 nisinaka Exp $
 */
public final class JunQTInterface {
	public static final String libraryName = "JunQT";
	public static final String requiredVersion = "2.0";
	public static final int requiredBuildNo = 14;

	private static JunQTInterface MovieInterface = null;

	static {
		try {
			System.loadLibrary(libraryName);
			String findVersion = GetVersion();
			String buildPrefix = requiredVersion + " build ";
			int buildNo;

			try {
				buildNo = Integer.parseInt(findVersion.substring(buildPrefix.length()));
			} catch (NumberFormatException e) {
				buildNo = 0;
			}

			if (findVersion.startsWith(buildPrefix) && (buildNo >= requiredBuildNo)) {
				System.out.println("JNI library '" + libraryName + "' version " + findVersion);
				InstallInterface();
			} else {
				System.err.println("JNI library '" + libraryName + "' version error.");
				System.err.println("Found library is version " + findVersion);
				System.err.println("Required library is version " + requiredVersion + ", build " + requiredBuildNo + " or later");
				throw new Error("JNI library '" + libraryName + "' version error");
			}
		} catch (UnsatisfiedLinkError e) {
			System.err.println("Can't find JNI library '" + libraryName + "'.");
			throw e;
		}
	}

	public static final int anyCodec = 0;
	public static final int codecNormalQuality = 0x00000200;
	public static final int createMovieFileDeleteCurFile = 0x80000000;
	public static final int createMovieFileDontCreateResFile = 0x10000000;
	public static final int fixed1 = 0x00010000;
	public static final int flattenAddMovieToDataFork = 1;
	public static final int kAnimationCodecType = 0x726c6520;
	public static final int kFullVolume = 0x0100;
	public static final int kNoVolume = 0;
	public static final int kOffsetBinary = 0x72617720;
	public static final int kQTMLNoIdleEvents = 2;
	public static final int kSyncSample = 0;
	public static final int kVideoTimeScale = 600;
	public static final int mcActionPrerollAndPlay = 55;
	public static final int mcNotVisible = 8;
	public static final int mcTopLeftMovie = 1;
	public static final int MovieFileType = 0x4d6f6f56;
	public static final int movieInDataForkResID = -1;
	public static final int nextTimeStep = 0x00000010;
	public static final int noErr = 0;
	public static final int sigMoviePlayer = 0x54564f44;
	public static final int smSystemScript = -1;
	public static final int VideoMediaType = 0x76696465;
	public static final int VisualMediaCharacteristic = 0x65796573;
	public static final int _MAX_PATH = 260;

	/**
	 * get the movie interface.
	 *
	 * @return jp.co.sra.jun.goodies.movie.framework.JunQTInterface
	 */
	public static JunQTInterface Interface() {
		if (MovieInterface == null) {
			InstallInterface();
		}
		return MovieInterface;
	}

	/**
	 * install the movie interface.
	 */
	public static void InstallInterface() {
		JunQTInterface anInterface = new JunQTInterface();
		anInterface.enterMovies();
		MovieInterface = anInterface;
	}

	/**
	 * uninstall the movie interface.
	 */
	public static void UninstallInterface() {
		if (MovieInterface != null) {
			try {
				MovieInterface.exitMovies();
			} catch (Exception e) {
				e.printStackTrace();
				System.err.println("Failed to uninstall " + MovieInterface.getClass().getName());
			}
		}
		MovieInterface = null;
	}

	/**
	 * Constainer Method.
	 */
	protected JunQTInterface() {
		// Singleton pattern
		super();
	}

	/**
	 * initialize the QuickTime Media Layer and then QuickTime itself.
	 *
	 * @throws Error
	 */
	public void enterMovies() {
		int err;

		err = this.qtInitializeQTML(0);
		if (err != noErr) {
			throw new Error("InitializeQTML failed");
		}

		err = this.qtEnterMovies();
		if (err != noErr) {
			throw new Error("EnterMovies failed");
		}
	}

	/**
	 * terminate the QuickTime itself and then QuickTime Media Layer.
	 *
	 * @throws Exception
	 */
	public void exitMovies() throws Exception {
		this.qtExitMovies();
		this.qtTerminateQTML();
	}

	/**
	 * Create port association (part 1).
	 *
	 * @param window java.awt.Component
	 * @return int
	 */
	public int createPortAssociation1(Component window) {
		try {
			Class aClass = Class.forName("javax.swing.JComponent");
			while (aClass.isInstance(window) && window.getParent() != null) {
				window = window.getParent();
			}
		} catch (ClassNotFoundException e) {
		}
		return this.qtCreatePortAssociation1(window, 0, kQTMLNoIdleEvents);
	}

	/**
	 * Create port association (part 2).
	 *
	 * @param viewPtr int
	 * @return int
	 */
	public int createPortAssociation2(int viewPtr) {
		return this.qtCreatePortAssociation2(viewPtr);
	}

	/**
	 * Destroy port association.
	 *
	 * @param viewPtr int
	 */
	public void destroyPortAssociation(int viewPtr) {
		this.qtDestroyPortAssociation(viewPtr);
	}

	/**
	 * flatten the movie.
	 *
	 * @param movie int
	 * @param filename java.io.File
	 */
	public void flattenMovie_filename_(int movie, File filename) {
		int newMovie = this.qtFlattenMovieData(movie, flattenAddMovieToDataFork,
				filename.getPath(), sigMoviePlayer, smSystemScript,
				createMovieFileDeleteCurFile | createMovieFileDontCreateResFile);
		if (newMovie != 0) {
			this.qtDisposeMovie(newMovie);
		}
	}

	/**
	 * answer true if getMovieFile() works.
	 *
	 * @return boolean
	 */
	public boolean canGetMovieFile() {
		String osName = System.getProperty("os.name");
		if (osName != null && osName.equals("Mac OS X")) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * open the standard file preview dialog and get the movie filename.
	 *
	 * @return java.lang.String
	 */
	public String getFilePreview() {
		int typeList[] = new int[1];
		typeList[0] = MovieFileType;
		return this.qtStandardGetFilePreview(typeList);
	}

	/**
	 * get the movie filename.
	 *
	 * @return java.lang.String
	 */
	public String getMovieFile() {
		return this.getFilePreview();
	}

	/**
	 * get the next interesting time of the movie.
	 *
	 * @param movie int
	 * @param time int
	 * @param rate int
	 * @return int
	 */
	public int nextInterestingTimeOfMovie(int movie, int time, int rate) {
		int[] mediaTypes = { VisualMediaCharacteristic };
		int[] array = this.qtGetMovieNextInterestingTime(
				movie, nextTimeStep, mediaTypes, time, rate);
		return array[0];
	}

	/**
	 * open the movie.
	 *
	 * @return int
	 */
	public int openMovie() {
		String filename = getMovieFile();
		if (filename != null && filename.length() > 0) {
			return this.openMovie(filename);
		}
		return 0;
	}

	/**
	 * implements for interface.
	 *
	 * @param filename java.io.File
	 * @return int
	 */
	public int openMovie(File filename) {
		return this.qtOpenMovie(filename.getPath());
	}

	/**
	 * implements for interface.
	 *
	 * @param filename java.lang.String
	 * @return int
	 */
	public int openMovie(String filename) {
		return this.qtOpenMovie(filename);
	}

	/**
	 * get rectangle of the movie.
	 *
	 * @param movie int
	 * @return java.awt.Rectangle
	 */
	public Rectangle getMovieBox(int movie) {
		int[] box = this.qtGetMovieBox(movie);
		return new Rectangle(box[0], box[1], box[2], box[3]);
	}

	/**
	 * implements for interface.
	 *
	 * @param movie int
	 * @param rect java.awt.Rectangle
	 */
	public void setMovieBox(int movie, Rectangle rect) {
		this.qtSetMovieBox(movie, rect.x, rect.y, rect.width, rect.height);
	}

	/**
	 * implements for interface.
	 *
	 * @param window java.awt.Component
	 * @return java.awt.Point
	 */
	public Point getMovieOffsetAwt(Component window) {
		String osName = System.getProperty("os.name");
		if (osName != null && osName.equals("Mac OS X")) {
			String javaVersion = System.getProperty("java.version");
			if (javaVersion != null && javaVersion.equals("1.4.1_01")) {
				Point point = new Point(0, 0);
				while (window != null && ! (window instanceof Window)) {
					Point offset = window.getLocation();
					point.translate(offset.x, offset.y);
					window = window.getParent();
				}
				return point;
			}
		}
		return new Point(0, 0);
	}

	/**
	 * implements for interface.
	 *
	 * @param window java.awt.Component
	 * @return java.awt.Point
	 */
	public Point getMovieOffsetSwing(Component window) {
		String osName = System.getProperty("os.name");
		if (osName != null && osName.equals("Mac OS X")) {
			String javaVersion = System.getProperty("java.version");
			if (javaVersion != null && ! javaVersion.equals("1.3.1")) {
				while (window != null) {
					if (window.getParent() instanceof Window) {
						return window.getLocation();
					}
					window = window.getParent();
				}
			}
		}
		return new Point(0, 0);
	}

	/**
	 * implements for interface.
	 *
	 * @param movie int
	 * @return boolean
	 */
	public boolean isMovieDone(int movie) {
		return this.qtIsMovieDone(movie);
	}

	/**
	 * implements for interface.
	 *
	 * @param movie int
	 * @param time int
	 * @param extent java.awt.Dimension
	 * @return java.awt.Image
	 */
	public Image getMovieImageAt(int movie, int time, Dimension extent) {
		int width = extent.width;
		int height = extent.height;
		int[] pixels = this.qtDrawMovieFrame(movie, time, width, height);
		if (pixels != null) {
			ColorModel colorModel = ColorModel.getRGBdefault();
			WritableRaster raster = colorModel.createCompatibleWritableRaster(width, height);
			BufferedImage image = new BufferedImage(colorModel, raster, false, null);
			image.setRGB(0, 0, width, height, pixels, 0, width);
			return image;
		}
		return null;
	}

	/**
	 * implements for interface.
	 *
	 * @param movie int
	 * @param start int
	 * @param duration int
	 * @return byte[]
	 */
	public byte[] convertMovieToSoundData(int movie, int start, int duration) {
		return qtConvertMovieToSoundData(movie, start, duration, kOffsetBinary, 1, 8, 0);
	}

	/**
	 * implements for interface.
	 *
	 * @param byteArray byte[]
	 * @param size int
	 * @return byte[][]
	 */
	public int[][] compactSoundData(byte[] byteArray, int size) {
		int[][] array = new int[2][size];
		for (int i = 0; i < size; ++i) {
			int startIndex = i * byteArray.length / size;
			int endIndex = (i + 1) * byteArray.length / size;
			int highValue = 128, lowValue = 128;
			for (int j = startIndex; j < endIndex; ++j) {
				int value = ((int) byteArray[j]) & 0xff;
				if (value > highValue) { highValue = value; }
				if (value < lowValue) { lowValue = value; }
			}
			array[0][i] = highValue;
			array[1][i] = lowValue;
		}
		return array;
	}

	/*
	 * Native Methods.
	 */

	public static native String GetVersion();

	/* --- Jun315WinQT --- */

	public native /* OSErr */ int qtInitializeQTML(
		/* long */ int flag);
	public native void qtTerminateQTML();

	public native /* void * */ int qtCreatePortAssociation1(
		/* HWND */ Component component,
		/* Ptr  */ int storage,
		/* long */ int flags);
	public native /* GrafPtr */ int qtCreatePortAssociation2(
		/* void * */ int viewPtr);
	public native void qtDestroyPortAssociation(
		/* void * */ int viewPtr);

	public native /* OSErr */ int qtEnterMovies();
	public native void qtExitMovies();

	public native void qtMoviesTask(
		/* Movie */ int movie,
		/* long  */ int maxMilliSecToUse);

	public native void qtPrerollMovie(
		/* Movie     */ int movie,
		/* TimeValue */ int time,
		/* Fixed     */ int rate);

	public native void qtSetMovieActive(int movie, boolean active);
	public native boolean qtGetMovieActive(int movie);

	public native void qtStartMovie(int movie);
	public native void qtStopMovie(int movie);

	public native void qtGoToBeginningOfMovie(int movie);
	public native void qtGoToEndOfMovie(int movie);

	public native boolean qtIsMovieDone(int movie);

	public native void qtShowMoviePoster(int movie);

	public native /* CGrafPtr, GDHandle */ int[] qtGetMovieGWorld(int movie);
	public native void qtSetMovieGWorld(
		/* Movie    */ int movie,
		/* CGrafPtr */ int port,
		/* GDHandle */ int gdh);

	public native /* OSErr */ int qtUpdateMovie(int movie);

	public native /* Rect */ int[] qtGetMovieBox(int movie);
	public native void qtSetMovieBox(
		/* Movie */ int movie,
		/* Rect  */ int x, int y, int width, int height);

	public native void qtDisposeMovie(int movie);

	public native /* TimeScale */ int qtGetMovieTimeScale(int movie);
	public native /* TimeValue */ int qtGetMovieDuration(int movie);

	public native /* Fixed */ int qtGetMovieRate(int movie);
	public native void qtSetMovieRate(
		/* Movie */ int movie,
		/* Fixed */ int rate);

	public native /* Fixed */ int qtGetMoviePreferredRate(int movie);

	public native /* short */ int qtGetMoviePreferredVolume(int movie);

	public native /* short */ int qtGetMovieVolume(int movie);
	public native void qtSetMovieVolume(
		/* Movie */ int movie,
		/* short */ int volume);

	public native /* TimeValue */ int qtGetMovieTimeValue(int movie);
	public native void qtSetMovieTimeValue(
		/* Movie     */ int movie,
		/* TimeValue */ int time);

	public native /* Movie */ int qtOpenMovie(String filename);

	public native String qtStandardGetFilePreview(
		/* OSType[] */ int[] typeList);

	/* --- Jun317WinQT --- */

	public native /* Pixel[] */ int[] qtDrawMovieFrame(
		/* Movie     */ int movie,
		/* TimeValue */ int time,
		/* short     */ int width,
		/* short     */ int height);

	/* --- Jun357WinQT --- */

	public native /* OSErr, short resRefNum, Movie */ int[] qtCreateMovieFile(
		/* FSSpec     */ String filename,
		/* OSType     */ int creator,
		/* ScriptCode */ int script,
		/* long       */ int flags);
	public native /* OSErr */ int qtAddMovieResource(
		/* Movie   */ int movie,
		/* short   */ int resRefNum,
		/* short * */ int resId,
		/* Str255  */ String resName);
	public native /* OSErr */ int qtCloseMovieFile(
		/* short */ int resRefNum);

	public native /* Track */ int qtNewMovieTrack(
		/* Movie */ int movie,
		/* Fixed */ int width,
		/* Fixed */ int height,
		/* short */ int trackVolume);
	public native /* OSErr */ int qtInsertMediaIntoTrack(
		/* Track     */ int track,
		/* TimeValue */ int trackStart,
		/* TimeValue */ int mediaTime,
		/* TimeValue */ int mediaDuration,
		/* Fixed     */ int mediaRate);

	public native /* Media */ int qtNewTrackMedia(
		/* Track     */ int track,
		/* OSType    */ int mediaType,
		/* TimeScale */ int timeScale,
		/* Handle    */ int dataRef,
		/* OSType    */ int dataRefType);
	public native /* OSErr */ int qtBeginMediaEdits(int media);
	public native /* OSErr */ int qtEndMediaEdits(int media);
	public native /* TimeValue */ int qtGetMediaDuration(int media);

	public native /* OSErr */ int qtAddMediaSampleImage(
		/* Media     */ int media,
		/* Pixel[]   */ int[] pixels,
		/* short     */ int width,
		/* short     */ int height,
		/* short     */ int destWidth,
		/* short     */ int destHeight,
		/* CodecQ    */ int quality,
		/* CodecType */ int codecType,
		/* TimeValue */ int duration);

	public native /* OSErr */ int qtGetMoviesError();
	public native /* Fixed */ int qtFixRatio(
		/* short */ int numer,
		/* short */ int denom);

	/* --- Jun366WinQT --- */

	public native /* TimeValue selectionTime, TimeValue selectionDuration */
			int[] qtGetMovieSelection(
		/* Movie */ int movie);
	public native void qtSetMovieSelection(
		/* Movie     */ int movie,
		/* TimeValue */ int selectionTime,
		/* TimeValue */ int selectionDuration);
	public native /* Movie */ int qtCopyMovieSelection(int movie);

	/* --- Jun368WinQT --- */

	public native /* Movie */ int qtNewMovie(
		/* long */ int flags);
	public native /* OSErr */ int qtInsertMovieSegment(
		/* Movie     */ int srcMovie,
		/* Movie     */ int dstMovie,
		/* TimeValue */ int srcIn,
		/* TimeValue */ int srcDuration,
		/* TimeValue */ int dstIn);
	public native /* Movie */ int qtFlattenMovieData(
		/* Movie      */ int movie,
		/* long       */ int movieFlattenFlags,
		/* FSSpec     */ String filename,
		/* OSType     */ int creator,
		/* ScriptCode */ int scriptTag,
		/* long       */ int createMovieFileFlags);

	/* --- Jun401WinQT --- */

	public native byte[] qtConvertMovieToSoundData(
		/* Movie         */ int movie,
		/* TimeValue     */ int start,
		/* TimeValue     */ int duration,
		/* OSType        */ int format,
		/* short         */ int numChannels,
		/* short         */ int sampleSize,
		/* UnsignedFixed */ int sampleRate);

	/* --- Jun404WinQT --- */

	public native /* Movie */ int qtNewMovieFromURL(String url);

	/* --- Jun422WinQT --- */

	public native /* TimeValue interestingTime, TimeValue interestingDuration */
			int[] qtGetMovieNextInterestingTime(
		/* Movie     */ int movie,
		/* short     */ int interestingTimeFlags,
		/* OSType[]  */ int[] mediaTypes,
		/* TimeValue */ int time,
		/* Fixed     */ int rate);

	/* --- Jun448WinQT --- */

	public native /* MovieController */ int qtNewMovieController(
		/* Movie */ int movie,
		/* Rect  */ int x, int y, int width, int height,
		/* long  */ int flags);
	public native void qtDisposeMovieController(
		/* MovieController */ int mc);

	public native /* ComponentResult */ int qtMCDoAction(
		/* MovieController */ int mc,
		/* short           */ int action,
		/* void *          */ int params);
	public native /* ComponentResult */ int qtMCIdle(
		/* MovieController */ int mc);
	public native /* ComponentResult */ int qtMCMovieChanged(
		/* MovieController */ int mc,
		/* Movie           */ int movie);
}
