package jp.co.sra.jun.vrml.support;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunVrmlScanner class
 * 
 *  @author    ASTI Beijing
 *  @created   UNKNOWN
 *  @updated   1999/06/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVrmlScanner.java,v 8.11 2008/02/20 06:33:17 nisinaka Exp $
 */
public abstract class JunVrmlScanner extends JunAbstractObject {
	//
	protected static final char CR = '\r';
	protected static final char LF = '\n';
	protected StReadStream source = null;
	protected int mark;
	protected Object token = null;
	protected char hereChar = 0;
	protected StSymbol tokenType;
	protected JunVrmlKeywordTable keywordTable = null;
	protected JunVrmlScannerTable typeTable = null;
	protected StBlockClosure failBlock = null;

	/**
	 * Create a new instance of <code>JunVrmlScanner</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunVrmlScanner() {
		super();
		this.initScanner();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aString DOCUMENT ME!
	 * 
	 * @return DOCUMENT ME!
	 */
	public Integer hexNumberFromString_(String aString) {
		int hexNumber;
		int stringSize;
		stringSize = aString.length();

		int nDigitNumber;
		int nPowerIndex;
		int nIndex;
		int nIntNumber = 0;
		String strToken;
		strToken = aString;

		for (nIndex = 0; nIndex < stringSize; nIndex++) {
			switch (strToken.charAt(nIndex)) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					nDigitNumber = strToken.charAt(nIndex) - '0';
					break;
				case 'A':
				case 'B':
				case 'C':
				case 'D':
				case 'E':
				case 'F':
					nDigitNumber = strToken.charAt(nIndex) - 'A' + 10;
					break;
				default:
					nDigitNumber = 0;
					break;
			}

			for (nPowerIndex = 0; nPowerIndex < (stringSize - 1 - nIndex); nPowerIndex++) {
				nDigitNumber = nDigitNumber * 16;
			}

			nIntNumber += nDigitNumber;
		}

		hexNumber = nIntNumber;

		return new Integer(hexNumber);
	}

	/**
	 * Initialize the scanner.
	 * 
	 * @category initialize-release
	 */
	protected void initScanner() {
		typeTable = this.defaultScannerTable();
		keywordTable = this.defaultKeywordTable();
		failBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + "near " + token.toString();
				String string = source.upToEnd().toString();

				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					if (string.length() > 30) {
						string = "--> " + string.substring(0, 30);
					} else {
						string = "-->" + string;
					}
				}

				throw SmalltalkException.Error(label + " : " + string);
			}
		};

	}

	/**
	 * Get multi characters of the type.
	 * 
	 * @param type jp.co.sra.smalltalk.StSymbol
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 */
	public void multiChar_(StSymbol type) {
		try {
			this.perform_(type.toString());
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Answer the next character.
	 * 
	 * @return char
	 */
	public char nextChar() {
		char ch = source.next();

		if (ch == LF) {
			return CR;
		}

		if (ch == CR) {
			if (source.peek() == LF) {
				source.next();
			}

			return CR;
		}

		if (source.atEnd()) {
			return typeTable.endChar();
		}

		return ch;
	}

	/**
	 * Get the next token.
	 * 
	 * @return java.lang.Object
	 */
	public Object nextToken() {
		mark = source.position();

		char ch = this.peekChar();
		tokenType = (StSymbol) typeTable.at(ch);

		while (tokenType == $("xDelimiter")) {
			this.nextChar();
			ch = this.peekChar();
			tokenType = (StSymbol) typeTable.at(ch);
		}

		ch = tokenType.toString().charAt(0);

		if (ch == 'x') {
			this.multiChar_(tokenType);
		} else {
			this.singleChar_(tokenType);
		}

		return token;
	}

	/**
	 * Read a number from the stream.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * 
	 * @return java.lang.String
	 */
	public String numberFrom_(StReadStream aStream) {
		StringBuffer buffer = new StringBuffer();

		while ((hereChar = aStream.next()) != 0) {
			if ((hereChar < '0') || (hereChar > '9')) {
				break;
			}

			buffer.append(hereChar);
		}

		token = buffer.toString();

		return (String) token;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @param inputStream jp.co.sra.smalltalk.StReadStream
	 */
	public void on_(StReadStream inputStream) {
		source = inputStream;
		mark = source.position();
	}

	/**
	 * Peek the next character.
	 * 
	 * @return char
	 */
	public char peekChar() {
		char ch = source.peek();

		if (ch == LF) {
			return CR;
		}

		if (ch == CR) {
			return CR;
		}

		if (ch == 0) {
			return typeTable.endChar();
		}

		return ch;
	}

	/**
	 * Get a single character.
	 * 
	 * @param type jp.co.sra.smalltalk.StSymbol
	 */
	public void singleChar_(StSymbol type) {
		this.nextChar();
		token = type;
	}

	/**
	 * Skip the line-splicing \.
	 * 
	 * @return char
	 */
	public char step() {
		char ch;

		while (true) {
			ch = hereChar;
			hereChar = source.next();

			if ((hereChar == '\\') && (source.peek() == CR)) {
				continue;
			}

			break;
		}

		return ch;
	}

	/**
	 * Read a string from the stream.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * 
	 * @return java.lang.String
	 */
	public String stringFrom_(StReadStream aStream) {
		StringBuffer buffer = new StringBuffer();
		char ch = aStream.next();

		if (ch == '"') {
			while ((ch = aStream.peek()) != 0) {
				if (ch == '\\') {
					aStream.next();
					ch = aStream.peek();

					if (ch != '"') {
						buffer.append('\\');
					}
				} else {
					if (ch == '"') {
						aStream.next();

						return buffer.toString();
					}
				}

				buffer.append(aStream.next());
			}
		}

		String string = aStream.upToEnd();

		if (string.length() > 100) {
			string = string.substring(0, 100);
		}

		try {
			return (String) failBlock.value_("Syntax error unmatched $\"");
		} catch (Exception e) {
			this.error_("failBlock value error");

			return "";
		}
	}

	/**
	 * Get a symbol from the stream.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 */
	public StSymbol symbolFrom_(StReadStream aStream) {
		StringBuffer buffer = new StringBuffer();
		char ch = aStream.peek();

		while (ch != 0) {
			StSymbol type = (StSymbol) typeTable.at(ch);

			if ((type == $("xSymbol")) || (type == $("xDigit")) || (type == $("xSign"))) {
				buffer.append(aStream.next());
				ch = aStream.peek();
			} else {
				break;
			}
		}

		if (buffer.toString().compareTo("nil") == 0) {
			return null;
		}

		return $(buffer.toString());
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.lang.Object
	 */
	public Object token() {
		return token();
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 */
	public StSymbol tokenType() {
		return tokenType;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 */
	public void unNextChar() {
		source.skip_(-1);
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 */
	public void unNextToken() {
		source.position_(mark);
	}

	/**
	 * Scan binary characters.
	 */
	public void xBinary() {
		failBlock.value_("Syntax error : " + source.peek());
	}

	/**
	 * Scan a comment.
	 */
	public void xComment() {
		char ch = this.nextChar();

		while (ch != CR) {
			if (ch == typeTable.endChar()) {
				this.nextToken();

				return;
			}

			ch = this.nextChar();
		}

		this.nextToken();

		return;
	}

	/**
	 * Scan a digit.
	 */
	public void xDigit() {
		String strToken = this.numberFrom_(source);

		if ((strToken.compareTo("0") == 0) && (hereChar == 'x')) {
			this.xHexadecimal();

			return;
		}

		if ((strToken.charAt(0) == '0') && (hereChar != '.') && (strToken.length() != 1)) {
			this.xOctal();

			return;
		}

		if (hereChar != 0) {
			if (Character.toUpperCase(hereChar) == 'U') {
				hereChar = source.next();

				if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'L')) {
					hereChar = source.next();
				}
			} else {
				if (Character.toUpperCase(hereChar) == 'L') {
					hereChar = source.next();

					if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'U')) {
						hereChar = source.next();
					}
				}
			}
		}

		if (hereChar != 0) {
			switch (hereChar) {
				case '.':
				case 'e':
				case 'E':
				case 'd':
				case 'D':
					token = this.xLimitedPrecisionReal_(strToken);
					break;
				default:
					token = new Integer(strToken);
					break;
			}
		}

		tokenType = $("number");
		this.unNextChar();
	}

	/**
	 * Scan a double quote.
	 */
	public void xDoubleQuote() {
		tokenType = $("string");
		token = this.stringFrom_(source);
	}

	/**
	 * Scan a hexadecimal value.
	 */
	public void xHexadecimal() {
		StringBuffer buffer = new StringBuffer();

		while (true) {
			boolean isHexDigit = false;
			hereChar = source.next();

			if (hereChar == 0) {
				break;
			}

			switch (hereChar) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'A':
				case 'B':
				case 'C':
				case 'D':
				case 'E':
				case 'F':
					isHexDigit = true;
					buffer.append(hereChar);

					break;
				default:
					break;
			}

			if (!isHexDigit) {
				break;
			}
		}

		if (hereChar != 0) {
			if (Character.toUpperCase(hereChar) == 'U') {
				hereChar = source.next();

				if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'L')) {
					hereChar = source.next();
				}
			}
		} else {
			if (Character.toUpperCase(hereChar) == 'L') {
				hereChar = source.next();

				if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'U')) {
					hereChar = source.next();
				}
			}
		}

		token = this.hexNumberFromString_(buffer.toString());
		tokenType = $("number");

		if (hereChar == 0) {
			this.step();
		} else {
			this.unNextChar();
		}
	}

	/**
	 * Scan a limited precisional real value.
	 * 
	 * @param prefixToken java.lang.String
	 * 
	 * @return java.lang.Number
	 */
	public Number xLimitedPrecisionReal_(String prefixToken) {
		char prevChar;
		StringBuffer realToken = new StringBuffer(prefixToken);

		if ((hereChar == '.') || (realToken.length() == 0)) {
			boolean fractionExists;
			fractionExists = false;

			if (hereChar == '.') {
				prevChar = hereChar;
				realToken.append(hereChar);
				hereChar = source.next();
			} else {
				realToken = new StringBuffer("0.");
			}

			while ((hereChar != 0) && (hereChar >= '0') && (hereChar <= '9')) {
				fractionExists = true;
				realToken.append(hereChar);
				hereChar = source.next();
			}

			if (!fractionExists) {
				realToken.append('0');
			}
		}

		if ((hereChar != 0) && ((hereChar == 'e') || (hereChar == 'E') || (hereChar == 'd') || hereChar == 'D')) {
			realToken.append('e');
			prevChar = hereChar;
			hereChar = source.next();

			if ((hereChar != 0) && (hereChar == '+' || hereChar == '-') && (hereChar != '+' || !(prevChar == 'd' || prevChar == 'D'))) {
				if (hereChar != '+') {
					realToken.append(hereChar);
				}

				hereChar = source.next();
			}

			while ((hereChar != 0) && (hereChar >= '0' && hereChar <= '9')) {
				realToken = realToken.append(hereChar);
				hereChar = source.next();
			}
		}

		if ((hereChar != 0) && (hereChar == 'f' || hereChar == 'F')) {
			hereChar = source.next();

			return new Double(realToken.toString());
		} else {
			if ((hereChar != 0) && (hereChar == 'l' || hereChar == 'L')) {
				hereChar = source.next();
			}

			return new Double(realToken.toString());
		}
	}

	/**
	 * Scan an octal value.
	 */
	public void xOctal() {
		if (hereChar != 0) {
			if (Character.toUpperCase(hereChar) == 'U') {
				hereChar = source.next();

				if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'L')) {
					hereChar = source.next();
				}
			} else {
				if (Character.toUpperCase(hereChar) == 'L') {
					hereChar = source.next();

					if ((hereChar != 0) && (Character.toUpperCase(hereChar) == 'U')) {
						hereChar = source.next();
					}
				}
			}
		}

		String strToken = (String) token;
		int tokenSize = strToken.length();

		for (int nIndex = 0; nIndex < tokenSize; nIndex++) {
			if ((strToken.charAt(nIndex) < '0') || (strToken.charAt(nIndex) > '7')) {
				this.error_("Expected Octal Constant");
			}
		}

		int nDigitNumber;
		int nPowerIndex;
		int nIntNumber = 0;

		for (int nIndex = 0; nIndex < tokenSize; nIndex++) {
			nDigitNumber = strToken.charAt(nIndex) - '0';

			for (nPowerIndex = 0; nPowerIndex < (tokenSize - 1 - nIndex); nPowerIndex++) {
				nDigitNumber = nDigitNumber * 8;
			}

			nIntNumber += nDigitNumber;
		}

		token = new Integer(nIntNumber);
		tokenType = $("number");
		this.unNextChar();
	}

	/**
	 * Scan a period.
	 */
	public void xPeriod() {
		hereChar = this.nextChar();

		char ch = this.peekChar();

		if (Character.isDigit(ch)) {
			token = this.xLimitedPrecisionReal_("");
			tokenType = $("number");

			return;
		}

		token = new Character('.');
		tokenType = $("period");
	}

	/**
	 * Scan a signed number.
	 */
	public void xSign() {
		char sign = this.nextChar();
		char ch = this.peekChar();

		if (Character.isDigit(ch)) {
			this.xDigit();

			if (sign == '-') {
				Number number = (Number) token;

				if (token instanceof Double) {
					token = new Double(-number.doubleValue());
				}

				if (token instanceof Float) {
					token = new Float(-number.floatValue());
				}

				if (token instanceof Integer) {
					token = new Integer(-number.intValue());
				}

				if (token instanceof Long) {
					token = new Long(-number.longValue());
				}
			}

			return;
		}

		if (ch == '.') {
			this.xPeriod();

			if ((tokenType == $("number")) && (sign == '-')) {
				Number number = (Number) token;

				if (token instanceof Double) {
					token = new Double(-number.doubleValue());
				}

				if (token instanceof Float) {
					token = new Float(-number.floatValue());
				}

				if (token instanceof Integer) {
					token = new Integer(-number.intValue());
				}

				if (token instanceof Long) {
					token = new Long(-number.longValue());
				}
			}

			return;
		}

		this.unNextChar();
		tokenType = $("symbol");
		token = this.symbolFrom_(source);
	}

	/**
	 * Scan a symbol.
	 */
	public void xSymbol() {
		token = this.symbolFrom_(source);

		if (keywordTable.hasSymbol_((StSymbol) token)) {
			tokenType = (StSymbol) token;
		} else {
			tokenType = $("identifier");
		}
	}

	/**
	 * Answer the default keyword table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlKeywordTable
	 */
	protected abstract JunVrmlKeywordTable defaultKeywordTable();

	/**
	 * Answer the default scanner table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlScannerTable
	 */
	protected abstract JunVrmlScannerTable defaultScannerTable();
}
