package jp.co.sra.jun.voronoi.twoD.diagram;

import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;

/**
 * JunVoronoi2dLine class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/01/25 (by NISHIHARA Satoshi)
 *  @updated   2006/11/07 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVoronoi2dLine.java,v 8.10 2008/02/20 06:33:14 nisinaka Exp $
 */
public class JunVoronoi2dLine extends JunVoronoi2dObject {

	protected double a;
	protected double b;
	protected double c;

	/**
	 * Create a new instance of JunVoronoi2dLine and initialize it.
	 *
	 * @param a double
	 * @param b double
	 * @param c double
	 * @category Instance creation
	 */
	public JunVoronoi2dLine(double a, double b, double c) {
		this.a_(a);
		this.b_(b);
		this.c_(c);
	}

	/**
	 * Create a new instance of JunVoronoi2dLine and initialize it.
	 *
	 * @param from jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @param to jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDot
	 * @category Instance creation
	 */
	public JunVoronoi2dLine(JunVoronoi2dDot from, JunVoronoi2dDot to) {
		double x1 = from.x();
		double y1 = from.y();
		double x2 = to.x();
		double y2 = to.y();
		double a = y1 - y2;
		double b = x2 - x1;
		double c = x1 * y2 - x2 * y1;
		if (Math.abs(a) < this.accuracy() && Math.abs(b) < this.accuracy()) {
			throw SmalltalkException.Error("two points are the same.");
		}

		this.a_(a);
		this.b_(b);
		this.c_(c);
	}

	/**
	 * Create a new instance of JunVoronoi2dLine and initialize it.
	 *
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @category Instance creation
	 */
	public JunVoronoi2dLine(Point fromPoint, Point toPoint) {
		this(new JunVoronoi2dDot(fromPoint), new JunVoronoi2dDot(toPoint));
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		a = Double.NaN;
		b = Double.NaN;
		c = Double.NaN;
	}

	/**
	 * Answer my current a value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double a() {
		return a;
	}

	/**
	 * Set my new a value.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void a_(double aNumber) {
		a = aNumber;
	}

	/**
	 * Answer my current b value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double b() {
		return b;
	}

	/**
	 * Set my new b value.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void b_(double aNumber) {
		b = aNumber;
	}

	/**
	 * Answer my current b value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double c() {
		return c;
	}

	/**
	 * Set my new c value.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void c_(double aNumber) {
		c = aNumber;
	}

	/**
	 * Answer the point at the specified x-coordinate.
	 * 
	 * @param x double
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @category accessing
	 */
	public JunVoronoi2dPoint pointAtX_(double x) {
		double y = (-c - a * x) / b;
		return new JunVoronoi2dPoint(x, y);
	}

	/**
	 * Answer the point at the specified y-coordinate.
	 * 
	 * @param y double
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @category accessing
	 */
	public JunVoronoi2dPoint pointAtY_(double y) {
		double x = (-c - b * y) / a;
		return new JunVoronoi2dPoint(x, y);
	}

	/**
	 * Answer the intersecting point with the specified line.
	 * 
	 * @param voronoi2dLine jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dLine
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dPoint
	 * @category functions 
	 */
	public JunVoronoi2dPoint intersectingPointWithLine_(JunVoronoi2dLine voronoi2dLine) {
		double a1 = this.a();
		double b1 = this.b();
		double c1 = this.c();
		double a2 = voronoi2dLine.a();
		double b2 = voronoi2dLine.b();
		double c2 = voronoi2dLine.c();
		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < this.accuracy()) {
			throw SmalltalkException.Error("two lines are parallel.");
		}
		double dinv = 1.0 / det;
		double x = (b1 * c2 - b2 * c1) * dinv;
		double y = (a2 * c1 - a1 * c2) * dinv;
		return new JunVoronoi2dPoint(x, y);
	}

	/**
	 * Answer true if the receiver is horizontal.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isHorizontal() {
		return Math.abs(a) < this.accuracy();
	}

	/**
	 * Answer true if the receiver is vertical.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isVertical() {
		return Math.abs(b) < this.accuracy();
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("line ( ");
		aWriter.write(String.valueOf(a));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(b));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(c));
		aWriter.write(" )");
	}

}
