package jp.co.sra.jun.topology.setoperators;

import java.awt.Color;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.collections.linkedlist.JunSingleLinkedList;
import jp.co.sra.jun.geometry.abstracts.JunCurve;
import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.Jun3dPolygon;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.topology.abstracts.JunSetOperator;
import jp.co.sra.jun.topology.elements.JunBody;
import jp.co.sra.jun.topology.elements.JunEdge;
import jp.co.sra.jun.topology.elements.JunLoop;
import jp.co.sra.jun.topology.elements.JunVertex;
import jp.co.sra.jun.topology.euleroperators.JunKDEV;
import jp.co.sra.jun.topology.euleroperators.JunKEL;
import jp.co.sra.jun.topology.euleroperators.JunKEML;
import jp.co.sra.jun.topology.euleroperators.JunKEV;
import jp.co.sra.jun.topology.euleroperators.JunKEVVL;
import jp.co.sra.jun.topology.euleroperators.JunKVE;
import jp.co.sra.jun.topology.euleroperators.JunMDEV;
import jp.co.sra.jun.topology.euleroperators.JunMEKL;
import jp.co.sra.jun.topology.euleroperators.JunMEL;
import jp.co.sra.jun.topology.euleroperators.JunMEV;
import jp.co.sra.jun.topology.euleroperators.JunMEVVL;
import jp.co.sra.jun.topology.euleroperators.JunMVE;
import jp.co.sra.jun.topology.globaloperators.JunADD;

/**
 * JunUNION class
 * 
 *  @author    ASTI Shanghai
 *  @created   UNKNOWN
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   1999/08/05 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun627 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunUNION.java,v 8.11 2008/02/20 06:33:13 nisinaka Exp $
 */
public class JunUNION extends JunSetOperator {

	public static final double ACCURACY = JunGeometry.ACCURACY * Math.pow(10, Math.max(0, Math.round(Math.log(1 / JunGeometry.ACCURACY) / Math.log(10) / 2 - 2)));

	/** Another JunBody on which the receiver operate. */
	protected JunBody body2 = null;

	/** Confliction bounds. */
	protected Jun3dBoundingBox conflictionBounds = null;

	/**
	 * The hashtable for vertexes on border. Its key is a kind of
	 * JunTopologicalElement. Its value is a Vector which contains
	 * JunTopologicalElements.
	 */
	protected Hashtable vertexesOnBorder = null;
	protected Vector _vertexesOnBorderKeys = null;

	/**
	 * The hashtable for vertexes on surface. Its key is a instance of
	 * JunVertex. Its value is a instance of JunSurface.
	 */
	protected Hashtable vertexesOnSurface = null;

	/** Confliction edges. */
	protected Vector conflictionEdges = null;

	/**
	 * The hashtable for surfaces. Its key is a instance of JunLoop. Its value
	 * is a kind of JunSurface.
	 */
	protected Hashtable surfaces = null;

	/**
	 * Create a new instance of JunUNION and initialize it with two bodies.
	 * 
	 * @param aBody1 jp.co.sra.jun.topology.elements.JunBody
	 * @param aBody2 jp.co.sra.jun.topology.elements.JunBody
	 */
	public JunUNION(JunBody aBody1, JunBody aBody2) {
		this.body_(aBody1);
		this.body2_(aBody2);
	}

	/**
	 * Create an instance of JunUNION with two bodies.
	 * 
	 * @param aBody1 jp.co.sra.jun.topology.elements.JunBody
	 * @param aBody2 jp.co.sra.jun.topology.elements.JunBody
	 * 
	 * @return jp.co.sra.jun.topology.setoperators.JunUNION
	 */
	public static final JunUNION Body_body_(JunBody aBody1, JunBody aBody2) {
		return new JunUNION(aBody1, aBody2);
	}

	/**
	 * Add the JunBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * 
	 * @return jp.co.sra.jun.topology.globaloperators.JunADD
	 */
	public JunADD addBody_(JunBody aBody) {
		JunADD add = new JunADD(body, aBody);
		add.doOperation();
		suboperators.addElement(add);

		return add;
	}

	/**
	 * Answer the second body of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 */
	public JunBody body2() {
		return body2;
	}

	/**
	 * Set the second body of the receiver.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 */
	public void body2_(JunBody aBody) {
		body2 = aBody;
	}

	/**
	 * Display the shells.
	 */
	public void displayShells() {
		JunLoop[][] shells = this.shells();

		for (int i = 0; i < shells.length; i++) {
			JunLoop[] loops = shells[i];
			Vector collection = new Vector(loops.length);

			for (int j = 0; j < loops.length; j++) {
				JunLoop loop = loops[i];
				JunOpenGL3dObject a3dObject = loop.asJunOpenGL3dObject();
				a3dObject.paint_(Color.blue);
				collection.addElement(a3dObject);
			}

			(new JunOpenGL3dCompoundObject(collection)).show();
		}
	}

	/**
	 * Execute the operation.
	 */
	public void execute() {
		if (suboperators == null) {
			suboperators = new Vector();
			surfaces = new Hashtable();

			final JunUNION self = this;
			final JunProgress progress = new JunProgress();
			progress.message_(JunSystem.$String("Executing set operation..."));
			progress.do_(new StBlockClosure() {
				public Object value() {
					self.createVertexesOnBorderWithProgress_from_to_(progress, 0.0f, 0.4f);
					self.addBody_((JunBody) self.body2());
					self.traceConflictionsWithProgress_from_to_(progress, 0.4f, 0.5f);
					progress.value_(0.5f);
					self.compressConflictionEdges();
					self.removeInsideEdgesWithProgress_from_to_(progress, 0.5f, 0.6f);
					self._flushVertexesOnBorder();
					self.unifySharedLoopsWithProgress_from_to_(progress, 0.6f, 0.9f);
					self.killInsideShellsWithProgress_from_to_(progress, 0.9f, 1.0f);
					surfaces = null;
					self._flushVertexesOnBorder();
					vertexesOnSurface = null;

					return null;
				}
			});
		} else {
			this.redo();
		}
	}

	/**
	 * Answer the suboperators.
	 * 
	 * @return suboperators
	 */
	public Vector suboperators() {
		return suboperators;
	}

	/**
	 * Flush the vertexesOnBorder.
	 */
	protected void _flushVertexesOnBorder() {
		vertexesOnBorder = null;
		_vertexesOnBorderKeys = null;
	}

	/**
	 * Initialize the vertexesOnBorder.
	 */
	protected void _initializeVertexesOnBorder() {
		vertexesOnBorder = new Hashtable();
		_vertexesOnBorderKeys = new Vector();
	}

	/**
	 * Put the key and the value to the vertexesOnBorder.
	 * 
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 */
	protected void _vertexesOnBorderPut(Object key, Object value) {
		vertexesOnBorder.put(key, value);

		if (_vertexesOnBorderKeys.contains(key) == false) {
			_vertexesOnBorderKeys.addElement(key);
		}
	}

	/**
	 * Remove the key from the vertexesOnBorder.
	 * 
	 * @param key java.lang.Object
	 */
	protected void _vertexesOnBorderRemove(Object key) {
		vertexesOnBorder.remove(key);
		_vertexesOnBorderKeys.removeElement(key);
	}

	/**
	 * Answer the basic surface of the Loop which contained in the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 */
	private JunSurface basicSurfaceOf_(JunLoop aLoop) {
		JunSurface aSurface = (JunSurface) surfaces.get(aLoop);

		if (aSurface == null) {
			aSurface = aLoop.surface();
		}

		return aSurface;
	}

	/**
	 * Remove the surface which corresponds to the Loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 */
	private void changedSurfaceOf_(JunLoop aLoop) {
		if (aLoop != null) {
			surfaces.remove(aLoop);
		}
	}

	/**
	 * Change the surface which corresponds to the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aSurface jp.co.sra.jun.geometry.abstracts.JunSurface
	 */
	private void changedSurfaceOf_as_(JunLoop aLoop, JunSurface aSurface) {
		surfaces.put(aLoop, aSurface);
	}

	/**
	 * Compress the confliction edges.
	 */
	private void compressConflictionEdges() {
		Vector newConflictionEdges = new Vector(conflictionEdges.size());

		for (int i = 0; i < conflictionEdges.size(); i++) {
			JunEdge edge = (JunEdge) conflictionEdges.elementAt(i);
			JunEdge[] leftLoopEdges = edge.leftLoop().edges();
			boolean confliction = false;

			for (int j = 0; j < leftLoopEdges.length; j++) {
				if (conflictionEdges.contains(leftLoopEdges[j]) == false) {
					confliction = true;

					break;
				}
			}

			if (confliction) {
				newConflictionEdges.addElement(edge);

				continue;
			}

			JunEdge[] rightLoopEdges = edge.rightLoop().edges();

			for (int j = 0; j < rightLoopEdges.length; j++) {
				if (conflictionEdges.contains(rightLoopEdges[j]) == false) {
					confliction = true;

					break;
				}
			}

			if (confliction) {
				newConflictionEdges.addElement(edge);
			}
		}

		conflictionEdges = newConflictionEdges;

		//
		StBlockClosure tipVertexBlock = new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				JunVertex vertex = (JunVertex) anObject1;
				JunEdge edge = (JunEdge) anObject2;
				JunEdge[] edges = vertex.edges();

				for (int i = 0; i < edges.length; i++) {
					if ((edges[i] != edge) && conflictionEdges.contains(edges[i])) {
						return Boolean.FALSE;
					}
				}

				return Boolean.TRUE;
			}
		};

		//
		Vector compressedConflictionEdges = conflictionEdges;

		do {
			conflictionEdges = compressedConflictionEdges;

			//
			Vector tipVertexes = new Vector();
			JunEdge[] conflictionEdgesArray = new JunEdge[conflictionEdges.size()];
			conflictionEdges.copyInto(conflictionEdgesArray);

			JunEdge edge = null;

			for (int i = 0; i < conflictionEdgesArray.length; i++) {
				edge = conflictionEdgesArray[i];

				if (tipVertexBlock.value_value_(edge.startVertex(), edge) == Boolean.TRUE) {
					tipVertexes.addElement(edge.startVertex());
				}

				if (tipVertexBlock.value_value_(edge.endVertex(), edge) == Boolean.TRUE) {
					tipVertexes.addElement(edge.endVertex());
				}

				if (tipVertexes.isEmpty() == false) {
					throw SmalltalkException.Error("internal error");
				}
			}

			compressedConflictionEdges = new Vector(conflictionEdgesArray.length);

			for (int j = 0; j < conflictionEdgesArray.length; j++) {
				edge = conflictionEdgesArray[j];

				if ((tipVertexBlock.value_value_(edge.startVertex(), edge) == Boolean.FALSE) && (tipVertexBlock.value_value_(edge.endVertex(), edge) == Boolean.FALSE)) {
					compressedConflictionEdges.addElement(edge);
				}
			}
		} while (compressedConflictionEdges.size() != conflictionEdges.size());
	}

	/**
	 * Check for the confliction vertex pair. The result JunVertex array should
	 * contain two elements.
	 * 
	 * @param aJunLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex[]
	 */
	private JunVertex[] conflictionVertexPairOnLoop_withLoop_fromVertex_vertex_(JunLoop aJunLoop1, JunLoop aJunLoop2, JunVertex aJunVertex1, JunVertex aJunVertex2) {
		JunVertex[] vertexPair = new JunVertex[2];

		vertexPair[0] = this.nextConflictingVertexOn_with_fromVertex_(aJunLoop1, aJunLoop2, aJunVertex1);

		if (vertexPair[0] != null) {
			vertexPair[1] = this.twinVertexOf_on_connectiveFrom_(vertexPair[0], aJunLoop2, aJunVertex2);

			return vertexPair;
		}

		vertexPair[1] = this.nextConflictingVertexOn_with_fromVertex_(aJunLoop2, aJunLoop1, aJunVertex2);

		if (vertexPair[1] != null) {
			vertexPair[0] = this.twinVertexOf_on_connectiveFrom_(vertexPair[1], aJunLoop1, aJunVertex1);

			return vertexPair;
		}

		return null;
	}

	/**
	 * Connect two vertexes on the loop.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 */
	private JunVertex connectVertex_withVertex_onLoop_(JunVertex aVertex1, JunVertex aVertex2, JunLoop aLoop) {
		if (aVertex1.hasEdgeToVertex_(aVertex2)) {
			return aVertex1;
		}

		if (aVertex1.isKilled() && aVertex2.isKilled()) {
			return this.mevvlPoint_point_(aVertex1.point(), aVertex2.point()).newVertex1();
		}

		if (aVertex1.isKilled()) {
			if (aVertex2.hasLoop_(aLoop)) {
				return this.mevVertex_loop_point_(aVertex2, aLoop, aVertex1.point()).newVertex();
			} else {
				return this.mevVertex_loop_point_(aVertex2, aVertex2.loop(), aVertex1.point()).newVertex();
			}
		}

		if (aVertex2.isKilled()) {
			if (aVertex1.hasLoop_(aLoop)) {
				this.mevVertex_loop_point_(aVertex1, aLoop, aVertex2.point());

				return aVertex1;
			} else {
				this.mevVertex_loop_point_(aVertex1, aVertex1.loop(), aVertex2.point());

				return aVertex1;
			}
		}

		if (aVertex1.hasLoop_(aLoop) && aVertex2.hasLoop_(aLoop)) {
			this.melLoop_vertex_vertex_(aLoop, aVertex1, aVertex2);

			return aVertex1;
		}

		if (aVertex1.hasLoop_(aLoop)) {
			this.meklKillLoop_aliveLoop_vertex_vertex_(aVertex2.loop(), aLoop, aVertex2, aVertex1);

			return aVertex1;
		}

		if (aVertex2.hasLoop_(aLoop)) {
			this.meklKillLoop_aliveLoop_vertex_vertex_(aVertex1.loop(), aLoop, aVertex1, aVertex2);

			return aVertex1;
		}

		JunLoop commonLoop = aVertex1.commonLoopWithVertex_(aVertex2);

		if (commonLoop != null) {
			this.melLoop_vertex_vertex_(commonLoop, aVertex1, aVertex2);
		} else {
			this.meklKillLoop_aliveLoop_vertex_vertex_(aVertex1.loop(), aVertex2.loop(), aVertex1, aVertex2);
		}

		return aVertex1;
	}

	/**
	 * Create the vertexesOnBorder.  While executing, show JunProgress.
	 * 
	 * @param aJunProgress jp.co.sra.jun.goodies.progress.JunProgress
	 * @param aNumber1 float
	 * @param aNumber2 float
	 */
	private void createVertexesOnBorderWithProgress_from_to_(JunProgress aJunProgress, float aNumber1, float aNumber2) {
		conflictionBounds = this.body().boundingBox().intersect_(this.body2().boundingBox()).expandedBy_(ACCURACY * 10.0d);

		JunEdge[] edges1Array = this.body().edges();
		JunEdge[] edges2Array = this.body2().edges();
		JunLoop[] loops1 = this.body().loops();
		JunLoop[] loops2 = this.body2().loops();
		Vector edges1 = new Vector(edges1Array.length);
		Vector curves1 = new Vector(edges1Array.length);
		Vector curveBounds1 = new Vector(edges1Array.length);

		for (int i = 0; i < edges1Array.length; i++) {
			edges1.addElement(edges1Array[i]);

			Jun3dLine curve = (Jun3dLine) edges1Array[i].curve();
			curves1.addElement(curve);
			curveBounds1.addElement(curve.boundingBox());
		}

		Vector edges2 = new Vector(edges2Array.length);
		Vector curves2 = new Vector(edges2Array.length);
		Vector curveBounds2 = new Vector(edges2Array.length);

		for (int i = 0; i < edges2Array.length; i++) {
			edges2.addElement(edges2Array[i]);

			Jun3dLine curve = (Jun3dLine) edges2Array[i].curve();
			curves2.addElement(edges2Array[i].curve());
			curveBounds2.addElement(curve.boundingBox());
		}

		this._initializeVertexesOnBorder();
		vertexesOnSurface = new Hashtable();

		int numLoops = loops1.length + loops2.length;
		int progressIndex = 0;

		for (int i = 0; i < loops1.length; i++) {
			JunLoop loop = loops1[i];
			aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * progressIndex / numLoops));
			progressIndex++;

			int edgeIndex = 0;
			Jun3dPolygon surface = (Jun3dPolygon) loop.surface();

			while (edgeIndex < edges2.size()) {
				JunEdge edge = (JunEdge) edges2.elementAt(edgeIndex);
				JunCurve curve = (Jun3dLine) curves2.elementAt(edgeIndex);
				Jun3dPoint intersectingPoint = null;

				if (surface.boundingBox().intersectsOrTouches_((Jun3dBoundingBox) curveBounds2.elementAt(edgeIndex)) && ((intersectingPoint = surface.intersectingPointWithLineSegment_((Jun3dLine) curve)) != null)) {
					JunVertex theVertex = null;

					if (intersectingPoint.distance_(edge.startVertex().point()) < ACCURACY) {
						theVertex = edge.startVertex();
					}

					if (intersectingPoint.distance_(edge.endVertex().point()) < ACCURACY) {
						theVertex = edge.endVertex();
					}

					if (theVertex == null) {
						JunMVE mve = this.mveBody_edge_point_(this.body2(), edge, intersectingPoint);
						theVertex = mve.newVertex();

						JunCurve newCurve = mve.newEdge().curve();
						edges2.addElement(mve.newEdge());
						curves2.addElement(newCurve);
						curveBounds2.addElement(((Jun3dLine) newCurve).boundingBox());
						curve = edge.curve();
						curves2.setElementAt(curve, edgeIndex);
						curveBounds2.setElementAt(((Jun3dLine) curve).boundingBox(), edgeIndex);
					}

					surfaces.put(loop, surface);
					vertexesOnSurface.put(theVertex, surface);

					Vector aVector = (Vector) vertexesOnBorder.get(loop);

					if (aVector == null) {
						aVector = new Vector();
						this._vertexesOnBorderPut(loop, aVector);
					}

					aVector.addElement(theVertex);
				}

				edgeIndex++;
			}
		}

		for (int i = 0; i < loops2.length; i++) {
			JunLoop loop = loops2[i];
			aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * progressIndex / numLoops));
			progressIndex++;

			int edgeIndex = 0;
			Jun3dPolygon surface = (Jun3dPolygon) loop.surface();

			while (edgeIndex < edges1.size()) {
				JunEdge edge = (JunEdge) edges1.elementAt(edgeIndex);
				JunCurve curve = (Jun3dLine) curves1.elementAt(edgeIndex);
				Jun3dPoint intersectingPoint = null;

				if (((Jun3dBoundingBox) curveBounds1.elementAt(edgeIndex)).intersectsOrTouches_(surface.boundingBox()) && ((intersectingPoint = surface.intersectingPointWithLineSegment_((Jun3dLine) curve)) != null)) {
					JunVertex theVertex = null;

					if (intersectingPoint.distance_(edge.startVertex().point()) < ACCURACY) {
						theVertex = edge.startVertex();
					}

					if (intersectingPoint.distance_(edge.endVertex().point()) < ACCURACY) {
						theVertex = edge.endVertex();
					}

					if (theVertex == null) {
						JunMVE mve = this.mveBody_edge_point_(this.body(), edge, intersectingPoint);
						theVertex = mve.newVertex();

						JunCurve newCurve = mve.newEdge().curve();
						edges1.addElement(mve.newEdge());
						curves1.addElement(newCurve);
						curveBounds1.addElement(((Jun3dLine) newCurve).boundingBox());
						curve = edge.curve();
						curves1.setElementAt(curve, edgeIndex);
						curveBounds1.setElementAt(((Jun3dLine) curve).boundingBox(), edgeIndex);
					}

					surfaces.put(loop, surface);
					vertexesOnSurface.put(theVertex, surface);

					Vector aVector = (Vector) vertexesOnBorder.get(loop);

					if (aVector == null) {
						aVector = new Vector();
						this._vertexesOnBorderPut(loop, aVector);
					}

					aVector.addElement(theVertex);
				}

				edgeIndex++;
			}
		}
	}

	/**
	 * Get the collection of loops which construct a shell. The specified loop
	 * will be one of the loops.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param checkedFlags java.util.Hashtable
	 * @param aCollectionOfLoop2 java.util.Vector
	 */
	private void getShellOf_from_into_(JunLoop aLoop, final Hashtable checkedFlags, Vector aCollectionOfLoop2) {
		if (checkedFlags.get(aLoop) == Boolean.TRUE) {
			return;
		}

		final JunSingleLinkedList loops = new JunSingleLinkedList();
		loops.add_(aLoop);

		while (!loops.isEmpty()) {
			JunLoop loop = (JunLoop) loops.first();
			loops.removeFirst();
			checkedFlags.put(loop, Boolean.TRUE);
			aCollectionOfLoop2.addElement(loop);
			loop.neighborLoopsDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunLoop neighborLoop = (JunLoop) anObject;

					if ((checkedFlags.get(neighborLoop) == Boolean.FALSE) && (loops.includes_(neighborLoop) == false)) {
						loops.addFirst_(neighborLoop);
					}

					return null;
				}
			});
		}
	}

	/**
	 * Create a JunKDEV as a suboperator of the receiver.
	 * 
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKDEV
	 */
	private JunKDEV kdevEdge_(JunEdge aJunEdge) {
		JunLoop loop1 = aJunEdge.leftLoop();
		JunLoop loop2 = aJunEdge.rightLoop();
		JunKDEV kdev = JunKDEV.Body_edge_(this.body(), aJunEdge);
		kdev.doOperation();
		this.changedSurfaceOf_(loop1);

		if (loop1 != loop2) {
			this.changedSurfaceOf_(loop2);
		}

		suboperators.addElement(kdev);

		return kdev;
	}

	/**
	 * Create a JunKEL as a suboperator of the receiver.
	 * 
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKEL
	 */
	private JunKEL kelEdge_(JunEdge aJunEdge) {
		JunKEL kel = JunKEL.Body_edge_(this.body(), aJunEdge);
		kel.doOperation();
		this.changedSurfaceOf_(kel.loop1());
		this.changedSurfaceOf_(kel.loop2());
		suboperators.addElement(kel);

		return kel;
	}

	/**
	 * Create a JunKEML as a suboperator of the receiver.
	 * 
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKEML
	 */
	private JunKEML kemlEdge_(JunEdge aJunEdge) {
		JunKEML keml = JunKEML.Body_edge_(this.body(), aJunEdge);
		keml.doOperation();
		this.changedSurfaceOf_(keml.loop1());
		this.changedSurfaceOf_(keml.loop2());
		suboperators.addElement(keml);

		return keml;
	}

	/**
	 * Create a JunKEV as a suboperator of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKEV
	 */
	private JunKEV kevEdge_vertex_(JunEdge anEdge, JunVertex aVertex) {
		JunKEV kev = JunKEV.Body_edge_vertex_(this.body(), anEdge, aVertex);
		kev.doOperation();
		this.changedSurfaceOf_(kev.loop());
		suboperators.addElement(kev);

		return kev;
	}

	/**
	 * Create a JunKEVVL as a suboperator of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKEVVL
	 */
	private JunKEVVL kevvlEdge_(JunEdge anEdge) {
		JunKEVVL kevvl = JunKEVVL.Body_edge_(this.body(), anEdge);
		kevvl.doOperation();
		this.changedSurfaceOf_(kevvl.loop());
		suboperators.addElement(kevvl);

		return kevvl;
	}

	/**
	 * Kill the specified edge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 */
	private void killEdge_(JunEdge anEdge) {
		if (anEdge.startVertex() == null) {
			return;
		}

		if (anEdge.leftLoop() == anEdge.rightLoop()) {
			if ((anEdge.startRightEdge() == anEdge) && (anEdge.endLeftEdge() == anEdge)) {
				this.kevvlEdge_(anEdge);

				return;
			}

			if (anEdge.startRightEdge() == anEdge) {
				this.kevEdge_vertex_(anEdge, anEdge.startVertex());

				return;
			}

			if (anEdge.endLeftEdge() == anEdge) {
				this.kevEdge_vertex_(anEdge, anEdge.endVertex());

				return;
			}

			this.kemlEdge_(anEdge);
		} else {
			this.kelEdge_(anEdge);
		}
	}

	/**
	 * Kill inside shells.
	 * 
	 * @param aJunProgress jp.co.sra.jun.goodies.progress.JunProgress
	 * @param aNumber1 float
	 * @param aNumber2 float
	 */
	private void killInsideShellsWithProgress_from_to_(JunProgress aJunProgress, float aNumber1, float aNumber2) {
		aJunProgress.value_(aNumber1);

		JunLoop[][] shells = this.shells();

		if (shells != null) {
			JunSurface[][] shellSurfaces = new JunSurface[shells.length][];

			for (int i = 0; i < shells.length; i++) {
				JunLoop[] shell = shells[i];
				shellSurfaces[i] = new JunSurface[shell.length];

				for (int j = 0; j < shell.length; j++) {
					shellSurfaces[i][j] = this.basicSurfaceOf_(shell[j]);
				}
			}

			for (int shellIndex1 = 0; shellIndex1 < shells.length; shellIndex1++) {
				aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * shellIndex1 / shells.length));

				JunLoop[] shell1 = shells[shellIndex1];
				JunSurface[] shellSurface1 = shellSurfaces[shellIndex1];
				final Vector shellPoints1 = new Vector();

				for (int i = 0; i < shell1.length; i++) {
					JunLoop loop = shell1[i];
					loop.vertexesDo_(new StBlockClosure() {
						public Object value_(Object vertex) {
							Jun3dPoint aPoint = ((JunVertex) vertex).point();

							if (shellPoints1.contains(aPoint) == false) {
								shellPoints1.addElement(aPoint);
							}

							return null;
						}
					});
				}

				for (int shellIndex2 = 0; shellIndex2 < shells.length; shellIndex2++) {
					if ((shellIndex2 != shellIndex1) && (shells[shellIndex1][0].isKilled() == false) && (shells[shellIndex2][0].isKilled() == false)) {
						JunLoop[] shell2 = shells[shellIndex2];
						JunSurface[] shellSurface2 = shellSurfaces[shellIndex2];
						final Vector shellPoints2 = new Vector();

						for (int index = 0; index < shell2.length; index++) {
							JunLoop loop = shell2[index];
							loop.vertexesDo_(new StBlockClosure() {
								public Object value_(Object vertex) {
									Jun3dPoint aPoint = ((JunVertex) vertex).point();

									if (shellPoints2.contains(aPoint) == false) {
										shellPoints2.addElement(aPoint);
									}

									return null;
								}
							});
						}

						StBlockClosure maxMinDistancePointBlock = new StBlockClosure() {
							public Object value_value_(Object myPoints, Object hisPoints) {
								StValueHolder thePoint = new StValueHolder();
								StValueHolder maxDistance = new StValueHolder(Double.NEGATIVE_INFINITY);

								for (int myIndex = 0; myIndex < ((Vector) myPoints).size(); myIndex++) {
									Jun3dPoint myPoint = (Jun3dPoint) ((Vector) myPoints).elementAt(myIndex);
									StValueHolder minDistance = new StValueHolder(Double.POSITIVE_INFINITY);

									for (int hisIndex = 0; hisIndex < ((Vector) hisPoints).size(); hisIndex++) {
										Jun3dPoint hisPoint = (Jun3dPoint) ((Vector) hisPoints).elementAt(hisIndex);
										double distance = myPoint.distance_(hisPoint);

										if (minDistance._doubleValue() > distance) {
											minDistance.value_(distance);
										}
									}

									if (minDistance._doubleValue() > maxDistance._doubleValue()) {
										thePoint.value_(myPoint);
										maxDistance.value_(minDistance._doubleValue());
									}
								}

								return thePoint.value();
							}
						};

						Jun3dPoint point1 = (Jun3dPoint) maxMinDistancePointBlock.value_value_(shellPoints1, shellPoints2);
						Jun3dPoint point2 = (Jun3dPoint) maxMinDistancePointBlock.value_value_(shellPoints2, shellPoints1);

						if (this.shellSurfaces_containsPoint_(shellSurface2, point1) && (!this.shellSurfaces_containsPoint_(shellSurface1, point2))) {
							this.killShellWithEdge_(shells[shellIndex1][0].edge());
						}
					}
				}
			}
		}

		aJunProgress.value_(aNumber2);
	}

	/**
	 * Kill the shell specified with the edge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 */
	private void killShellWithEdge_(JunEdge anEdge) {
		JunSingleLinkedList killList = new JunSingleLinkedList();
		killList.add_(anEdge);

		while (!killList.isEmpty()) {
			JunEdge edge = (JunEdge) killList.first();
			killList.removeFirst();

			if (!edge.isKilled()) {
				JunEdge nextEdge1 = edge.startRightEdge();
				JunEdge nextEdge2 = edge.endLeftEdge();
				this.killEdge_(edge);

				if (!killList.includes_(nextEdge1)) {
					killList.addFirst_(nextEdge1);
				}

				if (!killList.includes_(nextEdge2)) {
					killList.addFirst_(nextEdge2);
				}
			}
		}
	}

	/**
	 * Create a JunKVE as a suboperator of the receiver.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunKVE
	 */
	private JunKVE kveVertex_(JunVertex aVertex) {
		JunKVE kve = JunKVE.Body_vertex_(this.body(), aVertex);
		kve.doOperation();
		this.changedSurfaceOf_(kve.aliveEdge().leftLoop());
		this.changedSurfaceOf_(kve.aliveEdge().rightLoop());
		suboperators.addElement(kve);

		return kve;
	}

	/**
	 * Make the Vertex as a child on the Loop.
	 * 
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunLoop jp.co.sra.jun.topology.elements.JunLoop
	 */
	private void makeChild_on_(JunVertex aJunVertex, JunLoop aJunLoop) {
		final Jun3dPoint normalVector = ((JunPlane) aJunLoop.surface()).normalUnitVector();
		JunLoop theChildLoop = aJunVertex.detectLoop_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;
				JunPlane loopSurface = (JunPlane) loop.surface();
				return new Boolean(loopSurface != null && (loopSurface.normalUnitVector().dotProduct_(normalVector) + normalVector.ACCURACY) < 0);
			}
		}, new StBlockClosure());

		if (theChildLoop == null) {
			return;
		}

		JunVertex[] vertexes = theChildLoop.vertexes();
		final Vector pairVertexes = new Vector(vertexes.length);
		final StValueHolder theVertex = new StValueHolder();
		final StValueHolder minDistance = new StValueHolder();

		for (int i = 0; i < vertexes.length; i++) {
			JunVertex aVertex = vertexes[i];
			final Jun3dPoint point = aVertex.point();
			theVertex.setValue_(null);
			minDistance.value_(Double.MAX_VALUE);
			aJunLoop.vertexesDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunVertex vertex = (JunVertex) anObject;
					double distance = point.distance_(vertex.point());

					if (minDistance._doubleValue() > distance) {
						theVertex.setValue_(vertex);
						minDistance.value_(distance);
					}

					return null;
				}
			});
			pairVertexes.addElement(theVertex.value());
		}

		this.meklKillLoop_aliveLoop_vertex_vertex_(theChildLoop, aJunLoop, vertexes[0], (JunVertex) pairVertexes.elementAt(0));

		for (int i = 1; i < (vertexes.length - 1); i++) {
			JunVertex vertex1 = vertexes[i];
			JunVertex vertex2 = (JunVertex) pairVertexes.elementAt(i);
			this.connectVertex_withVertex_onLoop_(vertex1, vertex2, aJunLoop);
		}
	}

	/**
	 * Create a JunMDEV as a suboperator of the receiver.
	 * 
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunEdge1 jp.co.sra.jun.topology.elements.JunEdge
	 * @param aJunEdge2 jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMDEV
	 */
	private JunMDEV mdevVertex_edge_edge_(JunVertex aJunVertex, JunEdge aJunEdge1, JunEdge aJunEdge2) {
		JunMDEV mdev = JunMDEV.Body_vertex_edge_edge_(this.body(), aJunVertex, aJunEdge1, aJunEdge2);
		mdev.doOperation();
		suboperators.addElement(mdev);

		return mdev;
	}

	/**
	 * Create a JunMEKL as a suboperator of the receiver.
	 * 
	 * @param aJunLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMEKL
	 */
	private JunMEKL meklKillLoop_aliveLoop_vertex_vertex_(JunLoop aJunLoop1, JunLoop aJunLoop2, JunVertex aJunVertex1, JunVertex aJunVertex2) {
		JunMEKL mekl = JunMEKL.Body_killLoop_aliveLoop_vertex_vertex_(this.body(), aJunLoop1, aJunLoop2, aJunVertex1, aJunVertex2);
		mekl.doOperation();
		this.changedSurfaceOf_(mekl.loop1());
		this.changedSurfaceOf_(mekl.loop2());

		JunLoop killedLoop = mekl.killedLoop();
		JunLoop aliveLoop = mekl.aliveLoop();

		if ((vertexesOnBorder != null) && (killedLoop != null) && (vertexesOnBorder.containsKey(killedLoop))) {
			if (vertexesOnBorder.containsKey(aliveLoop)) {
				Vector aliveVertexes = (Vector) vertexesOnBorder.get(aliveLoop);
				Vector killedVertexes = (Vector) vertexesOnBorder.get(killedLoop);

				for (int index = 0; index < killedVertexes.size(); index++) {
					aliveVertexes.addElement(killedVertexes.elementAt(index));
				}
			} else {
				this._vertexesOnBorderPut(aliveLoop, vertexesOnBorder.get(killedLoop));
			}

			this._vertexesOnBorderRemove(killedLoop);
		}

		suboperators.addElement(mekl);

		return mekl;
	}

	/**
	 * Create a JunMEL as a suboperator of the receiver.
	 * 
	 * @param aJunLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMEL
	 */
	private JunMEL melLoop_vertex_vertex_(JunLoop aJunLoop, JunVertex aJunVertex1, JunVertex aJunVertex2) {
		JunMEL mel = JunMEL.Body_loop_vertex_vertex_(this.body(), aJunLoop, aJunVertex1, aJunVertex2);
		mel.doOperation();
		this.changedSurfaceOf_(mel.loop1());
		this.changedSurfaceOf_(mel.loop2());

		JunLoop originalLoop = mel.originalLoop();
		JunLoop newLoop = mel.newLoop();

		if ((vertexesOnBorder != null) && vertexesOnBorder.containsKey(originalLoop)) {
			Jun3dPolygon originalSurface = (Jun3dPolygon) this.basicSurfaceOf_(originalLoop);
			Jun3dPolygon newSurface = (Jun3dPolygon) this.basicSurfaceOf_(newLoop);
			Vector vertexesOnBorderAtOriginalLoop = (Vector) vertexesOnBorder.get(originalLoop);
			JunVertex[] vertexes = new JunVertex[vertexesOnBorderAtOriginalLoop.size()];
			vertexesOnBorderAtOriginalLoop.copyInto(vertexes);

			Vector vertexesOnNewLoop = new Vector();

			if (newSurface != null) {
				for (int i = 0; i < vertexes.length; i++) {
					JunVertex vertex = vertexes[i];

					if ((vertex.isKilled() == false) && newSurface.containsPoint_(newSurface.projectionOfPoint_(vertex.point()))) {
						vertexesOnNewLoop.addElement(vertex);
					}
				}
			}

			Vector vertexesOnOriginalLoop = new Vector();

			if (originalSurface != null) {
				for (int i = 0; i < vertexes.length; i++) {
					JunVertex vertex = vertexes[i];

					if ((vertex.isKilled() == false) && originalSurface.containsPoint_(originalSurface.projectionOfPoint_(vertex.point()))) {
						vertexesOnOriginalLoop.addElement(vertex);
					}
				}
			}

			if (vertexesOnNewLoop.isEmpty() == false) {
				this._vertexesOnBorderPut(newLoop, vertexesOnNewLoop);
			}

			if (vertexesOnOriginalLoop.isEmpty() == true) {
				this._vertexesOnBorderRemove(originalLoop);
			} else {
				this._vertexesOnBorderPut(originalLoop, vertexesOnOriginalLoop);
			}
		}

		suboperators.addElement(mel);

		return mel;
	}

	/**
	 * Create a JunMEV as a suboperator of the receiver.
	 * 
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMEV
	 */
	private JunMEV mevVertex_loop_point_(JunVertex aJunVertex, JunLoop aJunLoop, Jun3dPoint aJun3dPoint) {
		JunMEV mev = JunMEV.Body_vertex_loop_point_(this.body(), aJunVertex, aJunLoop, aJun3dPoint);
		mev.doOperation();
		this.changedSurfaceOf_(mev.loop());
		suboperators.addElement(mev);

		return mev;
	}

	/**
	 * Create a JunMEVVL as a suboperator of the receiver.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMEVVL
	 */
	private JunMEVVL mevvlPoint_point_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2) {
		JunMEVVL mevvl = JunMEVVL.Body_point_point_(this.body(), aJun3dPoint1, aJun3dPoint2);
		mevvl.doOperation();
		this.changedSurfaceOf_(mevvl.loop());
		suboperators.addElement(mevvl);

		return mevvl;
	}

	/**
	 * Create a JunMVE as a suboperator of the receiver.
	 * 
	 * @param aJunBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMVE
	 */
	private JunMVE mveBody_edge_point_(JunBody aJunBody, JunEdge aJunEdge, Jun3dPoint aJun3dPoint) {
		return this.mveBody_edge_point_newEdgeOnVertex_(aJunBody, aJunEdge, aJun3dPoint, null);
	}

	/**
	 * Create a JunMVE as a suboperator of the receiver.
	 * 
	 * @param aJunBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMVE
	 */
	private JunMVE mveBody_edge_point_newEdgeOnVertex_(JunBody aJunBody, JunEdge aJunEdge, Jun3dPoint aJun3dPoint, JunVertex aJunVertex) {
		JunMVE mve = JunMVE.Body_edge_point_newEdgeOnVertex_(aJunBody, aJunEdge, aJun3dPoint, aJunVertex);
		mve.doOperation();
		this.changedSurfaceOf_(mve.newEdge().leftLoop());
		this.changedSurfaceOf_(mve.newEdge().rightLoop());
		suboperators.addElement(mve);

		return mve;
	}

	/**
	 * Create a JunMVE as a suboperator of the receiver.
	 * 
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMVE
	 */
	private JunMVE mveEdge_point_(JunEdge aJunEdge, Jun3dPoint aJun3dPoint) {
		return this.mveBody_edge_point_newEdgeOnVertex_(this.body(), aJunEdge, aJun3dPoint, null);
	}

	/**
	 * Create a JunMVE as a suboperator of the receiver.
	 * 
	 * @param aJunEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basics.Jun3dPoint
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.euleroperators.JunMVE
	 */
	private JunMVE mveEdge_point_newEdgeOnVertex_(JunEdge aJunEdge, Jun3dPoint aJun3dPoint, JunVertex aJunVertex) {
		return this.mveBody_edge_point_newEdgeOnVertex_(this.body(), aJunEdge, aJun3dPoint, aJunVertex);
	}

	/**
	 * Answer the next conflicting vertex on two JunLoops start from the
	 * JunVertex.
	 * 
	 * @param aJunLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 */
	private JunVertex nextConflictingVertexOn_with_fromVertex_(JunLoop aJunLoop1, JunLoop aJunLoop2, JunVertex aJunVertex) {
		Jun3dPolygon surface1 = (Jun3dPolygon) this.surfaceOf_(aJunLoop1);
		Jun3dPolygon surface2 = (Jun3dPolygon) this.surfaceOf_(aJunLoop2);

		if ((surface1 == null) || (surface2 == null)) {
			return null;
		}

		Vector vertexes = (Vector) vertexesOnBorder.get(aJunLoop2);

		if (vertexes == null) {
			return null;
		}

		JunVertex theVertex = null;
		double minDistance = Double.POSITIVE_INFINITY;

		for (int i = 0; i < vertexes.size(); i++) {
			JunVertex vertex = (JunVertex) vertexes.elementAt(i);

			if ((vertex.isKilled() == false) && (vertex != aJunVertex) && vertex.hasLoop_(aJunLoop1)) {
				JunEdge edge = vertex.edgeToVertex_(aJunVertex);

				if ((edge == null) || (conflictionEdges.contains(edge) == false)) {
					Jun3dLine lineSegment = new Jun3dLine(surface2.projectionOfPoint_(vertex.point()), surface2.projectionOfPoint_(aJunVertex.point()));

					if (surface2.containsLineSegment_(lineSegment)) {
						lineSegment = new Jun3dLine(surface1.projectionOfPoint_(vertex.point()), surface1.projectionOfPoint_(aJunVertex.point()));

						if (surface1.containsLineSegment_(lineSegment)) {
							double distance = vertex.point().distance_(aJunVertex.point());

							if (distance < minDistance) {
								theVertex = vertex;
								minDistance = distance;
							}
						}
					}
				}
			}
		}

		return theVertex;
	}

	/**
	 * Remove the inside edges.  While executing, show JunProgress.
	 * 
	 * @param aJunProgress jp.co.sra.jun.goodies.progress.JunProgress
	 * @param aNumber1 float
	 * @param aNumber2 float
	 */
	private void removeInsideEdgesWithProgress_from_to_(JunProgress aJunProgress, float aNumber1, float aNumber2) {
		Vector isolatedEdges = new Vector(vertexesOnBorder.size());
		Vector crossingVertexes = new Vector(vertexesOnBorder.size());
		Vector crossingEdges = new Vector(vertexesOnBorder.size());
		int numConflictions = vertexesOnSurface.size();
		int progressIndex = 0;
		Enumeration keys = vertexesOnSurface.keys();

		while (keys.hasMoreElements()) {
			aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * progressIndex / numConflictions));
			progressIndex++;

			final JunVertex vertex = (JunVertex) keys.nextElement();
			JunEdge[] edges = vertex.edges();
			Vector myConflictionEdges = new Vector(edges.length);

			for (int i = 0; i < edges.length; i++) {
				if (conflictionEdges.contains(edges[i])) {
					myConflictionEdges.addElement(edges[i]);
				}
			}

			int size = myConflictionEdges.size();
			final Vector myConflictionEdges2 = new Vector(size);

			for (int i = 0; i < size; i++) {
				JunEdge conflictionEdge1 = (JunEdge) myConflictionEdges.elementAt(i);
				JunEdge conflictionEdge2 = (JunEdge) myConflictionEdges.elementAt((i + 1) % size);
				JunEdge conflictionEdge3 = (JunEdge) myConflictionEdges.elementAt((i + 2) % size);

				if ((conflictionEdge1.nextEdgeWithVertex_(vertex) == conflictionEdge2) || (conflictionEdge2.nextEdgeWithVertex_(vertex) == conflictionEdge3)) {
					continue;
				}

				myConflictionEdges2.addElement(myConflictionEdges.elementAt(i));
			}

			myConflictionEdges = myConflictionEdges2;

			if (myConflictionEdges.isEmpty() == false) {
				JunPlane surface = (JunPlane) vertexesOnSurface.get(vertex);
				final Jun3dPoint normalVector = surface.normalVector();
				final StValueHolder maxAngle = new StValueHolder(normalVector.ACCURACY);
				final StValueHolder maxAngledEdge = new StValueHolder();
				vertex.edgesDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						JunEdge edge = (JunEdge) anObject;

						if (myConflictionEdges2.contains(edge) == false) {
							JunVertex oppositeVertex = edge.oppositeVertexOfVertex_(vertex);
							Jun3dPoint vector = (Jun3dPoint) vertex.point().minus_(oppositeVertex.point());

							if (vector.length() > vector.ACCURACY) {
								vector = (Jun3dPoint) vector.unitVector();

								double dotProduct = vector.dotProduct_(normalVector);

								if (dotProduct > maxAngle._doubleValue()) {
									maxAngledEdge.value_(edge);
									maxAngle.value_(dotProduct);
								}
							}
						}

						return null;
					}
				});

				if (maxAngledEdge.value() != null) {
					crossingVertexes.addElement(vertex);
					crossingEdges.addElement(maxAngledEdge.value());

					JunEdge theEdge = ((JunEdge) maxAngledEdge.value()).nextEdgeWithVertex_(vertex);

					while ((theEdge != maxAngledEdge.value()) && (myConflictionEdges.contains(theEdge) == false)) {
						crossingVertexes.addElement(vertex);
						crossingEdges.addElement(theEdge);
						theEdge = theEdge.nextEdgeWithVertex_(vertex);
					}

					theEdge = theEdge.nextEdgeWithVertex_(vertex);

					while (myConflictionEdges.contains(theEdge) == false) {
						theEdge = theEdge.nextEdgeWithVertex_(vertex);
					}

					theEdge = theEdge.nextEdgeWithVertex_(vertex);

					while (theEdge != maxAngledEdge.value()) {
						crossingVertexes.addElement(vertex);
						crossingEdges.addElement(theEdge);
						theEdge = theEdge.nextEdgeWithVertex_(vertex);
					}
				}
			}
		}

		int size = crossingVertexes.size();

		for (int i = 0; i < size; i++) {
			JunVertex vertex = (JunVertex) crossingVertexes.elementAt(i);
			JunEdge crossingEdge = (JunEdge) crossingEdges.elementAt(i);

			if (crossingEdge.includesVertex_(vertex)) {
				JunMVE mve = this.mveEdge_point_newEdgeOnVertex_(crossingEdge, vertex.point(), vertex);
				JunEdge splitEdge = mve.edge1();
				isolatedEdges.addElement(mve.edge2());

				if (splitEdge.startVertex().point().equals(splitEdge.endVertex().point()) == false) {
					throw SmalltalkException.Error("internal error");
				}

				this.killEdge_(splitEdge);
			}
		}

		for (int i = 0; i < isolatedEdges.size(); i++) {
			this.killShellWithEdge_((JunEdge) isolatedEdges.elementAt(i));
		}
	}

	/**
	 * Answer the shells of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop[][]
	 */
	private JunLoop[][] shells() {
		JunLoop[] loops = this.body().loops();
		Hashtable checkedFlag = new Hashtable(loops.length);

		for (int i = 0; i < loops.length; i++) {
			checkedFlag.put(loops[i], Boolean.FALSE);
		}

		Vector shells = new Vector();

		for (int i = 0; i < loops.length; i++) {
			Vector shell = new Vector();
			this.getShellOf_from_into_(loops[i], checkedFlag, shell);

			int numberOfLoops = shell.size();

			if (numberOfLoops > 0) {
				JunLoop[] arrayOfLoop = new JunLoop[numberOfLoops];
				shell.copyInto(arrayOfLoop);
				shells.addElement(arrayOfLoop);
			}
		}

		int numberOfShells = shells.size();
		JunLoop[][] arrayOfShell = new JunLoop[numberOfShells][];
		shells.copyInto(arrayOfShell);

		return arrayOfShell;
	}

	/**
	 * Answer true if the one of the surfaces contains the point, otherwise
	 * false.
	 * 
	 * @param anArrayOfSurfaces jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return boolean
	 */
	private boolean shellSurfaces_absContainsPoint_(JunSurface[] anArrayOfSurfaces, Jun3dPoint aJun3dPoint) {
		if (anArrayOfSurfaces.length < 3) {
			return false;
		}

		int numIntersections = 0;
		double x = aJun3dPoint.x();
		double y = aJun3dPoint.y();
		double z = aJun3dPoint.z();

		for (int i = 0; i < anArrayOfSurfaces.length; i++) {
			JunSurface surface = anArrayOfSurfaces[i];

			if (surface != null) {
				// Each surface have to be a JunPlane, so far.
				JunPlane plane = (JunPlane) surface;

				if (plane.containsPoint_(aJun3dPoint)) {
					return true;
				}

				if (plane.c() > plane.ACCURACY) {
					double t = -((plane.a() * x) + (plane.b() * y) + (plane.c() * z) + plane.d()) / plane.c();

					if ((t >= 0) && plane.containsPoint_(new Jun3dPoint(x, y, z + t))) {
						numIntersections++;
					}
				}
			}
		}

		return (numIntersections % 2 == 1);
	}

	/**
	 * Answer true if the one of the surfaces contains the point, otherwise
	 * false.
	 * 
	 * @param anArrayOfSurfaces jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return boolean
	 */
	private boolean shellSurfaces_containsPoint_(JunSurface[] anArrayOfSurfaces, Jun3dPoint aJun3dPoint) {
		if (anArrayOfSurfaces.length < 3) {
			return false;
		}

		double minDistance = Double.POSITIVE_INFINITY;
		double referenceAngle = Double.NaN;

		for (int index = 0; index < anArrayOfSurfaces.length; index++) {
			JunSurface surface = anArrayOfSurfaces[index];

			if (surface != null) {
				Jun3dPoint point = ((Jun3dPolygon) surface).nearestPointFromPoint_(aJun3dPoint);
				double distance = point.distance_(aJun3dPoint);

				if (distance < Jun3dPoint.Accuracy()) {
					return true;
				}

				double angle = ((Jun3dPolygon) surface).normalUnitVector().dotProduct_(point.minus_(aJun3dPoint).unitVector());

				if (((Math.abs(minDistance - distance) < Jun3dPoint.ACCURACY) && (Math.abs(angle) > Math.abs(referenceAngle))) || (minDistance > distance)) {
					referenceAngle = angle;
					minDistance = distance;
				}
			}
		}

		if (Double.isNaN(referenceAngle)) {
			return false;
		}

		return (referenceAngle + Jun3dPoint.ACCURACY > 0);
	}

	/**
	 * Answer the surface of the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 */
	private JunSurface surfaceOf_(JunLoop aLoop) {
		JunSurface surface = (JunSurface) surfaces.get(aLoop);

		if (surface == null) {
			surface = aLoop.surface();
			surfaces.put(aLoop, surface);
		}

		return surface;
	}

	/**
	 * Trace conflication on two loops with two vertexes.
	 * 
	 * @param aJunLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 */
	private JunVertex traceConflictionOnLoop_vertex_withLoop_vertex_(JunLoop aJunLoop1, JunVertex aJunVertex1, JunLoop aJunLoop2, JunVertex aJunVertex2) {
		JunVertex[] vertexPair = this.conflictionVertexPairOnLoop_withLoop_fromVertex_vertex_(aJunLoop1, aJunLoop2, aJunVertex1, aJunVertex2);

		if (vertexPair == null) {
			return null;
		}

		JunVertex tmpConflictingVertex1 = vertexPair[0];
		JunVertex tmpConflictingVertex2 = vertexPair[1];
		JunVertex conflictingVertex1 = this.connectVertex_withVertex_onLoop_(tmpConflictingVertex1, aJunVertex1, aJunLoop1);
		JunVertex conflictingVertex2 = this.connectVertex_withVertex_onLoop_(tmpConflictingVertex2, aJunVertex2, aJunLoop2);
		JunLoop[] nextLoops1 = null;

		if (conflictingVertex1.hasLoop_(aJunLoop1)) {
			nextLoops1 = conflictingVertex1.loops();
		} else {
			nextLoops1 = new JunLoop[] { aJunLoop1 };
		}

		JunLoop[] nextLoops2 = null;

		if (conflictingVertex2.hasLoop_(aJunLoop2)) {
			nextLoops2 = conflictingVertex2.loops();
		} else {
			nextLoops2 = new JunLoop[] { aJunLoop2 };
		}

		Object element = conflictingVertex1.edgeToVertex_(this.twinVertexOf_on_connectiveFrom_(aJunVertex1, aJunLoop1, conflictingVertex1));

		if ((element != null) && (conflictionEdges.contains(element) == false)) {
			conflictionEdges.addElement(element);
		}

		element = conflictingVertex2.edgeToVertex_(this.twinVertexOf_on_connectiveFrom_(aJunVertex2, aJunLoop2, conflictingVertex2));

		if ((element != null) && (conflictionEdges.contains(element) == false)) {
			conflictionEdges.addElement(element);
		}

		for (int index1 = 0; index1 < nextLoops1.length; index1++) {
			for (int index2 = 0; index2 < nextLoops2.length; index2++) {
				Jun3dPolygon nextSurface1 = (Jun3dPolygon) nextLoops1[index1].surface();
				Jun3dPolygon nextSurface2 = (Jun3dPolygon) nextLoops2[index2].surface();

				if ((nextSurface1 != null) && (nextSurface2 != null) && (Math.abs(nextSurface1.normalUnitVector().innerProduct_(nextSurface2.normalUnitVector())) < (1 - nextSurface1.ACCURACY))) {
					this.traceConflictionOnLoop_vertex_withLoop_vertex_(nextLoops1[index1], conflictingVertex1, nextLoops2[index2], conflictingVertex2);
				}
			}
		}

		return this.twinVertexOf_on_connectiveFrom_(aJunVertex1, aJunLoop1, conflictingVertex1);
	}

	/**
	 * Trace the conflictions.
	 * 
	 * @return boolean
	 */
	private boolean traceConflictions() {
		for (int keyIndex = 0; keyIndex < _vertexesOnBorderKeys.size(); keyIndex++) {
			JunLoop loop1 = (JunLoop) _vertexesOnBorderKeys.elementAt(keyIndex);
			Vector vertexes = (Vector) vertexesOnBorder.get(loop1);
			Jun3dPolygon surface1 = (Jun3dPolygon) this.surfaceOf_(loop1);

			if (surface1 != null) {
				Jun3dPoint normalVector1 = surface1.normalUnitVector();

				for (int i = 0; i < vertexes.size(); i++) {
					JunVertex vertex2 = (JunVertex) vertexes.elementAt(i);
					JunLoop[] loops2 = vertex2.loops();

					for (int j = 0; j < loops2.length; j++) {
						JunLoop loop2 = loops2[j];

						if ((loop1.isKilled() == false) && (loop2.isKilled() == false)) {
							Jun3dPolygon surface2 = (Jun3dPolygon) this.surfaceOf_(loop2);

							if (surface2 != null) {
								if (Math.abs(surface2.normalUnitVector().innerProduct_(normalVector1)) < (1 - normalVector1.ACCURACY)) {
									JunVertex vertex1 = this.twinVertexOf_on_connectiveFrom_(vertex2, loop1, null);
									JunVertex endVertex = this.traceConflictionOnLoop_vertex_withLoop_vertex_(loop1, vertex1, loop2, vertex2);

									if ((endVertex != null) && (endVertex.isKilled() == false) && endVertex.point().equals(vertex1.point()) && (endVertex.hasLoop_(loop1) == false)) {
										this.makeChild_on_(endVertex, loop1);
									}

									if (endVertex != null) {
										return true;
									}
								}
							}
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @param aJunProgress jp.co.sra.jun.goodies.progress.JunProgress
	 * @param aNumber1 float
	 * @param aNumber2 float
	 */
	private void traceConflictionsWithProgress_from_to_(JunProgress aJunProgress, float aNumber1, float aNumber2) {
		conflictionEdges = new Vector();

		if (vertexesOnBorder.isEmpty()) {
			return;
		}

		int progressIndex = 0;

		do {
			aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * progressIndex / vertexesOnBorder.size()));
			progressIndex++;
		} while (this.traceConflictions());
	}

	/**
	 * Get a twin vertex of the vertex on the loop.
	 * 
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aJunLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 */
	private JunVertex twinVertexOf_on_connectiveFrom_(final JunVertex aJunVertex, JunLoop aJunLoop, JunVertex aJunVertex2) {
		final StBlockClosure testBlock = new StBlockClosure() {
			public Object value_value_(Object p1, Object p2) {
				return new Boolean(((Jun3dPoint) p1).distance_((Jun3dPoint) p2) < ACCURACY);
			}
		};

		Object result = aJunLoop.vertexesDo_(new StBlockClosure() {
			public Object value_(Object vertex) {
				if (((Boolean) testBlock.value_value_(aJunVertex.point(), ((JunVertex) vertex).point())).booleanValue()) {
					return vertex;
				}

				return null;
			}
		});

		if (result != null) {
			return (JunVertex) result;
		}

		if ((aJunVertex2 != null) && (aJunVertex2.isKilled() == false)) {
			result = aJunVertex2.loopsDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunLoop loop = (JunLoop) anObject;

					return loop.vertexesDo_(new StBlockClosure() {
						public Object value_(Object vertex) {
							if (((Boolean) testBlock.value_value_(aJunVertex.point(), ((JunVertex) vertex).point())).booleanValue()) {
								return vertex;
							}

							return null;
						}
					});
				}
			});

			if (result != null) {
				return (JunVertex) result;
			}
		}

		return JunVertex.Point_(aJunVertex.point());
	}

	/**
	 * Unify the loops.
	 * 
	 * @param loop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param loop2 jp.co.sra.jun.topology.elements.JunLoop
	 */
	private void unifyLoop_with_(JunLoop loop1, JunLoop loop2) {
		JunVertex[][] pairs = this.vertexPairsFrom_and_(loop1, loop2);

		if (pairs == null) {
			return;
		}

		JunEdge[] edges = loop1.edges(); // edges := loop1 edges asSet asArray.

		// pairs := pairs asSet asArray.
		this.meklKillLoop_aliveLoop_vertex_vertex_(loop1, loop2, pairs[0][0], pairs[0][1]);

		for (int pairIndex = 1; pairIndex < pairs.length; pairIndex++) {
			JunVertex v1 = pairs[pairIndex][0];
			JunVertex v2 = pairs[pairIndex][1];
			JunLoop loop = this.body().loopForVertex_and_(v1, v2);
			this.melLoop_vertex_vertex_(loop, v1, v2);
		}

		for (int i = 0; i < edges.length; i++) {
			this.kelEdge_(edges[i]);
		}

		for (int i = 0; i < pairs.length; i++) {
			JunVertex vertex = pairs[i][0];

			if (vertex.isKilled() == false) {
				int numEdges = vertex.numberOfEdges();

				if (numEdges == 1) {
					this.kevEdge_vertex_(vertex.edge(), vertex);
				}

				if (numEdges == 2) {
					this.kveVertex_(vertex);
				}

				if (numEdges > 2) {
					this.kdevEdge_(vertex.edgeToVertex_(pairs[i][1]));
				}
			}
		}
	}

	/**
	 * Unify the loops. While executing, show JunProgress.
	 * 
	 * @param aJunProgress jp.co.sra.jun.goodies.progress.JunProgress
	 * @param aNumber1 float
	 * @param aNumber2 float
	 */
	private void unifySharedLoopsWithProgress_from_to_(JunProgress aJunProgress, float aNumber1, float aNumber2) {
		JunLoop[] bodyLoops = this.body().loops();
		Vector loops = new Vector(bodyLoops.length);

		for (int i = 0; i < bodyLoops.length; i++) {
			JunVertex vertex = bodyLoops[i].detectVertex_ifNone_(new StBlockClosure() {
				public Object value_(Object anObject) {
					JunVertex vertex2 = (JunVertex) anObject;

					return new Boolean(conflictionBounds.containsOrTouchesPoint_(vertex2.point()));
				}
			}, new StBlockClosure());

			if (vertex != null) {
				loops.addElement(bodyLoops[i]);
			}
		}

		int numLoops = loops.size();
		int progressIndex = 0;

		for (int i = 0; i < numLoops; i++) {
			aJunProgress.value_(aNumber1 + ((aNumber2 - aNumber1) * progressIndex / numLoops));
			progressIndex++;

			JunLoop loop1 = (JunLoop) loops.elementAt(i);

			if (loop1.isKilled() == false) {
				for (int j = i + 1; j < numLoops; j++) {
					if (loop1.isKilled() == false) {
						JunLoop loop2 = (JunLoop) loops.elementAt(j);

						if (loop2.isKilled() == false) {
							this.unifyLoop_with_(loop1, loop2);
						}
					}
				}
			}
		}
	}

	/**
	 * Answer the pair of vertexes, one from a JunLoop and one from another
	 * JunLoop.
	 * 
	 * @param aJunLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aJunLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex[][]
	 */
	private JunVertex[][] vertexPairsFrom_and_(JunLoop aJunLoop1, JunLoop aJunLoop2) {
		final Jun3dPoint p1 = aJunLoop2.vertex().point();
		Object result = aJunLoop1.detectVertex_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunVertex v = (JunVertex) anObject;

				return new Boolean((Math.abs(v.point().x() - p1.x()) < p1.ACCURACY) && (Math.abs(v.point().y() - p1.y()) < p1.ACCURACY) && (Math.abs(v.point().z() - p1.z()) < p1.ACCURACY));
			}
		}, new StBlockClosure());

		if (result == null) {
			return null;
		}

		//
		int numVertexes = aJunLoop1.numberOfVertexes();

		if (numVertexes != aJunLoop2.numberOfVertexes()) {
			return null;
		}

		JunVertex[] vertexes1 = aJunLoop1.vertexes();
		JunVertex[] vertexes2 = aJunLoop2.vertexes();

		for (int i = 0; i < (vertexes2.length / 2); i++) {
			int j = vertexes2.length - i - 1;
			JunVertex tmp = vertexes2[i];
			vertexes2[i] = vertexes2[j];
			vertexes2[j] = tmp;
		}

		int offset2 = numVertexes;

		for (int offset = 0; offset < numVertexes; offset++) {
			boolean matched = true;

			for (int index = 0; index < numVertexes; index++) {
				if (vertexes1[index] == vertexes2[(index + offset) % numVertexes]) {
					return null;
				}

				if (vertexes1[index].point().distance_(vertexes2[(index + offset) % numVertexes].point()) >= JunGeometry.Accuracy()) {
					matched = false;

					break;
				}
			}

			if (matched) {
				offset2 = offset;
			}
		}

		if (offset2 == numVertexes) {
			//haven't matched vertex
			return null;
		}

		JunVertex[][] pairs = new JunVertex[vertexes1.length][2];

		for (int index1 = 0; index1 < vertexes1.length; index1++) {
			int index2 = (index1 + offset2) % vertexes2.length;
			pairs[index1][0] = vertexes1[index1];
			pairs[index1][1] = vertexes2[index2];
		}

		return pairs;
	}

	/**
	 * Answer true if the one of the surfaces contains the point, otherwise
	 * false.
	 * 
	 * @param anArrayOfSurfaces jp.co.sra.jun.geometry.abstracts.JunSurface[]
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return boolean
	 */
	private boolean XshellSurfaces_containsPoint_(JunSurface[] anArrayOfSurfaces, Jun3dPoint aJun3dPoint) {
		if (anArrayOfSurfaces.length < 2) {
			return false;
		}

		// The surface has to be a JunPolygon, so far.
		Jun3dPoint min = (Jun3dPoint) ((Jun3dPolygon) anArrayOfSurfaces[0]).boundingBox().origin();

		for (int i = 1; i < anArrayOfSurfaces.length; i++) {
			min = (Jun3dPoint) min.min_(((Jun3dPolygon) anArrayOfSurfaces[i]).boundingBox().origin());
		}

		Jun3dPoint outerPoint = (Jun3dPoint) min.minus_(1);
		boolean condition1 = !this.shellSurfaces_absContainsPoint_(anArrayOfSurfaces, outerPoint);
		boolean condition2 = this.shellSurfaces_absContainsPoint_(anArrayOfSurfaces, aJun3dPoint);

		return (condition1 != condition2);
	}
}
