package jp.co.sra.jun.topology.graph;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.goodies.calendar.JunCalendarModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunElementalStuff class
 * 
 *  @author    nisinaka
 *  @created   2006/04/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun598 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalStuff.java,v 8.11 2008/02/20 06:33:02 nisinaka Exp $
 */
public abstract class JunElementalStuff extends JunAbstractObject implements StDisplayable {

	protected static Date _UniqueNumberDate = null;
	protected static int _UniqueNumberSequence = 0;
	protected static SimpleDateFormat _UniqueNumberDateFormat = new SimpleDateFormat("yyyyMMddHHmmssSSS");
	protected static DecimalFormat _UniqueNumberDecimalFormat = new DecimalFormat("000");

	protected StSymbol uniqueNumber;
	protected JunAttributeTable myAttributes;

	/**
	 * Answer the default label string.
	 * 
	 * @return java.lang.String
	 * @category Defaults
	 */
	protected static String DefaultLabelString() {
		return $String("Untitled");
	}

	/**
	 * Answer the default label emphasis.
	 * 
	 * @return boolean
	 * @category Defaults
	 */
	protected static boolean DefaultLabelEmphasis() {
		return false;
	}

	/**
	 * Answer the default location point.
	 * 
	 * @return java.awt.Point
	 * @category Defaults
	 */
	protected static Point DefaultLocationPoint() {
		return new Point(0, 0);
	}

	/**
	 * Answer the default border width.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultBorderWidth() {
		return 1;
	}

	/**
	 * Answer the default border color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultBorderColor() {
		return StColorValue.VeryDarkGray;
	}

	/**
	 * Answer the default foreground color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultForegroundColor() {
		return Color.black;
	}

	/**
	 * Answer the default background color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultBackgroundColor() {
		return Color.white;
	}

	/**
	 * Answer the default selection border color.
	 * 
	 * @return java.awt.Color
	 * @category Defaults
	 */
	protected static Color DefaultSelectionBorderColor() {
		return Color.red;
	}

	/**
	 * Answer the default selection border width.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultSelectionBorderWidth() {
		return 2;
	}

	/**
	 * Generate a symbol as a unique number.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Utilities
	 */
	protected StSymbol UniqueNumber() {
		return PrivateUniqueNumberFrom_(new Date());
	}

	/**
	 * Generate a symbol as private unique number from the specified string.
	 * 
	 * @param aDate java.util.Date
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Private
	 */
	protected static StSymbol PrivateUniqueNumberFrom_(Date aDate) {
		if (aDate.equals(_UniqueNumberDate)) {
			_UniqueNumberSequence++;
		} else {
			_UniqueNumberDate = aDate;
			_UniqueNumberSequence = 0;
		}

		return $(_UniqueNumberDateFormat.format(_UniqueNumberDate) + _UniqueNumberDecimalFormat.format(_UniqueNumberSequence));
	}

	/**
	 * Create a new instance of JunElementalStuff and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalStuff() {
		super();
	}

	/**
	 * Create a new instance of JunElementalStuff and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunElementalStuff(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		Date aDate = new Date();
		uniqueNumber = PrivateUniqueNumberFrom_(aDate);
		String aString = JunCalendarModel.StringFromDateAndTime_(aDate);
		myAttributes = new JunAttributeTable();
		myAttributes.at_put_($("creationDate"), aString);
		myAttributes.at_put_($("modificationDate"), aString);
	}

	/**
	 * Answer my unique number.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol uniqueNumber() {
		return uniqueNumber;
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public abstract StRectangle boundingBox();

	/**
	 * Answer the bounding box of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return this.boundingBox().toRectangle();
	}

	/**
	 * Answer my attributes.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category attributes
	 */
	public JunAttributeTable attributes() {
		return myAttributes;
	}

	/**
	 * Add the attribute.
	 * 
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 * @category attributes
	 */
	public void addAttribute_(Object key, Object value) {
		myAttributes.at_put_(key, value);
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Answer my attribute value at the specified key.
	 * 
	 * @param key java.lang.Object
	 * @return java.lang.Object
	 * @category attributes
	 */
	public Object attributeAt_(Object key) {
		return myAttributes.at_(key);
	}

	/**
	 * Answer my label string.
	 * 
	 * @return java.lang.String
	 * @category attributes
	 */
	public String labelString() {
		String aString = (String) myAttributes.at_($("labelString"));
		if (aString == null) {
			aString = this.defaultLabelString();
			this.setLabelString_(aString);
		}
		return aString;
	}

	/**
	 * Set my label string.
	 * 
	 * @param aString java.lang.String
	 * @category attributes
	 */
	public void labelString_(String aString) {
		this.setLabelString_(aString);
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Answer my label emphasis.
	 * 
	 * @return java.lang.String
	 * @category attributes
	 */
	public boolean labelEmphasis() {
		Boolean aBoolean = (Boolean) myAttributes.at_($("labelEmphasis"));
		if (aBoolean == null) {
			aBoolean = Boolean.valueOf(this.defaultLabelEmphasis());
			myAttributes.at_put_($("labelEmphasis"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set my label emphasis.
	 * 
	 * @param aBoolean boolean
	 * @category attributes
	 */
	public void labelEmphasis_(boolean aBoolean) {
		myAttributes.at_put_($("labelEmphasis"), Boolean.valueOf(aBoolean));
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Answer my location point.
	 * 
	 * @return java.awt.Point
	 * @category attributes
	 */
	public Point locationPoint() {
		Point aPoint = (Point) myAttributes.at_($("locationPoint"));
		if (aPoint == null) {
			aPoint = this.defaultLocationPoint();
			myAttributes.at_put_($("locationPoint"), aPoint);
		}
		return aPoint;
	}

	/**
	 * Set my location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category attributes
	 */
	public void locationPoint_(Point aPoint) {
		myAttributes.at_put_($("locationPoint"), aPoint);
		this.updateModificationDate();
		this.flushBoundingBox();
	}

	/**
	 * Answre my border width.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int borderWidth() {
		Number aNumber = (Number) myAttributes.at_($("borderWidth"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultBorderWidth());
			myAttributes.at_put_($("borderWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my border width.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void borderWidth_(int anInteger) {
		myAttributes.at_put_($("borderWidth"), new Integer(anInteger));
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Answer my border color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color borderColor() {
		Color aColor = (Color) myAttributes.at_($("borderColor"));
		if (aColor == null) {
			aColor = this.defaultBorderColor();
			myAttributes.at_put_($("borderColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my border color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void borderColor_(Color aColor) {
		myAttributes.at_put_($("borderColor"), aColor);
		this.updateModificationDate();
		this.flushBoundingBox();
	}

	/**
	 * Answer my foreground color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color foregroundColor() {
		Color aColor = (Color) myAttributes.at_($("foregroundColor"));
		if (aColor == null) {
			aColor = this.defaultForegroundColor();
			myAttributes.at_put_($("foregroundColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void foregroundColor_(Color aColor) {
		myAttributes.at_put_($("foregroundColor"), aColor);
		this.updateModificationDate();
		this.flushBoundingBox();
	}

	/**
	 * Answer my background color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color backgroundColor() {
		Color aColor = (Color) myAttributes.at_($("backgroundColor"));
		if (aColor == null) {
			aColor = this.defaultBackgroundColor();
			myAttributes.at_put_($("backgroundColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my background color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void backgroundColor_(Color aColor) {
		myAttributes.at_put_($("backgroundColor"), aColor);
		this.updateModificationDate();
		this.flushBoundingBox();
	}

	/**
	 * Answer my selection border color.
	 * 
	 * @return java.awt.Color
	 * @category attributes
	 */
	public Color selectionBorderColor() {
		Color aColor = (Color) myAttributes.at_($("selectionBorderColor"));
		if (aColor == null) {
			aColor = this.defaultSelectionBorderColor();
			myAttributes.at_put_($("selectionBorderColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Set my selection border color.
	 * 
	 * @param aColor java.awt.Color
	 * @category attributes
	 */
	public void selectionBorderColor_(Color aColor) {
		myAttributes.at_put_($("selectionBorderColor"), aColor);
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Answre my selection border width.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int selectionBorderWidth() {
		Number aNumber = (Number) myAttributes.at_($("selectionBorderWidth"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultSelectionBorderWidth());
			myAttributes.at_put_($("selectionBorderWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my selection border width.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void selectionBorderWidth_(int anInteger) {
		myAttributes.at_put_($("selectionBorderWidth"), new Integer(anInteger));
		this.updateModificationDate();
		this.flushVisualObject();
	}

	/**
	 * Reset attributes by defaults.
	 * 
	 * @category attributes
	 */
	protected void resetAttributesByDefaults() {
		StSymbol[] symbols = this.attributeSymbolsToReset();
		for (int i = 0; i < symbols.length; i++) {
			this.attributes().removeKey_(symbols[i]);
			try {
				this.perform_(symbols[i].toString());
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Answer the attribute symbols to reset.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol[]
	 * @category attributes
	 */
	protected abstract StSymbol[] attributeSymbolsToReset();

	/**
	 * Update the modification date.
	 * 
	 * @category dates
	 */
	protected void updateModificationDate() {
		String aString = JunCalendarModel.StringFromDateAndTime();
		myAttributes.at_put_($("modificationDate"), aString);
		if (myAttributes.at_($("creationDate")) == null) {
			myAttributes.at_put_($("creationDate"), aString);
		}
	}

	/**
	 * Flush the visual object.
	 * 
	 * @category flushing
	 */
	protected void flushVisualObject() {
		this.flushBoundingBox();
	}

	/**
	 * Flush the bounding box.
	 * 
	 * @category flushing
	 */
	protected void flushBoundingBox() {
		// nothing to do
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category defaults
	 */
	protected abstract JunAttributeTable defaultAttributes();

	/**
	 * Answer the default label string.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	protected String defaultLabelString() {
		String aString = (String) this.defaultAttributes().at_($("labelString"));
		if (aString == null) {
			aString = DefaultLabelString();
			this.defaultAttributes().at_put_($("labelString"), aString);
		}
		return aString;
	}

	/**
	 * Answer the default label emphasis.
	 * 
	 * @return boolean
	 * @category defaults
	 */
	protected boolean defaultLabelEmphasis() {
		Boolean aBoolean = (Boolean) this.defaultAttributes().at_($("labelEmphasis"));
		if (aBoolean == null) {
			aBoolean = Boolean.valueOf(DefaultLabelEmphasis());
			this.defaultAttributes().at_put_($("labelEmphasis"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Answer the default location point.
	 * 
	 * @return java.awt.Point
	 * @category defaults
	 */
	protected Point defaultLocationPoint() {
		Point aPoint = (Point) this.defaultAttributes().at_($("locationPoint"));
		if (aPoint == null) {
			aPoint = DefaultLocationPoint();
			this.defaultAttributes().at_put_($("locationPoint"), aPoint);
		}
		return aPoint;
	}

	/**
	 * Answer the default border width.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultBorderWidth() {
		Number aNumber = (Number) this.defaultAttributes().at_($("borderWidth"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultBorderWidth());
			this.defaultAttributes().at_put_($("borderWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Answer the default border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultBorderColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("borderColor"));
		if (aColor == null) {
			aColor = DefaultBorderColor();
			this.defaultAttributes().at_put_($("borderColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default foreground color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultForegroundColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("foregroundColor"));
		if (aColor == null) {
			aColor = DefaultForegroundColor();
			this.defaultAttributes().at_put_($("foregroundColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default background color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultBackgroundColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("backgroundColor"));
		if (aColor == null) {
			aColor = DefaultBackgroundColor();
			this.defaultAttributes().at_put_($("backgroundColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default selection border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultSelectionBorderColor() {
		Color aColor = (Color) this.defaultAttributes().at_($("selectionBorderColor"));
		if (aColor == null) {
			aColor = DefaultSelectionBorderColor();
			this.defaultAttributes().at_put_($("selectionBorderColor"), aColor);
		}
		return aColor;
	}

	/**
	 * Answer the default selection border width.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultSelectionBorderWidth() {
		Number aNumber = (Number) this.defaultAttributes().at_($("selectionBorderWidth"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultSelectionBorderWidth());
			this.defaultAttributes().at_put_($("selectionBorderWidth"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public abstract void displayOn_at_(Graphics aGraphics, Point aPoint);

	/**
	 * Answer my display string.
	 * 
	 * @return java.lang.String
	 * @category printing 
	 */
	public String displayString() {
		return "(" + this.labelString() + ")";
	}

	/**
	 * Answer the receiver as StImage.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		StRectangle aBox = this.boundingBox();
		StImage anImage = new StImage(aBox.width(), aBox.height());
		Graphics aGraphics = null;
		try {
			aGraphics = anImage.image().getGraphics();
			aGraphics.translate(-aBox.x(), -aBox.y());
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
		return anImage;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return this._className();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.uniqueNumberFromLispList(aList);
		this.myAttributesFromLispList(aList);
	}

	/**
	 * Convert my attributes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons aList = new JunLispCons(this.kindName());
		aList.add_(this.uniqueNumberToLispList());
		aList.add_(this.myAttributesToLispList());
		return aList;
	}

	/**
	 * Get my unique number from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected void uniqueNumberFromLispList(JunLispList aList) {
		JunLispCons uniqueNumberList = aList._findSublistWithHead($("uniqueNumber"));
		if (uniqueNumberList == null) {
			return;
		}

		this.setUniqueNumber_((StSymbol) uniqueNumberList.tail());
	}

	/**
	 * Convert my unique number to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons uniqueNumberToLispList() {
		return new JunLispCons($("uniqueNumber"), uniqueNumber);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected void myAttributesFromLispList(JunLispList aList) {
		JunLispCons myAttributesList = aList._findSublistWithHead($("myAttributes"));
		if (myAttributesList == null) {
			return;
		}

		this.setMyAttributes_(new JunAttributeTable((JunLispList) myAttributesList.tail()));
	}

	/**
	 * Convert my attributes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons myAttributesToLispList() {
		return new JunLispCons($("myAttributes"), myAttributes.toLispList());
	}

	/**
	 * Set my label string.
	 * 
	 * @param aString java.lang.String
	 * @category private
	 */
	protected void setLabelString_(String aString) {
		myAttributes.at_put_($("labelString"), aString);
	}

	/**
	 * Set my unique number.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category private
	 */
	protected void setUniqueNumber_(StSymbol aSymbol) {
		uniqueNumber = aSymbol;
	}

	/**
	 * Set my attributes.
	 * 
	 * @param anAttributeTable 
	 * @category private
	 */
	protected void setMyAttributes_(JunAttributeTable anAttributeTable) {
		myAttributes = anAttributeTable;
	}

}
