package jp.co.sra.jun.terrain.editor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunTerrainEditController class
 * 
 *  @author    nisinaka
 *  @created   2001/11/19 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2007/08/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun680 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTerrainEditController.java,v 8.13 2008/02/20 06:32:59 nisinaka Exp $
 */
public class JunTerrainEditController extends JunAbstractController implements MouseListener, MouseMotionListener {

	protected Jun2dPoint _currentPoint;
	protected Jun2dPoint _previousPoint;
	protected Cursor _savedCursor;
	protected Point _startPoint;
	protected Point _mousePointOnScreen;
	protected Point[] _lineSegment;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StController#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		super.initialize();
		_currentPoint = null;
		_previousPoint = null;
		_savedCursor = null;
		_startPoint = null;
		_mousePointOnScreen = null;
		_lineSegment = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * Remove the listeners of the view here as well.
	 * 
	 * @see jp.co.sra.smalltalk.StController#release()
	 * @category initialize-release
	 */
	public void release() {
		Component drawable = this.getTerrainEditView().getOpenGLDrawable().toComponent();
		drawable.removeMouseListener(this);
		drawable.removeMouseMotionListener(this);

		super.release();
	}

	/**
	 * Add myself as a listener of the view.
	 * 
	 * @param newView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView newView) {
		Component drawable = this.getTerrainEditView().getOpenGLDrawable().toComponent();
		drawable.addMouseListener(this);
		drawable.addMouseMotionListener(this);
	}

	/**
	 * Answer my model as JunTerrainEditModel.
	 * 
	 * @return jp.co.sra.jun.terrain.editor.JunTerrainEditModel
	 * @category view accessing
	 */
	protected JunTerrainEditModel getTerrainEditModel() {
		return (JunTerrainEditModel) this.model();
	}

	/**
	 * Answer my view as JunTerrainEditView.
	 * 
	 * @return jp.co.sra.jun.terrain.editor.JunTerrainEditView
	 * @category view accessing
	 */
	protected JunTerrainEditView getTerrainEditView() {
		return (JunTerrainEditView) this.view();
	}

	/**
	 * Invoked when a mouse is entered on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseEntered(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseEntered(MouseEvent e) {
		JunTerrainEditModel theModel = (JunTerrainEditModel) this.model();
		StSymbol state = theModel.buttonState();

		if (state == null) {
			return;
		}

		Component mainView = this.getTerrainEditView().getOpenGLDrawable().toComponent();
		_savedCursor = mainView.getCursor();

		if (state == $("peak")) {
			mainView.setCursor(JunCursors.CrossCursor());
		} else if (state == $("drag")) {
			mainView.setCursor(JunCursors.QuartersCursor());
		} else if (state == $("show")) {
			mainView.setCursor(JunCursors.ReadCursor());
		}
	}

	/**
	 * Invoked when a mouse is exited on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseExited(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseExited(MouseEvent e) {
		this.getTerrainEditView().getOpenGLDrawable().toComponent().setCursor(_savedCursor);
	}

	/**
	 * Invoked when a mouse button is pressed on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent e) {
		_startPoint = e.getPoint();
		_mousePointOnScreen = e.getComponent().getLocationOnScreen();
		_mousePointOnScreen.translate(_startPoint.x, _startPoint.y);
		_currentPoint = this.regularizePoint_(_startPoint);
		_previousPoint = _currentPoint;
	}

	/**
	 * Invoked when a mouse button is released on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent e) {
		JunTerrainEditModel theModel = (JunTerrainEditModel) this.model();
		StSymbol state = theModel.buttonState();

		if (state == $("peak")) {
			theModel.plotPeak_(this);
		} else if (state == $("show")) {
			this._drawLineSegment();
			theModel.showTerrain_(this);
			_lineSegment = null;
		}

		_startPoint = null;
		_mousePointOnScreen = null;
	}

	/**
	 * Invoked when a mouse button is dragged on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent e) {
		JunTerrainEditModel theModel = (JunTerrainEditModel) this.model();
		StSymbol state = theModel.buttonState();

		if (state == $("drag")) {
			_currentPoint = this.regularizePoint_(e.getPoint());

			if (_currentPoint.equals(_previousPoint) == false) {
				theModel.drag_xy_(_previousPoint, _currentPoint);
				_previousPoint = _currentPoint;
			}
		} else if (state == $("show")) {
			if (_startPoint != null) {
				this._drawLineSegment();
				_lineSegment = new Point[] { _startPoint, e.getPoint() };
				this._drawLineSegment();
			}
		}
	}

	/**
	 * Draw the line segment if exist.
	 * 
	 * @category private
	 */
	protected void _drawLineSegment() {
		if (_lineSegment == null) {
			return;
		}

		Graphics aGraphics = this.getTerrainEditView().getOpenGLDrawable().toComponent().getGraphics();
		try {
			aGraphics.setColor(Color.white);
			aGraphics.setXORMode(Color.black);
			aGraphics.drawLine(_lineSegment[0].x, _lineSegment[0].y, _lineSegment[1].x, _lineSegment[1].y);
		} finally {
			aGraphics.dispose();
		}
	}

	/**
	 * Answer the specified line segment as Jun2dLine.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category private
	 */
	protected Jun2dLine mouse2dLineSegment() {
		if (_lineSegment == null) {
			return null;
		}

		return new Jun2dLine(this.regularizePoint_(_lineSegment[0]), this.regularizePoint_(_lineSegment[1]));
	}

	/**
	 * Answer the normalized mouse point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint mouse2dPoint() {
		return _currentPoint;
	}

	/**
	 * Normalize the mouse point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint regularizePoint_(Point aPoint) {
		return this.getTerrainEditModel().regularizePoint_in_(aPoint, this.getTerrainEditView().boundsOfCanvas());
	}

}
