package jp.co.sra.jun.terrain.display;

import java.awt.Color;
import java.awt.Point;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StRadioButtonGroup;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.opengl.cosmos.JunOpenGLMicrocosmModel;
import jp.co.sra.jun.opengl.display.JunOpenGL3dView;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.terrain.support.JunTerrainField;

/**
 * JunTerrainDisplayModel class
 * 
 *  @author    nisinaka
 *  @created   2002/01/22 (by nisinaka)
 *  @updated   2003/05/15 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2007/08/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun642 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTerrainDisplayModel.java,v 8.12 2008/02/20 06:32:59 nisinaka Exp $
 */
public class JunTerrainDisplayModel extends JunOpenGLMicrocosmModel {

	protected JunTerrainField terrainField;
	protected double inches;
	protected double defaultZoomHeight;

	/**
	 * Create a new instance of JunTerrainDisplayModel and initialize it.
	 *
	 * @param aTerrainField jp.co.sra.jun.terrain.support.JunTerrainField
	 * @category Instance creation
	 */
	public JunTerrainDisplayModel(JunTerrainField aTerrainField) {
		terrainField = aTerrainField;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		terrainField = null;
		inches = 1.0d;
		defaultZoomHeight = 1.0d;
	}

	/**
	 * Answer the current inches.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double inches() {
		return inches;
	}

	/**
	 * Set the new inches.
	 * 
	 * @param newInches double
	 * @category accessing
	 */
	public void inches_(double newInches) {
		inches = newInches;
		this.moveTo_(this.displayProjector().eyePoint());
	}

	/**
	 * Answer the current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject displayObject() {
		if (openGL3dObject == null) {
			openGL3dObject = this.computeDisplayObject();
		}
		return openGL3dObject;
	}

	/**
	 * Answer my display projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	public JunOpenGLProjector displayProjector() {
		if (openGLProjector == null) {
			openGLProjector = new JunOpenGLProjector();
			openGLProjector.projection_(new JunTerrainProjection());
			openGLProjector.eyePoint_(this.defaultEyePoint());
			openGLProjector.sightPoint_(this.defaultSightPoint());
			openGLProjector.upVector_(this.defaultUpVector());
			openGLProjector.viewFactor_(this.defaultViewFactor());
			openGLProjector.far_(this.defaultFar());
			openGLProjector.near_(this.defaultNear());
		}
		return openGLProjector;
	}

	/**
	 * Answer the array of all display lights.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @category lighting
	 */
	public JunOpenGLDisplayLight[] displayLights() {
		if (displayLights == null) {
			displayLights = new JunOpenGLDisplayLight[5];
			displayLights[0] = JunOpenGLDisplayLight.ParallelLight_color_position_(true, this.defaultLightColor(), this.defaultLightPoint());
			displayLights[1] = new JunOpenGLDisplayLight();
			displayLights[2] = new JunOpenGLDisplayLight();
			displayLights[3] = new JunOpenGLDisplayLight();
			displayLights[4] = JunOpenGLDisplayLight.AmbientLight_color_(false, this.defaultLightColor());
			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].compute_(new StBlockClosure() {
					public Object value() {
						updateLightMenuIndication();
						changed_($("light"));
						return null;
					}
				});
			}
		}
		return displayLights;
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void grab_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		this.grab_x_(from2dPoint, to2dPoint);
		this.grab_y_(from2dPoint, to2dPoint);
		this.changed_($("projection"));
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	protected void grab_x_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjector projector = this.displayProjector();
		JunAngle rotationAngle = JunAngle.FromRad_((to2dPoint.x() - from2dPoint.x()) / 2.0d);

		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(projector.eyePoint(), projector.eyePoint().plus_(new Jun3dPoint(0, 0, 1)));
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.mul_(-1), rotationAxis);
			projector.sightPoint_(transformationInv.applyTo_(projector.sightPoint()));
			projector.upVector_(new Jun3dPoint(0, 0, 1));
		}
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	protected void grab_y_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjector projector = this.displayProjector();
		JunOpenGLProjection projection = projector.projection();
		JunAngle rotationAngle = JunAngle.FromRad_((to2dPoint.y() - from2dPoint.y()) / 2.0d);

		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(projection.eyePoint(), projection.eyePoint().plus_(projection.rightVector()));
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
			projector.sightPoint_(transformationInv.applyTo_(projection.sightPoint()));
			projector.upVector_(new Jun3dPoint(0, 0, 1));
		}
	}

	/**
	 * Dolly with the specified factor.
	 * 
	 * @param factor double
	 * @category projection
	 */
	public void dolly_(double factor) {
		double dollyFactor = factor;
		JunOpenGLProjection projection = this.displayProjection();
		Jun3dPoint dollyVector = projection.sightVector().negated().multipliedBy_(dollyFactor);
		Jun3dPoint eyePoint = projection.eyePoint();
		this.moveTo_(new Jun2dPoint(eyePoint.x() + dollyVector.x(), eyePoint.y() + dollyVector.y()));
	}

	/**
	 * Fit the 3d object to the current view size.
	 * 
	 * @category projection
	 */
	public void fitSilently() {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjector projector = this.displayProjector();
		projector.sightPoint_(this.defaultSightPoint());
		projector.eyePoint_(this.defaultEyePoint());
		projector.upVector_(this.defaultUpVector());
		projector.viewFactor_(this.defaultViewFactor());
		projector.far_(this.defaultFar());
		projector.near_(this.defaultNear());
		projector.zoomHeight_(this.defaultZoomHeight());
	}

	/**
	 * Answer the default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		StView aView = super.defaultView();
		aView.controller_(new JunTerrainDisplayController());
		((JunOpenGL3dView) aView).getOpenGLDrawable().toComponent().setBackground(Color.getHSBColor(0.5f, 0.5f, 1.0f));
		return aView;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Terrain");
	}

	/**
	 * Answer the default far.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultFar() {
		return this.defaultEyePoint().distance_(this.defaultSightPoint()) * this.defaultViewFactor();
	}

	/**
	 * Answer the default near.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultNear() {
		return this.defaultFar() / 10000.0d;
	}

	/**
	 * Answer the default light color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	public Color defaultLightColor() {
		return Color.getHSBColor(0.19f, 0.5f, 1.0f);
	}

	/**
	 * Answer the default light point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultLightPoint() {
		return new Jun3dPoint(0, 0.5, 1);
	}

	/**
	 * Answer the default view factor.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultViewFactor() {
		return 5.0d;
	}

	/**
	 * Answer the current default zoom height.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultZoomHeight() {
		return defaultZoomHeight;
	}

	/**
	 * Set the new default zoom height.
	 * 
	 * @param newDefaultZoomHeight double
	 * @category defaults
	 */
	public void defaultZoomHeight_(double newDefaultZoomHeight) {
		defaultZoomHeight = newDefaultZoomHeight;
	}

	/**
	 * Answer the current dolly button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel dollyButton() {
		if (dollyButton == null) {
			JunButtonModel button = new JunButtonModel();
			button.value_(true);
			button.visual_(JunCursors.DollyCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					return model;
				}
			});
			dollyButton = button;
		}
		return dollyButton;
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu(JunSystem.$String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem(JunSystem.$String("Quit"), new MenuPerformer(this, "quitDoing")));
		return fileMenu;
	}

	/**
	 * Create a "View" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createViewMenu() {
		StMenu viewMenu = new StMenu(JunSystem.$String("View"), $("viewMenu"));

		// Inches...
		viewMenu.add(new StMenuItem(JunSystem.$String("Inch", "Inches") + "...", new MenuPerformer(this, "changeInches")));

		viewMenu.addSeparator();

		// Zooming
		StMenu zoomingMenu = new StMenu(JunSystem.$String("Zooming"), $("zoomingMenu"));
		zoomingMenu.add(new StMenuItem(JunSystem.$String("Fit zoom"), new MenuPerformer(this, "fitZoom")));
		zoomingMenu.add(new StMenuItem(JunSystem.$String("Close up"), new MenuPerformer(this, "closeUp")));
		zoomingMenu.addSeparator();
		zoomingMenu.add(new StMenuItem(JunSystem.$String("Zoom"), new MenuPerformer(this, "zoom")));
		zoomingMenu.add(new StMenuItem(JunSystem.$String("Pan"), new MenuPerformer(this, "pan")));
		viewMenu.add(zoomingMenu);

		// Pointing
		StMenu pointingMenu = new StMenu(JunSystem.$String("Pointing"));
		pointingMenu.add(new StMenuItem(JunSystem.$String("Fit sight"), new MenuPerformer(this, "fitSight")));
		pointingMenu.addSeparator();
		pointingMenu.add(new StMenuItem(JunSystem.$String("Eye point") + "...", new MenuPerformer(this, "changeEyePoint")));
		pointingMenu.add(new StMenuItem(JunSystem.$String("Sight point") + "...", new MenuPerformer(this, "changeSightPoint")));
		pointingMenu.add(new StMenuItem(JunSystem.$String("Up vector") + "...", new MenuPerformer(this, "changeUpVector")));
		pointingMenu.addSeparator();
		pointingMenu.add(new StMenuItem(JunSystem.$String("Zoom height") + "...", new MenuPerformer(this, "changeZoomHeight")));
		pointingMenu.add(new StMenuItem(JunSystem.$String("View factor") + "...", new MenuPerformer(this, "changeViewFactor")));
		viewMenu.add(pointingMenu);

		viewMenu.addSeparator();

		// Projection
		StMenu projectionMenu = new StMenu(JunSystem.$String("Projection"), $("projectionMenu"));
		StRadioButtonGroup projectionGroup = new StRadioButtonGroup();
		projectionMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Perspective Projection", "Perspective"), $("perspectiveProjectionMenu"), projectionGroup, new MenuPerformer(this, "perspectiveProjection")));
		projectionMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Parallel Projection", "Parallel"), $("parallelProjectionMenu"), projectionGroup, new MenuPerformer(this, "parallelProjection")));
		viewMenu.add(projectionMenu);

		// Presentation
		StMenu presentationMenu = new StMenu(JunSystem.$String("Presentation"), $("presentationMenu"));
		StRadioButtonGroup presentationGroup = new StRadioButtonGroup();
		presentationMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Solid Presentation", "Solid"), $("solidPresentationMenu"), presentationGroup, new MenuPerformer(this, "solidPresentation")));
		presentationMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Wireframe Presentation", "Wireframe"), $("wireframePresentationMenu"), presentationGroup, new MenuPerformer(this, "wireframePresentation")));
		presentationMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Hidden-line Presentation", "Hidden-line"), $("hiddenlinePresentation"), presentationGroup, new MenuPerformer(this, "hiddenlinePresentation")));
		viewMenu.add(presentationMenu);

		// Shading
		StMenu shadingMenu = new StMenu(JunSystem.$String("Shading"), $("shadingMenu"));
		StRadioButtonGroup shadingGroup = new StRadioButtonGroup();
		shadingMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Flat Shading", "Flat"), $("flatShadingMenu"), shadingGroup, new MenuPerformer(this, "flatShading")));
		shadingMenu.add(new StRadioButtonMenuItem(JunSystem.$String("Smooth Shading", "Smooth"), $("smoothShadingMenu"), shadingGroup, new MenuPerformer(this, "smoothShading")));
		viewMenu.add(shadingMenu);

		// Smoothing
		StMenu smoothingMenu = new StMenu(JunSystem.$String("Smoothing"), $("smoothingMenu"));
		smoothingMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Line smooth"), $("lineSmoothMenu"), new MenuPerformer(this, "lineSmooth")));
		smoothingMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Polygon smooth"), $("polygonSmoothMenu"), new MenuPerformer(this, "polygonSmooth")));
		viewMenu.add(smoothingMenu);

		viewMenu.addSeparator();

		// and others.
		viewMenu.add(new StMenuItem(JunSystem.$String("Reset"), $("resetMenu"), new MenuPerformer(this, "resetView")));
		viewMenu.add(this._createPresetProjectionMenu());

		return viewMenu;
	}

	/**
	 * Change inches.
	 * 
	 * @category menu messages
	 */
	public void changeInches() {
		String numberString = JunDialog.Request_(JunSystem.$String("Input your inches."), String.valueOf(this.inches()));
		if (numberString == null || numberString.length() == 0) {
			return;
		}

		try {
			double newInches = Double.parseDouble(numberString);
			this.inches_(newInches);
		} catch (NumberFormatException e) {
			JunDialog.Warn_(numberString + JunSystem.$String(" is invalid value."));
		}
	}

	/**
	 * Show the display object.
	 * 
	 * @category menu messages
	 */
	public void showDisplayObject() {
		this.displayObject().show();
	}

	/**
	 * Spawn the object.
	 * 
	 * @category menu messages
	 */
	public void spawnObject() {
		JunTerrainDisplayModel displayModel = new JunTerrainDisplayModel(terrainField);
		displayModel.defaultSightPoint_(this.displayProjection().sightPoint());
		displayModel.defaultEyePoint_(this.displayProjection().eyePoint());
		displayModel.defaultZoomHeight_(this.defaultZoomHeight());
		displayModel.defaultUpVector_(new Jun3dPoint(0, 0, 1));
		displayModel.inches_(this.inches());

		StView view = this.getView();
		if (view == null) {
			displayModel.open();
		} else {
			StRectangle box = new StRectangle(view.topComponent().getBounds());
			StRectangle area = new StRectangle(0, 0, box.width(), box.height());
			area = area.align_with_(area.topLeft(), new Point(box.right() + 5, box.top()));
			displayModel.openIn_(area.toRectangle());
		}

		displayModel.resetView();
		displayModel.changed_($("object"));
	}

	/**
	 * Create the current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject computeDisplayObject() {
		if (terrainField == null) {
			return null;
		}

		JunOpenGL3dObject newDisplayObject = terrainField.asJunOpenGL3dObject();
		newDisplayObject.paint_(Color.green);
		return newDisplayObject;
	}

	/**
	 * Answer the height of the eye at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category private
	 */
	protected double eyeHeightAtPoint_(Jun2dPoint aPoint) {
		double z = terrainField.zAt2dPoint_(aPoint);
		if (Double.isNaN(z)) {
			z = 0.0d;
		}
		return z + this.inches();
	}

	/**
	 * Move your eyes to the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected void moveTo_(Jun2dPoint aPoint) {
		JunOpenGLProjector projector = this.displayProjector();
		Jun3dPoint currentEyePoint = projector.eyePoint();
		Jun3dPoint newEyePoint = new Jun3dPoint(aPoint, this.eyeHeightAtPoint_(aPoint));
		Jun3dPoint newSightPoint = projector.sightPoint().plus_(newEyePoint).minus_(currentEyePoint);
		projector.sightPoint_(newSightPoint);
		projector.eyePoint_(newEyePoint);
		projector.upVector_(new Jun3dPoint(0, 0, 1));
		this.changed_($("projection"));
	}

	/**
	 * Move your eyes to the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void moveTo_(Jun3dPoint aPoint) {
		this.moveTo_(new Jun2dPoint(aPoint.x(), aPoint.y()));
	}

	/**
	 * Move your eyes to the specified point with the sight.
	 * 
	 * @param eye2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param sight2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	public void moveTo_sight_(Jun2dPoint eye2dPoint, Jun2dPoint sight2dPoint) {
		JunOpenGLProjector projector = this.displayProjector();
		Jun3dPoint newEyePoint = new Jun3dPoint(eye2dPoint, this.eyeHeightAtPoint_(eye2dPoint));
		Jun3dPoint newSightPoint = new Jun3dPoint(sight2dPoint, terrainField.zAt2dPoint_(sight2dPoint));
		projector.sightPoint_(newSightPoint);
		projector.eyePoint_(newEyePoint);
		projector.upVector_(new Jun3dPoint(0, 0, 1));
		this.changed_($("projection"));
	}

}