package jp.co.sra.jun.opengl.roughsketch;

import java.awt.BasicStroke;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.util.ArrayList;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayController;

/**
 * JunOpenGLRoughSketchController class
 * 
 *  @author    nisinaka
 *  @created   2005/10/25 (by nisinaka)
 *  @updated   2007/08/24 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun536 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLRoughSketchController.java,v 8.11 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunOpenGLRoughSketchController extends JunOpenGLDisplayController {

	protected ArrayList points;

	/**
	 * Answer my model as JunOpenGLRoughSketch.
	 * 
	 * @return jp.co.sra.jun.opengl.roughsketch.JunOpenGLRoughSketch
	 * @category model accessing
	 */
	public JunOpenGLRoughSketch getRoughSketch() {
		return (JunOpenGLRoughSketch) this.model();
	}

	/**
	 * Answer my current points.
	 * 
	 * @return java.awt.Point[]
	 * @category accessing
	 */
	public Point[] getPoints() {
		return (Point[]) this.points().toArray(new Point[this.points().size()]);
	}

	/**
	 * Add a point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void addPoint(Point aPoint) {
		if (this.points().size() > 0) {
			Point p = (Point) this.points().get(this.points().size() - 1);
			if (Distance(p, aPoint) <= JunOpenGLRoughSketch.MinimumDistance) {
				return;
			}
		}

		this.points().add(aPoint);
		this.drawPolyline();
	}

	/**
	 * Answer the array list of the points.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	protected ArrayList points() {
		if (points == null) {
			points = new ArrayList();
		}
		return points;
	}

	/**
	 * Flush the points.
	 * 
	 * @category accessing
	 */
	protected void flushPoints() {
		points = null;
	}

	/**
	 * Draw the polyline.
	 * 
	 * @category drawing
	 */
	protected void drawPolyline() {
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) this.getOpenGLDrawable().toComponent().getGraphics();
			aGraphics.setColor(JunOpenGLRoughSketch.NibColor);
			aGraphics.setStroke(new BasicStroke(JunOpenGLRoughSketch.NibWidth));

			Point[] points = this.getPoints();
			for (int i = 1; i < points.length; i++) {
				aGraphics.drawLine(points[i - 1].x, points[i - 1].y, points[i].x, points[i].y);
			}

		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Invoked when a mouse button has been pressed on the view.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayController#mousePressed(java.awt.event.MouseEvent)
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent event) {
		if (this.getRoughSketch().buttonState() != $("pencil")) {
			super.mousePressed(event);
			return;
		}

		this.flushPoints();
		this.addPoint(event.getPoint());
	}

	/**
	 * Invoked when a mouse button has been released on the view.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayController#mouseReleased(java.awt.event.MouseEvent)
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent event) {
		if (this.getRoughSketch().buttonState() != $("pencil")) {
			super.mouseReleased(event);
			return;
		}

		Point[] points = this.getPoints();
		Dimension size = this.getOpenGLDrawable().toComponent().getSize();
		if (Distance(points[0], points[points.length - 1]) > Math.max(size.width, size.height) / 2) {
			this.getRoughSketch().pencilOpenedPoints(points, this);
		} else {
			this.getRoughSketch().pencilClosedPoints(points, this);
		}

		this.flushPoints();
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param event java.awt.event.MouseEvent 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayController#mouseDragged(java.awt.event.MouseEvent)
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent event) {
		if (this.getRoughSketch().buttonState() != $("pencil")) {
			super.mouseDragged(event);
			return;
		}

		this.addPoint(event.getPoint());
	}

	/**
	 * Invoked when a mouse is entered on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseEntered(MouseEvent e) {
		super.mouseEntered(e);

		if (this.getRoughSketch().buttonState() == $("pencil")) {
			this.getOpenGLDrawable().toComponent().setCursor(JunCursors.PencilCursor());
		}
	}

	/**
	 * Regularize the point on the current view.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayController#regularizePoint_(java.awt.Point)
	 * @category private
	 */
	protected Jun2dPoint regularizePoint_(Point aPoint) {
		// TODO Auto-generated method stub
		return super.regularizePoint_(aPoint);
	}

	/**
	 * Answer the distance between the two points.
	 * 
	 * @param p1 java.awt.Point
	 * @param p2 java.awt.Point
	 * @return double
	 * @category Utilities
	 */
	protected static double Distance(Point p1, Point p2) {
		int dx = p1.x - p2.x;
		int dy = p1.y - p2.y;
		return Math.sqrt(dx * dx + dy * dy);
	}

}
