package jp.co.sra.jun.opengl.rotation;

import java.awt.Point;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.Vector;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.topology.elements.JunBody;
import jp.co.sra.jun.topology.globaloperators.JunROTATE;

/**
 * JunOpenGLRotationModel class
 * 
 *  @author    nisinaka
 *  @created   1998/12/09 (by nisinaka)
 *  @updated   2004/09/22 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLRotationModel.java,v 8.12 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunOpenGLRotationModel extends JunApplicationModel {
	/** The polyline. */
	protected JunOpenGLRotationPolyline polyline;

	/** The scale. */
	protected int scale;

	/** The collection of indices of the selected vertices. */
	protected Vector selection;

	/** The flag is true if you want to swap X-axis and Y-axis. */
	protected boolean swapXY;

	/** The flag is true if you want to show each axis on a view. */
	protected boolean showXAxis;
	protected boolean showYAxis;

	/** The grid size. */
	protected double grid;

	/** The last divisions. */
	protected int lastDivisions;

	/** The blockclosure evaluated when opening a file. */
	protected StBlockClosure openBlock;

	/** menu model. */
	protected StMenuBar _menuBar;

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		polyline = null;
		scale = 50;
		selection = null;
		swapXY = false;
		showXAxis = true;
		showYAxis = true;
		grid = Double.NaN;
		lastDivisions = 10;
		openBlock = null;
		_menuBar = null;
	}

	/**
	 * Answer the de-scaled vertex at the specified Point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint descaledPoint_(Point aPoint) {
		double scale = this.scale();

		if (swapXY) {
			return new Jun2dPoint(aPoint.y / scale * -1, aPoint.x / scale);
		} else {
			return new Jun2dPoint(aPoint.x / scale, aPoint.y / scale * -1);
		}
	}

	/**
	 * Answer the grid size of the receiver.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double grid() {
		return grid;
	}

	/**
	 * Answer the last divisions.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int lastDivisions() {
		return lastDivisions;
	}

	/**
	 * Set the last divisions.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void lastDivisions_(int aNumber) {
		lastDivisions = aNumber;
	}

	/**
	 * Answer the last vertex of the polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint lastVertex() {
		return this.polyline().last();
	}

	/**
	 * Answer the block closure for reading a LRT file.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public StBlockClosure openBlock() {
		return openBlock;
	}

	/**
	 * Answer the block closure for reading a LRT file.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void openBlock_(StBlockClosure aBlock) {
		openBlock = aBlock;
	}

	/**
	 * Answer the polyline of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.rotation.JunOpenGLRotationPolyline
	 * @category accessing
	 */
	public JunOpenGLRotationPolyline polyline() {
		if (polyline == null) {
			polyline = new JunOpenGLRotationPolyline();
			polyline.addDependent_(this);
		}

		return polyline;
	}

	/**
	 * Set the polyline of the receiver.
	 * 
	 * @param aPolyline jp.co.sra.jun.opengl.rotation.JunOpenGLRotationPolyline
	 * @category accessing
	 */
	public void polyline_(JunOpenGLRotationPolyline aPolyline) {
		if (polyline != null) {
			polyline.removeDependent_(this);
		}

		polyline = aPolyline;

		if (polyline != null) {
			polyline.addDependent_(this);
		}

		this.updateLoopMenu();
		this.changed_($("polyline"));
	}

	/**
	 * Answer the scale value of the receiver.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int scale() {
		return scale;
	}

	/**
	 * Set the scale value of the receiver and notify the modification to the
	 * dependents.
	 * 
	 * @param anInteger int
	 * @category accessing
	 */
	public void scale_(int anInteger) {
		scale = anInteger;
		this.changed_($("scale"));
	}

	/**
	 * Answer the scaled point of the Jun2dPoint.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point scaledPoint_(Jun2dPoint aJun2dPoint) {
		double scale = this.scale();

		if (swapXY) {
			return new Point((int) (aJun2dPoint.y() * scale), (int) (aJun2dPoint.x() * scale * -1));
		} else {
			return new Point((int) (aJun2dPoint.x() * scale), (int) (aJun2dPoint.y() * scale * -1));
		}
	}

	/**
	 * Answer true if the receiver wants to show X-axis on a view.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean showXAxis() {
		return showXAxis;
	}

	/**
	 * Set the flag whether the receiver wants to show X-axis on a view or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void showXAxis_(boolean aBoolean) {
		showXAxis = aBoolean;
		this.update_(new DependentEvent(this, $("showXAxis"), null));
	}

	/**
	 * Answer true if the receiver wants to show X-axis on a view.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean showYAxis() {
		return showYAxis;
	}

	/**
	 * Set the flag whether the receiver wants to show Y-axis on a view or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void showYAxis_(boolean aBoolean) {
		showYAxis = aBoolean;
		this.update_(new DependentEvent(this, $("showYAxis"), null));
	}

	/**
	 * Answer true if X-axis and Y-axis are needed to be swapped.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean swapXY() {
		return swapXY;
	}

	/**
	 * Set the flag for the axes direction.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void swapXY_(boolean aBoolean) {
		swapXY = aBoolean;
		this.update_(new DependentEvent(this, $("swapXY"), null));
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLRotationViewAwt(this);
		} else {
			return new JunOpenGLRotationViewSwing(this);
		}
	}

	/**
	 * Create the windows for the receiver. .
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Rotation Model");
	}

	/**
	 * Convert the divisions to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList divisionsToLispList() {
		return new JunLispCons($("divisions"), new Integer(this.lastDivisions()));
	}

	/**
	 * Initialize the attributes of the receiver with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public void fromLispList_(JunLispCons aList) {
		if (aList.head() != this.kindName()) {
			throw SmalltalkException.Error("unexpected error");
		}

		final JunOpenGLRotationModel self = this;
		((JunLispList) aList.tail()).do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLispCons each = (JunLispCons) anObject;

				if (each.head() == $("scale")) {
					self.scale_(((Number) each.tail()).intValue());
				}

				if (each.head() == $("swapXY")) {
					self.swapXY_(((Boolean) each.tail()).booleanValue());
				}

				if (each.head() == $("showXAxis")) {
					self.showXAxis_(((Boolean) each.tail()).booleanValue());
				}

				if (each.head() == $("showYAxis")) {
					self.showYAxis_(((Boolean) each.tail()).booleanValue());
				}

				if (each.head() == $("grid")) {
					if (each.tail() == null) {
						self.grid_(Double.NaN);
					} else {
						self.grid_(((Number) each.tail()).doubleValue());
					}
				}

				if (each.head() == $("divisions")) {
					self.lastDivisions_(((Number) each.tail()).intValue());
				}

				if (each.head() == $("JunOpenGLRotationPolyline")) {
					self.polyline_(JunOpenGLRotationPolyline.FromLispList_(each));
				}

				return null;
			}
		});
	}

	/**
	 * Convert the grid to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList gridToLispList() {
		return new JunLispCons($("grid"), (Double.isNaN(this.grid()) ? null : new Double(this.grid())));
	}

	/**
	 * Convert the scale to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList scaleToLispList() {
		return new JunLispCons($("scale"), new Integer(this.scale()));
	}

	/**
	 * Convert the flag showXAxis to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList showXAxisToLispList() {
		return new JunLispCons($("showXAxis"), new Boolean(this.showXAxis()));
	}

	/**
	 * Convert the flag showYAxis to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList showYAxisToLispList() {
		return new JunLispCons($("showYAxis"), new Boolean(this.showYAxis()));
	}

	/**
	 * Convert the flag swapXY to a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList swapXYToLispList() {
		return new JunLispCons($("swapXY"), new Boolean(this.swapXY()));
	}

	/**
	 * Answer the list representation of the JunOpenGLRotationModel.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = JunLispCons.Cell();
		list.head_(this.kindName());
		list.add_(this.scaleToLispList());
		list.add_(this.swapXYToLispList());
		list.add_(this.showXAxisToLispList());
		list.add_(this.showYAxisToLispList());
		list.add_(this.gridToLispList());
		list.add_(this.divisionsToLispList());
		list.add_(this.polyline().toLispList());

		return list;
	}

	/**
	 * Menu message - update loop menu.
	 * 
	 * @category menu accessing
	 */
	protected void updateLoopMenu() {
		StCheckBoxMenuItem aMenuItem = (StCheckBoxMenuItem) this._menuBar().atNameKey_($("toggleLoop"));
		aMenuItem.beSelected(this.polyline().loop());
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		this.updateLoopMenu();
		this.updateShowXAxisMenu();
		this.updateShowYAxisMenu();
		this.updateSwapXYMenu();
	}

	/**
	 * Menu message - update show x axis menu.
	 * 
	 * @category menu accessing
	 */
	protected void updateShowXAxisMenu() {
		StCheckBoxMenuItem aMenuItem = (StCheckBoxMenuItem) this._menuBar().atNameKey_($("toggleShowXAxis"));
		aMenuItem.beSelected(this.showXAxis());
	}

	/**
	 * Menu message - update show y axis menu.
	 * 
	 * @category menu accessing
	 */
	protected void updateShowYAxisMenu() {
		StCheckBoxMenuItem aMenuItem = (StCheckBoxMenuItem) this._menuBar().atNameKey_($("toggleShowYAxis"));
		aMenuItem.beSelected(this.showYAxis());
	}

	/**
	 * Menu message - update swap xy menu.
	 * 
	 * @category menu accessing
	 */
	protected void updateSwapXYMenu() {
		StCheckBoxMenuItem aMenuItem = (StCheckBoxMenuItem) this._menuBar().atNameKey_($("toggleSwapXY"));
		aMenuItem.beSelected(this.swapXY());
	}

	/**
	 * Menu message - delete the selected vertices.
	 * 
	 * @category menu messages
	 */
	public void deleteSelection() {
		Vector selection = this.selectedVertices();
		this.deselectAll();
		this.removeVertices_(selection);
	}

	/**
	 * Menu message - open a file.
	 * 
	 * @throws jp.co.sra.smalltlak.SmalltalkException
	 * @category menu messages
	 */
	public void doOpen() {
		File file = JunFileRequesterDialog.RequestFile($String("Select a <1p> file.", null, "lrt"), new File("rotation.lrt"));
		if (file == null) {
			return;
		}

		try {
			if (this.numVertices() > 0) {
				if (this.openBlock() != null) {
					this.openBlock().value_(file);
				} else {
					JunOpenGLRotationModel aRotationModel = new JunOpenGLRotationModel();
					aRotationModel.readFromLRT10_(file);
					if (aRotationModel.numVertices() > 0) {
						aRotationModel.open();
					}
				}
			} else {
				this.readFromLRT10_(file);
			}
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Menu message - rotate.
	 * 
	 * @category menu messages
	 */
	public void doRotate() {
		String divisions;
		JunBody aBody;

		divisions = new Integer(lastDivisions).toString();
		divisions = JunDialog.Request_(JunSystem.$String("How many divisions?"), divisions);

		if (divisions != null) {
			try {
				aBody = this.rotatedBody_(new Integer(divisions).intValue());

				if (aBody != null) {
					aBody.show();
				}
			} catch (NumberFormatException e) {
			}
		}
	}

	/**
	 * Menu message - save to a file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void doSave() {
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "lrt"), new File("rotation.lrt"));
		if (file == null) {
			return;
		}

		try {
			this.writeToLRT10_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Menu message - invert the selection.
	 * 
	 * @category menu messages
	 */
	public void invertSelection() {
		int size = this.numVertices();

		for (int i = 0; i < size; i++) {
			if (this.indexSelected_(i)) {
				this.deselectIndex_(i);
			} else {
				this.selectIndex_(i);
			}
		}
	}

	/**
	 * Menu message - reverse the vertices.
	 * 
	 * @category menu messages
	 */
	public void reverseVertices() {
		Vector oldSelection = (Vector) this.selectedIndices().clone();
		this.polyline().reverseVertices();
		this.deselectAll();

		int numVertices = this.numVertices();

		for (int i = 0; i < oldSelection.size(); i++) {
			int each = ((Number) oldSelection.elementAt(i)).intValue();
			this.selectIndex_(numVertices - each);
		}
	}

	/**
	 * Menu message - select all vertices.
	 * 
	 * @category menu messages
	 */
	public void selectAll() {
		int size = this.numVertices();

		for (int each = 0; each < size; each++) {
			this.selectIndex_(each);
		}
	}

	/**
	 * Menu message - set the grid.
	 * 
	 * @category menu messages
	 */
	public void setGrid() {
		String newGrid;

		if (!Double.isNaN(grid)) {
			newGrid = new Double(grid).toString();
		} else {
			newGrid = "none";
		}

		newGrid = JunDialog.Request_(JunSystem.$String("New grid value ('none' for no grid)?"), newGrid);

		if (newGrid != null) {
			try {
				this.grid_(new Double(newGrid).doubleValue());
			} catch (NumberFormatException e) {
				this.grid_(Double.NaN);
			}
		}
	}

	/**
	 * Menu message - swap the selected vertices.
	 * 
	 * @category menu messages
	 */
	public void swapVertices() {
		Jun2dPoint tempVertex;

		if (this.numSelection() == 2) {
			tempVertex = this.selectedVertexAt_(0);
			this.vertexAt_put_(this.selectedIndexAt_(0), this.selectedVertexAt_(1));
			this.vertexAt_put_(this.selectedIndexAt_(1), tempVertex);
		}
	}

	/**
	 * Menu message - toggle the loop.
	 * 
	 * @category menu messages
	 */
	public void toggleLoop() {
		this.polyline().loop_(!this.polyline().loop());
	}

	/**
	 * Menu message - toggle the showXAxis.
	 * 
	 * @category menu messages
	 */
	public void toggleShowXAxis() {
		this.showXAxis_(!this.showXAxis());
	}

	/**
	 * Menu message - toggle the showYAxis.
	 * 
	 * @category menu messages
	 */
	public void toggleShowYAxis() {
		this.showYAxis_(!this.showYAxis());
	}

	/**
	 * Menu message - toggle the swapXY.
	 * 
	 * @category menu messages
	 */
	public void toggleSwapXY() {
		this.swapXY_(!this.swapXY());
	}

	/**
	 * Zoom in.
	 * 
	 * @category menu messages
	 */
	public void zoomIn() {
		this.scale_(this.scale() * 2);
	}

	/**
	 * Zoom out.
	 * 
	 * @category menu messages
	 */
	public void zoomOut() {
		this.scale_(this.scale() / 2);
	}

	/**
	 * Load the rotational model from the reader.
	 * 
	 * @param aReader java.lang.String
	 * @exception IOException
	 * @category reading
	 */
	public void loadFromLRT10_(BufferedReader aReader) throws IOException {
		StringWriter aWriter = new StringWriter();
		try {
			int ch;
			while ((ch = aReader.read()) > 0) {
				aWriter.write(ch);
			}
			aWriter.flush();
			this.fromLispList_((JunLispCons) JunLispParser.Parse_(aWriter.toString()));
		} finally {
			aWriter.close();
		}
	}

	/**
	 * Read the rotational model from the file.
	 * 
	 * @param aFile java.io.File
	 * @exception IOException
	 * @category reading
	 */
	public void readFromLRT10_(File aFile) throws IOException {
		FileReader aFileReader;
		try {
			aFileReader = new FileReader(aFile);
		} catch (FileNotFoundException e) {
			return;
		}

		BufferedReader aReader = new BufferedReader(aFileReader);
		try {
			this.loadFromLRT10_(aReader);
		} finally {
			aReader.close();
		}
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createEditMenu());
		}
		return _menuBar;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.addAll(this._editMenu().menuItems());
		return aPopupMenu;
	}

	/**
	 * Create an "Edit" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createEditMenu() {
		StMenu editMenu = new StMenu(JunSystem.$String("Edit"), $("editMenu"));
		editMenu.add(new StMenuItem(JunSystem.$String("Rotate") + "...", new MenuPerformer(this, "doRotate")));
		editMenu.addSeparator();
		editMenu.add(new StMenuItem(JunSystem.$String("Open") + "...", new MenuPerformer(this, "doOpen")));
		editMenu.add(new StMenuItem(JunSystem.$String("Save") + "...", new MenuPerformer(this, "doSave")));
		editMenu.addSeparator();
		editMenu.add(new StMenuItem(JunSystem.$String("Select all"), new MenuPerformer(this, "selectAll")));
		editMenu.add(new StMenuItem(JunSystem.$String("Deselect all"), new MenuPerformer(this, "deselectAll")));
		editMenu.add(new StMenuItem(JunSystem.$String("Invert selection"), new MenuPerformer(this, "invertSelection")));
		editMenu.addSeparator();
		editMenu.add(new StMenuItem(JunSystem.$String("Reverse all vertices"), new MenuPerformer(this, "reverseVertices")));
		editMenu.add(new StMenuItem(JunSystem.$String("Swap selected vertices"), new MenuPerformer(this, "swapVertices")));
		editMenu.add(new StMenuItem(JunSystem.$String("Delete selected vertices"), new MenuPerformer(this, "deleteSelection")));
		editMenu.addSeparator();
		editMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Loop polyline"), $("toggleLoop"), new MenuPerformer(this, "toggleLoop")));
		editMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Swap XY axes"), $("toggleSwapXY"), new MenuPerformer(this, "toggleSwapXY")));
		editMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Show X-Axis", "Show X axis"), $("toggleShowXAxis"), new MenuPerformer(this, "toggleShowXAxis")));
		editMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Show Y-Axis", "Show Y axis"), $("toggleShowYAxis"), new MenuPerformer(this, "toggleShowYAxis")));
		editMenu.add(new StMenuItem(JunSystem.$String("Grid") + "...", new MenuPerformer(this, "setGrid")));
		editMenu.addSeparator();
		editMenu.add(new StMenuItem(JunSystem.$String("Zoom in"), new MenuPerformer(this, "zoomIn")));
		editMenu.add(new StMenuItem(JunSystem.$String("Zoom out"), new MenuPerformer(this, "zoomOut")));
		return editMenu;
	}

	/**
	 * Answer my edit menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _editMenu() {
		return (StMenu) this._menuBar().atNameKey_($("editMenu"));
	}

	/**
	 * Create a rotated body from the receiver.
	 * 
	 * @param divisions int
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category rotating
	 */
	public JunBody rotatedBody_(int divisions) {
		if ((divisions < 3) || (this.numVertices() < 2)) {
			return null;
		}

		Object result = this.verticesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dPoint each = (Jun2dPoint) anObject;
				if (each.y() < 0) {
					return Boolean.FALSE;
				}
				return null;
			}
		});

		if (result == Boolean.FALSE) {
			JunDialog.Warn_(JunSystem.$String("There is a vertex whose y < 0."));
			return null;
		}

		final JunOpenGLRotationModel self = this;
		result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				final Jun2dLine edge = (Jun2dLine) anObject;
				if (edge.from().equals(edge.to()) == false) {
					Object result2 = self.edgesDo_(new StBlockClosure() {
						public Object value_(Object anObject) {
							Jun2dLine edge2 = (Jun2dLine) anObject;
							if (edge2.from().equals(edge2.to()) == false) {
								Jun2dPoint point = edge.lineSegmentIntersectingPointWithLineSegment_(edge2);
								if ((point != null) && (point.equals(edge.from()) == false) && (point.equals(edge.to()) == false) && (point.equals(edge2.from()) == false) && (point.equals(edge2.to()) == false)) {
									return Boolean.FALSE;
								}
							}
							return null;
						}
					});

					if (result2 != null) {
						return result2;
					}
				}
				return null;
			}
		});

		if (result == Boolean.FALSE) {
			JunDialog.Warn_(JunSystem.$String("Some segments intersect."));
			return null;
		}

		final StValueHolder sumOfProductsHolder = new StValueHolder(0.0d);
		final StValueHolder lastEdgeHolder = new StValueHolder();
		this.edgesDo_flag_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dLine edge = (Jun2dLine) anObject;
				Jun2dLine lastEdge = (Jun2dLine) lastEdgeHolder.value();
				if (lastEdge != null) {
					double sumOfProducts = sumOfProductsHolder._doubleValue();
					sumOfProducts += ((Jun2dPoint) lastEdge.to().minus_(lastEdge.from())).product_((Jun2dPoint) edge.to().minus_(edge.from()));
					sumOfProductsHolder.value_(sumOfProducts);
				}
				lastEdgeHolder.value_(edge);
				return null;
			}
		}, 2);

		Jun2dPoint[] vertices = this.allVertices();
		if (sumOfProductsHolder._doubleValue() < 0) {
			Jun2dPoint[] reversedVertices = new Jun2dPoint[vertices.length];
			for (int i = 0; i < vertices.length; i++) {
				reversedVertices[i] = vertices[vertices.length - i - 1];
			}
			vertices = reversedVertices;
		}

		JunBody body = JunBody.RotationSweep2dPoints_divisions_(vertices, divisions);
		if (this.swapXY() == true) {
			JunROTATE.Body_point_vector_angle_(body, new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 1, 0), JunAngle.FromDeg_(-90)).doOperation();
		}
		lastDivisions = divisions;
		return body;
	}

	/**
	 * Convert the collection of the selected indices as an array of Integer.
	 * 
	 * @return java.lang.Integer[]
	 * @category selection accessing
	 */
	public Integer[] _selectedIndicesAsArray() {
		int size = selection.size();
		Integer[] anArray = new Integer[size];
		selection.copyInto(anArray);

		return anArray;
	}

	/**
	 * De-select all of the selected vertices.
	 * 
	 * @category selection accessing
	 */
	public void deselectAll() {
		for (int i = this.selectedVertices().size(); (--i) >= 0;) {
			Jun2dPoint each = (Jun2dPoint) this.selectedVertices().elementAt(i);
			this.deselectVertex_(each);
		}
	}

	/**
	 * De-select the vertex specified with the index.
	 * 
	 * @param index int
	 * @category selection accessing
	 */
	public void deselectIndex_(int index) {
		Integer anIndex = new Integer(index);

		if (this.selectedIndices().contains(anIndex)) {
			this.selectedIndices().removeElement(anIndex);
			this.changed_($("selection"));
		}
	}

	/**
	 * De-select the vertex.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category selection accessing
	 */
	public void deselectVertex_(Jun2dPoint aJun2dPoint) {
		int index = this.vertexIndex_(aJun2dPoint);

		if (index >= 0) {
			this.deselectIndex_(index);
		}
	}

	/**
	 * Answer true if the index is one of the selected indices, otherwise false.
	 * 
	 * @param index int
	 * @return boolean
	 * @category selection accessing
	 */
	public boolean indexSelected_(int index) {
		return this.selectedIndices().contains(new Integer(index));
	}

	/**
	 * Answer the number of the selected vertices.
	 * 
	 * @return int
	 * @category selection accessing
	 */
	public int numSelection() {
		return this.selectedIndices().size();
	}

	/**
	 * Answer the index of the selected vertex with the specified index.
	 * 
	 * @param index int
	 * @return int
	 * @category selection accessing
	 */
	public int selectedIndexAt_(int index) {
		return ((Number) this.selectedIndices().elementAt(index)).intValue();
	}

	/**
	 * Answer the collection of index of the selected vertices.
	 * 
	 * @return java.util.Vector
	 * @category selection accessing
	 */
	public Vector selectedIndices() {
		if (selection == null) {
			selection = new Vector();
		}
		return selection;
	}

	/**
	 * Answer the selected vertex with the specified index.
	 * 
	 * @param anIndex int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category selection accessing
	 */
	public Jun2dPoint selectedVertexAt_(int anIndex) {
		return this.vertexAt_(this.selectedIndexAt_(anIndex));
	}

	/**
	 * Answer the collection of the selected vertices.
	 * 
	 * @return java.util.Vector
	 * @category selection accessing
	 */
	public Vector selectedVertices() {
		int size = this.numSelection();
		Vector vertices = new Vector(size);

		for (int each = 0; each < size; each++) {
			vertices.addElement(this.selectedVertexAt_(each));
		}

		return vertices;
	}

	/**
	 * Select the vertex specified with the index.
	 * 
	 * @param index int
	 * @category selection accessing
	 */
	public void selectIndex_(int index) {
		if ((0 <= index) && (index < this.numVertices())) {
			Integer anIndex = new Integer(index);

			if (this.selectedIndices().contains(anIndex)) {
				this.selectedIndices().removeElement(anIndex);
			}

			this.selectedIndices().addElement(anIndex);
			this.changed_($("selection"));
		}
	}

	/**
	 * Select the vertex.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category selection accessing
	 */
	public void selectVertex_(Jun2dPoint aJun2dPoint) {
		int index = this.vertexIndex_(aJun2dPoint);

		if (index >= 0) {
			this.selectIndex_(index);
		}
	}

	/**
	 * Answer true if the Jun2dPoint is selected, otherwise false.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category selection accessing
	 */
	public boolean vertexSelected_(Jun2dPoint aJun2dPoint) {
		return this.selectedVertices().contains(aJun2dPoint);
	}

	/**
	 * Action for the update notification.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol anAspectSymbol = evt.getAspect();

		if (anAspectSymbol == $("loop")) {
			this.updateLoopMenu();
		}

		if (anAspectSymbol == $("swapXY")) {
			this.updateSwapXYMenu();
		}

		if (anAspectSymbol == $("showXAxis")) {
			this.updateShowXAxisMenu();
		}

		if (anAspectSymbol == $("showYAxis")) {
			this.updateShowYAxisMenu();
		}

		this.changed_(anAspectSymbol);
	}

	/**
	 * Add the vertex to the receiver.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category vertices accessing
	 */
	public void addVertex_(Jun2dPoint aJun2dPoint) {
		this.polyline().add_(aJun2dPoint);
	}

	/**
	 * Insert the vertex right before the specified index.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param index int
	 * @category vertices accessing
	 */
	public void addVertex_beforeIndex_(Jun2dPoint aJun2dPoint, int index) {
		this.polyline().add_beforeIndex_(aJun2dPoint, index);
	}

	/**
	 * Answer the array of all vertices.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category vertices accessing
	 */
	public Jun2dPoint[] allVertices() {
		if (this.polyline().loop() == true) {
			Vector vertices = this.polyline().vertices();
			int size = vertices.size();
			Jun2dPoint[] arrayOfVertices = new Jun2dPoint[size + 1];
			vertices.copyInto(arrayOfVertices);
			arrayOfVertices[size] = this.firstVertex();

			return arrayOfVertices;
		} else {
			return this.polyline()._verticesAsArray();
		}
	}

	/**
	 * Enumerate all edges of the polyline and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category vertices accessing
	 */
	public Object edgesDo_(StBlockClosure aBlock) {
		return this.polyline().edgesDo_(aBlock);
	}

	/**
	 * Enumerate all edges of the polyline and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aFlag int
	 * @return java.lang.Object
	 * @category vertices accessing
	 */
	public Object edgesDo_flag_(StBlockClosure aBlock, int aFlag) {
		return this.polyline().edgesDo_flag_(aBlock, aFlag);
	}

	/**
	 * Answer the first vertex of the polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category vertices accessing
	 */
	public Jun2dPoint firstVertex() {
		return this.polyline().first();
	}

	/**
	 * Set the grid size of the receiver and notify the modification to the dependents.
	 * 
	 * @param aNumber double
	 * @category vertices accessing
	 */
	public void grid_(double aNumber) {
		grid = aNumber;
		this.changed_($("grid"));
	}

	/**
	 * Answer the index of the nearest vertex from the Jun2dPoint.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return int
	 * @category vertices accessing
	 */
	public int nearestVertexIndex_(final Jun2dPoint aJun2dPoint) {
		int numSelection = this.numSelection();

		if (numSelection == 2) {
			int index = this.selectedIndexAt_(0);
			int index2 = this.selectedIndexAt_(1);

			if (Math.abs(index - index2) == 1) {
				return Math.max(index, index2);
			}
		}

		if (numSelection == 1) {
			int index = this.selectedIndexAt_(0);

			if (index == (this.numVertices() - 1)) {
				return index + 1;
			}

			if (index == 0) {
				return index;
			}

			Jun2dLine line = new Jun2dLine(this.vertexAt_(index), this.vertexAt_(index - 1));
			double delta = line.lineSegmentDistanceFromPoint_(aJun2dPoint);
			line = new Jun2dLine(this.vertexAt_(index), this.vertexAt_(index + 1));

			double delta2 = line.lineSegmentDistanceFromPoint_(aJun2dPoint);

			if (delta2 < delta) {
				return index + 1;
			}

			return index;
		}

		final StValueHolder index = new StValueHolder(0);
		final StValueHolder index2 = new StValueHolder(0);
		final StValueHolder delta = new StValueHolder();
		this.edgesDo_flag_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dLine edge = (Jun2dLine) anObject;
				double delta2;

				if (edge.from().equals(edge.to())) {
					delta2 = edge.from().minus_(aJun2dPoint).length();
				} else {
					delta2 = edge.lineSegmentDistanceFromPoint_(aJun2dPoint);
				}

				if ((delta.value() == null) || (delta2 < delta._doubleValue())) {
					index.value_(index2.value());
					delta.value_(delta2);
				}

				index2.value_(index2._intValue() + 1);

				return null;
			}
		}, 1);

		return index._intValue();
	}

	/**
	 * Answer the number of the vertices of the polyline.
	 * 
	 * @return int
	 * @category vertices accessing
	 */
	public int numVertices() {
		return this.polyline().size();
	}

	/**
	 * Remove the vertex from the receiver.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category vertices accessing
	 */
	public void removeVertex_(Jun2dPoint aJun2dPoint) {
		int index = this.vertexIndex_(aJun2dPoint);

		if (index >= 0) {
			this.removeVertexAtIndex_(index);
		}
	}

	/**
	 * Remove the vertex specified with the index from the receiver.
	 * 
	 * @param index int
	 * @category vertices accessing
	 */
	public void removeVertexAtIndex_(int index) {
		this.polyline().removeAtIndex_(index);
		this.deselectIndex_(index);

		for (int each = 0; each < this.numSelection(); each++) {
			int anIndex = this.selectedIndexAt_(each);

			if (anIndex > index) {
				this.selectedIndices().setElementAt(new Integer(anIndex - 1), each);
			}
		}
	}

	/**
	 * Remove the vertices.
	 * 
	 * @param arrayOfJun2dPoint java.util.Vector
	 * @category vertices accessing
	 */
	public void removeVertices_(Vector arrayOfJun2dPoint) {
		int size = arrayOfJun2dPoint.size();

		for (int i = 0; i < size; i++) {
			Jun2dPoint each = (Jun2dPoint) arrayOfJun2dPoint.elementAt(i);
			this.removeVertex_(each);
		}
	}

	/**
	 * Answer the Jun2dPoint as a vertex on the polyline with the specified
	 * index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category vertices accessing
	 */
	public Jun2dPoint vertexAt_(int index) {
		return this.polyline().at_(index);
	}

	/**
	 * Put the vertex at the index.
	 * 
	 * @param index int
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category vertices accessing
	 */
	public void vertexAt_put_(int index, Jun2dPoint aJun2dPoint) {
		this.polyline().at_put_(index, aJun2dPoint);
	}

	/**
	 * Answer the index of the vertex specified with the Jun2dPoint. If there
	 * is no such vertex, return -1.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return int
	 * @category vertices accessing
	 */
	public int vertexIndex_(Jun2dPoint aJun2dPoint) {
		for (int each = 0; each < this.numVertices(); each++) {
			if (aJun2dPoint.equals(this.vertexAt_(each))) {
				return each;
			}
		}

		return -1;
	}

	/**
	 * Enumerate all vertices of the polyline and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category vertices accessing
	 */
	public Object verticesDo_(StBlockClosure aBlock) {
		return this.polyline().verticesDo_(aBlock);
	}

	/**
	 * Answer the default stamp string for LRT1.0.
	 * 
	 * @return java.lang.String
	 * @category writing
	 */
	protected String defaultStampForLRT10() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("%LRT V1.0 List Rotation Transmission (Lisp S Expression)");
		pw.println("% This file was created by " + JunSystem.System() + JunSystem.Version());
		pw.println("% " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Save the receiver to the writer.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @exception IOException
	 * @category writing
	 */
	public void saveToLRT10_(BufferedWriter aWriter) throws IOException {
		aWriter.write(this.defaultStampForLRT10());
		this.toLispList().saveOn_(aWriter);
	}

	/**
	 * Write the receiver to the file.
	 * 
	 * @param aFile java.io.File
	 * @exception IOException
	 * @category writing
	 */
	public void writeToLRT10_(File aFile) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));

		try {
			this.saveToLRT10_(aWriter);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}
}
