package jp.co.sra.jun.opengl.projection;

import java.awt.Rectangle;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunOpenGLProjection class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/10/27 (by MATSUDA Ryouichi)
 *  @updated   2004/10/22 (by nisinaka)
 *  @updated   2006/10/13 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLProjection.java,v 8.13 2008/02/20 06:32:48 nisinaka Exp $
 */
public abstract class JunOpenGLProjection extends JunAbstractObject {

	protected String description;
	protected Jun3dPoint eyePoint;
	protected Jun3dPoint sightPoint;
	protected Jun3dPoint upVector;
	protected double viewFactor;
	protected double near;
	protected double far;

	public static Jun3dPoint DefaultEyePoint = new Jun3dPoint(30, 30, 30);
	public static Jun3dPoint DefaultSightPoint = new Jun3dPoint(0, 0, 0);
	public static Jun3dPoint DefaultUpVector = new Jun3dPoint(0, 0, 1);
	public static double DefaultViewFactor = 50;
	public static double DefaultNear = 0.001;
	public static double DefaultFar = 1000;

	/**
	 * Create a new default projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category Defaults
	 */
	public static JunOpenGLProjection Default() {
		return DefaultPerspectiveProjection();
	}

	/**
	 * Create a new default perspective peojection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category Defaults
	 */
	public static JunOpenGLProjection DefaultPerspectiveProjection() {
		return new JunOpenGLPerspectiveProjection();
	}

	/**
	 * Create a new instance of JunOpenGLProjection and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLProjection() {
		super();
	}

	/**
	 * Create a new parallel projection and set parameters and answer it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param height double
	 * @param near double
	 * @param far double
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @category Instance creation
	 * @deprecated since Jun497
	 */
	public static JunOpenGLParallelProjection ParallelProjectionEyePoint_sightPoint_upVector_height_near_far_(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double height, double near, double far) {
		return new JunOpenGLParallelProjection(eyePoint, sightPoint, upVector, height, near, far);
	}

	/**
	 * Create a new perspective projection and set parameters and answer it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fovy jp.co.sra.jun.geometry.basic.JunAngle
	 * @param near double
	 * @param far double
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @category Instance creation
	 * @deprecated since Jun497
	 */
	public static JunOpenGLPerspectiveProjection PerspectiveProjectionEyePoint_sightPoint_upVector_fovy_near_far_(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, JunAngle fovy, double near, double far) {
		return new JunOpenGLPerspectiveProjection(eyePoint, sightPoint, upVector, fovy, near, far);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		viewFactor = DefaultViewFactor;
	}

	/**
	 * Answer my projection type as StSymbol.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public abstract StSymbol type();

	/**
	 * Answer my current description.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String description() {
		return description;
	}

	/**
	 * Set my new description.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void description_(String aString) {
		description = aString;
	}

	/**
	 * Answer my current eyePoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint eyePoint() {
		return eyePoint;
	}

	/**
	 * Set my new eyePoint.
	 * 
	 * @param newEyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void eyePoint_(Jun3dPoint newEyePoint) {
		this.setEyePoint_(newEyePoint);
		this.fitDistance();
		this.normalizeUpVector();
	}

	/**
	 * Answer my current sightPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint sightPoint() {
		return sightPoint;
	}

	/**
	 * Set my new sightPoint.
	 * 
	 * @param newSightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void sightPoint_(Jun3dPoint newSightPoint) {
		this.setSightPoint_(newSightPoint);
		this.fitDistance();
		this.normalizeUpVector();
	}

	/**
	 * Set the eye point and the sight point at the same time.
	 * 
	 * @param pointOfEye jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointOfSight jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void eyePoint_sightPoint_(Jun3dPoint pointOfEye, Jun3dPoint pointOfSight) {
		if (pointOfEye.equal_(pointOfSight)) {
			return;
		}

		this.setEyePoint_(pointOfEye);
		this.setSightPoint_(pointOfSight);
	}

	/**
	 * Answer my current upVector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint upVector() {
		return upVector;
	}

	/**
	 * Set my new upVector.
	 * 
	 * @param newUpVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void upVector_(Jun3dPoint newUpVector) {
		this.setUpVector_(newUpVector);
		this.normalizeUpVector();
	}

	/**
	 * Answer my current viewFactor.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double viewFactor() {
		return viewFactor;
	}

	/**
	 * Set my new viewFactor.
	 * 
	 * @param newViewFactor double
	 * @category accessing
	 */
	public void viewFactor_(double newViewFactor) {
		this.setViewFactor_(newViewFactor);
		this.fitDistance();
		this.normalizeUpVector();
	}

	/**
	 * Answer my current near.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double near() {
		return near;
	}

	/**
	 * Set my new near.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void near_(double aNumber) {
		near = aNumber;
	}

	/**
	 * Answer my current far.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double far() {
		return far;
	}

	/**
	 * Set my new far.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void far_(double aNumber) {
		far = aNumber;
	}

	/**
	 * Answer my regular height.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double regularHeight();

	/**
	 * Answer a distance.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double distance() {
		return this.sightPoint().minus_(this.eyePoint()).abs().length();
	}

	/**
	 * Answer the center point of the near.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint nearCenter() {
		return this.eyePoint().plus_(this.unitSightVector().multipliedBy_(this.near()));
	}

	/**
	 * Answer the center point of the far.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint farCenter() {
		return this.eyePoint().plus_(this.unitSightVector().multipliedBy_(this.far()));
	}

	/**
	 * Answer my current right vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint rightVector() {
		return this.sightVector().product_(this.upVector());
	}

	/**
	 * Set my new right vector.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void rightVector_(Jun3dPoint aJun3dPoint) {
		this.upVector_(aJun3dPoint.product_(this.sightVector()).unitVector());
	}

	/**
	 * Answer my current sight vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint sightVector() {
		return this.sightPoint().minus_(this.eyePoint());
	}

	/**
	 * Answer my unit upVector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitUpVector() {
		return this.upVector().dividedBy_(this.upVector().length());
	}

	/**
	 * Answer my unit rightVector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitRightVector() {
		Jun3dPoint rightVector = this.rightVector();
		return rightVector.dividedBy_(rightVector.length());
	}

	/**
	 * Answer my unit sightVector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitSightVector() {
		Jun3dPoint sightVector = this.sightVector();
		return sightVector.dividedBy_(sightVector.length());
	}

	/**
	 * Answer true if I am a parallel projection.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isParallel() {
		return false;
	}

	/**
	 * Answer true if I am a perspective projection.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPerspective() {
		return false;
	}

	/**
	 * Project on a RenderingContext.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category projection
	 */
	public abstract void projectOn_(JunOpenGLRenderingContext aRenderingContext);

	/**
	 * Move the eyePoint to aNumber forward to the center
	 * 
	 * @param aNumber double
	 * @category moving
	 */
	public void moveForward_(double aNumber) {
		this.eyePoint_(this.eyePoint().plus_(this.unitSightVector().multipliedBy_(aNumber)));
	}

	/**
	 * Move the eyePoint to aNumber backward from the center
	 * 
	 * @param aNumber double
	 * @category moving
	 */
	public void moveBackward_(double aNumber) {
		this.eyePoint_(this.eyePoint().minus_(this.unitSightVector().multipliedBy_(aNumber)));
	}

	/**
	 * Move the eyePoint to aNumber backward from the center
	 * 
	 * @param aNumber double
	 * @category moving
	 */
	public void moveLeft_(double aNumber) {
		this.eyePoint_(this.eyePoint().minus_(this.unitRightVector().multipliedBy_(aNumber)));
	}

	/**
	 * Move the eyePoint to aNumber backward from the center
	 * 
	 * @param aNumber double
	 * @category moving
	 */
	public void moveRight_(double aNumber) {
		this.eyePoint_(this.eyePoint().plus_(this.unitRightVector().multipliedBy_(aNumber)));
	}

	/**
	 * Pan the projection.
	 * 
	 * @category zooming
	 */
	public void pan() {
		this.pan_(2);
	}

	/**
	 * Pan the projection with the factor.
	 * 
	 * @param factor double
	 * @category zooming
	 */
	public abstract void pan_(double factor);

	/**
	 * Zoom the projection.
	 * 
	 * @category zooming
	 */
	public void zoom() {
		this.zoom_(2);
	}

	/**
	 * Zoom the projection with the factor.
	 * 
	 * @param factor double
	 * @category zooming
	 */
	public abstract void zoom_(double factor);

	/**
	 * Answer my current zoom height.
	 *
	 * @return double
	 * @category zooming
	 */
	public abstract double zoomHeight();

	/**
	 * Set my new zoom height.
	 * 
	 * @param aNumber double
	 * @category zooming
	 */
	public abstract void zoomHeight_(double aNumber);

	/**
	 * Answer a Block of boundary condition for a Rectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category boundary
	 */
	public StBlockClosure boundaryConditionForRectangle_(Rectangle aRectangle) {
		final Jun3dPoint topLeft = this.translateTo3dPointInFarBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x, aRectangle.y));
		final Jun3dPoint topRight = this.translateTo3dPointInFarBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x + aRectangle.width, aRectangle.y));
		final Jun3dPoint bottomLeft = this.translateTo3dPointInFarBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x, aRectangle.y + aRectangle.height));
		final Jun3dPoint bottomRight = this.translateTo3dPointInFarBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x + aRectangle.width, aRectangle.y + aRectangle.height));
		final Jun3dPoint anotherTopLeft = this.translateTo3dPointInNearBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x, aRectangle.y));
		final Jun3dPoint anotherBottomRight = this.translateTo3dPointInNearBoundaryPlaneFromPoint_(new Jun2dPoint(aRectangle.x + aRectangle.width, aRectangle.y + aRectangle.height));
		final StBlockClosure leftBlock = this.boundaryConditionFrom3dPoint_and_and_includes_(topLeft, bottomLeft, anotherTopLeft, topRight);
		final StBlockClosure rightBlock = this.boundaryConditionFrom3dPoint_and_and_includes_(topRight, bottomRight, anotherBottomRight, topLeft);
		final StBlockClosure topBlock = this.boundaryConditionFrom3dPoint_and_and_includes_(topLeft, topRight, anotherTopLeft, bottomLeft);
		final StBlockClosure bottomBlock = this.boundaryConditionFrom3dPoint_and_and_includes_(bottomLeft, bottomRight, anotherBottomRight, topLeft);

		return new StBlockClosure() {
			public Object value_(Object o) {
				if (((Boolean) leftBlock.value_(o)).booleanValue()) {
					if (((Boolean) rightBlock.value_(o)).booleanValue()) {
						if (((Boolean) topBlock.value_(o)).booleanValue()) {
							if (((Boolean) bottomBlock.value_(o)).booleanValue()) {
								return Boolean.TRUE;
							}
						}
					}
				}
				return Boolean.FALSE;
			}
		};
	}

	/**
	 * Answer a Block of boundary condition from parameters.
	 * 
	 * @param a3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint4 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category boundary
	 */
	public StBlockClosure boundaryConditionFrom3dPoint_and_and_includes_(Jun3dPoint a3dPoint1, Jun3dPoint a3dPoint2, Jun3dPoint a3dPoint3, Jun3dPoint a3dPoint4) {
		if (a3dPoint1.equals(a3dPoint2) || a3dPoint2.equals(a3dPoint3) || a3dPoint3.equals(a3dPoint1)) {
			return new StBlockClosure() {
				public Object value_(Object o) {
					return Boolean.FALSE;
				}
			};
		}

		JunPlane plane = new JunPlane(a3dPoint1, a3dPoint2, a3dPoint3);
		final double a = plane.a();
		final double b = plane.b();
		final double c = plane.c();
		final double d = plane.d();
		double leftHand = (a * a3dPoint4.x()) + (b * a3dPoint4.y()) + (c * a3dPoint4.z()) + d;
		if (leftHand < 0) {
			return new StBlockClosure() {
				public Object value_(Object o) {
					Jun3dPoint p = (Jun3dPoint) o;
					return new Boolean(((a * p.x()) + (b * p.y()) + (c * p.z()) + d) < 0);
				}
			};
		} else if (leftHand > 0) {
			return new StBlockClosure() {
				public Object value_(Object o) {
					Jun3dPoint p = (Jun3dPoint) o;
					return new Boolean(((a * p.x()) + (b * p.y()) + (c * p.z()) + d) > 0);
				}
			};
		} else {
			return new StBlockClosure() {
				public Object value_(Object o) {
					Jun3dPoint p = (Jun3dPoint) o;
					return new Boolean(((a * p.x()) + (b * p.y()) + (c * p.z()) + d) == 0);
				}
			};
		}
	}

	/**
	 * Convert to a paralalel projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @category converting
	 */
	public abstract JunOpenGLParallelProjection asParallelProjection();

	/**
	 * Convert to a perspective projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @category converting
	 */
	public abstract JunOpenGLPerspectiveProjection asPerspectiveProjection();

	/**
	 * Convert to a projection for picking at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category converting
	 */
	public JunOpenGLProjection asPickingProjectionAt_(Jun2dPoint aPoint) {
		Jun3dPoint pickingCenter = this.translateTo3dPointFromPoint_(aPoint);
		double pickingZoomHeight = pickingCenter.minus_(this.translateTo3dPointFromPoint_(aPoint.plus_(new Jun2dPoint(0, -0.01d)))).length();
		JunOpenGLProjection projection = (JunOpenGLProjection) this.copy();
		projection.sightPoint_(pickingCenter);
		if (this.isParallel()) {
			projection.eyePoint_(this.eyePoint().plus_(pickingCenter).minus_(this.sightPoint()));
		}
		projection.zoomHeight_(pickingZoomHeight);
		return projection;
	}

	/**
	 * Convert to a 3D Transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category converting
	 */
	public abstract Jun3dTransformation asTransformation();

	/**
	 * Convert to a 3D transformation for eye.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category converting
	 */
	public abstract Jun3dTransformation asEyeTransformation();

	/**
	 * Apply the transformation to the 3dPoint.
	 * 
	 * @param anObject jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category applying transformation
	 */
	public Jun3dPoint applyTo_(Jun3dPoint anObject) {
		return (Jun3dPoint) anObject.transform_(this.asTransformation());
	}

	/**
	 * Apply the transformation to the 3dPoint.
	 * 
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category applying transformation
	 */
	public Jun3dPoint applyToPoint_(Jun3dPoint a3dPoint) {
		return this.translateToPointFrom3dPoint_(a3dPoint);
	}

	/**
	 * Translate the Jun2dPoint with the depth to a Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public abstract Jun3dPoint translateTo3dPointFromPoint_depth_(Jun2dPoint aPoint, double aNumber);

	/**
	 * Translate the Jun2dPoint to a Jun3dLine from near to far.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category utilities
	 */
	public Jun3dLine translateTo3dLineFromPoint_(Jun2dPoint aPoint) {
		Jun3dPoint from = this.translateTo3dPointFromPoint_depth_(aPoint, this.near());
		Jun3dPoint to = this.translateTo3dPointFromPoint_depth_(aPoint, this.far());
		return new Jun3dLine(from, to);
	}

	/**
	 * Translate the Jun2dPoint to a Jun3dPoint with the depth from near to far.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint translateTo3dPointFromPoint_(Jun2dPoint aPoint) {
		return this.translateTo3dPointFromPoint_depth_(aPoint, this.distance());
	}

	/**
	 * Translate the Jun2dPoint to a Jun3dPoint in the near boundary plane.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint translateTo3dPointInNearBoundaryPlaneFromPoint_(Jun2dPoint aPoint) {
		return this.translateTo3dPointFromPoint_depth_(aPoint, this.near());
	}

	/**
	 * Translate the Jun2dPoint to Jun3dPoint in the far boundary plane.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint translateTo3dPointInFarBoundaryPlaneFromPoint_(Jun2dPoint aPoint) {
		return this.translateTo3dPointFromPoint_depth_(aPoint, this.far());
	}

	/**
	 * Translate the Jun3dPoint.
	 * 
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category utilities
	 */
	public Jun3dPoint translateToPointFrom3dPoint_(Jun3dPoint a3dPoint) {
		return (Jun3dPoint) a3dPoint.transform_(this.asTransformation());
	}

	/**
	 * Fit the distance.
	 * 
	 * @category private
	 */
	protected void fitDistance() {
		double factor = this.viewFactor();
		far = this.distance() * factor;
		near = this.distance() / factor;
	}

	/**
	 * Normalize the up vector.
	 * 
	 * @category private
	 */
	protected void normalizeUpVector() {
		this.setUpVector_(this.rightVector().product_(this.sightVector()).unitVector());
	}

	/**
	 * Set the eye point in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setEyePoint_(Jun3dPoint aPoint) {
		eyePoint = aPoint;
	}

	/**
	 * Set the sight point in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setSightPoint_(Jun3dPoint aPoint) {
		sightPoint = aPoint;
	}

	/**
	 * Set the up vector in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setUpVector_(Jun3dPoint aPoint) {
		upVector = aPoint;
	}

	/**
	 * Set the view factor in private.
	 * 
	 * @param factor double
	 * @category private
	 */
	protected void setViewFactor_(double factor) {
		viewFactor = factor;
	}

	/**
	 * Set the zoom height in private.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected abstract void setZoomHeight_(double aNumber);

	/**
	 * Set the projection in private.
	 * 
	 * @category private
	 */
	protected void setProjection() {
		this.fitDistance();
		this.normalizeUpVector();
	}

}
