package jp.co.sra.jun.opengl.projection;

import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGLParallelProjection class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/10/29 (by MATSUDA Ryouichi)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2004/10/22 (by nisinaka)
 *  @updated   2006/10/13 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun618 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLParallelProjection.java,v 8.12 2008/02/20 06:32:48 nisinaka Exp $
 */
public class JunOpenGLParallelProjection extends JunOpenGLProjection {

	protected double height;

	public static StSymbol Type = $("parallelProjection");
	public static double DefaultHeight = 13.681728885883d;

	/**
	 * Create a new instance of JunOpenGLParallelProjection and initialize it.
	 * Replacement of Default() method.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLParallelProjection() {
		super();
		this.setEyePoint_sightPoint_upVector_height_near_far_(DefaultEyePoint, DefaultSightPoint, DefaultUpVector, DefaultHeight, DefaultNear, DefaultFar);
	}

	/**
	 * Create a new instance of JunOpenGLParallelProjection and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param height double
	 * @param near double
	 * @param far double
	 * @category Instance creation
	 */
	public JunOpenGLParallelProjection(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double height, double near, double far) {
		super();
		this.setEyePoint_sightPoint_upVector_height_near_far_(eyePoint, sightPoint, upVector, height, near, far);
	}

	/**
	 * Create a new projection and set parameters and answer it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param height double
	 * @param near double
	 * @param far double
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @deprecated since Jun497, use the constructor.
	 * @category Instance creation
	 */
	public static JunOpenGLParallelProjection EyePoint_sightPoint_upVector_height_near_far_(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double height, double near, double far) {
		return new JunOpenGLParallelProjection(eyePoint, sightPoint, upVector, height, near, far);
	}

	/**
	 * Create a new default projection and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category Defaults
	 * @deprecated since Jun497, use the constructor.
	 */
	public static JunOpenGLProjection Default() {
		return new JunOpenGLParallelProjection();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
	}

	/**
	 * Answer my projection type as StSymbol
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#type()
	 * @category accessing
	 */
	public StSymbol type() {
		return Type;
	}

	/**
	 * Answer my current height.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double height() {
		return height;
	}

	/**
	 * Set my new height.
	 * 
	 * @param newHeight double
	 * @category accessing
	 */
	public void height_(double newHeight) {
		height = newHeight;
	}

	/**
	 * Answer a regular height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#regularHeight()
	 * @category accessing
	 */
	public double regularHeight() {
		return this.height();
	}

	/**
	 * Convert to a paralalel projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asParallelProjection()
	 * @category converting
	 */
	public JunOpenGLParallelProjection asParallelProjection() {
		return (JunOpenGLParallelProjection) this.copy();
	}

	/**
	 * Convert to a perspective projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asPerspectiveProjection()
	 * @category converting
	 */
	public JunOpenGLPerspectiveProjection asPerspectiveProjection() {
		JunAngle fovy = JunAngle.FromRad_(Math.atan(this.height() / 2 / this.distance()) * 2);
		return new JunOpenGLPerspectiveProjection(this.eyePoint(), this.sightPoint(), this.upVector(), fovy, this.near(), this.far());
	}

	/**
	 * Convert to a 3D Transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asTransformation()
	 * @category converting
	 */
	public Jun3dTransformation asTransformation() {
		Jun3dPoint transformedSightPoint = new Jun3dPoint(0, 0, 1);
		Jun3dPoint transformedEyePoint = new Jun3dPoint(0, 0, 0);
		Jun3dPoint transformedUpVector = new Jun3dPoint(0, 1, 0);
		Jun3dPoint transformedRightVector = new Jun3dPoint(1, 0, 0);
		double scale = Math.abs(this.height()) / 2;
		Jun3dPoint[][] alignPoints = new Jun3dPoint[][] {
				{ this.sightPoint(), transformedSightPoint },
				{ this.eyePoint(), transformedEyePoint },
				{ this.sightPoint().plus_(this.unitUpVector().multipliedBy_(scale)), transformedUpVector },
				{ this.sightPoint().plus_(this.unitRightVector().multipliedBy_(scale)), transformedRightVector } };
		return Jun3dTransformation.AlignPoints_(alignPoints);
	}

	/**
	 * Convert to a 3D transformation for eye.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asEyeTransformation()
	 * @category converting
	 */
	public Jun3dTransformation asEyeTransformation() {
		Jun3dPoint transformedSightPoint = new Jun3dPoint(0, 0, 0);
		Jun3dPoint transformedEyePoint = new Jun3dPoint(0, 0, 1);
		Jun3dPoint transformedUpVector = new Jun3dPoint(0, 1, 0);
		Jun3dPoint transformedRightVector = new Jun3dPoint(1, 0, 0);
		double scale = Math.abs(this.height()) / 2;
		Jun3dPoint[][] alignPoints = new Jun3dPoint[][] {
				{ this.sightPoint(), transformedSightPoint },
				{ this.sightPoint().minus_(this.unitSightVector().multipliedBy_(scale)), transformedEyePoint },
				{ this.sightPoint().plus_(this.unitUpVector().multipliedBy_(scale)), transformedUpVector },
				{ this.sightPoint().plus_(this.unitRightVector().multipliedBy_(scale)), transformedRightVector } };
		return Jun3dTransformation.AlignPoints_(alignPoints);
	}

	/**
	 * Project on a RenderingContext.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#projectOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category projection
	 */
	public void projectOn_(JunOpenGLRenderingContext aRenderingContext) {
		aRenderingContext.parallelProjection_(this);
	}

	/**
	 * Answer true if I am a parallel projection.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#isParallel()
	 * @category testing
	 */
	public boolean isParallel() {
		return true;
	}

	/**
	 * Pan the projection.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#pan_(double)
	 * @category zooming
	 */
	public void pan_(double factor) {
		height *= factor;
	}

	/**
	 * Zoom the projection
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoom_(double)
	 * @category zooming
	 */
	public void zoom_(double factor) {
		height /= factor;
	}

	/**
	 * Answer my current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoomHeight()
	 * @category zooming
	 */
	public double zoomHeight() {
		return this.height();
	}

	/**
	 * Set my new zoom height.
	 * 
	 * @param newZoomHeight double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoomHeight_(double)
	 * @category zooming
	 */
	public void zoomHeight_(double newZoomHeight) {
		this.setZoomHeight_(newZoomHeight);
	}

	/**
	 * Translate the Jun2dPoint with the depth to a Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#translateTo3dPointFromPoint_depth_(jp.co.sra.jun.geometry.basic.Jun2dPoint, double)
	 * @category utilities
	 */
	public Jun3dPoint translateTo3dPointFromPoint_depth_(Jun2dPoint aPoint, double aNumber) {
		double scale = Math.abs(this.height()) / 2;
		Jun3dPoint forward = this.unitSightVector().multipliedBy_(aNumber);
		Jun3dPoint right = this.unitRightVector().multipliedBy_(aPoint.x()).multipliedBy_(scale);
		Jun3dPoint up = this.unitUpVector().multipliedBy_(aPoint.y()).multipliedBy_(scale);
		return this.eyePoint().plus_(forward).plus_(right).plus_(up);
	}

	/**
	 * Set eye point with parameters.
	 * 
	 * @param a3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber1 double
	 * @param aNumber2 double
	 * @param aNumber3 double
	 * @category private
	 */
	protected void setEyePoint_sightPoint_upVector_height_near_far_(Jun3dPoint a3dPoint1, Jun3dPoint a3dPoint2, Jun3dPoint a3dPoint3, double aNumber1, double aNumber2, double aNumber3) {
		eyePoint = a3dPoint1;
		sightPoint = a3dPoint2;
		upVector = a3dPoint3;
		height = aNumber1;
		near = aNumber2;
		far = aNumber3;
		this.normalizeUpVector();
	}

	/**
	 * Set the zoom height in private.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#setZoomHeight_(double)
	 * @category private
	 */
	protected void setZoomHeight_(double aNumber) {
		this.height_(aNumber);
	}

}
