package jp.co.sra.jun.opengl.plotter;

import java.awt.Color;
import java.util.Collection;
import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.goodies.pen.JunPen;
import jp.co.sra.jun.goodies.pen.JunPenDirection;
import jp.co.sra.jun.goodies.pen.JunPenLocation;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLPlotter class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1999/01/05 (by Hirotsugu Kondo)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2000/01/19 (by MATSUDA Ryouichi)
 *  @updated   2004/02/05 (by Nobuto Matsubara)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2006/11/02 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLPlotter.java,v 8.11 2008/02/20 06:32:48 nisinaka Exp $
 */
public class JunOpenGLPlotter extends JunOpenGLDisplayModel {

	/**
	 * Answer the array of the plot program selectors.
	 * 
	 * @return java.lang.String[]
	 * @category accessing
	 */
	protected String[] programSelectors() {
		String[] programSelectors = new String[13];
		programSelectors[0] = "curveOfSinAndCos1";
		programSelectors[1] = "curveOfSinAndCos2";
		programSelectors[2] = "penExample";
		programSelectors[3] = "sample1_triangles";
		programSelectors[4] = "sample2_star";
		programSelectors[5] = "sample3_dragon";
		programSelectors[6] = "sample4_mandala";
		programSelectors[7] = "sample5_spirals";
		programSelectors[8] = "sample6_spirals";
		programSelectors[9] = "sample7_polylines";
		programSelectors[10] = "sample8_fill";
		programSelectors[11] = "sample9_spirals";
		programSelectors[12] = "sampleA_combination";
		return programSelectors;
	}

	/**
	 * Answer this window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Pen Plotter");
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createPlotMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createFileMenu()
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem($String("New"), new MenuPerformer(this, "newModel")));

		StMenu saveAsMenu = new StMenu($String("Save as..."), $("saveAsMenu"));
		saveAsMenu.add(new StMenuItem("LST1.0...", new MenuPerformer(this, "saveLST10")));
		saveAsMenu.add(new StMenuItem("VRML97...", new MenuPerformer(this, "saveWRL97")));
		fileMenu.add(saveAsMenu);

		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Save as image..."), $("saveAsImageMenu"), new MenuPerformer(this, "saveAsImage")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "quitDoing")));

		return fileMenu;
	}

	/**
	 * Create a "Plot" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createPlotMenu() {
		StMenu plotMenu = new StMenu($String("Plot"), $("plotMenu"));

		plotMenu.add(new StMenuItem($String("Clear"), new MenuPerformer(this, "clearPaper")));

		StMenu executePlotMenu = new StMenu($String("Execute") + "...");
		String[] selectors = this.programSelectors();
		for (int index = 0; index < selectors.length; index++) {
			String selector = selectors[index];
			executePlotMenu.add(new StMenuItem(selector, new MenuPerformer(this, selector)));
		}
		plotMenu.add(executePlotMenu);

		return plotMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu(JunSystem.$String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem(JunSystem.$String("Spawn"), $("spawnMenu"), new MenuPerformer(this, "spawnObject")));
		miscMenu.add(new StMenuItem(JunSystem.$String("Viewport"), $("viewportMenu"), new MenuPerformer(this, "spawnViewport")));
		return miscMenu;
	}

	/**
	 * Clear the paper.
	 * 
	 * @category menu messages
	 */
	public void clearPaper() {
		this.displayObject_(new JunOpenGL3dCompoundObject());
		this.changed_($("clear"));
	}

	/**
	 * Program to draw a curve of sin and cos.
	 * 
	 * @category programs
	 */
	public void curveOfSinAndCos1() {
		this.clearPaper();

		final JunPen pen1 = this.newPen(Color.red);
		final JunPen pen2 = this.newPen(Color.blue);
		final Jun2dPoint scale = new Jun2dPoint(30, 30);

		final StBlockClosure curve = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				final JunPen pen = (JunPen) o1;
				final StSymbol function = (StSymbol) o2;

				pen.up();
				for (int i = -300; i <= 300; i += 6) {
					final Double n = new Double(i / 100.0);
					JunOpenGLPlotter.this.do_forMilliseconds_(new StBlockClosure() {
						public Object value() {
							double x = n.doubleValue() * Math.PI;
							double y = x;
							if (function == $("sin")) {
								y = Math.sin(y);
							} else if (function == $("cos")) {
								y = Math.cos(y);
							}

							Jun2dPoint p = new Jun2dPoint(x, y);
							p = p.scaledBy_(scale);
							pen.goto_(p);

							if (pen.isUp()) {
								pen.down();
							}
							return null;
						}
					}, 100);
				}

				return null;
			}
		};

		Thread aThread1 = new Thread() {
			public void run() {
				curve.value_value_(pen1, $("sin"));
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				curve.value_value_(pen2, $("cos"));
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Program to draw a curve of sin and cos with axes.
	 * 
	 * @category programs
	 */
	public void curveOfSinAndCos2() {
		this.curveOfSinAndCos1();

		final JunPen pen0 = this.newPen(Color.gray);
		final Jun2dPoint scale = new Jun2dPoint(30, 30);

		Thread aThread1 = new Thread() {
			public void run() {
				pen0.location_(new Jun2dPoint(-3 * Math.PI, 0).scaledBy_(scale));
				pen0.goto_(new Jun2dPoint(3 * Math.PI, 0).scaledBy_(scale));
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				pen0.location_(new Jun2dPoint(0, -1).scaledBy_(scale));
				pen0.goto_(new Jun2dPoint(0, 1).scaledBy_(scale));
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Program of a pen example.
	 * 
	 * @category programs
	 */
	public void penExample() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(4);
		aPen.color_(Color.gray);

		for (int i = 1; i <= 50; i++) {
			final Integer ii = new Integer(i);
			this.do_forMilliseconds_(new StBlockClosure() {
				public Object value() {
					aPen.go_(ii.intValue() * 4);
					aPen.turn_(89);
					return null;
				}
			}, 100);
		}
	}

	/**
	 * Program of triangles.
	 * 
	 * @category programs
	 */
	public void sample1_triangles() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		int increment = 45;

		for (int i = 0; i < 360; i += increment) {
			for (int j = 0; j < 3; j++) {
				this.do_forMilliseconds_(new StBlockClosure() {
					public Object value() {
						aPen.go_(100);
						aPen.turn_(120);
						return null;
					}
				}, 100);
			}
			aPen.turn_(increment);
		}
	}

	/**
	 * Program of a star.
	 * 
	 * @category programs
	 */
	public void sample2_star() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.color_(Color.red);
		aPen.spiral_angle_tick_(200, 144, 10);
	}

	/**
	 * Program of a dragon.
	 * 
	 * @category programs
	 */
	public void sample3_dragon() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		aPen.dragon_distance_tick_(9, 5, 3);
	}

	/**
	 * Program of a mandala.
	 * 
	 * @category programs
	 */
	public void sample4_mandala() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		aPen.mandala_diameter_tick_(24, 250, 25);
	}

	/**
	 * Program of a mandala.
	 * 
	 * @category programs
	 */
	public void sample5_spirals() {
		this.clearPaper();

		JunPen[] pen = new JunPen[5];
		pen[0] = this.newPen(Color.red);
		pen[1] = this.newPen(Color.red);
		pen[2] = this.newPen(Color.red);
		pen[3] = this.newPen(Color.red);
		pen[4] = this.newPen(Color.red);

		pen[0].location_(new Jun2dPoint(-75, -75));
		pen[1].location_(new Jun2dPoint(75, -75));
		pen[2].location_(new Jun2dPoint(75, 75));
		pen[3].location_(new Jun2dPoint(-75, 75));
		pen[4].location_(new Jun2dPoint(0, 0));

		for (int i = 0; i < pen.length; i++) {
			pen[i].spiral_angle_tick_(100, 88, 10);
		}
	}

	/**
	 * Program of sprials.
	 * 
	 * @category programs
	 */
	public void sample6_spirals() {
		this.clearPaper();

		JunPen[] pen = new JunPen[5];
		pen[0] = this.newPen(Color.red);
		pen[1] = this.newPen(Color.red);
		pen[2] = this.newPen(Color.red);
		pen[3] = this.newPen(Color.red);
		pen[4] = this.newPen(Color.red);

		pen[0].location_(new Jun2dPoint(-75, -75));
		pen[1].location_(new Jun2dPoint(75, -75));
		pen[2].location_(new Jun2dPoint(75, 75));
		pen[3].location_(new Jun2dPoint(-75, 75));
		pen[4].location_(new Jun2dPoint(0, 0));

		for (int i = 0; i < pen.length; i++) {
			final JunPen aPen = pen[i];
			Thread aThread = new Thread() {
				public void run() {
					aPen.spiral_angle_tick_(100, 88, 10);
				}
			};
			aThread.setPriority(Thread.NORM_PRIORITY - 1);
			aThread.start();
		}
	}

	/**
	 * Program of polylines.
	 * 
	 * @category programs
	 */
	public void sample7_polylines() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, -50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.turn_(90);
		}
		aPen.location_(JunPenLocation.FromArray_(new double[] { 50, -50, -50 }));
		aPen.direction_(JunPenDirection.Longitude_(90));
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.tilt_(90);
		}
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, 50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.tilt_(90);
		}
	}

	/**
	 * Program of fill.
	 * 
	 * @category programs
	 */
	public void sample8_fill() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, -50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.red);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.turn_(90);
				}
				return null;
			}
		});
		aPen.location_(JunPenLocation.FromArray_(new double[] { 50, -50, -50 }));
		aPen.direction_(JunPenDirection.Longitude_(90));
		aPen.color_(Color.blue);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.tilt_(90);
				}
				return null;
			}
		});
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, 50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.green);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.tilt_(90);
				}
				return null;
			}
		});
	}

	/**
	 * Program of spirals
	 * 
	 * @category programs
	 */
	public void sample9_spirals() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(1);
		aPen.color_(Color.blue);

		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				final double sign = ((Number) anObject).doubleValue();
				aPen.home();
				for (int i = 1; i <= 50; i++) {
					final Integer n = new Integer(i);
					JunOpenGLPlotter.this.do_forMilliseconds_(new StBlockClosure() {
						public Object value() {
							JunPenLocation from = aPen.location();
							aPen.up();
							aPen.go_(n.doubleValue() * 4 * sign);
							aPen.goto_(aPen.location().plus_(JunPenLocation.X_y_z_(0, 2 * sign, 0)));
							JunPenLocation to = aPen.location();
							aPen.location_(from);
							aPen.down();
							aPen.goto_(to);
							aPen.tilt_(89);
							return null;
						}
					}, 100);

				}
				return null;
			}
		};

		aBlock.value_(new Double(1));
		aBlock.value_(new Double(-1));
	}

	/**
	 * Program of combination
	 * 
	 * @category programs
	 */
	public void sampleA_combination() {
		Thread aThread1 = new Thread() {
			public void run() {
				sample4_mandala();
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				sample9_spirals();
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Create a new pen.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPen
	 * @category private
	 */
	protected JunPen newPen() {
		JunPen aPen = new JunPen();
		aPen.compute_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGLPlotter.this.interpretAs3dObject_((Collection) anObject);
				JunOpenGLPlotter.this.changed_with_($("draw"), anObject);
				if (StInputState.Default().shiftDown()) {
					JunOpenGLPlotter.this.resetView();
				}
				return null;
			}
		});
		return aPen;
	}

	/**
	 * Create a new pen with the specified color.
	 * 
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.goodies.pen.JunPen
	 * @category private
	 */
	protected JunPen newPen(Color aColor) {
		JunPen aPen = this.newPen();
		aPen.color_(aColor);
		return aPen;
	}

	/**
	 * Interpret the argument as JunOpenGL3dObject
	 * 
	 * @param aCollection java.util.Collection
	 * @category private
	 */
	protected void interpretAs3dObject_(Collection aCollection) {
		if (aCollection == null || aCollection.isEmpty()) {
			return;
		}

		Object[] anArray = aCollection.toArray();
		StSymbol tagSymbol = (StSymbol) anArray[0];
		if (tagSymbol == $("stroke")) {
			int nibWidth = ((Number) anArray[1]).intValue();
			Color colorValue = (Color) anArray[2];
			JunPenLocation fromPoint = (JunPenLocation) anArray[3];
			JunPenLocation toPoint = (JunPenLocation) anArray[4];
			Jun3dPoint[] locationArray = new Jun3dPoint[2];
			locationArray[0] = JunPenLocation.Coerce_(fromPoint)._as3dPoint();
			locationArray[1] = JunPenLocation.Coerce_(toPoint)._as3dPoint();
			JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(locationArray);
			aPolyline.paint_(colorValue);
			aPolyline.lineWidth_(nibWidth);
			((JunOpenGL3dCompoundObject) this.displayObject()).add_(aPolyline);
		} else if (tagSymbol == $("fill")) {
			Color colorValue = (Color) anArray[1];
			Vector locationArray = (Vector) anArray[2];
			Jun3dPoint[] vertexes = new Jun3dPoint[locationArray.size()];
			for (int index = 0; index < locationArray.size(); index++) {
				vertexes[index] = JunPenLocation.Coerce_(locationArray.elementAt(index))._as3dPoint();
			}
			JunOpenGL3dPolygon aPolygon = new JunOpenGL3dPolygon(vertexes);
			aPolygon.paint_(colorValue);
			((JunOpenGL3dCompoundObject) this.displayObject()).add_(aPolygon);
		}

		this.updateMenuIndication();
	}

}
