package jp.co.sra.jun.opengl.objects;

import java.awt.Color;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.curves.JunNurbsCurve;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGL3dNurbsCurve class
 * 
 *  @author    nisinaka
 *  @created   1998/10/29 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2004/05/14 (by nisinaka)
 *  @updated   2004/09/29 (by m-asada)
 *  @updated   2004/11/15 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun500 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dNurbsCurve.java,v 8.10 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dNurbsCurve extends JunOpenGL3dPrimitiveObject {

	/** The line width. */
	protected float lineWidth;

	/** The stipple factor. */
	protected int stippleFactor;

	/** The stipple pattern. */
	protected short stipplePattern;

	/** The array of the control points. */
	protected Jun3dPoint[] controlPoints;

	/** The array of weight numbers. */
	protected double[] weights;

	/** The knot vector. */
	protected double[] knotVector;

	/**
	 * Create an instance of JunOpenGL3dNurbsCurve and initialize it with the arguments.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsCurve BezierControlPoints_(Jun3dPoint[] anArrayOfPoints) {
		int numberOfPoints = anArrayOfPoints.length;
		double[] knotVector = new double[numberOfPoints * 2];
		for (int i = 0; i < numberOfPoints; i++) {
			knotVector[i] = 0;
			knotVector[i + numberOfPoints] = 1;
		}
		return BSplineControlPoints_knotVector_(anArrayOfPoints, knotVector);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsCurve and initialize it with the arguments.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfNumbers double[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsCurve BSplineControlPoints_knotVector_(Jun3dPoint[] anArrayOfPoints, double[] anArrayOfNumbers) {
		int numberOfPoints = anArrayOfPoints.length;
		double[] weights = new double[numberOfPoints];
		for (int i = 0; i < numberOfPoints; i++) {
			weights[i] = 1.0;
		}
		return new JunOpenGL3dNurbsCurve(anArrayOfPoints, weights, anArrayOfNumbers);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsCurve and initialize it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve
	 * @category Instance creation
	 */
	public static final JunOpenGL3dObject Circle() {
		Jun3dPoint[] controlPoints = new Jun3dPoint[9];
		controlPoints[0] = new Jun3dPoint(0, -1, 0);
		controlPoints[1] = new Jun3dPoint(1, -1, 0);
		controlPoints[2] = new Jun3dPoint(1, 0, 0);
		controlPoints[3] = new Jun3dPoint(1, 1, 0);
		controlPoints[4] = new Jun3dPoint(0, 1, 0);
		controlPoints[5] = new Jun3dPoint(-1, 1, 0);
		controlPoints[6] = new Jun3dPoint(-1, 0, 0);
		controlPoints[7] = new Jun3dPoint(-1, -1, 0);
		controlPoints[8] = new Jun3dPoint(0, -1, 0);

		double[] weights = new double[9];
		weights[0] = 1.0;
		weights[1] = Math.sqrt(0.5);
		weights[2] = 1.0;
		weights[3] = Math.sqrt(0.5);
		weights[4] = 1.0;
		weights[5] = Math.sqrt(0.5);
		weights[6] = 1.0;
		weights[7] = Math.sqrt(0.5);
		weights[8] = 1.0;

		double[] knots = { 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 4 };

		return new JunOpenGL3dNurbsCurve(controlPoints, weights, knots);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsCurve and initialize it with the arguments.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfNumber1 double[]
	 * @param anArrayOfNumber2 double[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve
	 * @deprecated since Jun454, user the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsCurve ControlPoints_weights_knotVector_(Jun3dPoint[] anArrayOfPoints, double[] anArrayOfNumber1, double[] anArrayOfNumber2) {
		return new JunOpenGL3dNurbsCurve(anArrayOfPoints, anArrayOfNumber1, anArrayOfNumber2);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsCurve and initialize it with the arguments.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[] as control points
	 * @param anArrayOfNumber1 double[] as weights
	 * @param anArrayOfNumber2 double[] as knot vector
	 * @category Instance creation
	 */
	public JunOpenGL3dNurbsCurve(Jun3dPoint[] anArrayOfPoints, double[] anArrayOfNumber1, double[] anArrayOfNumber2) {
		this.controlPoints_(anArrayOfPoints);
		this.weights_(anArrayOfNumber1);
		this.knotVector_(anArrayOfNumber2);
	}

	/**
	 * Create a new instance of JunOpenGL3dNurbsCurve and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dNurbsCurve(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		lineWidth = this.defaultLineWidth();
		stippleFactor = 1;
		stipplePattern = Short.MIN_VALUE;
		controlPoints = null;
		weights = null;
		knotVector = null;
	}

	/**
	 * Set the specified halftone scale.
	 * 
	 * @param halftoneScale double
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#halftone_(double)
	 * @category accessing
	 */
	public void halftone_(double halftoneScale) {
		int patternNumber = JunImageProcessor.Stipple_(Math.max(0.0d, Math.min(halftoneScale, 1.0d)));
		stipplePattern = (short) patternNumber;
	}

	/**
	 * Answer the line width.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float lineWidth() {
		return lineWidth;
	}

	/**
	 * Set a number as the line width.
	 * 
	 * @param aNumber float
	 * @category accessing
	 */
	public void lineWidth_(float aNumber) {
		lineWidth = aNumber;
	}

	/**
	 * Answer the stipple factor.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int stippleFactor() {
		return stippleFactor;
	}

	/**
	 * Set a number as the stipple factor.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void stippleFactor_(int aNumber) {
		stippleFactor = aNumber;
	}

	/**
	 * Answer the stipple pattern
	 * 
	 * @return short
	 * @category accessing
	 */
	public short stipplePattern() {
		return stipplePattern;
	}

	/**
	 * Set a number as the stipple pattern.
	 * 
	 * @param aNumber short
	 * @category accessing
	 */
	public void stipplePattern_(short aNumber) {
		stipplePattern = aNumber;
	}

	/**
	 * Answer the control points of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] controlPoints() {
		return controlPoints;
	}

	/**
	 * Set the control points of the receiver.
	 * 
	 * @param anArray jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public void controlPoints_(Jun3dPoint[] anArray) {
		controlPoints = anArray;
	}

	/**
	 * Answer the array of weight number.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] weights() {
		return weights;
	}

	/**
	 * Set the weight numbers.
	 * 
	 * @param anArrayOfDouble double[]
	 * @category accessing
	 */
	public void weights_(double[] anArrayOfDouble) {
		weights = anArrayOfDouble;
	}

	/**
	 * Answer the knot vector.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] knotVector() {
		return knotVector;
	}

	/**
	 * Set the knot vector.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void knotVector_(double[] anArray) {
		knotVector = anArray;
	}

	/**
	 * Answer true if this JunOpenGL3dObject has a stipple, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasStipple()
	 * @category testing
	 */
	public boolean hasStipple() {
		return stipplePattern != Short.MIN_VALUE;
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunRenderingContext
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		if (this.hasPaint()) {
			aRenderingContext.paint_(this.paint());
		}
		if (this.hasAlpha()) {
			aRenderingContext.alpha_(this.alpha());
		}
		aRenderingContext.lineWidth_(lineWidth);
		if (this.hasStipple()) {
			aRenderingContext.lineStippleFactor_pattern_(this.stippleFactor(), this.stipplePattern());
			aRenderingContext.enableLineStipple();
		}

		aRenderingContext.displayNurbsCurve_(this._asJunNurbsCurve());

		if (this.hasStipple()) {
			aRenderingContext.disableLineStipple();
		}
	}

	/**
	 * Enumerate every geometries and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		return aBlock.value_(this._asJunNurbsCurve());
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		for (int i = 0; i < controlPoints.length; i++) {
			aBlock.value_(controlPoints[i]);
		}
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dNurbsCurve theNurbsCurve = (JunOpenGL3dNurbsCurve) this.copy();
		for (int i = 0; i < theNurbsCurve.controlPoints.length; i++) {
			Jun3dPoint transformedPoint = theNurbsCurve.controlPoints[i].transform_(aTransformation);
			theNurbsCurve.controlPoints[i] = transformedPoint;
		}
		return theNurbsCurve;
	}

	/**
	 * Convert the receiver as a JunNurbsCurve.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category converting
	 */
	public JunNurbsCurve _asJunNurbsCurve() {
		return new JunNurbsCurve(controlPoints, weights, knotVector);
	}

	/**
	 * Answer the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		throw new SmalltalkException("not implemented yet");
	}

	/**
	 * Answer the control points object of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject controlPointsObject() {
		JunOpenGL3dCompoundObject controlPointsObject = new JunOpenGL3dCompoundObject();

		if (controlPoints != null) {
			for (int i = 0; i < controlPoints.length; i++) {
				JunOpenGL3dVertex aVertex = new JunOpenGL3dVertex(controlPoints[i], Color.blue);
				aVertex.size_(5);
				controlPointsObject.add_(aVertex);
			}
			controlPointsObject.add_(new JunOpenGL3dPolyline(controlPoints, Color.blue));
		}

		return controlPointsObject;
	}

	/**
	 * Answer myself and my control points object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject yourselfAndControlPointsObject() {
		return new JunOpenGL3dCompoundObject(this, this.controlPointsObject());
	}

	/**
	 * Finish doing whatever is required, beyond a shallowCopy, to implement 'copy'.
	 * Return the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();
		controlPoints = (Jun3dPoint[]) controlPoints.clone();
		weights = (double[]) weights.clone();
		knotVector = (double[]) knotVector.clone();
		return this;
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @category utilities
	 */
	public int numberOfPolygons() {
		return 0;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("NurbsCurve");
	}

	/**
	 * Convert the receiver as a JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.lineWidth() > 1) {
			list.add_(this.lineWidthToLispList());
		}
		if (this.hasStipple()) {
			list.add_(this.stippleToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.controlsToLispList());
		list.add_(this.weightsToLispList());
		list.add_(this.knotsToLispList());

		return list;
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList stippleToLispList() {
		if (!this.hasStipple()) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("stipple"));
		list.add_(new Integer(this.stippleFactor()));
		list.add_(new Short(this.stipplePattern()));
		return list;
	}

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList textureToLispList() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList lineWidthToLispList() {
		if (this.lineWidth <= 1) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("width"));
		list.tail_(new Float(this.lineWidth()));
		return list;
	}

	/**
	 * Convert the receiver's contol points as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList controlsToLispList() {
		Object[] collection = new Object[controlPoints.length + 1];
		collection[0] = $("controls");
		for (int i = 0; i < controlPoints.length; i++) {
			collection[i + 1] = controlPoints[i];
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's weight numbers as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList weightsToLispList() {
		Object[] collection = new Object[weights.length + 1];
		collection[0] = $("weights");
		for (int i = 0; i < weights.length; i++) {
			collection[i + 1] = new Double(weights[i]);
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's knot vector as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList knotsToLispList() {
		Object[] collection = new Object[knotVector.length + 1];
		collection[0] = $("knots");
		for (int i = 0; i < knotVector.length; i++) {
			collection[i + 1] = new Double(knotVector[i]);
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.lineWidthFromLispList(aList);
		this.stippleFromLispList(aList);
		this.controlsFromLispList(aList);
		this.weightsFromLispList(aList);
		this.knotsFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Get my controlsPoints from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void controlsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("controls")));
			}
		}, new StBlockClosure());

		Object[] tails = (list == null) ? ((JunLispCons) aList.tail()).asArray() : ((JunLispCons) list.tail()).asArray();
		Jun3dPoint[] points = new Jun3dPoint[tails.length];
		for (int i = 0; i < points.length; i++) {
			points[i] = (Jun3dPoint) tails[i];
		}
		this.controlPoints_(points);
	}

	/**
	 * Get my knotVector from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void knotsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("knots")));
			}
		}, new StBlockClosure());

		Object[] numbers = (list == null) ? ((JunLispList) aList.tail()).asArray() : ((JunLispList) list.tail()).asArray();
		double[] doubles = new double[numbers.length];
		for (int i = 0; i < doubles.length; i++) {
			doubles[i] = ((Number) numbers[i]).doubleValue();
		}
		this.knotVector_(doubles);
	}

	/**
	 * Get my lineWidth from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void lineWidthFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("width")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.lineWidth_(((Number) list.tail()).floatValue());
	}

	/**
	 * Get my stipple from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#stippleFromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	protected void stippleFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("stipple")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.stippleFactor_(((Number) ((JunLispCons) list.tail()).nth_(1)).intValue());
		this.stipplePattern_(((Number) ((JunLispCons) list.tail()).nth_(2)).shortValue());
	}

	/**
	 * Get my wights from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void weightsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("weights")));
			}
		}, new StBlockClosure());

		Object[] numbers = (list == null) ? ((JunLispList) aList.tail()).asArray() : ((JunLispList) list.tail()).asArray();
		double[] doubles = new double[numbers.length];
		for (int i = 0; i < doubles.length; i++) {
			doubles[i] = ((Number) numbers[i]).doubleValue();
		}
		this.weights_(doubles);
	}

	/**
	 * Write the VRML1.0 string of the receiver on the writer.
	 * 
	 * @param writer java.io.Writer
	 * @category vrml support
	 */
	public void vrml10On_(Writer writer) {
		throw new SmalltalkException("not implemented yet");
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param writer java.io.Writer
	 * @category vrml support
	 */
	public void vrml20On_(Writer writer) {
		throw new SmalltalkException("not implemented yet");
	}

}
