package jp.co.sra.jun.opengl.objects;

import java.awt.Cursor;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc;
import jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraph;
import jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode;
import jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;

/**
 * JunOpenGL3dCompoundObject class
 * 
 *  @author    nisinaka
 *  @created   1998/10/13 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2004/05/13 (by nisinaka)
 *  @updated   2004/09/29 (by nisinaka)
 *  @updated   2007/09/10 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun682 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dCompoundObject.java,v 8.17 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGL3dCompoundObject extends JunOpenGL3dObject implements JunOpenGLPickingObjects {

	protected ArrayList components;
	protected JunOpenGLTexture texture;
	protected JunAngle smoothingAngle;

	/**
	 * Create a new instance of JunOpenGL3dCompoundObject.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dCompoundObject() {
		this(0);
	}

	/**
	 * Create an instance of JunOpenGL3dCompoundObject and initialize its initial capacity.
	 * 
	 * @param size int
	 * @category Instance creation
	 */
	public JunOpenGL3dCompoundObject(int size) {
		super();
		this.initialize_(size);
	}

	/**
	 * Create a new instance of JunOpenGL3dCompoundObject and initialize it with a collection of JunOpenGL3dObject.
	 * 
	 * @param aCollection java.util.Collection
	 * @category Instance creation
	 */
	public JunOpenGL3dCompoundObject(Collection aCollection) {
		super();
		this.components_(aCollection);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with a JunOpenGL3dObject.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation 
	 */
	public JunOpenGL3dCompoundObject(JunOpenGL3dObject a3dObject) {
		this(1);
		this.add_(a3dObject);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with two JunOpenGL3dObject.
	 * 
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation 
	 */
	public JunOpenGL3dCompoundObject(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2) {
		this(2);
		this.add_(a3dObject1);
		this.add_(a3dObject2);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with three JunOpenGL3dObject.
	 * 
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject3 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation 
	 */
	public JunOpenGL3dCompoundObject(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2, JunOpenGL3dObject a3dObject3) {
		this(3);
		this.add_(a3dObject1);
		this.add_(a3dObject2);
		this.add_(a3dObject3);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with four JunOpenGL3dObject.
	 * 
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject3 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject4 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation 
	 */
	public JunOpenGL3dCompoundObject(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2, JunOpenGL3dObject a3dObject3, JunOpenGL3dObject a3dObject4) {
		this(4);
		this.add_(a3dObject1);
		this.add_(a3dObject2);
		this.add_(a3dObject3);
		this.add_(a3dObject4);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with an array of JunOpenGL3dObject.
	 * 
	 * @param anArray jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @category Instance creation 
	 */
	public JunOpenGL3dCompoundObject(JunOpenGL3dObject[] anArray) {
		super();
		this.addAll_(anArray);
	}

	/**
	 * Create a new instance of JunOpenGL3dCompoundObject and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dCompoundObject(JunLispList aList) {
		this();

		StSymbol kindName = (StSymbol) aList.head();
		if (kindName == $("SkinningBody")) {
			this.skinningBodyFromLispList(aList);
		} else {
			this.fromLispList(aList);
		}
	}

	/**
	 * Create a new instance of JunOpenGL3dCompoundObject and initialize it with a collection of JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @param aCollection java.util.Vector
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation
	 */
	public static JunOpenGL3dCompoundObject Components_(Vector aCollection) {
		return new JunOpenGL3dCompoundObject(aCollection);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with a JunOpenGL3dObject.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation 
	 */
	public static JunOpenGL3dCompoundObject With_(JunOpenGL3dObject a3dObject) {
		return new JunOpenGL3dCompoundObject(a3dObject);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with two JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation 
	 */
	public static JunOpenGL3dCompoundObject With_with_(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2) {
		return new JunOpenGL3dCompoundObject(a3dObject1, a3dObject2);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with three JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject3 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation 
	 */
	public static JunOpenGL3dCompoundObject With_with_with_(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2, JunOpenGL3dObject a3dObject3) {
		return new JunOpenGL3dCompoundObject(a3dObject1, a3dObject2, a3dObject3);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with four JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @param a3dObject1 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject2 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject3 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param a3dObject4 jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation 
	 */
	public static JunOpenGL3dCompoundObject With_with_with_with_(JunOpenGL3dObject a3dObject1, JunOpenGL3dObject a3dObject2, JunOpenGL3dObject a3dObject3, JunOpenGL3dObject a3dObject4) {
		return new JunOpenGL3dCompoundObject(a3dObject1, a3dObject2, a3dObject3, a3dObject4);
	}

	/**
	 * Create a new JunOpenGL3dCompoundObject and initialize it with an array of JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @param anArray jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @deprecated since Jun454, use the constructor.
	 * @category Instance creation 
	 */
	public static JunOpenGL3dCompoundObject WithAll_(JunOpenGL3dObject[] anArray) {
		return new JunOpenGL3dCompoundObject(anArray);
	}

	/**
	 * Create a JunOpenGL3dCompoundObject from the lisp list.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @deprecated since Jun454, use the constructor
	 * @category Lisp support 
	 */
	public static JunOpenGL3dObject FromLispList_(JunLispCons aList) {
		StSymbol kindName = (StSymbol) aList.head();
		if (kindName == $("PatchedBody")) {
			return new JunOpenGL3dPatchedObject(aList);
		} else {
			return new JunOpenGL3dCompoundObject(aList);
		}
	}

	/**
	 * Create a JunOpenGL3dObject from the lisp list.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category Lisp support 
	 */
	public static JunOpenGL3dObject ObjectFromLispList_(JunLispCons aList) {
		StSymbol kindName = (StSymbol) aList.head();
		if (kindName == $("CompoundBody")) {
			return new JunOpenGL3dCompoundObject(aList);
		} else if (kindName == $("PatchedBody")) {
			return new JunOpenGL3dPatchedObject(aList);
		} else if (kindName == $("SkinningBody")) {
			return new JunOpenGL3dCompoundObject(aList);
		} else if (kindName == $("Polygon")) {
			return new JunOpenGL3dPolygon(aList);
		} else if (kindName == $("Polyline")) {
			return new JunOpenGL3dPolyline(aList);
		} else if (kindName == $("PolylineLoop")) {
			return new JunOpenGL3dPolylineLoop(aList);
		} else if (kindName == $("Spot")) {
			return new JunOpenGL3dVertex(aList);
		} else if (kindName == $("NurbsCurve")) {
			return new JunOpenGL3dNurbsCurve(aList);
		} else if (kindName == $("NurbsSurface")) {
			return new JunOpenGL3dNurbsSurface(aList);
		} else if (kindName == $("Cone")) {
			return new JunOpenGL3dCone(aList);
		} else if (kindName == $("Cylinder")) {
			return new JunOpenGL3dCylinder(aList);
		} else if (kindName == $("Sphere")) {
			return new JunOpenGL3dSphere(aList);
		} else if (kindName == $("Arc")) {
			return new JunOpenGL3dArc(aList);
		} else if (kindName == $("Graph")) {
			return new JunOpenGL3dGraph(aList);
		} else if (kindName == $("Node")) {
			return new JunOpenGL3dNode(aList);
		}
		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		this.initialize_(-1);
	}

	/**
	 * Initialize the receiver. Create a component with its initial capacity 'size'.
	 * 
	 * @param size int
	 * @category initialize-release
	 */
	protected void initialize_(int size) {
		super.initialize();
		components = (size < 0) ? null : new ArrayList(size);
		texture = null;
		smoothingAngle = null;
	}

	/**
	 * Answer my current components as ArrayList.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	protected ArrayList _components() {
		if (components == null) {
			components = new ArrayList();
		}
		return components;
	}

	/**
	 * Answer my current components.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @see jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects#components()
	 * @category accessing
	 */
	public JunOpenGL3dObject[] components() {
		return (JunOpenGL3dObject[]) this._components().toArray(new JunOpenGL3dObject[this.size()]);
	}

	/**
	 * Set my new components.
	 * 
	 * @param anArray jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @category accessing
	 */
	public void components_(JunOpenGL3dObject[] anArray) {
		if (anArray == null) {
			components = new ArrayList(0);
		} else {
			components = new ArrayList(anArray.length);
			this.addAll_(anArray);
		}
	}

	/**
	 * Set my new components.
	 * 
	 * @param aCollection java.util.Collection
	 * @category accessing
	 */
	public void components_(Collection aCollection) {
		JunOpenGL3dObject[] anArray = null;
		if (aCollection != null) {
			anArray = new JunOpenGL3dObject[aCollection.size()];
			aCollection.toArray(anArray);
		}
		this.components_(anArray);
	}

	/**
	 * Answer the texture.
	 * 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#texture()
	 * @category accessing
	 */
	public JunOpenGLTexture texture() {
		return texture;
	}

	/**
	 * Set a texture.
	 * 
	 * @param aTexture jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#texture_(jp.co.sra.jun.opengl.texture.JunOpenGLTexture)
	 * @category accessing
	 */
	public void texture_(JunOpenGLTexture aTexture) {
		texture = aTexture;
		super.texture_(aTexture);
	}

	/**
	 * Answer my current smoothing angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#smoothingAngle()
	 * @category accessing
	 */
	public JunAngle smoothingAngle() {
		return smoothingAngle;
	}

	/**
	 * Set my new smoothing angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	protected void smoothingAngle_(JunAngle anAngle) {
		smoothingAngle = anAngle;
	}

	/**
	 * Answer my size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this._components().size();
	}

	/**
	 * Establish all normal vectors with the specified smoothing angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#establishAllNormalVectorsWithSmoothingAngle_(jp.co.sra.jun.geometry.basic.JunAngle)
	 * @category vector accessing
	 */
	public void establishAllNormalVectorsWithSmoothingAngle_(JunAngle anAngle) {
		final JunGeometry.HashEqualityDictionary aMap = new JunGeometry.HashEqualityDictionary();
		this.polygonsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dPolygon aPolygon = (JunOpenGL3dPolygon) anObject;
				Jun3dPoint[] normalVectors = new Jun3dPoint[aPolygon.numberOfVertexes()];
				for (int i = 0; i < normalVectors.length; i++) {
					normalVectors[i] = aPolygon.normalVector();
				}
				aPolygon.normalVectors_(normalVectors);

				Jun3dPoint[] vertexes = aPolygon.vertexes();
				for (int i = 0; i < vertexes.length; i++) {
					ArrayList aList = (ArrayList) aMap.get(vertexes[i]);
					if (aList == null) {
						aList = new ArrayList();
						aMap.put(vertexes[i], aList);
					}
					aList.add(aPolygon);
				}
				return null;
			}
		});

		Iterator i = aMap.keySet().iterator();
		while (i.hasNext()) {
			Jun3dPoint vertex = (Jun3dPoint) i.next();
			ArrayList polygonList = (ArrayList) aMap.get(vertex);
			this._establishNormalVectors(vertex, polygonList, anAngle);
		}

		this.smoothingAngle_(anAngle);
	}

	/**
	 * Establish the normal vectors.
	 * 
	 * @param vertex jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param polygonList java.util.ArrayList
	 * @param creaseAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category vector accessing
	 */
	protected void _establishNormalVectors(Jun3dPoint vertex, ArrayList polygonList, JunAngle creaseAngle) {
		if (polygonList == null || polygonList.isEmpty()) {
			return;
		}

		JunOpenGL3dPolygon polygon = (JunOpenGL3dPolygon) polygonList.remove(0);
		Jun3dPoint vector = polygon.normalVector();
		ArrayList group = new ArrayList();
		group.add(polygon);

		// Collect polygons of the group to the right.
		JunOpenGL3dPolygon polygon1 = polygon;
		JunOpenGL3dPolygon polygon2 = null;
		while ((polygon2 = polygon1._findAdjacentOnRight(vertex, polygonList)) != null) {
			if (group.contains(polygon2)) {
				break;
			}
			double cos = polygon1.normalVector().dotProduct_(polygon2.normalVector());
			if (cos < creaseAngle.cos()) {
				break;
			}

			group.add(polygon2);
			vector = vector.plus_(polygon2.normalVector());
			polygonList.remove(polygon2);
			polygon1 = polygon2;
		}

		// Collect polygons of the group to the left.
		polygon1 = polygon;
		polygon2 = null;
		while ((polygon2 = polygon1._findAdjacentOnLeft(vertex, polygonList)) != null) {
			if (group.contains(polygon2)) {
				break;
			}
			double cos = polygon1.normalVector().dotProduct_(polygon2.normalVector());
			if (cos < creaseAngle.cos()) {
				break;
			}

			group.add(polygon2);
			vector = vector.plus_(polygon2.normalVector());
			polygonList.remove(polygon2);
			polygon1 = polygon2;
		}

		// Set the vertex normals of the group.
		if (group.size() > 1) {
			vector = vector.unitVector();
			for (int i = 0; i < group.size(); i++) {
				polygon = (JunOpenGL3dPolygon) group.get(i);
				int index = polygon.indexOfVertex(vertex);
				polygon.normalVectors()[index] = vector;
			}
		}

		// Recurse while a polygon exist.
		if (polygonList.isEmpty() == false) {
			this._establishNormalVectors(vertex, polygonList, creaseAngle);
		}
	}

	/**
	 * Flush all alphas.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllAlphas()
	 * @category flushing
	 */
	public void flushAllAlphas() {
		this.flushAlpha();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllAlphas();
		}
	}

	/**
	 * Flush all bounds.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllBounds()
	 * @category flushing
	 */
	public void flushAllBounds() {
		this.flushBounds();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllBounds();
		}
	}

	/**
	 * Flush all colors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllColors()
	 * @category flushing
	 */
	public void flushAllColors() {
		this.flushColors();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllColors();
		}
	}

	/**
	 * Flush all names.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllNames()
	 * @category flushing
	 */
	public void flushAllNames() {
		this.flushName();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllNames();
		}
	}

	/**
	 * Flush all normal vectors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllNormalVectors()
	 * @category flushing
	 */
	public void flushAllNormalVectors() {
		this.flushNormalVectors();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllNormalVectors();
		}
	}

	/**
	 * Flush all paints.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllPaints()
	 * @category flushing
	 */
	public void flushAllPaints() {
		this.flushPaint();

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].flushAllPaints();
		}
	}

	/**
	 * Flush the components.
	 * 
	 * @category flushing
	 */
	public void flushComponents() {
		components = null;
	}

	/**
	 * Flush the current smoothing angle.
	 * 
	 * @category flushing
	 */
	public void flushSmoothingAngle() {
		smoothingAngle = null;
	}

	/**
	 * Answer true if the texture is specified, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasTexture()
	 * @category testing
	 */
	public boolean hasTexture() {
		return (texture != null);
	}

	/**
	 * Answer true if the receiver is a compound object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#isCompound()
	 * @category testing
	 */
	public boolean isCompound() {
		return true;
	}

	/**
	 * Answer true if the receiver is a compound body, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCompoundBody() {
		if (components == null) {
			return true;
		}
		if (components.isEmpty()) {
			return true;
		}

		JunOpenGL3dObject[] anArray = this.components();
		for (int i = 0; i < anArray.length; i++) {
			JunOpenGL3dObject each = anArray[i];
			if (each.hasColor()) {
				return true;
			}
			if (each.hasStipple()) {
				return true;
			}
			if (each instanceof JunOpenGL3dPolygon == false) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this._components().isEmpty();
	}

	/**
	 * Answer true if the receiver is a patched body, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPatchedBody() {
		if (this.isCompoundBody()) {
			return false;
		}

		int total = 0;
		Vector points = new Vector(1024);
		int min = Integer.MAX_VALUE;
		int max = Integer.MIN_VALUE;
		JunOpenGL3dObject[] anArray = this.components();
		for (int i = 0; i < anArray.length; i++) {
			JunOpenGL3dPolygon polygon = (JunOpenGL3dPolygon) anArray[i];
			if (polygon.hasColors() || polygon.hasNormalVectors()) {
				return false;
			}

			Jun3dPoint[] vertexes = polygon.vertexes();
			int size = vertexes.length;
			total += size;
			for (int j = 0; j < size; j++) {
				Jun3dPoint p = vertexes[j];
				int index = -1;
				for (int k = 0; k < points.size(); k++) {
					if (p.equals((Jun3dPoint) points.elementAt(k))) {
						index = k;
						break;
					}
				}
				if (index < 0) {
					points.add(p);
				}
			}
			min = Math.min(min, size);
			max = Math.max(max, size);
		}

		if (min == max && points.size() * 2 > total) {
			return false;
		}

		return true;
	}

	/**
	 * Answer true if the receiver is a skinning body, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSkinningBody() {
		if (this.isCompoundBody()) {
			return false;
		}

		return (!this.isPatchedBody());
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		if (components != null) {
			components = (ArrayList) components.clone();
		}

		return this;
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		int oldTextureCoordinateIndex = Integer.MIN_VALUE;

		if (this.hasTexture()) {
			oldTextureCoordinateIndex = aRenderingContext.textureCoordinateIndex();
			aRenderingContext.textureCoordinateIndex_(0);
			this.texture().enableTextureOn_(aRenderingContext);
		}

		if (this.hasTransparency()) {
			aRenderingContext.enableCullFace();
			aRenderingContext.disableDepthMask();
			aRenderingContext.enableBlending();
		}

		this.renderOpaqueComponentsOn_(aRenderingContext);
		this.renderTransparentComponentsOn_(aRenderingContext);

		if (this.hasTransparency()) {
			aRenderingContext.disableCullFace();
			aRenderingContext.enableDepthMask();
			aRenderingContext.disableBlending();
		}

		if (this.hasTexture()) {
			this.texture().disableTextureOn_(aRenderingContext);
			aRenderingContext.textureCoordinateIndex_(oldTextureCoordinateIndex);
		}
	}

	/**
	 * Render the opaque components on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderOpaqueComponentsOn_(JunOpenGLRenderingContext aRenderingContext) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			if (!myComponents[i].hasTransparency()) {
				if (this.hasPaint()) {
					aRenderingContext.paint_(this.paint());
				}
				if (this.hasAlpha()) {
					aRenderingContext.alpha_(this.alpha());
				}
				if (this.hasTexture()) {
					this.texture().enableTextureOn_(aRenderingContext);
				}
				myComponents[i].renderOn_(aRenderingContext);
			}
		}
	}

	/**
	 * Render the transparent components on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	protected void renderTransparentComponentsOn_(JunOpenGLRenderingContext aRenderingContext) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			if (myComponents[i].hasTransparency()) {
				if (this.hasPaint()) {
					aRenderingContext.paint_(this.paint());
				}
				if (this.hasAlpha()) {
					aRenderingContext.alpha_(this.alpha());
				}
				if (this.hasTexture()) {
					this.texture().enableTextureOn_(aRenderingContext);
				}
				myComponents[i].renderOn_(aRenderingContext);
			}
		}
	}

	/**
	 * Add a JunOpenGL3dObject to the JunOpenGL3dCompoundObject.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category adding
	 */
	public void add_(JunOpenGL3dObject a3dObject) {
		if (a3dObject == null) {
			return;
		}

		this._components().add(a3dObject);
		this.flushBounds();
	}

	/**
	 * Add all of the array elements.
	 * 
	 * @param anArray jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @category adding
	 */
	public void addAll_(JunOpenGL3dObject[] anArray) {
		if (anArray == null) {
			return;
		}

		for (int i = 0; i < anArray.length; i++) {
			this.add_(anArray[i]);
		}
	}

	/**
	 * Add all of the collection elements.
	 * 
	 * @param aCollection java.util.Collection
	 * @category adding
	 */
	public void addAll_(Collection aCollection) {
		if (aCollection == null) {
			return;
		}

		JunOpenGL3dObject[] anArray = new JunOpenGL3dObject[aCollection.size()];
		aCollection.toArray(anArray);
		this.addAll_(anArray);
	}

	/**
	 * Remove a JunOpenGL3dObject from the JunOpenGL3dCompoundObject.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category removing
	 */
	public void remove_(JunOpenGL3dObject a3dObject) {
		if (a3dObject == null) {
			return;
		}

		this._components().remove(a3dObject);
		this.flushBounds();
	}

	/**
	 * Remove all of the array elements from the JunOpenGL3dCompoundObject.
	 * 
	 * @param anArray jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @category removing
	 */
	public void removeAll_(JunOpenGL3dObject[] anArray) {
		if (anArray == null) {
			return;
		}

		for (int i = 0; i < anArray.length; i++) {
			this.remove_(anArray[i]);
		}
	}

	/**
	 * Remove all of the collection elements from the receiver.
	 * 
	 * @param aCollection java.util.Collection
	 * @category removing
	 */
	public void removeAll_(Collection aCollection) {
		if (aCollection == null) {
			return;
		}

		JunOpenGL3dObject[] anArray = new JunOpenGL3dObject[aCollection.size()];
		aCollection.toArray(anArray);
		this.removeAll_(anArray);
	}

	/**
	 * Enumerate each objects and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating 
	 */
	public Object do_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			Object result = aBlock.value_(myComponents[i]);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Enumerate each objects and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#objectsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating 
	 */
	public Object objectsDo_(StBlockClosure aBlock) {
		aBlock.value_(this);

		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			Object result = myComponents[i].objectsDo_(aBlock);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Enumerate each geometries and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#geometriesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating 
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			Object result = myComponents[i].geometriesDo_(aBlock);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].pointsDo_(aBlock);
		}
	}

	/**
	 * Enumerate all of the primitive objects and evaluate the Block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#primitivesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating 
	 */
	public Object primitivesDo_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].primitivesDo_(aBlock);
		}
		return null;
	}

	/**
	 * Enumerate every polygons and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#polygonsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void polygonsDo_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].polygonsDo_(aBlock);
		}
	}

	/**
	 * Enumerate every polylines and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#polylinesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void polylinesDo_(StBlockClosure aBlock) {
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			myComponents[i].polylinesDo_(aBlock);
		}
	}

	/**
	 * Answer the new JunOpenGL3dCompoundObject transformed with aTransformation.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming 
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dCompoundObject newObject = (JunOpenGL3dCompoundObject) this.copy();

		if (components != null) {
			JunOpenGL3dObject[] myComponents = this.components();
			JunOpenGL3dObject[] newArray = new JunOpenGL3dObject[myComponents.length];
			for (int i = 0; i < myComponents.length; i++) {
				newArray[i] = myComponents[i].transform_(aTransformation);
			}
			newObject.components_(newArray);
		}

		return newObject;
	}

	/**
	 * Convert the receiver to JunOpenGL3dCompoundObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#asCompoundObject()
	 * @category converting
	 */
	public JunOpenGL3dCompoundObject asCompoundObject() {
		return this;
	}

	/**
	 * Convert the receiver to JunOpenGL3dPatchedObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dPatchedObject
	 * @category converting
	 */
	public JunOpenGL3dPatchedObject asPatchedObject() {
		JunOpenGL3dPatchedObject patchedObject = new JunOpenGL3dPatchedObject();
		JunOpenGL3dObject[] components = this.components();
		if (components.length > 0) {
			for (int i = 0; i < components.length; i++) {
				if (components[i] instanceof JunOpenGL3dPolygon == false) {
					return null;
				}
				patchedObject.add_((JunOpenGL3dPolygon) components[i]);
			}
		}
		if (name != null) {
			patchedObject.name = name;
		}
		if (paint != null) {
			patchedObject.paint = paint;
		}
		patchedObject.alpha = alpha;
		if (texture != null) {
			patchedObject.texture = texture;
		}

		return patchedObject;
	}

	/**
	 * Anser the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#reversed()
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		JunOpenGL3dObject[] myComponents = this.components();
		JunOpenGL3dCompoundObject newObject = (JunOpenGL3dCompoundObject) _New(this.getClass(), new Integer(myComponents.length));
		for (int i = 0; i < myComponents.length; i++) {
			newObject.add_(myComponents[i].reversed());
		}
		return newObject;
	}

	/**
	 * Answer myself and my normal vectors object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#yourselfAndNormalVectorsObject()
	 * @category converting
	 */
	public JunOpenGL3dObject yourselfAndNormalVectorsObject() {
		JunOpenGL3dCompoundObject yourselfAndNormalVectorsObject = new JunOpenGL3dCompoundObject();

		JunOpenGL3dObject[] components = this.components();
		for (int i = 0; i < components.length; i++) {
			yourselfAndNormalVectorsObject.add_(components[i].yourselfAndNormalVectorsObject());
		}
		this._copyInto(yourselfAndNormalVectorsObject);

		return yourselfAndNormalVectorsObject;
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPolygons()
	 * @category utilities
	 */
	public int numberOfPolygons() {
		int number = 0;
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			number += myComponents[i].numberOfPolygons();
		}
		return number;
	}

	/**
	 * Answer the number of primitives.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPrimitives()
	 * @category utilities
	 */
	public int numberOfPrimitives() {
		int number = 0;
		JunOpenGL3dObject[] myComponents = this.components();
		for (int i = 0; i < myComponents.length; i++) {
			number += myComponents[i].numberOfPrimitives();
		}
		return number;
	}

	/**
	 * Collect all the patches.
	 * 
	 * @param patches java.util.Vector
	 * @param points java.util.Vector
	 * @param aPolygon jp.co.sra.smalltalk.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected void patcheCollection_from_for_(Vector patches, Vector points, JunOpenGL3dObject aPolygon) {
		if (aPolygon instanceof JunOpenGL3dPolygon) {
			Jun3dPoint[] vertexes = ((JunOpenGL3dPolygon) aPolygon).vertexes();
			Vector collection = new Vector(vertexes.length);
			for (int i = 0; i < vertexes.length; i++) {
				int index = points.indexOf(vertexes[i]);
				if (index >= 0) {
					collection.addElement(new Integer(index + 1));
				}
			}
			patches.addElement(collection);
		}
	}

	/**
	 * Collect all the points of the Polygon. The Vector "points" could already
	 * contains some points and avoid adding the same point to the vector.
	 * 
	 * @param points java.util.Vector
	 * @param aPolygon jp.co.sra.smalltalk.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected void pointCollection_for_(Vector points, JunOpenGL3dObject aPolygon) {
		if (aPolygon instanceof JunOpenGL3dPolygon) {
			Jun3dPoint[] vertexes = ((JunOpenGL3dPolygon) aPolygon).vertexes();
			for (int i = 0; i < vertexes.length; i++) {
				if (points.indexOf(vertexes[i]) < 0) {
					points.addElement(vertexes[i]);
				}
			}
		}
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("CompoundBody");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		if (!this.isCompoundBody()) {
			return this.patchedOrSkinningBodyToLispList();
		}

		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.componentsToLispList());

		return list;
	}

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList textureToLispList() {
		if (!this.hasTexture()) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("texture"));
		list.tail_(this.texture().toLispList());

		return list;
	}

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList componentsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("bodies"));
		JunOpenGL3dObject[] components = this.components();
		for (int i = 0; i < components.length; i++) {
			list.add_(components[i].toLispList());
		}
		return list;
	}

	/**
	 * Convert the receiver (which is a patched body) as a JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons patchedBodyToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("PatchedBody"));

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}

		Vector points;
		Object[] patches;

		JunCursors _cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try {
			_cursor._show();

			JunOpenGL3dObject[] polygons = this.components();
			points = new Vector(1024);
			for (int i = 0; i < polygons.length; i++) {
				this.pointCollection_for_(points, polygons[i]);
			}

			Vector patcheCollection = new Vector(1024);
			for (int i = 0; i < polygons.length; i++) {
				this.patcheCollection_from_for_(patcheCollection, points, polygons[i]);
			}

			patches = patcheCollection.toArray();
			for (int i = 0; i < patches.length; i++) {
				patches[i] = JunLispCons.List_((Vector) patches[i]);
			}
		} finally {
			_cursor._restore();
		}

		JunLispList pointList = JunLispCons.List_(points);
		JunLispList patcheList = JunLispCons.List_(patches);
		pointList = JunLispCons.Head_tail_($("points"), pointList);
		patcheList = JunLispCons.Head_tail_($("patches"), patcheList);
		list.add_(pointList);
		list.add_(patcheList);

		return list;
	}

	/**
	 * Convert the receiver (which is a skinning body) as a JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons skinningBodyToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("SkinningBody"));

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}

		JunOpenGL3dObject[] myComponents = this.components();
		JunLispList[] patches = new JunLispList[myComponents.length];
		for (int i = 0; i < patches.length; i++) {
			patches[i] = myComponents[i].toLispList();
		}
		JunLispList patcheList = JunLispCons.List_(patches);
		patcheList = JunLispCons.Head_tail_($("polygons"), patcheList);
		list.add_(patcheList);

		return list;
	}

	/**
	 * Convert the receiver as an appropriate LispList. 
	 * The receiver could be either a patched body or a skinning body.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons patchedOrSkinningBodyToLispList() {
		if (this.isSkinningBody()) {
			return this.skinningBodyToLispList();
		} else {
			return this.asPatchedObject().toLispList();
		}
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.componentsFromLispList(aList);
		this.textureFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Get my components from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void componentsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("bodies")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] objects = ((JunLispList) list.tail()).asArray();
		for (int i = 0; i < objects.length; i++) {
			this.add_(JunOpenGL3dCompoundObject.ObjectFromLispList_((JunLispCons) objects[i]));
		}
	}

	/**
	 * Get my texture from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void textureFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("texture")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		if (list.tail() instanceof JunLispCons) {
			this.texture_(new JunOpenGLTexture((JunLispCons) list.tail()));
		}
	}

	/**
	 * Get my attributes from the lisp list and set it as a patched body.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @deprecated since Jun454, use JunOpenGL3dPatchedObject
	 * @category lisp support
	 */
	protected void patchedBodyFromLispList(JunLispList aList) {
		this.nameFromLispList(aList);
		this.colorFromLispList(aList);

		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("points")));
			}
		}, new StBlockClosure());
		Object[] points = (list == null) ? null : ((JunLispCons) list.tail()).asArray();

		list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("patches")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		if (points == null) {
			Object[] objects = ((JunLispCons) list.tail()).asArray();
			for (int i = 0; i < objects.length; i++) {
				this.add_(new JunOpenGL3dPolygon((JunLispCons) objects[i]));
			}
		} else {
			Object[] objects = ((JunLispCons) list.tail()).asArray();
			Object[] patches = new Object[objects.length];
			for (int i = 0; i < objects.length; i++) {
				patches[i] = ((JunLispList) objects[i]).asArray();
			}
			for (int i = 0; i < patches.length; i++) {
				Object[] indexes = (Object[]) patches[i];
				int size = indexes.length;
				Jun3dPoint[] vertexes = new Jun3dPoint[size];
				for (int j = 0; j < size; j++) {
					vertexes[j] = (Jun3dPoint) points[((Number) indexes[j]).intValue() - 1];
				}
				this.add_(new JunOpenGL3dPolygon(vertexes));
			}
		}

		this.textureFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Get my attributes from the lisp list and set it as a patched body.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void skinningBodyFromLispList(JunLispList aList) {
		this.nameFromLispList(aList);
		this.colorFromLispList(aList);

		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("polygons")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] objects = ((JunLispList) list.tail()).asArray();
		for (int i = 0; i < objects.length; i++) {
			this.add_(new JunOpenGL3dPolygon((JunLispCons) objects[i]));
		}

		this.textureFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Write my VRML1.0 string on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			bw.write("DEF ");
			bw.write(this.legalName());
			bw.newLine();
			bw.write("Separator {");
			bw.newLine();
			this.vrml10ColorOn_(bw);
			this.vrml10ComponentsOn_(bw);
			bw.write("} #Separator");
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return String
	 * @category vrml support
	 */
	protected String vrml10IndexedSetName() {
		return "IndexedFaceSet";
	}

	/**
	 * Write my components as VRML1.0 on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @category vrml support
	 */
	protected void vrml10ComponentsOn_(Writer aWriter) {
		if (!this.isCompoundBody()) {
			this.vrml10IndexedFaceSetOn_(aWriter);
			return;
		}

		JunOpenGL3dObject[] components = this.components();
		for (int index = 0; index < components.length; index++) {
			components[index].vrml10On_(aWriter);
		}
	}

	/**
	 * Write my IndexedFaceSet as VRML1.0 on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException 
	 * @category vrml support
	 */
	protected void vrml10IndexedFaceSetOn_(Writer aWriter) {
		JunOpenGL3dObject[] polygons = this.components();
		Vector points = new Vector();
		for (int index = 0; index < polygons.length; index++) {
			this.pointCollection_for_(points, polygons[index]);
		}
		Vector patches = new Vector();
		for (int index = 0; index < polygons.length; index++) {
			this.patcheCollection_from_for_(patches, points, polygons[index]);
		}

		if (points.isEmpty()) {
			return;
		}

		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			bw.write("\t");
			bw.write("Coordinate3 {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("point [");
			bw.newLine();

			for (int index = 0; index < points.size(); index++) {
				Jun3dPoint point = (Jun3dPoint) points.elementAt(index);
				bw.write("\t\t\t");
				bw.write(String.valueOf((float) point.x()));
				bw.write(" ");
				bw.write(String.valueOf((float) point.y()));
				bw.write(" ");
				bw.write(String.valueOf((float) point.z()));
				if (index != (points.size() - 1)) {
					bw.write(",");
				}
				bw.write(" #");
				bw.write(String.valueOf(index));
				bw.newLine();
			}

			bw.write("\t\t");
			bw.write("] #point");
			bw.newLine();
			bw.write("\t");
			bw.write("} #Coordinate3");
			bw.newLine();

			if (this.hasTexture()) {
				bw.write("\t");
				bw.write("Texture2 {");
				bw.newLine();
				bw.write("\t\t");
				bw.write("image ");
				this.texture.vrmlSFImageOn_(bw);
				bw.newLine();
				bw.write("\t");
				bw.write("} #Texture2");
				bw.newLine();
			}

			if (this.hasTexture() && (this.texture().hasCoordinates())) {
				bw.write("\t");
				bw.write("TextureCoordinate2 {");
				bw.newLine();
				bw.write("\t\t");
				bw.write("point [");
				bw.newLine();

				Jun2dPoint[] coordinates = this.texture().coordinates();
				for (int index = 0; index < coordinates.length; index++) {
					Jun2dPoint point = coordinates[index];
					bw.write("\t\t\t");
					bw.write(String.valueOf((float) point.x()));
					bw.write(" ");
					bw.write(String.valueOf((float) point.y()));
					if (index != (coordinates.length - 1)) {
						bw.write(",");
					}
					bw.write(" #");
					bw.write(String.valueOf(index));
					bw.newLine();
				}

				bw.write("\t\t");
				bw.write("]");
				bw.newLine();
				bw.write("\t");
				bw.write("} #TextureCoordinate2");
				bw.newLine();
			}

			bw.write("\t");
			bw.write(this.vrml10IndexedSetName());
			bw.write(" {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("coordIndex [");
			bw.newLine();
			for (int nth = 0; nth < patches.size(); nth++) {
				Vector indexes = (Vector) patches.elementAt(nth);
				bw.write("\t\t\t");
				for (int i = 0; i < indexes.size(); i++) {
					int index = ((Integer) indexes.elementAt(i)).intValue();
					bw.write(String.valueOf(index - 1));
					bw.write(", ");
				}
				bw.write("-1");
				if (nth != (patches.size() - 1)) {
					bw.write(",");
				}
				bw.newLine();
			}

			bw.write("\t\t");
			bw.write("] #coordIndex");
			bw.newLine();

			if (this.hasTexture() && (this.texture().hasCoordinates())) {
				bw.write("\t\t");
				bw.write("textureCoordIndex [");
				bw.newLine();

				int i = 0;
				for (int nth = 1; nth < (patches.size() + 1); nth++) {
					Vector indexes = (Vector) patches.elementAt(nth - 1);
					bw.write("\t\t\t");
					for (int index = i; index < (i + indexes.size() - 1); index++) {
						bw.write(String.valueOf(index));
						bw.write(", ");
					}
					bw.write("-1");
					i = i + indexes.size();
					if (nth != patches.size()) {
						bw.write(",");
					}
					bw.newLine();
				}

				bw.write("\t\t");
				bw.write("] #textureCoordIndex");
				bw.newLine();
			}

			bw.write("\t");
			bw.write("} #");
			bw.write(this.vrml10IndexedSetName());
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	public void vrml20On_(PrintWriter pw, String leader) {
		pw.println(leader + "Transform {");

		pw.println(leader + INDENT + "children [");
		if (this.isCompoundBody()) {
			this.vrml20ComponentsOn_(pw, leader + INDENT + INDENT);
		} else {
			this.vrml20ShapeOn_(pw, leader + INDENT + INDENT);
		}
		pw.println(leader + INDENT + "] # children");

		Jun3dBoundingBox box = this.boundingBox();
		Jun3dPoint center = (Jun3dPoint) box.center();
		pw.print(leader + INDENT + "bboxCenter ");
		pw.print((float) center.x());
		pw.print(' ');
		pw.print((float) center.y());
		pw.print(' ');
		pw.print((float) center.z());
		pw.println();
		pw.print(leader + INDENT + "bboxSize ");
		pw.print((float) box.width());
		pw.print(' ');
		pw.print((float) box.height());
		pw.print(' ');
		pw.print((float) box.depth());
		pw.println();

		pw.println(leader + "} #Transform");
		pw.flush();
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return String
	 * @category vrml support
	 */
	protected String vrml20IndexedSetName() {
		return "IndexedFaceSet";
	}

	/**
	 * Write my components as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @throws SmalltalkException 
	 * @category vrml support
	 */
	protected void vrml20ComponentsOn_(PrintWriter pw, String leader) {
		JunOpenGL3dObject[] components = this.components();
		for (int i = 0; i < components.length; i++) {
			components[i].vrml20On_(pw, leader);
			if (i < components.length - 1) {
				pw.println(leader + ',');
			}
		}
	}

	/**
	 * Write my geoemtry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		JunOpenGL3dObject[] polygons = this.components();
		Vector points = new Vector();
		for (int index = 0; index < polygons.length; index++) {
			this.pointCollection_for_(points, polygons[index]);
		}
		Vector patches = new Vector();
		for (int index = 0; index < polygons.length; index++) {
			this.patcheCollection_from_for_(patches, points, polygons[index]);
		}
		if (points.isEmpty()) {
			return;
		}

		pw.println(leader + "geometry " + this.vrml20IndexedSetName() + " {");

		pw.println(leader + INDENT + "coord Coordinate {");
		pw.println(leader + INDENT + INDENT + "point [");
		for (int i = 0; i < points.size(); i++) {
			Jun3dPoint point = (Jun3dPoint) points.elementAt(i);
			pw.print(leader + INDENT + INDENT + INDENT);
			pw.print((float) point.x());
			pw.print(' ');
			pw.print((float) point.y());
			pw.print(' ');
			pw.print((float) point.z());
			if (i < points.size() - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}
		pw.println(leader + INDENT + INDENT + "] #point");
		pw.println(leader + INDENT + "} #Coordinate");

		pw.println(leader + INDENT + "coordIndex [ #patches: " + patches.size());
		for (int i = 0; i < patches.size(); i++) {
			Vector indexes = (Vector) patches.elementAt(i);
			pw.print(leader + INDENT + INDENT);
			for (int j = 0; j < indexes.size(); j++) {
				int index = ((Integer) indexes.elementAt(j)).intValue();
				pw.print(index - 1);
				pw.print(", ");
			}
			pw.write("-1");
			if (i < patches.size() - 1) {
				pw.print(',');
			}
			pw.println();
		}
		pw.println(leader + INDENT + "] #coordIndex");

		if (this.hasTexture() && (this.texture().hasCoordinates())) {
			pw.println(leader + INDENT + "texCoord TextureCoordinate {");
			pw.println(leader + INDENT + INDENT + "point [");
			Jun2dPoint[] coordinates = this.texture().coordinates();
			for (int i = 0; i < coordinates.length; i++) {
				Jun2dPoint point = coordinates[i];
				pw.print(leader + INDENT + INDENT + INDENT);
				pw.print((float) point.x());
				pw.print(' ');
				pw.print((float) point.y());
				if (i < coordinates.length - 1) {
					pw.print(',');
				}
				pw.println(" #" + i);
			}
			pw.println(leader + INDENT + INDENT + "] #point");
			pw.println(leader + INDENT + "} #TextureCoordinate");

			pw.println(leader + INDENT + "texCoordIndex [");
			int i = 0;
			for (int nth = 0; nth < patches.size(); nth++) {
				Vector indexes = (Vector) patches.elementAt(nth);
				pw.print(leader + INDENT + INDENT);
				for (int index = i; index <= i + indexes.size() - 1; index++) {
					pw.print(index);
					pw.print(", ");
				}
				pw.print("-1");
				i = i + indexes.size();
				if (nth != patches.size()) {
					pw.print(',');
				}
				pw.println();
			}
			pw.println(leader + INDENT + "] #texCoordIndex");
		}

		pw.println(leader + INDENT + "solid FALSE");
		pw.println(leader + "} #" + this.vrml20IndexedSetName());
	}

}
