package jp.co.sra.jun.opengl.grapher;

import java.awt.Color;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;

/**
 * JunOpenGL3dNode class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/12/07 (by Hirotsugu Kondo)
 *  @updated   2004/11/12 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dNode.java,v 8.10 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGL3dNode extends JunOpenGL3dGraphAbstract {
	protected Jun3dPoint nodeLocation;
	protected Jun3dPoint nodeExtent;
	protected Object nodeEntity;
	protected JunOpenGLTexture nodeTexture;

	/**
	 * Create a new instance of JunOpenGL3dNode with a extent point.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Extent_(Jun3dPoint extentPoint) {
		return Extent_color_(extentPoint, Color.gray);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a extent point and a color.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param colorValue java.awt.Color
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Extent_color_(Jun3dPoint extentPoint, Color colorValue) {
		return Location_extent_color_(new Jun3dPoint(0, 0, 0), extentPoint, colorValue);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a extent point, a color and a entity object.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param colorValue java.awt.Color
	 * @param anObject java.lang.Object
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Extent_color_entity_(Jun3dPoint extentPoint, Color colorValue, Object anObject) {
		return Location_extent_color_entity_(new Jun3dPoint(0, 0, 0), extentPoint, colorValue, anObject);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a extent point and a entity object.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param extentPoint jp.co.sra.jun.opengl.basic.Jun3dPoint
	 * @param anObject java.lang.Object
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Extent_entity_(Jun3dPoint extentPoint, Object anObject) {
		return Location_extent_color_entity_(new Jun3dPoint(0, 0, 0), extentPoint, null, anObject);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode from a JunLispList.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @deprecated since Jun454, use the constructor
	 * @category Lisp support
	 */
	public static JunOpenGL3dGraphAbstract FromLispList_(JunLispList aList) {
		return new JunOpenGL3dNode(aList);
	}

	/**
	 * Load a JunOpenGL3dNode from the object.
	 * 
	 * @param anObject java.lang.Object
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Lisp support
	 */
	public static JunOpenGL3dObject LoadFrom_(Object anObject) {
		JunLispList aList = (JunLispList) JunLispParser.Parse_(anObject);
		return new JunOpenGL3dNode(aList);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a location point and a extent point.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param locationPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Location_extent_(Jun3dPoint locationPoint, Jun3dPoint extentPoint) {
		return Location_extent_color_entity_(locationPoint, extentPoint, null, null);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a location point, a extent point and a color.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param locationPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param colorValue java.awt.Color
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Location_extent_color_(Jun3dPoint locationPoint, Jun3dPoint extentPoint, Color colorValue) {
		return Location_extent_color_entity_(locationPoint, extentPoint, colorValue, null);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a location point, a extent point, a color and entity object.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param locationPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param colorValue java.awt.Color
	 * @param anObject java.lang.Object
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Location_extent_color_entity_(Jun3dPoint locationPoint, Jun3dPoint extentPoint, Color colorValue, Object anObject) {
		JunOpenGL3dNode node = new JunOpenGL3dNode();
		node.location_(locationPoint);
		node.extent_(extentPoint);
		node.color_(colorValue);
		node.entity_(anObject);

		return node;
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with a location point, a extent point and entity object.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param locationPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param extentPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param anObject java.lang.Object
	 * @category Instance creation
	 */
	public static JunOpenGL3dNode Location_extent_entity_(Jun3dPoint locationPoint, Jun3dPoint extentPoint, Object anObject) {
		return Location_extent_color_entity_(locationPoint, extentPoint, null, anObject);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dNode() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dNode from a JunLispList.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dNode(JunLispList aList) {
		super(aList);
	}

	/**
	 * Answer the receiver's corner point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint corner() {
		Jun3dPoint location = this.location();
		return new Jun3dPoint(location.x() + this.width(), location.y() + (this.height() / 2), location.z() + (this.depth() / 2));
	}

	/**
	 * Create receiver's display object and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category displaying
	 */
	protected JunOpenGL3dObject createDisplayObject() {
		JunOpenGL3dObject object = JunOpenGL3dObject.Origin_corner_(this.origin(), this.corner());

		if (this.hasColor()) {
			object.paint_(this.color());
		}
		if (this.hasTexture()) {
			object.texture_(this.texture());
			if (this.texture().coordinates() == null) {
				this.texture().zPositiveParameters();
			}
		}

		return object;
	}

	/**
	 * Answer the receiver's depth size.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double depth() {
		return this.extent().z();
	}

	/**
	 * Get the receiver's entity object.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object entity() {
		return this.nodeEntity;
	}

	/**
	 * Set the receiver's entity object.
	 * 
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void entity_(Object anObject) {
		this.nodeEntity = anObject;
	}

	/**
	 * Get my name from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void entityFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("entity")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.entity_(list.tail());
	}

	/**
	 * Convert the receiver's entity as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons entityToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("entity"));
		list.tail_(this.entity());
		return list;
	}

	/**
	 * Get the receiver's extent point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint extent() {
		Jun3dPoint extent = this.nodeExtent;
		if (extent == null) {
			extent = new Jun3dPoint(1, 1, 1);
			this.nodeExtent = extent;
		}
		return extent;
	}

	/**
	 * Set the receiver's extent point.
	 * 
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void extent_(Jun3dPoint a3dPoint) {
		this.nodeExtent = a3dPoint;
		this.flushDisplayObject();
	}

	/**
	 * Get my extent from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void extentFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("extent")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.extent_((Jun3dPoint) list.tail());
	}

	/**
	 * Convert the receiver's extent as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons extentToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("extent"));
		list.tail_(this.extent());
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.locationFromLispList_(aList);
		this.extentFromLispList_(aList);
		this.entityFromLispList_(aList);
		this.textureFromLispList_(aList);
	}

	/**
	 * Answer true if the receiver has a entity, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasEntity() {
		return this.entity() != null;
	}

	/**
	 * Answer true if the receiver has a texture, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasTexture() {
		return this.texture() != null;
	}

	/**
	 * Answer the receiver's height size.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double height() {
		return this.extent().y();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		nodeLocation = null;
		nodeExtent = null;
		nodeTexture = null;
		nodeEntity = null;
	}

	/**
	 * Answer true if the receiver is a node object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNode() {
		return true;
	}

	/**
	 * Answer true if the receiver is a kind of JunOpenGL3dPrimitiveObject, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPrimitive() {
		return true;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Node");
	}

	/**
	 * Get the receiver's location point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint location() {
		Jun3dPoint location = this.nodeLocation;

		if (location == null) {
			location = new Jun3dPoint(0, 0, 0);
			this.nodeLocation = location;
		}

		return location;
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void location_(Jun3dPoint a3dPoint) {
		this.nodeLocation = a3dPoint;
		this.flushDisplayObject();
	}

	/**
	 * Get my location from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void locationFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("location")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.location_((Jun3dPoint) list.tail());
	}

	/**
	 * Convert the receiver's location as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons locationToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("location"));
		list.tail_(this.location());

		return list;
	}

	/**
	 * Answer the receiver's origin point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint origin() {
		Jun3dPoint location = this.location();
		return new Jun3dPoint(location.x(), location.y() - (this.height() / 2), location.z() - (this.depth() / 2));
	}

	/**
	 * Answer the texture of the receiver. Does not have a texture as a default.
	 * 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category accessing
	 */
	public JunOpenGLTexture texture() {
		return this.nodeTexture;
	}

	/**
	 * Set the texture of the receiver.
	 * 
	 * @param aTexture jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @category accessing
	 */
	public void texture_(JunOpenGLTexture aTexture) {
		this.nodeTexture = aTexture;
	}

	/**
	 * Get my texture from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void textureFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("texture")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.texture_(new JunOpenGLTexture((JunLispList) list.tail()));
	}

	/**
	 * Convert the receiver's texture as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispList textureToLispList() {
		if (!this.hasTexture()) {
			return this.lispNil();
		} else {
			JunLispCons list = this.lispCons();
			list.head_($("texture"));
			list.tail_(this.texture().toLispList());

			return list;
		}
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		list.add_(this.locationToLispList());
		list.add_(this.extentToLispList());
		list.head_(this.kindName());
		if (this.hasEntity()) {
			list.add_(this.entityToLispList());
		}
		list.head_(this.kindName());
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}

		return list;
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		this.location_((Jun3dPoint) this.location().transform_(aTransformation));
		return this;
	}

	/**
	 * Answer the receiver's width size.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double width() {
		return this.extent().x();
	}
}
