package jp.co.sra.jun.opengl.grapher;

import java.awt.Color;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StReadStream;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGL3dGraphAbstract class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/12/07 (by Hirotsugu Kondo)
 *  @updated   2004/11/12 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun500 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dGraphAbstract.java,v 8.13 2008/02/20 06:32:34 nisinaka Exp $
 */
public abstract class JunOpenGL3dGraphAbstract extends JunOpenGL3dObject {

	protected JunOpenGL3dObject displayObject;
	protected Color displayColor;

	/**
	 * Answer the default arc class.
	 * 
	 * @return java.lang.Class
	 * @category Defaults
	 */
	public static Class DefaultArcClass() {
		return JunOpenGL3dArc.class;
	}

	/**
	 * Answer the default graph class.
	 * 
	 * @return java.lang.Class
	 * @category Defaults
	 */
	public static Class DefaultGraphClass() {
		return JunOpenGL3dGraph.class;
	}

	/**
	 * Answer the default node class.
	 * 
	 * @return java.lang.Class
	 * @category Defaults
	 */
	public static Class DefaultNodeClass() {
		return JunOpenGL3dNode.class;
	}

	/**
	 * Load a graph element of the specified class from the read stream.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract
	 * @param aClass java.lang.Class
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category Lisp support
	 */
	public static JunOpenGL3dGraphAbstract LoadFrom_(Class aClass, StReadStream aStream) {
		JunLispList list = (JunLispList) JunLispParser.Parse_(aStream);
		try {
			JunOpenGL3dGraphAbstract graph = (JunOpenGL3dGraphAbstract) _PerformWith(aClass, "FromLispList_", list);
			return graph;
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Create a new instance of JunOpenGL3dGraphAbstract.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dGraphAbstract() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dGraphAbstract and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	protected JunOpenGL3dGraphAbstract(JunLispList aList) {
		super(aList);
	}

	/**
	 * Convert the receiver to JunOpenGL3dCompoundObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#asCompoundObject()
	 * @category converting
	 */
	public JunOpenGL3dCompoundObject asCompoundObject() {
		return new JunOpenGL3dCompoundObject(this.displayObject());
	}

	/**
	 * Convert the receiver to an array of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint3d[]
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#asPointArray()
	 * @category converting
	 */
	public Jun3dPoint[] asPointArray() {
		return this.displayObject().asPointArray();
	}

	/**
	 * Answer my bouding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#boundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return this.displayObject().boundingBox();
	}

	/**
	 * Get the color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color color() {
		return this.displayColor;
	}

	/**
	 * Set the color.
	 * 
	 * @param colorValue java.awt.Color
	 * @category accessing
	 */
	public void color_(Color colorValue) {
		this.displayColor = colorValue;
		this.flushDisplayObject();
	}

	/**
	 * Establish all normal vectors with the specified smoothing angle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#establishAllNormalVectorsWithSmoothingAngle_(jp.co.sra.jun.geometry.basic.JunAngle)
	 * @category vector accessing
	 */
	public void establishAllNormalVectorsWithSmoothingAngle_(JunAngle anAngle) {
		if (displayObject != null) {
			displayObject.establishAllNormalVectorsWithSmoothingAngle_(anAngle);
		}
	}

	/**
	 * Flush all alphas.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllAlphas()
	 * @category flushing
	 */
	public void flushAllAlphas() {
		this.flushAlpha();

		if (displayObject != null) {
			displayObject.flushAllAlphas();
		}
	}

	/**
	 * Flush all bounds.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllBounds()
	 * @category flushing
	 */
	public void flushAllBounds() {
		this.flushBounds();

		if (displayObject != null) {
			displayObject.flushAllBounds();
		}
	}

	/**
	 * Flush all colors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllColors()
	 * @category flushing
	 */
	public void flushAllColors() {
		this.flushColors();

		if (displayObject != null) {
			displayObject.flushAllColors();
		}
	}

	/**
	 * Flush all names.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllNames()
	 * @category flushing
	 */
	public void flushAllNames() {
		this.flushName();

		if (displayObject != null) {
			displayObject.flushAllNames();
		}
	}

	/**
	 * Flush all normal vectors.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllNormalVectors()
	 * @category flushing
	 */
	public void flushAllNormalVectors() {
		this.flushNormalVectors();

		if (displayObject != null) {
			displayObject.flushAllNormalVectors();
		}
	}

	/**
	 * Flush all paints.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#flushAllPaints()
	 * @category flushing
	 */
	public void flushAllPaints() {
		this.flushPaint();

		if (displayObject != null) {
			displayObject.flushAllPaints();
		}
	}

	/**
	 * Get my color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#colorFromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	protected void colorFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("color")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		list = (JunLispCons) list.tail();
		Number red = (Number) list.nth_(1);
		if (red instanceof Integer) {
			red = new Float(Math.max(0, Math.min(red.floatValue() / 255, 1)));
		}
		Number green = (Number) list.nth_(2);
		if (green instanceof Integer) {
			green = new Float(Math.max(0, Math.min(green.floatValue() / 255, 1)));
		}
		Number blue = (Number) list.nth_(3);
		if (blue instanceof Integer) {
			blue = new Float(Math.max(0, Math.min(blue.floatValue() / 255, 1)));
		}
		this.color_(new Color(red.floatValue(), green.floatValue(), blue.floatValue()));
	}

	/**
	 * Convert the receiver's color as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#colorToLispList()
	 * @category lisp support
	 */
	protected JunLispList colorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("color"));

		Color colorValue = this.color();
		list.add_(new Float(colorValue.getRed() / 255f));
		list.add_(new Float(colorValue.getGreen() / 255f));
		list.add_(new Float(colorValue.getBlue() / 255f));
		if (this.hasAlpha()) {
			list.add_(new Float(this.alpha()));
		}
		return list;
	}

	/**
	 * Make a copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		// return (JunOpenGL3dGraphAbstract) _PerformWith(this.getClass(), "FromLispList_", this.toLispList());
		// can not copy because of the image.
		return this;
	}

	/**
	 * Create receiver's display object and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category displaying
	 */
	protected abstract JunOpenGL3dObject createDisplayObject();

	/**
	 * Answer the receiver's JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category displaying
	 */
	public JunOpenGL3dObject displayObject() {
		if (displayObject == null) {
			displayObject = this.createDisplayObject();
		}
		return displayObject;
	}

	/**
	 * Flush receiver's display object.
	 * 
	 * @category displaying
	 */
	protected void flushDisplayObject() {
		this.displayObject = null;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
	}

	/**
	 * Answer the receiver's unique id string.
	 * 
	 * @return java.lang.String
	 * @category lisp support
	 */
	protected String generateIdString() {
		return JunOpenGL3dObject.UniqueId();
	}

	/**
	 * Enumerate every geometries and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#geometriesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return null;
		}
		return this.displayObject().geometriesDo_(aBlock);
	}

	/**
	 * Answer true if the receiver has a color, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasColor()
	 * @category testing
	 */
	public boolean hasColor() {
		return this.color() != null;
	}

	/**
	 * Answer true if the receiver is an arc object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isArc() {
		return false;
	}

	/**
	 * Answer true if the receiver is a graph object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isGraph() {
		return false;
	}

	/**
	 * Answer true if the receiver is a node object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNode() {
		return false;
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPolygons()
	 * @category utilities
	 */
	public int numberOfPolygons() {
		if (this.displayObject() == null) {
			return 0;
		}
		return this.displayObject().numberOfPolygons();
	}

	/**
	 * Answer the number of primitives.
	 * 
	 * @return int
	 * @category utilities
	 */
	public int numberOfPrimitives() {
		if (this.displayObject() == null) {
			return 0;
		}
		return this.displayObject().numberOfPrimitives();
	}

	/**
	 * Enumerate every objects and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#objectsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object objectsDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return null;
		}
		return this.displayObject().objectsDo_(aBlock);
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return;
		}
		this.displayObject().pointsDo_(aBlock);
	}

	/**
	 * Enumerate every polygons and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#polygonsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void polygonsDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return;
		}
		this.displayObject().polygonsDo_(aBlock);
	}

	/**
	 * Enumerate every polylines and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#polylinesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void polylinesDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return;
		}
		this.displayObject().polylinesDo_(aBlock);
	}

	/**
	 * Enumerate all of the primitive objects and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#primitivesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object primitivesDo_(StBlockClosure aBlock) {
		if (this.displayObject() == null) {
			return null;
		}
		return this.displayObject().primitivesDo_(aBlock);
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		this.displayObject().renderOn_(aRenderingContext);
	}

	/**
	 * Anser the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#reversed()
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		return this.displayObject().reversed();
	}

	/**
	 * Set the receiver's value.
	 * 
	 * @param a3dObject java.lang.Object
	 * @category accessing
	 */
	public void setValue_(Object a3dObject) {
		this.displayObject = (JunOpenGL3dObject) a3dObject;
	}

	/**
	 * Answer the transformation of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transformation()
	 * @category transforming
	 */
	public Jun3dTransformation transformation() {
		return this.displayObject().transformation();
	}

	/**
	 * Get the receiver's value.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object value() {
		return (Object) this.displayObject();
	}

	/**
	 * Write the VRML1.0 string of the receiver on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}
		displayObject.vrml10On_(aWriter);
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml20On_(Writer aWriter) {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}
		displayObject.vrml20On_(aWriter);
	}

	/**
	 * Write the VRML string of the receiver on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrmlOn_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrmlOn_(Writer aWriter) {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}
		displayObject.vrmlOn_(aWriter);
	}

}
