package jp.co.sra.jun.opengl.grapher;

import java.awt.Color;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;

/**
 * JunOpenGL3dArc class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/12/07 (by Hirotsugu Kondo)
 *  @updated   2004/11/12 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dArc.java,v 8.10 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGL3dArc extends JunOpenGL3dGraphAbstract {
	protected boolean crankArc;
	protected JunOpenGL3dNode fromNode;
	protected JunOpenGL3dNode toNode;

	/**
	 * Create a new instance of JunOpenGL3dNode with fron node and to node.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc
	 * @param fromNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param toNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category Instance creation
	 */
	public static JunOpenGL3dArc From_to_(JunOpenGL3dNode fromNode, JunOpenGL3dNode toNode) {
		return From_to_color_(fromNode, toNode, Color.gray);
	}

	/**
	 * Create a new instance of JunOpenGL3dNode with fron node, to node and color.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc
	 * @param fromNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param toNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param colorValue java.awt.Color
	 * @category Instance creation
	 */
	public static JunOpenGL3dArc From_to_color_(JunOpenGL3dNode fromNode, JunOpenGL3dNode toNode, Color colorValue) {
		JunOpenGL3dArc arc = new JunOpenGL3dArc();
		arc.from_to_(fromNode, toNode);
		arc.color_(colorValue);

		return arc;
	}

	/**
	 * Create a new instance of JunOpenGL3dNode from a JunLispList.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @deprecated since Jun454, use the constructor
	 * @category Lisp support
	 */
	public static JunOpenGL3dGraphAbstract FromLispList_(JunLispList aList) {
		return new JunOpenGL3dArc(aList);
	}

	/**
	 * Load a JunOpenGL3dArc from the object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param anObject java.lang.Object
	 * @category Lisp support
	 */
	public static JunOpenGL3dObject LoadFrom_(Object anObject) {
		JunLispList aList = (JunLispList) JunLispParser.Parse_(anObject);
		return new JunOpenGL3dArc(aList);
	}

	/**
	 * Create a new instance of JunOpenGL3dArc and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dArc() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dNode from a JunLispList.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dArc(JunLispList aList) {
		super(aList);
	}

	/**
	 * Answer true if the receiver is crank, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean crank() {
		return crankArc;
	}

	/**
	 * Set the crank.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void crank_(boolean aBoolean) {
		crankArc = aBoolean;
	}

	/**
	 * Get my crank from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void crankFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("crank")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.crank_(((Boolean) list.tail()).booleanValue());
	}

	/**
	 * Convert the receiver's crank as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons crankToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("crank"));
		list.tail_(Boolean.valueOf(this.crank()));

		return list;
	}

	/**
	 * Create receiver's display object and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category displaying
	 */
	protected JunOpenGL3dObject createDisplayObject() {
		if (this.crank()) {
			Jun3dPoint p1 = (Jun3dPoint) this.from().location().plus_(new Jun3dPoint(this.from().width(), 0, 0));
			Jun3dPoint p4 = this.to().location();
			Jun3dPoint p2 = new Jun3dPoint(p1.x() + ((p4.x() - (p1.x())) / 2), p1.y(), p1.z());
			Jun3dPoint p3 = new Jun3dPoint(p1.x() + ((p4.x() - (p1.x())) / 2), p4.y(), p4.z());
			Jun3dPoint[] vertexes = new Jun3dPoint[4];
			vertexes[0] = p1;
			vertexes[1] = p2;
			vertexes[2] = p3;
			vertexes[3] = p4;
			JunOpenGL3dPolyline object = new JunOpenGL3dPolyline(vertexes);
			if (this.hasColor()) {
				object.paint_(this.color());
			}
			return (JunOpenGL3dObject) object;
		} else {
			Jun3dPoint p1 = (Jun3dPoint) this.from().location().plus_(new Jun3dPoint(this.from().width(), 0, 0));
			Jun3dPoint p4 = this.to().location();
			Jun3dPoint[] vertexes = new Jun3dPoint[2];
			vertexes[0] = p1;
			vertexes[1] = p4;
			JunOpenGL3dPolyline object = new JunOpenGL3dPolyline(vertexes);
			if (this.hasColor()) {
				object.paint_(this.color());
			}
			return (JunOpenGL3dObject) object;
		}
	}

	/**
	 * Get the receiver's from node.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category accessing
	 */
	public JunOpenGL3dNode from() {
		return fromNode;
	}

	/**
	 * Set the receiver's from node.
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category accessing
	 */
	public void from_(JunOpenGL3dNode aNode) {
		fromNode = aNode;
		this.flushDisplayObject();
	}

	/**
	 * Set the receiver's from node and to node.
	 * 
	 * @param aNode1 jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param aNode2 jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category accessing
	 */
	public void from_to_(JunOpenGL3dNode aNode1, JunOpenGL3dNode aNode2) {
		fromNode = aNode1;
		toNode = aNode2;
		this.flushDisplayObject();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.crankFromLispList_(aList);
		this.fromNodeFromLispList_(aList);
		this.toNodeFromLispList_(aList);
	}

	/**
	 * Get my from node from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void fromNodeFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("from")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		try {
			this.from_(new JunOpenGL3dNode((JunLispList) list.tail()));
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Convert the receiver's from node as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fromNodeToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("from"));
		list.tail_(this.from().toLispList());

		return list;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		crankArc = false;
		fromNode = null;
		toNode = null;
	}

	/**
	 * Answer true if the receiver is an arc object, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isArc() {
		return true;
	}

	/**
	 * Answer true if the receiver is a kind of JunOpenGL3dPrimitiveObject, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPrimitive() {
		return true;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Arc");
	}

	/**
	 * Get the receiver's to node and to node.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category accessing
	 */
	public JunOpenGL3dNode to() {
		return this.toNode;
	}

	/**
	 * Set the receiver's to node and to node.
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category accessing
	 */
	public void to_(JunOpenGL3dNode aNode) {
		this.toNode = aNode;
		this.flushDisplayObject();
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}

		list.add_(this.crankToLispList());
		list.add_(this.fromNodeToLispList());
		list.add_(this.toNodeToLispList());

		return list;
	}

	/**
	 * Convert the receiver without from node and to node, as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispListWithoutFromNodeAndToNode() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}

		list.add_(this.crankToLispList());

		return list;
	}

	/**
	 * Get my to node from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void toNodeFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("to")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		try {
			this.to_(new JunOpenGL3dNode((JunLispList) list.tail()));
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Convert the receiver's to node as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons toNodeToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("to"));
		list.tail_(this.to().toLispList());

		return list;
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		this.flushDisplayObject();
		return this;
	}
}
