package jp.co.sra.jun.opengl.display;

import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.progress.JunCursorAnimator;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.goodies.utilities.JunSensorUtility;
import jp.co.sra.jun.goodies.wheels.JunThumbWheel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dTransformedObject;
import jp.co.sra.jun.opengl.picking.JunOpenGLObjectPicker;
import jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.jun.vrml.support.JunVrmlCompiler;
import jp.co.sra.jun.vrml.support.JunVrmlGenerator10;
import jp.co.sra.jun.vrml.support.JunVrmlGenerator97;
import jp.co.sra.jun.vrml.support.JunVrmlParser10;
import jp.co.sra.jun.vrml.support.JunVrmlParser97;

/**
 * JunOpenGLDisplayModel class
 * 
 *  @author    He Weijie
 *  @created   1998/10/08 (by MATSUDA Ryouichi)
 *  @updated   1998/10/27 (by He Weijie)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2000/01/06 (by MATSUDA Ryouichi)
 *  @updated   2003/05/13 (by nisinaka)
 *  @updated   2003/07/01 (by nisinaka)
 *  @updated   2004/03/23 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2007/08/23 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun696 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLDisplayModel.java,v 8.28 2008/02/20 06:32:19 nisinaka Exp $
 */
public class JunOpenGLDisplayModel extends JunOpenGL3dModel {

	protected HashMap pushButtons;
	protected JunButtonModel dollyButton;

	/**
	 * Create a new instance of JunOpenGLDisplayModel.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLDisplayModel() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel and set the display object.
	 * This is the replacement of the class method "JunOpenGLDisplayModel class >> displayObject:" of the Smalltalk version.
	 *
	 * @param anOpenGL3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public JunOpenGLDisplayModel(JunOpenGL3dObject anOpenGL3dObject) {
		this();
		this.displayObject_(anOpenGL3dObject);
	}

	/**
	 * Create a new instance of the specified class and set the display object.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @param aClass java.lang.Class
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayModel DisplayObject_(Class aClass, JunOpenGL3dObject a3dObject) {
		JunOpenGLDisplayModel displayModel = (JunOpenGLDisplayModel) _New(aClass);
		displayModel.displayObject_(a3dObject);
		return displayModel;
	}

	/**
	 * Answer the normal image for the pick button.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category Images
	 */
	public static BufferedImage PickNormalImage() {
		return JunCursors.NormalCursorImage();
	}

	/**
	 * Answer the bounds image for the pick button.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category Images
	 */
	public static BufferedImage PickBoundsImage() {
		return JunCursors.BoundsCursorImage();
	}

	/**
	 * Answer the pencil image for the pick button.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category Images
	 */
	public static BufferedImage PickPencilImage() {
		return JunCursors.PencilCursorImage();
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_(JunOpenGL3dObject a3dObject) {
		return Show_eyePoint_(a3dObject, null);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint) {
		return Show_eyePoint_sightPoint_(a3dObject, eyePoint, null);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint) {
		return Show_eyePoint_sightPoint_upVector_(a3dObject, eyePoint, sightPoint, null);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_upVector_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector) {
		return Show_eyePoint_sightPoint_upVector_viewFactor_(a3dObject, eyePoint, sightPoint, upVector, Double.NaN);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param viewFactor double
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_upVector_viewFactor_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double viewFactor) {
		return Show_eyePoint_sightPoint_upVector_viewFactor_zoomHeight_(a3dObject, eyePoint, sightPoint, upVector, viewFactor, Double.NaN);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param viewFactor double
	 * @param zoomHeight double
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_upVector_viewFactor_zoomHeight_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double viewFactor, double zoomHeight) {
		return Show_eyePoint_sightPoint_upVector_viewFactor_zoomHeight_in_(a3dObject, eyePoint, sightPoint, upVector, viewFactor, zoomHeight, null);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param viewFactor double
	 * @param zoomHeight double
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_eyePoint_sightPoint_upVector_viewFactor_zoomHeight_in_(JunOpenGL3dObject a3dObject, Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double viewFactor, double zoomHeight, Rectangle aRectangle) {
		Hashtable projectionTable = new Hashtable();
		if (eyePoint != null) {
			projectionTable.put($("eyePoint"), eyePoint);
		}
		if (sightPoint != null) {
			projectionTable.put($("sightPoint"), sightPoint);
		}
		if (upVector != null) {
			projectionTable.put($("upVector"), upVector);
		}
		if (Double.isNaN(viewFactor) == false) {
			projectionTable.put($("viewFactor"), new Double(viewFactor));
		}
		if (Double.isNaN(zoomHeight) == false) {
			projectionTable.put($("zoomHeight"), new Double(zoomHeight));
		}
		return Show_projectionTable_in_(a3dObject, projectionTable, aRectangle);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param displayRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_in_(JunOpenGL3dObject a3dObject, Rectangle displayRectangle) {
		return Show_projectionTable_in_(a3dObject, null, displayRectangle);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aMap java.util.Map
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_projectionTable_(JunOpenGL3dObject a3dObject, Map aMap) {
		return Show_projectionTable_in_(a3dObject, aMap, null);
	}

	/**
	 * Create a new instance of JunOpenGLDisplayModel with the display object and open it.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aMap java.util.Map
	 * @param displayRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Utilities
	 */
	public static JunOpenGLDisplayModel Show_projectionTable_in_(JunOpenGL3dObject a3dObject, Map aMap, Rectangle displayRectangle) {
		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel(a3dObject);
		if (aMap != null) {
			displayModel.defaultProjectionTable_(aMap);
		}
		if (displayRectangle == null) {
			displayModel.open();
		} else {
			displayModel.openIn_(displayRectangle);
		}
		return displayModel;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		pushButtons = null;
		dollyButton = null;
	}

	/**
	 * Answer my current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject displayObject() {
		return super.openGL3dObject();
	}

	/**
	 * Set my new display object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public void displayObject_(JunOpenGL3dObject a3dObject) {
		super.openGL3dObject_(a3dObject);

		if (this.showModel() != null) {
			this.showModel().flushObject();
		}
	}

	/**
	 * Answer my current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public JunOpenGLProjection displayProjection() {
		return super.openGLProjection();
	}

	/**
	 * Set my new projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public void displayProjection_(JunOpenGLProjection aProjection) {
		super.openGLProjection_(aProjection);
	}

	/**
	 * Answer my current projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	public JunOpenGLProjector displayProjector() {
		return super.openGLProjector();
	}

	/**
	 * Answer my show model.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @category accessing
	 */
	public JunOpenGLShowModel showModel() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunOpenGLShowModel) {
				return (JunOpenGLShowModel) dependents[i];
			}
		}
		return null;
	}

	/**
	 * Answer my current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openGL3dObject()
	 * @category accessing
	 */
	public final JunOpenGL3dObject openGL3dObject() {
		return this.displayObject();
	}

	/**
	 * Set my new display object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openGL3dObject_(jp.co.sra.jun.opengl.objects.JunOpenGL3dObject)
	 * @category accessing
	 */
	public final void openGL3dObject_(JunOpenGL3dObject a3dObject) {
		this.displayObject_(a3dObject);
	}

	/**
	 * Answer my current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openGLProjection()
	 * @category accessing
	 */
	public final JunOpenGLProjection openGLProjection() {
		return this.displayProjection();
	}

	/**
	 * Set my new projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openGLProjection_(jp.co.sra.jun.opengl.projection.JunOpenGLProjection)
	 * @category accessing
	 */
	public final void openGLProjection_(JunOpenGLProjection aProjection) {
		this.displayProjection_(aProjection);
	}

	/**
	 * Answer my current projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openGLProjector()
	 * @category accessing
	 */
	public final JunOpenGLProjector openGLProjector() {
		return this.displayProjector();
	}

	/**
	 * Process the JunOpenGL3dObject picked with the alt key down.
	 *
	 * @param pickedObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category manipulating
	 */
	protected void altClick_(JunOpenGL3dObject pickedObject) {
		// Nothing to do as a default.
	}

	/**
	 * Process the JunOpenGL3dObject picked with the control key down.
	 *
	 * @param pickedObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category manipulating
	 */
	protected void ctrlClick_(JunOpenGL3dObject pickedObject) {
		// Nothing to do as a default.
	}

	/**
	 * Change the focus area.
	 * 
	 * @param originalRectangle jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @param newRectangle jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category manipulating
	 */
	public void focus_to_(Jun2dBoundingBox originalRectangle, Jun2dBoundingBox newRectangle) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.displayProjection();
		Jun3dPoint originalCenter = aProjection.translateTo3dPointFromPoint_((Jun2dPoint) originalRectangle.origin().plus_(originalRectangle.corner()).dividedBy_(2));
		Jun3dPoint newCenter = aProjection.translateTo3dPointFromPoint_((Jun2dPoint) newRectangle.origin().plus_(newRectangle.corner()).dividedBy_(2));
		aProjection.sightPoint_((Jun3dPoint) aProjection.sightPoint().plus_(newCenter.minus_(originalCenter)));
		Jun3dPoint originalTop = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, originalRectangle.origin().y()));
		Jun3dPoint originalBottom = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, originalRectangle.corner().y()));
		Jun3dPoint originalLeft = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(originalRectangle.origin().x(), 0));
		Jun3dPoint originalRight = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(originalRectangle.corner().x(), 0));
		double originalHeight = originalTop.minus_(originalBottom).length();
		double originalWidth = originalLeft.minus_(originalRight).length();
		Jun3dPoint newTop = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, newRectangle.origin().y()));
		Jun3dPoint newBottom = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, newRectangle.corner().y()));
		Jun3dPoint newLeft = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(newRectangle.origin().x(), 0));
		Jun3dPoint newRight = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(newRectangle.corner().x(), 0));
		double newHeight = newTop.minus_(newBottom).length();
		double newWidth = newLeft.minus_(newRight).length();
		double newViewHeight = aProjection.regularHeight() * newHeight / originalHeight;
		double newViewWidth = newViewHeight * newWidth / newHeight;
		double newZoomHeight = newViewHeight;
		if (originalWidth / originalHeight * newViewHeight < newViewWidth) {
			double tmpViewHeight = originalWidth * newViewHeight / (originalHeight * newViewWidth);
			newZoomHeight = newViewHeight * tmpViewHeight;
		}
		if (((originalHeight * 0.01d) < newHeight) && ((newHeight * 0.01d) < originalHeight)) {
			aProjection.zoomHeight_(newZoomHeight);
		}
		this.displayProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Answer the JunOpenGL3dObject at the specified point on the view.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param event java.awt.event.MouseEvent
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category manipulating
	 */
	public JunOpenGL3dObject pick_with_(Jun2dPoint aPoint, MouseEvent event) {
		if (this.displayObject() == null) {
			return null;
		}

		if (this.displayObject().isPrimitive()) {
			return null;
		}

		JunOpenGL3dObject pickedObject = JunOpenGLObjectPicker.PickObjectAt_fromCompound_projection_(aPoint, (JunOpenGLPickingObjects) this.displayObject(), this.displayProjection());
		if (pickedObject == null) {
			if (this.selectedObjects().isEmpty() == false) {
				this.clearSelectedObjects();
				this.changed_($("selection"));
			}
			this.updateMenuIndication();
			return null;
		} else {
			if (event.isShiftDown()) {
				if (this.selectedObjects().contains(pickedObject)) {
					this.removeSelectedObject_(pickedObject);
					this.changed_($("selection"));
				} else {
					this.addSelectedObject_(pickedObject);
					this.changed_($("selection"));
				}
			} else {
				if (event.isControlDown()) {
					this.ctrlClick_(pickedObject);
				} else if (event.isAltDown()) {
					this.altClick_(pickedObject);
				} else {
					this.clearSelectedObjects();
					this.addSelectedObject_(pickedObject);
					this.changed_($("selection"));
				}
			}
			this.updateMenuIndication();
			return pickedObject;
		}
	}

	/**
	 * Set the movement vector.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void movementVector_(Jun2dPoint aPoint) {
		StController aController = this.getController();
		if (aController != null && aController instanceof JunOpenGLDisplayController) {
			JunOpenGLDisplayController anOpenGLDisplayController = (JunOpenGLDisplayController) aController;
			try {
				anOpenGLDisplayController.flushMovementVector();
				anOpenGLDisplayController.movementVector_(aPoint);
			} finally {
				anOpenGLDisplayController.createMovementProcess();
			}
		}
	}

	/**
	 * Select all objects.
	 * 
	 * @category selecting
	 */
	public void selectAll() {
		if (this.openGL3dObject() == null) {
			return;
		}

		if (this.openGL3dObject().isPrimitive()) {
			return;
		}

		this.clearSelectedObjects();

		if (this.openGL3dObject().isCompound()) {
			JunOpenGL3dObject[] objects = ((JunOpenGL3dCompoundObject) this.openGL3dObject()).components();
			for (int i = 0; i < objects.length; i++) {
				this.selectedObjects().add(objects[i]);
			}
		} else {
			this.selectedObjects().add(this.openGL3dObject());
		}

		this.changed_($("selection"));
		this.updateMenuIndication();
	}

	/**
	 * Answer true if the receiver uses the movement process, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean useMovementProcess() {
		Boolean aBoolean = (Boolean) this.preferenceTable().at_($("useMovementProcess"));
		if (aBoolean == null) {
			aBoolean = Boolean.TRUE;
			this.preferenceTable().at_put_($("useMovementProcess"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set whether the receiver uses the movement process or not.
	 * 
	 * @param aBoolean boolean
	 * @category preferences
	 */
	public void useMovementProcess_(boolean aBoolean) {
		this.preferenceTable().at_put_($("useMovementProcess"), Boolean.valueOf(aBoolean));
	}

	/**
	 * Answer true if the receiver uses the press activity, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean usePressActivity() {
		Boolean aBoolean = (Boolean) this.preferenceTable().at_($("usePressActivity"));
		if (aBoolean == null) {
			aBoolean = Boolean.FALSE;
			this.preferenceTable().at_put_($("usePressActivity"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set whether the receiver uses the press activity or not.
	 * 
	 * @param aBoolean boolean
	 * @category preferences
	 */
	public void usePressActivity_(boolean aBoolean) {
		this.preferenceTable().at_put_($("usePressActivity"), Boolean.valueOf(aBoolean));
	}

	/**
	 * Answer true if the receiver uses the yellow button menu, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean useYellowButtonMenu() {
		Boolean aBoolean = (Boolean) this.preferenceTable().at_($("useYellowButtonMenu"));
		if (aBoolean == null) {
			aBoolean = Boolean.FALSE;
			this.preferenceTable().at_put_($("useYellowButtonMenu"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set whether the receiver uses the yellow button menu (popup menu) or not.
	 * 
	 * @param aBoolean boolean
	 * @category preferences
	 */
	public void useYellowButtonMenu_(boolean aBoolean) {
		this.preferenceTable().at_put_($("useYellowButtonMenu"), Boolean.valueOf(aBoolean));
	}

	/**
	 * Answer true if the receiver has no control, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean noControl() {
		Boolean aBoolean = (Boolean) this.preferenceTable().at_($("noControl"));
		if (aBoolean == null) {
			aBoolean = Boolean.FALSE;
			this.preferenceTable().at_put_($("noControl"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set whether the receiver has no control or not.
	 * 
	 * @param aBoolean boolean
	 * @category preferences
	 */
	public void noControl_(boolean aBoolean) {
		this.preferenceTable().at_put_($("noControl"), Boolean.valueOf(aBoolean));
	}

	/**
	 * Action for the wheel movement.
	 * If the wheel moves up, unitsToScroll will be a negative number.
	 * 
	 * @param unitsToScroll int
	 * @param mouseCursorPoint java.awt.Point
	 * @param viewDisplayBox java.awt.Rectangle
	 * @category wheel actions
	 */
	public void wheel_at_in_(int unitsToScroll, Point mouseCursorPoint, Rectangle viewDisplayBox) {
		StSymbol aSymbol = null;
		Map nineBoxes = NineBoxesOf_(viewDisplayBox);
		Iterator iterator = nineBoxes.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			StRectangle box = (StRectangle) entry.getValue();
			if (box.containsPoint_(mouseCursorPoint)) {
				aSymbol = (StSymbol) entry.getKey();
				break;
			}
		}

		if (aSymbol == null) {
			return;
		} else if (aSymbol == $("topLeft")) {
			this.wheelActionInTopLeftBox_(unitsToScroll);
		} else if (aSymbol == $("topCenter")) {
			this.wheelActionInTopCenterBox_(unitsToScroll);
		} else if (aSymbol == $("topRight")) {
			this.wheelActionInTopRightBox_(unitsToScroll);
		} else if (aSymbol == $("leftCenter")) {
			this.wheelActionInLeftCenterBox_(unitsToScroll);
		} else if (aSymbol == $("center")) {
			this.wheelActionInCenterBox_(unitsToScroll);
		} else if (aSymbol == $("rightCenter")) {
			this.wheelActionInRightCenterBox_(unitsToScroll);
		} else if (aSymbol == $("bottomLeft")) {
			this.wheelActionInBottomLeftBox_(unitsToScroll);
		} else if (aSymbol == $("bottomCenter")) {
			this.wheelActionInBottomCenterBox_(unitsToScroll);
		} else if (aSymbol == $("bottomRight")) {
			this.wheelActionInBottomRightBox_(unitsToScroll);
		}
	}

	/**
	 * Action for the wheel movement in the top left box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInTopLeftBox_(int unitsToScroll) {
		double value = JunSensorUtility.AltDown() ? 0.01 : 0.1;
		Jun2dPoint factor = (unitsToScroll < 0) ? new Jun2dPoint(-value, value) : new Jun2dPoint(value, -value);
		this.grab_(factor);
	}

	/**
	 * Action for the wheel movement in the top center box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInTopCenterBox_(int unitsToScroll) {
		int angle = JunSensorUtility.AltDown() ? 1 : 5;
		if (unitsToScroll > 0) {
			angle *= -1;
		}
		Jun3dTransformation transformation = JunAngle.FromDeg_(angle).transformationToRotate_(this.sightPoint().to_(this.eyePoint()));
		Jun3dLine line = this.sightPoint().to_(this.sightPoint().plus_(this.upVector().normalVector())).transform_(transformation);
		Jun3dPoint vector = line.to().minus_(line.from());
		this.upVector_(vector);
	}

	/**
	 * Action for the wheel movement in the top right box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInTopRightBox_(int unitsToScroll) {
		double value = JunSensorUtility.AltDown() ? 0.01 : 0.1;
		Jun2dPoint factor = (unitsToScroll < 0) ? new Jun2dPoint(value, value) : new Jun2dPoint(-value, -value);
		this.grab_(factor);
	}

	/**
	 * Action for the wheel movement in the left center box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInLeftCenterBox_(int unitsToScroll) {
		double value = JunSensorUtility.AltDown() ? 0.01 : 0.1;
		Jun2dPoint factor = (unitsToScroll < 0) ? new Jun2dPoint(0, value) : new Jun2dPoint(0, -value);
		this.grab_(factor);
	}

	/**
	 * Action for the wheel movement in the center box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInCenterBox_(int unitsToScroll) {
		this.wheelActionInRightCenterBox_(unitsToScroll);
	}

	/**
	 * Action for the wheel movement in the right center box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInRightCenterBox_(int unitsToScroll) {
		double value = JunSensorUtility.AltDown() ? 0.01 : 0.1;
		double factor = (unitsToScroll < 0) ? 1 - value : 1 + value;
		this.zoom_(factor);
	}

	/**
	 * Action for the wheel movement in the bottom left box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInBottomLeftBox_(int unitsToScroll) {
		this.wheelActionInTopRightBox_(unitsToScroll);
	}

	/**
	 * Action for the wheel movement in the bottom center box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInBottomCenterBox_(int unitsToScroll) {
		double value = JunSensorUtility.AltDown() ? 0.01 : 0.1;
		Jun2dPoint factor = (unitsToScroll < 0) ? new Jun2dPoint(-value, 0) : new Jun2dPoint(value, 0);
		this.grab_(factor);
	}

	/**
	 * Action for the wheel movement in the bottom right box.
	 * 
	 * @param unitsToScroll int
	 * @category wheel actions
	 */
	protected void wheelActionInBottomRightBox_(int unitsToScroll) {
		this.wheelActionInTopLeftBox_(unitsToScroll);
	}

	/**
	 * Answer the button model for the dolly button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel dollyButton() {
		if (dollyButton == null) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.DollyCursorImage());
			button.action_(this.dollyButtonAction());
			dollyButton = button;
		}

		return dollyButton;
	}

	/**
	 * Answer the button action for the dolly button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure dollyButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				model.value_(!model.value());
				return null;
			}
		};
	}

	/**
	 * Answer the button model for the drag button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel dragButton() {
		if (this.pushButtons().containsKey($("drag")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.QuartersCursorImage());
			button.action_(this.dragButtonAction());
			this.pushButtons().put($("drag"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("drag"));
	}

	/**
	 * Answer the button action for the drag button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure dragButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunOpenGLDisplayModel.this.setButtonState($("drag"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer the button model for the focus button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel focusButton() {
		if (this.pushButtons().containsKey($("focus")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.GlassCursorImage());
			button.action_(this.focusButtonAction());
			this.pushButtons().put($("focus"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("focus"));
	}

	/**
	 * Answer the button action for the focus button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure focusButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunOpenGLDisplayModel.this.setButtonState($("focus"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer the button model for the grab button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel grabButton() {
		if (this.pushButtons().containsKey($("grab")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.HandCursorImage());
			button.action_(this.grabButtonAction());
			this.pushButtons().put($("grab"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("grab"));
	}

	/**
	 * Answer the button action for the grab button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure grabButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunOpenGLDisplayModel.this.setButtonState($("grab"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer the button model for the pick button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel pickButton() {
		if (this.pushButtons().containsKey($("pick")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(PickNormalImage());
			button.action_(this.pickButtonAction());
			this.pushButtons().put($("pick"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("pick"));
	}

	/**
	 * Answer the button action for the pick button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure pickButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunOpenGLDisplayModel.this.setButtonState($("pick"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer my current button status.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category buttons
	 */
	public synchronized StSymbol buttonState() {
		for (Iterator i = this.pushButtons().entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			JunButtonModel buttonModel = (JunButtonModel) entry.getValue();
			if (buttonModel.value()) {
				return (StSymbol) entry.getKey();
			}
		}

		return $("none");
	}

	/**
	 * Set my new button status.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aBoolean boolean
	 * @category buttons
	 */
	protected synchronized void setButtonState(StSymbol aSymbol, boolean aBoolean) {
		if (aBoolean == false) {
			((JunButtonModel) this.pushButtons().get(aSymbol)).value_(aBoolean);
		} else {
			for (Iterator i = this.pushButtons().entrySet().iterator(); i.hasNext();) {
				Map.Entry entry = (Map.Entry) i.next();
				((JunButtonModel) entry.getValue()).value_(entry.getKey() == aSymbol);
			}
		}
		this.changed_($("state"));
	}

	/**
	 * Answer the map for the push buttons.
	 * 
	 * @return java.util.Map
	 * @category buttons
	 */
	protected Map pushButtons() {
		if (pushButtons == null) {
			pushButtons = new HashMap();
		}
		return pushButtons;
	}

	/**
	 * Answer a xThumbWheel.
	 * 
	 * @return jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category wheels
	 */
	public JunThumbWheel xThumbWheel() {
		JunThumbWheel thumbWheel = new JunThumbWheel(true);
		thumbWheel.compute_(new StBlockClosure() {
			public Object value_(Object value) {
				Jun2dPoint factor = new Jun2dPoint(0, -((Number) value).doubleValue() / 50);
				grab_(factor);
				return null;
			}
		});

		return thumbWheel;
	}

	/**
	 * Answer a yThumbWheel.
	 * 
	 * @return jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category wheels
	 */
	public JunThumbWheel yThumbWheel() {
		JunThumbWheel thumbWheel = new JunThumbWheel(false);
		thumbWheel.compute_(new StBlockClosure() {
			public Object value_(Object value) {
				Jun2dPoint factor = new Jun2dPoint(((Number) value).doubleValue() / 50, 0);
				grab_(factor);
				return null;
			}
		});

		return thumbWheel;
	}

	/**
	 * Answer a zThumbWheel.
	 * 
	 * @return jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category wheels
	 */
	public JunThumbWheel zThumbWheel() {
		JunThumbWheel thumbWheel = new JunThumbWheel(true);
		thumbWheel.compute_(new StBlockClosure() {
			public Object value_(Object value) {
				if (dollyButton().value() == true) {
					double factor = ((Number) value).doubleValue() / 400;
					dolly_(factor);
				} else {
					double factor = 1 + (((Number) value).doubleValue() / 100);
					zoom_(factor);
				}
				return null;
			}
		});

		return thumbWheel;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLDisplayViewAwt(this);
		} else {
			return new JunOpenGLDisplayViewSwing(this);
		}
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultViewWithoutWidgets() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunOpenGLDisplayViewAwt.WithoutWidgets(this);
		} else {
			return JunOpenGLDisplayViewSwing.WithoutWidgets(this);
		}
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		JunOpenGLDisplayLight[] lights = this.displayLights();
		for (int i = 0; i < lights.length; i++) {
			lights[i].closeRequest();
		}

		if (this.showModel() != null) {
			this.showModel().closeRequest();
		}

		super.noticeOfWindowClose(e);
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));

		// New
		fileMenu.add(new StMenuItem($String("New"), new MenuPerformer(this, "newModel")));

		// Open...
		fileMenu.add(new StMenuItem($String("Open") + "...", new MenuPerformer(this, "openLST")));

		// Open as...
		StMenu openAsMenu = new StMenu($String("Open as..."));
		openAsMenu.add(new StMenuItem("VRML1.0...", new MenuPerformer(this, "openWRL10")));
		openAsMenu.add(new StMenuItem("VRML97...", new MenuPerformer(this, "openWRL97")));
		fileMenu.add(openAsMenu);

		//
		fileMenu.addSeparator();

		// Save...
		fileMenu.add(new StMenuItem($String("Save") + "...", $("saveMenu"), new MenuPerformer(this, "saveLST")));

		// Save as...
		StMenu saveAsMenu = new StMenu($String("Save as..."), $("saveAsMenu"));
		saveAsMenu.add(new StMenuItem("VRML1.0...", new MenuPerformer(this, "saveWRL10")));
		saveAsMenu.add(new StMenuItem("VRML97...", new MenuPerformer(this, "saveWRL97")));
		fileMenu.add(saveAsMenu);

		// Save as image...
		fileMenu.add(new StMenuItem($String("Save as image..."), $("saveAsImageMenu"), new MenuPerformer(this, "saveAsImage")));

		//
		fileMenu.addSeparator();

		// Quit
		fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "quitDoing")));

		return fileMenu;
	}

	/**
	 * Create an "Edit" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createEditMenu() {
		StMenu editMenu = new StMenu($String("Edit"), $("editMenu"));
		editMenu.add(new StMenuItem($String("Copy"), $("copyMenu"), new MenuPerformer(this, "copyObject")));
		editMenu.add(new StMenuItem($String("Cut"), $("cutMenu"), new MenuPerformer(this, "cutObject")));
		editMenu.add(new StMenuItem($String("Paste"), $("pasteMenu"), new MenuPerformer(this, "pasteObject")));
		editMenu.add(new StMenuItem($String("Clear"), $("clearMenu"), new MenuPerformer(this, "clearObject")));
		editMenu.addSeparator();
		editMenu.add(new StMenuItem($String("Select all"), $("selectAllMenu"), new MenuPerformer(this, "selectAll")));
		return editMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu($String("Misc"), $("miscMenu"));

		// Texture
		StMenu textureMenu = new StMenu($String("Texture"), $("textureMenu"));
		textureMenu.add(new StMenuItem($String("From user..."), $("textureFromUserMenu"), new MenuPerformer(this, "textureFromUser")));
		textureMenu.add(new StMenuItem($String("From file..."), $("textureFromFileMenu"), new MenuPerformer(this, "textureFromFile")));
		textureMenu.add(new StMenuItem($String("Take away"), new MenuPerformer(this, "textureTakeAway")));
		textureMenu.addSeparator();
		textureMenu.add(new StCheckBoxMenuItem($String("Linear approx."), $("textureLinearMenu"), new MenuPerformer(this, "textureLinear")));
		textureMenu.add(new StCheckBoxMenuItem($String("Clamp"), $("textureClampMenu"), new MenuPerformer(this, "textureClamp")));
		textureMenu.add(new StCheckBoxMenuItem($String("Modulate"), $("textureModulateMenu"), new MenuPerformer(this, "textureModulate")));
		textureMenu.add(new StCheckBoxMenuItem($String("Mipmap"), $("textureMipmapMenu"), new MenuPerformer(this, "textureMipmap")));
		miscMenu.add(textureMenu);

		// Spawn
		miscMenu.add(new StMenuItem($String("Spawn"), $("spawnMenu"), new MenuPerformer(this, "spawnObject")));

		// Viewport
		miscMenu.add(new StMenuItem($String("Viewport"), $("viewportMenu"), new MenuPerformer(this, "spawnViewport")));

		// Bounds
		miscMenu.add(new StMenuItem($String("Bounds"), $("boundsMenu"), new MenuPerformer(this, "showBounds")));

		return miscMenu;
	}

	/**
	 * Menu message: Clear the display object.
	 * 
	 * @category menu messages
	 */
	public void clearObject() {
		if (this.selectedObjects().isEmpty()) {
			return;
		}

		if (this.displayObject().isCompound() == false) {
			return;
		}

		JunCursorAnimator cursorAnimator = JunCursorAnimator.ClockCursors();
		try {
			cursorAnimator._show();

			Object[] objects = this.selectedObjects().toArray();
			for (int i = 0; i < objects.length; i++) {
				((JunOpenGL3dCompoundObject) this.displayObject()).remove_((JunOpenGL3dObject) objects[i]);
			}
			this.clearSelectedObjects();
			this.changed_($("object"));
		} finally {
			cursorAnimator._restore();
		}

		this.updateMenuIndication();
	}

	/**
	 * Menu message: Copy the object.
	 * 
	 * @category menu messages
	 */
	public void copyObject() {
		if (JunSensorUtility.ShiftDown()) {
			this.projectionTableToCopyBuffer();
			return;
		}

		if (this.openGL3dObject() == null) {
			return;
		}

		JunCursorAnimator cursorAnimator = JunCursorAnimator.ClockCursors();
		try {
			cursorAnimator._show();

			JunOpenGL3dObject object = this.spawningObject();
			String string = this.defaultStampForLST10() + object.toLispList().saveString();
			Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(string), this);
		} finally {
			cursorAnimator._restore();
		}
	}

	/**
	 * Menu message: Cut the selected objects.
	 * 
	 * @category menu messages
	 */
	public void cutObject() {
		if (this.selectedObjects().isEmpty()) {
			return;
		}

		if (this.displayObject().isCompound() == false) {
			return;
		}

		JunCursorAnimator cursorAnimator = JunCursorAnimator.ClockCursors();
		try {
			cursorAnimator._show();

			this.copyObject();
			Object[] objects = this.selectedObjects().toArray();
			for (int i = 0; i < objects.length; i++) {
				((JunOpenGL3dCompoundObject) this.displayObject()).remove_((JunOpenGL3dObject) objects[i]);
			}
			this.clearSelectedObjects();
			this.changed_($("object"));
		} finally {
			cursorAnimator._restore();
		}

		this.updateMenuIndication();
	}

	/**
	 * Menu message: Open an LST file.
	 * 
	 * @category menu messages
	 */
	public void openLST() {
		this.openLST10();
	}

	/**
	 * Menu message: Open an LST1.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void openLST10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "LST"), new String[] { "*.lst", "*.LST" }) };
		File file = JunFileRequesterDialog.Request($String("Select an <1p> file.", null, "LST"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		JunOpenGL3dObject object = null;
		try {
			object = this.readFromLST10_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
		if (object == null) {
			return;
		}

		this.displayObject_(object);

		this.resetView();
		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Menu message: Open a VRML1.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void openWRL10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, "VRML"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.readFromWRL10_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}

		this.resetView();
		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Menu message: Open a VRML2.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void openWRL20() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, "VRML"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.readFromWRL20_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}

		this.resetView();
		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Menu message: Open a VRML97 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void openWRL97() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, "VRML"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.readFromWRL97_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Menu message: Paste the object from the clipboard.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void pasteObject() {
		if (JunSensorUtility.ShiftDown()) {
			this.projectionTableFromCopyBuffer();
			return;
		}

		Transferable content = Toolkit.getDefaultToolkit().getSystemClipboard().getContents(this);
		String string = null;
		try {
			string = (String) content.getTransferData(DataFlavor.stringFlavor);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
		if (string == null) {
			return;
		}

		if (string.startsWith(this.defaultStampForLST10().substring(0, 9)) == false) {
			JunDialog.Warn_($String("The contents of copy buffer is not <1p>.", null, "LST"));
			return;
		}

		JunCursorAnimator cursorAnimator = JunCursorAnimator.ClockCursors();
		try {
			cursorAnimator._show();

			JunOpenGL3dObject object = JunOpenGL3dObject.LoadFrom_(string);
			if (this.displayObject() == null) {
				this.displayObject_(object);
				this.clearSelectedObjects();
				this.addSelectedObject_(object);
				this.resetView();
			} else {
				if (this.displayObject().isCompound()) {
					((JunOpenGL3dCompoundObject) this.displayObject()).add_(object);
					this.clearSelectedObjects();
					this.addSelectedObject_(object);
					this.changed_($("object"));
				} else {
					object = new JunOpenGL3dCompoundObject(this.displayObject(), object);
					this.displayObject_(object);
					this.clearSelectedObjects();
					this.addSelectedObject_(object);
					this.changed_($("object"));
				}
			}
		} finally {
			cursorAnimator._restore();
		}

		this.updateMenuIndication();
	}

	/**
	 * Menu message: Close all windows to quit.
	 * 
	 * @category menu messages
	 */
	public void quitDoing() {
		this.closeRequest();
	}

	/**
	 * Menu message: Save the display object to an LST file.
	 * 
	 * @category menu messages
	 */
	public void saveLST() {
		this.saveLST10();
	}

	/**
	 * Menu message: Save the display object to an LST1.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void saveLST10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "LST"), new String[] { "*.lst", "*.LST" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input an <1p> file.", null, "LST"), new File(this.displayObject().name() + ".lst"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToLST10_object_(file, this.displayObject());
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Mene message: Save the display object to the VRML1.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void saveWRL10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "VRML"), new File(this.displayObject().name() + ".wrl"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToWRL10_object_(file, this.displayObject());
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Save the display object to the VRML2.0 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void saveWRL20() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "VRML"), new File(this.displayObject().name() + ".wrl"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToWRL20_object_(file, this.displayObject());
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Save the display object to the VRML97 file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void saveWRL97() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "VRML"), new File(this.displayObject().name() + ".wrl"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToWRL97_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Menu message: Show the bounding box.
	 * 
	 * @category menu messages
	 */
	public void showBounds() {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGL3dObject spawningObject = this.spawningObject();
		JunOpenGL3dObject boundingObject = this.boundingObjectFor_(spawningObject);
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(boundingObject);
		compoundObject.add_(spawningObject);

		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel();
		displayModel.displayObject_(compoundObject);
		displayModel.displayProjection_((JunOpenGLProjection) this.displayProjection().copy());
		displayModel.open();
		displayModel.changed_($("object"));
	}

	/**
	 * Menu message: Spawn the display object.
	 * 
	 * @category menu messages
	 */
	public void spawnObject() {
		JunOpenGLDisplayModel displayModel = null;
		try {
			displayModel = (JunOpenGLDisplayModel) this.classToSpawn().newInstance();
		} catch (InstantiationException e) {
		} catch (IllegalAccessException e) {
		}
		displayModel.displayObject_(this.spawningObject());
		displayModel.defaultProjectionTable_(this.projectionTable());

		StView view = this.getView();
		if (view == null) {
			displayModel.open();
		} else {
			StRectangle box = new StRectangle(view.topComponent().getBounds());
			StRectangle area = new StRectangle(0, 0, box.width(), box.height());
			area = area.align_with_(area.topLeft(), new Point(box.right() + 5, box.top()));
			displayModel.openIn_(area.toRectangle());
		}

		displayModel.changed_($("object"));
	}

	/**
	 * Menu message: Spawn the viewport.
	 * 
	 * @category menu messages
	 */
	public void spawnViewport() {
		JunOpenGLShowModel model = this.showModel();
		if (model == null) {
			model = new JunOpenGLShowModel(this);
		}

		if (model.builder().windows().length > 0) {
			model._windowExpandAndRaise();
		} else {
			StView view = this.getView();
			if (view == null) {
				model.open();
			} else {
				StRectangle box = new StRectangle(view.topComponent().getBounds());
				StRectangle area = new StRectangle(0, 0, box.width(), box.height());
				area = area.align_with_(area.topLeft(), new Point(box.right() + 5, box.top()));
				model.openIn_(area.toRectangle());
			}
		}
	}

	/**
	 * Menu message: Toggle a texture clamp effect on a projector.
	 * 
	 * @category menu messages
	 */
	public void textureClamp() {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}

		final Boolean aBoolean = Boolean.valueOf(!this.clamp());
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					texture.clamp_(aBoolean.booleanValue());
					each.texture_(texture);
				}
				return null;
			}
		});

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Get an image from a file and set it as a texture.
	 * 
	 * @category menu messages
	 */
	public void textureFromFile() {
		if (this.displayObject() == null) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, $String("Image")), JunSystem.DefaultImageExtensionPatterns()), JunFileModel.FileType.All($String("All files")) };
		File file = JunFileRequesterDialog.Request($String("Select an <1p> file.", null, $String("Image")), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		StImage anImage = JunImageProcessor.ImageFromFile_(file.getPath());
		if (anImage == null) {
			// JunDialog.Warn_(file + $String(" is not an image file."));
			return;
		}

		JunOpenGLTexture aTexture = new JunOpenGLTexture(anImage);
		if (this.hasTextureMapping()) {
			aTexture.linear_(this.linear());
			aTexture.clamp_(this.clamp());
			aTexture.modulate_(this.modulate());
			aTexture.mipmap_(this.mipmap());
		}

		if (this.selectedObjects().isEmpty()) {
			this.displayObject().texture_(aTexture);
		} else {
			Object[] objects = this.selectedObjects().toArray();
			for (int i = 0; i < objects.length; i++) {
				((JunOpenGL3dObject) objects[i]).texture_(aTexture);
			}
		}

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Menu message: Get an image from a user and set it as a texture.
	 * 
	 * @category menu messages
	 */
	public void textureFromUser() {
		if (this.displayObject() == null) {
			return;
		}

		StImage anImage = StImage._FromUser();
		if (anImage == null) {
			return;
		}

		JunOpenGLTexture aTexture = new JunOpenGLTexture(anImage);
		if (this.hasTextureMapping()) {
			aTexture.linear_(this.linear());
			aTexture.clamp_(this.clamp());
			aTexture.modulate_(this.modulate());
			aTexture.mipmap_(this.mipmap());
		}

		if (this.selectedObjects().isEmpty()) {
			this.displayObject().texture_(aTexture);
		} else {
			Object[] objects = this.selectedObjects().toArray();
			for (int i = 0; i < objects.length; i++) {
				((JunOpenGL3dObject) objects[i]).texture_(aTexture);
			}
		}

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Menu message: Toggle a texture linear effect on a projector.
	 * 
	 * @category menu messages
	 */
	public void textureLinear() {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}

		final Boolean aBoolean = Boolean.valueOf(!this.linear());
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					texture.linear_(aBoolean.booleanValue());
					each.texture_(texture);
				}
				return null;
			}
		});

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Menu message: Toggle a texture mipmap effect on a projector.
	 * 
	 * @category menu messages
	 */
	public void textureMipmap() {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}

		final Boolean aBoolean = Boolean.valueOf(!this.mipmap());
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					texture.mipmap_(aBoolean.booleanValue());
					each.texture_(texture);
				}
				return null;
			}
		});

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Menu message: Toggle a texture modulate effect on a projector.
	 * 
	 * @category menu messages
	 */
	public void textureModulate() {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}

		final Boolean aBoolean = Boolean.valueOf(!this.modulate());
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					texture.modulate_(aBoolean.booleanValue());
					each.texture_(texture);
				}
				return null;
			}
		});

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Menu message: Take a texture away if exists.
	 * 
	 * @category menu messages
	 */
	public void textureTakeAway() {
		if (this.displayObject() == null) {
			return;
		}

		if (this.selectedObjects().isEmpty()) {
			this.displayObject().texture_(null);
		} else {
			Object[] objects = this.selectedObjects().toArray();
			for (int i = 0; i < objects.length; i++) {
				((JunOpenGL3dObject) objects[i]).texture_(null);
			}
		}

		this.updateMiscMenuIndication();
		this.changed_($("texture"));
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();

		this.updateFileMenuIndication();
		this.updateEditMenuIndication();
		this.updateMiscMenuIndication();

		if (this.showModel() != null) {
			this.showModel().updateMenuIndication();
		}
	}

	/**
	 * Update the file menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateFileMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu fileMenu = (StMenu) this._menuBar().atNameKey_($("fileMenu"));
		if (fileMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Save...
		menuItem = fileMenu.atNameKey_($("saveMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Save as...
		menuItem = fileMenu.atNameKey_($("saveAsMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Save as image
		menuItem = fileMenu.atNameKey_($("saveAsImageMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}
	}

	/**
	 * Update the edit menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateEditMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu editMenu = (StMenu) this._menuBar().atNameKey_($("editMenu"));
		if (editMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();
		boolean selectedObjectsIsNotEmpty = !this.selectedObjects().isEmpty();

		menuItem = editMenu.atNameKey_($("copyMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		menuItem = editMenu.atNameKey_($("cutMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty && selectedObjectsIsNotEmpty);
		}

		menuItem = editMenu.atNameKey_($("clearMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty && selectedObjectsIsNotEmpty);
		}

		menuItem = editMenu.atNameKey_($("selectAllMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

	}

	/**
	 * Update the misc menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateMiscMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu miscMenu = (StMenu) this._menuBar().atNameKey_($("miscMenu"));
		if (miscMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Texture
		StMenu textureMenu = (StMenu) miscMenu.atNameKey_($("textureMenu"));
		if (textureMenu != null) {
			textureMenu.beEnabled(displayObjectIsNotEmpty);
			if (displayObjectIsNotEmpty) {
				StCheckBoxMenuItem checkBoxMenuItem;

				// Mipmap
				checkBoxMenuItem = (StCheckBoxMenuItem) textureMenu.atNameKey_($("textureMipmapMenu"));
				if (checkBoxMenuItem != null) {
					checkBoxMenuItem.beSelected(this.mipmap());
				}

				// Modulate
				checkBoxMenuItem = (StCheckBoxMenuItem) textureMenu.atNameKey_($("textureModulateMenu"));
				if (checkBoxMenuItem != null) {
					checkBoxMenuItem.beSelected(this.modulate());
				}

				// Clamp
				checkBoxMenuItem = (StCheckBoxMenuItem) textureMenu.atNameKey_($("textureClampMenu"));
				if (checkBoxMenuItem != null) {
					checkBoxMenuItem.beSelected(this.clamp());
				}

				// Linear
				checkBoxMenuItem = (StCheckBoxMenuItem) textureMenu.atNameKey_($("textureLinearMenu"));
				if (checkBoxMenuItem != null) {
					checkBoxMenuItem.beSelected(this.linear());
				}
			}
		}

		// Spawn
		menuItem = miscMenu.atNameKey_($("spawnMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Viewport
		menuItem = miscMenu.atNameKey_($("viewportMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Bounds
		menuItem = miscMenu.atNameKey_($("boundsMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}
	}

	/**
	 * Flush the movement vector.
	 * 
	 * @category flushing
	 */
	public void flushMovementVector() {
		StView aView = this.getView();
		if (aView != null) {
			((JunOpenGLDisplayController) ((JunOpenGLDisplayView) aView).controller()).flushMovementVector();
		}
	}

	/**
	 * Load an LST1.0 file and create a JunOpenGL3dObject.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category reading
	 */
	public JunOpenGL3dObject loadFromLST10_(StReadStream aStream) {
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			return JunOpenGL3dObject.LoadFrom_(aStream);
		} catch (SmalltalkException e) {
			JunDialog.Warn_(e.getMessage());
			return null;
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML1.0 file and create a JunOpenGL3dObject.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading
	 */
	public void loadFromWRL10_(StReadStream aStream) {
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			JunVrmlCompiler.Source_parser_generator_displayModel_(aStream, new JunVrmlParser10(), new JunVrmlGenerator10(), this);
		} catch (SmalltalkException e) {
			JunDialog.Warn_(e.getMessage());
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML2.0 file and create a JunOpenGL3dObject.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading
	 */
	public void loadFromWRL20_(StReadStream aStream) {
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			JunVrmlCompiler.Source_parser_generator_displayModel_(aStream, new JunVrmlParser97(), new JunVrmlGenerator97(), this);
		} catch (SmalltalkException e) {
			JunDialog.Warn_(e.getMessage());
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML97 file and create a JunOpenGL3dObject.
	 * 
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 * @category reading
	 */
	public void loadFromWRL97_(StReadStream aStream) {
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			JunVrmlCompiler.Source_parser_generator_displayModel_(aStream, new JunVrmlParser97(), new JunVrmlGenerator97(), this);
		} catch (SmalltalkException e) {
			JunDialog.Warn_(e.getMessage());
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load an LST1.0 file and create a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aFile java.io.File
	 * @exception java.io.IOException
	 * @category reading
	 */
	public JunOpenGL3dObject readFromLST10_(File aFile) throws IOException {
		if (aFile == null) {
			return null;
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			final long size = aFile.length();
			final StReadStream stream = new StReadStream(aFile);
			try {
				if (size <= 100 * 1000) {
					return this.loadFromLST10_(stream);
				} else {
					// Show the progress.
					final StValueHolder result = new StValueHolder();
					final JunProgress progress = new JunProgress();
					progress.message_($String("reading <1p>...", null, "LST"));
					progress.do_(new StBlockClosure() {
						public Object value() {
							progress.value_(0);

							Thread thread = new Thread() {
								public void run() {
									while (progress.value() < 1) {
										float value = Math.round((float) stream.position() / size / 0.005) * 0.005f;
										if (progress.value() != value) {
											progress.value_(value);
										}
										try {
											Thread.sleep(1000);
										} catch (InterruptedException e) {
										}
									}
								}
							};
							thread.setPriority(Thread.NORM_PRIORITY + 1);
							thread.start();

							try {
								result.value_(JunOpenGLDisplayModel.this.loadFromLST10_(stream));
							} finally {
								progress.value_(1);
							}

							return null;
						}
					});
					return (JunOpenGL3dObject) result.value();
				}
			} finally {
				stream.close();
			}
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML1.0 file and create a JunOpenGL3dObject.
	 * 
	 * @param aFile java.io.File
	 * @exception java.io.IOException
	 * @category reading
	 */
	public void readFromWRL10_(File aFile) throws IOException {
		if (aFile == null) {
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			final long size = aFile.length();
			final StReadStream stream = new StReadStream(aFile);
			try {
				if (size <= 100 * 1000) {
					this.loadFromWRL10_(stream);
				} else {
					// Show the progress.
					final JunProgress progress = new JunProgress();
					progress.message_($String("reading <1p>...", null, "WRL"));
					progress.do_(new StBlockClosure() {
						public Object value() {
							progress.value_(0);

							Thread thread = new Thread() {
								public void run() {
									while (progress.value() < 1) {
										float value = Math.round((float) stream.position() / size / 0.005) * 0.005f;
										if (progress.value() != value) {
											progress.value_(value);
										}
										try {
											Thread.sleep(1000);
										} catch (InterruptedException e) {
										}
									}
								}
							};
							thread.setPriority(Thread.NORM_PRIORITY + 1);
							thread.start();

							try {
								JunOpenGLDisplayModel.this.loadFromWRL10_(stream);
							} finally {
								progress.value_(1);
							}

							return null;
						}
					});
				}
			} finally {
				stream.close();
			}
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML2.0 file and create a JunOpenGL3dObject.
	 * 
	 * @param aFile java.io.File
	 * @exception java.io.IOException
	 * @category reading
	 */
	public void readFromWRL20_(File aFile) throws IOException {
		if (aFile == null) {
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			final long size = aFile.length();
			final StReadStream stream = new StReadStream(aFile);
			try {
				if (size <= 100 * 1000) {
					this.loadFromWRL20_(stream);
				} else {
					// Show the progress.
					final JunProgress progress = new JunProgress();
					progress.message_($String("reading <1p>...", null, "WRL"));
					progress.do_(new StBlockClosure() {
						public Object value() {
							progress.value_(0);

							Thread thread = new Thread() {
								public void run() {
									while (progress.value() < 1) {
										float value = Math.round((float) stream.position() / size / 0.005) * 0.005f;
										if (progress.value() != value) {
											progress.value_(value);
										}
										try {
											Thread.sleep(1000);
										} catch (InterruptedException e) {
										}
									}
								}
							};
							thread.setPriority(Thread.NORM_PRIORITY + 1);
							thread.start();

							try {
								JunOpenGLDisplayModel.this.loadFromWRL20_(stream);
							} finally {
								progress.value_(1);
							}

							return null;
						}
					});
				}
			} finally {
				stream.close();
			}
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Load a VRML97 file and create a JunOpenGL3dObject.
	 * 
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category reading
	 */
	public void readFromWRL97_(File aFile) throws IOException {
		if (aFile == null) {
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			final long size = aFile.length();
			final StReadStream stream = new StReadStream(aFile);
			try {
				if (size <= 100 * 1000) {
					this.loadFromWRL97_(stream);
				} else {
					// Show the progress.
					final JunProgress progress = new JunProgress();
					progress.message_($String("reading <1p>...", null, "WRL"));
					progress.do_(new StBlockClosure() {
						public Object value() {
							progress.value_(0);

							Thread thread = new Thread() {
								public void run() {
									while (progress.value() < 1) {
										float value = Math.round((float) stream.position() / size / 0.005) * 0.005f;
										if (progress.value() != value) {
											progress.value_(value);
										}
										try {
											Thread.sleep(1000);
										} catch (InterruptedException e) {
										}
									}
								}
							};
							thread.setPriority(Thread.NORM_PRIORITY + 1);
							thread.start();

							try {
								JunOpenGLDisplayModel.this.loadFromWRL97_(stream);
							} finally {
								progress.value_(1);
							}

							return null;
						}
					});
				}
			} finally {
				stream.close();
			}
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the writer as LST1.0.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param anObject jp.co.sra.jun.opengl.flux.JunOpenGL3dObject
	 * @exception IOException
	 * @category writing
	 */
	public void saveToLST10_object_(BufferedWriter aWriter, JunOpenGL3dObject anObject) throws IOException {
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();

			aWriter.write(this.defaultStampForLST10());
			anObject.saveOn_(aWriter);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the writer as VRML1.0.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param anObject jp.co.sra.jun.opengl.flux.JunOpenGL3dObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void saveToWRL10_object_(BufferedWriter aWriter, JunOpenGL3dObject anObject) throws IOException {
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();

			aWriter.write(this.defaultStampForWRL10());
			anObject.vrml10On_(aWriter);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the writer as VRML2.0.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param anObject jp.co.sra.jun.opengl.flux.JunOpenGL3dObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void saveToWRL20_object_(BufferedWriter aWriter, JunOpenGL3dObject anObject) throws IOException {
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();

			double size = anObject.boundingBox().width();
			size = Math.max(size, anObject.boundingBox().height());
			size = Math.max(size, anObject.boundingBox().depth());
			aWriter.write(this.defaultStampForWRL20());
			aWriter.write("Viewpoint {");
			aWriter.newLine();
			aWriter.write("\t");
			aWriter.write("position 0.0 0.0 ");
			aWriter.write(String.valueOf((float) size * 2));
			aWriter.newLine();
			aWriter.write("}");
			aWriter.newLine();
			aWriter.newLine();
			aWriter.write("NavigationInfo {");
			aWriter.newLine();
			aWriter.write("\t");
			aWriter.write("speed ");
			aWriter.write(String.valueOf((float) size / 10));
			aWriter.newLine();
			aWriter.write("}");
			aWriter.newLine();
			aWriter.newLine();
			anObject.vrml20On_(aWriter);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Write the display object to the writer as VRML97.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void saveToWRL97_(BufferedWriter aWriter) throws IOException {
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();
			this.vrml97On_(aWriter);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the LST1.0 file.
	 * 
	 * @param aFile java.io.File
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToLST10_object_(File aFile, JunOpenGL3dObject anObject) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));
		try {
			this.saveToLST10_object_(aWriter, anObject);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the VRML1.0 file.
	 * 
	 * @param aFile java.io.File
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToWRL10_object_(File aFile, JunOpenGL3dObject anObject) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));
		try {
			this.saveToWRL10_object_(aWriter, anObject);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the VRML2.0 file.
	 * 
	 * @param aFile java.io.File
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToWRL20_object_(File aFile, JunOpenGL3dObject anObject) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));
		try {
			this.saveToWRL20_object_(aWriter, anObject);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Write the display object as VRML97 to the file.
	 * 
	 * @param aFile java.io.File
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToWRL97_(File aFile) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));
		try {
			this.saveToWRL97_(aWriter);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Write my VRML97 representation on the writer.
	 *
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97On_(BufferedWriter aWriter) throws IOException {
		aWriter.write(this.defaultStampForWRL97());
		aWriter.newLine();
		this.vrml97WorldInfoOn_(aWriter);
		aWriter.newLine();
		this.vrml97NavigationInfoOn_(aWriter);
		aWriter.newLine();
		this.vrml97BackgroundOn_(aWriter);
		aWriter.newLine();
		this.vrml97ViewpointOn_(aWriter);
		aWriter.newLine();
		this.displayObject().vrml20On_(aWriter);
	}

	/**
	 * Write the VRML97 WorldInfo node to the writer.
	 *
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97WorldInfoOn_(BufferedWriter aWriter) throws IOException {
		aWriter.write("WorldInfo {");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("info [");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write('\t');
		aWriter.write("\"This file was created by " + JunSystem.System() + JunSystem.Version() + " for Java\"");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write('\t');
		aWriter.write("\"" + DateFormat.getInstance().format(new Date()) + "\"");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write(']');
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("title \"");
		if (this.displayObject() != null) {
			aWriter.write(this.displayObject().name() + ": ");
		}
		aWriter.write("Sample of " + this._className() + "\"");
		aWriter.newLine();
		aWriter.write('}');
		aWriter.newLine();
	}

	/**
	 * Write the VRML97 NavigationInfo node to the writer.
	 *
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97NavigationInfoOn_(BufferedWriter aWriter) throws IOException {
		aWriter.write("NavigationInfo {");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("type [\"EXAMINE\", \"ANY\"]");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("avatarSize [0.25, 1.6, 0.75");
		aWriter.write(", " + (float) this.sightPoint().x());
		aWriter.write(", " + (float) this.sightPoint().y());
		aWriter.write(", " + (float) this.sightPoint().z());
		aWriter.write(']');
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("visibilityLimit " + (float) this.displayProjector().far());
		aWriter.newLine();
		aWriter.write('}');
		aWriter.newLine();
	}

	/**
	 * Write the VRML97 NavigationInfo node to the writer.
	 *
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97BackgroundOn_(BufferedWriter aWriter) throws IOException {
		Color aColor = this.backgroundColor();
		if (aColor == null) {
			return;
		}

		aWriter.write("Background {");
		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("skyColor [");
		aWriter.write(" " + (float) aColor.getRed() / 255);
		aWriter.write(" " + (float) aColor.getGreen() / 255);
		aWriter.write(" " + (float) aColor.getBlue() / 255);
		aWriter.write(" ]");
		aWriter.newLine();
		aWriter.write('}');
		aWriter.newLine();
	}

	/**
	 * Write the VRML97 Viewpoint node to the writer.
	 *
	 * @param aWriter java.io.BufferedWriter
	 * @exception java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97ViewpointOn_(BufferedWriter aWriter) throws IOException {
		if (_presetProjections == null) {
			this.vrml97ViewpointOn_(aWriter, this.openGLProjection());
		} else {
			for (int i = 0; i < _presetProjections.length; i++) {
				this.vrml97ViewpointOn_(aWriter, _presetProjections[i]);
			}
		}
	}

	/**
	 * Write my current projection as a VRML97 Viewpoint node to the writer.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @throws IOException java.io.IOException
	 * @category vrml support
	 */
	protected void vrml97ViewpointOn_(BufferedWriter aWriter, JunOpenGLProjection aProjection) throws IOException {
		if (aProjection == null) {
			return;
		}

		Jun3dTransformation transformation = this.transformationForVrml97(aProjection);
		Jun3dPoint rotationVector = transformation.rotationVector();
		JunAngle rotationAngle = transformation.rotationAngle();
		aWriter.write("Viewpoint {");

		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("fieldOfView " + (float) Math.atan(aProjection.zoomHeight() / aProjection.eyePoint().distance_(aProjection.sightPoint()) / 2) * 2);

		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("orientation");
		aWriter.write(" " + (float) rotationVector.x());
		aWriter.write(" " + (float) rotationVector.y());
		aWriter.write(" " + (float) rotationVector.z());
		aWriter.write(" " + (float) rotationAngle.rad());

		aWriter.newLine();
		aWriter.write('\t');
		aWriter.write("position");
		aWriter.write(" " + (float) aProjection.eyePoint().x());
		aWriter.write(" " + (float) aProjection.eyePoint().y());
		aWriter.write(" " + (float) aProjection.eyePoint().z());

		if (aProjection.description() != null) {
			aWriter.newLine();
			aWriter.write('\t');
			aWriter.write("description \"" + aProjection.description() + '"');
		}

		aWriter.newLine();
		aWriter.write('}');
		aWriter.newLine();
	}

	/**
	 * Answer the transformation for the VRML97 Viewpoint node.
	 *
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category vrml support
	 */
	private Jun3dTransformation transformationForVrml97(JunOpenGLProjection aProjection) {
		Jun3dPoint vz = aProjection.eyePoint().minus_(aProjection.sightPoint()).unitVector();
		Jun3dPoint vy = aProjection.upVector();
		Jun3dPoint vx = (new Jun3dPoint(vy.y() * vz.z() - vy.z() * vz.y(), vy.z() * vz.x() - vy.x() * vz.z(), vy.x() * vz.y() - vy.y() * vz.x())).unitVector();
		double[] collection = new double[16];
		collection[0] = vx.x();
		collection[1] = vy.x();
		collection[2] = vz.x();
		collection[3] = 0;
		collection[4] = vx.y();
		collection[5] = vy.y();
		collection[6] = vz.y();
		collection[7] = 0;
		collection[8] = vx.z();
		collection[9] = vy.z();
		collection[10] = vz.z();
		collection[11] = 0;
		collection[12] = 0;
		collection[13] = 0;
		collection[14] = 0;
		collection[15] = 1;
		return Jun3dTransformation.Translate_((new Jun3dPoint(0, 0, 0)).minus_(aProjection.sightPoint())).product_(Jun3dTransformation.FromArray_(collection));
	}

	/**
	 * Answer the stamp string for the LST 1.0 file format.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultStampForLST10() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("%LST V1.0 List Shape Transmission (Lisp S Expression)");
		pw.println("% This file was created by " + JunSystem.System() + JunSystem.Version());
		pw.println("% " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Answer the stamp string for the VRML 1.0 file format.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultStampForWRL10() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("#VRML V1.0 ascii");
		pw.println("# This file was created by " + JunSystem.System() + JunSystem.Version());
		pw.println("# " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Answer the stamp string for the VRML 2.0 file format.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultStampForWRL20() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("#VRML V2.0 utf8");
		pw.println();
		pw.println("WorldInfo {");
		pw.print("\t");
		pw.println("info [");
		pw.print("\t\t");
		pw.println("\"This file was created by " + JunSystem.System() + JunSystem.Version() + "\"");
		pw.print("\t\t");
		pw.println("\"" + DateFormat.getInstance().format(new Date()) + "\"");
		pw.print("\t");
		pw.println("]");
		pw.print("\t");
		pw.print("title \"");

		if (this.displayObject() != null) {
			pw.print(this.displayObject().name());
		}

		pw.print(": Sample of ");
		pw.print(this.getClass().getName());
		pw.println("\"");
		pw.println("}");
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Answer the stamp string for the VRML 97 file format.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultStampForWRL97() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("#VRML V2.0 utf8");
		pw.println("# This file was created by " + JunSystem.System() + JunSystem.Version() + " for Java");
		pw.println("# " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Answer the bounding object for the specified object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	public JunOpenGL3dObject boundingObjectFor_(JunOpenGL3dObject a3dObject) {
		Jun3dBoundingBox aBox = a3dObject.boundingBox();
		Jun3dPoint originPoint = aBox.origin();
		Jun3dPoint cornerPoint = aBox.corner();
		Jun3dPoint[] fourPoints = new Jun3dPoint[4];
		fourPoints[0] = new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z());
		fourPoints[3] = new Jun3dPoint(originPoint.x(), originPoint.y(), originPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop1 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop1.alpha_(0.5f);
		fourPoints[3] = new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z());
		fourPoints[0] = new Jun3dPoint(cornerPoint.x(), originPoint.y(), originPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop2 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop2.alpha_(0.5f);
		fourPoints[0] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z());
		fourPoints[3] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop3 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop3.alpha_(0.5f);
		fourPoints[3] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), originPoint.z());
		fourPoints[0] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), originPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop4 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop4.alpha_(0.5f);
		fourPoints[0] = new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[3] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop5 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop5.alpha_(0.5f);
		fourPoints[0] = new Jun3dPoint(originPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[1] = new Jun3dPoint(cornerPoint.x(), originPoint.y(), cornerPoint.z());
		fourPoints[2] = new Jun3dPoint(cornerPoint.x(), cornerPoint.y(), cornerPoint.z());
		fourPoints[3] = new Jun3dPoint(originPoint.x(), cornerPoint.y(), cornerPoint.z());

		JunOpenGL3dPolylineLoop polylineLoop6 = new JunOpenGL3dPolylineLoop(fourPoints, Color.gray);
		polylineLoop6.alpha_(0.5f);

		JunOpenGL3dObject anAxes = JunOpenGL3dObject.Axes2();
		anAxes = anAxes.transform_(Jun3dTransformation.Scale_(cornerPoint.minus_(originPoint)));
		anAxes = anAxes.transform_(Jun3dTransformation.Translate_(originPoint));

		JunOpenGL3dCompoundObject boundingObject = new JunOpenGL3dCompoundObject();
		boundingObject.add_(polylineLoop1);
		boundingObject.add_(polylineLoop2);
		boundingObject.add_(polylineLoop3);
		boundingObject.add_(polylineLoop4);
		boundingObject.add_(polylineLoop5);
		boundingObject.add_(polylineLoop6);
		boundingObject.add_(anAxes);

		return boundingObject;
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeSightPoint()
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		if (this.selectedObjects().isEmpty()) {
			return super.computeSightPoint();
		} else {
			int size = this.selectedObjects().size();
			JunOpenGL3dObject[] selectedObjects = new JunOpenGL3dObject[size];
			this.selectedObjects().copyInto(selectedObjects);

			double x = 0;
			double y = 0;
			double z = 0;

			for (int i = 0; i < size; i++) {
				Jun3dPoint center = (Jun3dPoint) selectedObjects[i].boundingBox().center();
				x += center.x();
				y += center.y();
				z += center.z();
			}

			return new Jun3dPoint(x / size, y / size, z / size);
		}
	}

	/**
	 * Answer the array of the Jun3dPoint, which is a collection between 'fromValue' and 'toValue' divided into 'numberOfValues'.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param numberOfValues int
	 * @param centerValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fromValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param toValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint[] getCollection_centerValue_fromValue_toValue_(int numberOfValues, Jun3dPoint centerValue, Jun3dPoint fromValue, Jun3dPoint toValue) {
		int howMany = numberOfValues - 1;
		double aDistance = centerValue.distance_(fromValue);
		Jun3dLine aLine = new Jun3dLine(centerValue, toValue);
		aLine = aLine.normalizedLine();
		Jun3dPoint stepValue = aLine.atT_(aDistance).minus_(fromValue).dividedBy_(howMany);
		Jun3dPoint[] values = new Jun3dPoint[numberOfValues];
		Jun3dPoint aValue = fromValue;
		values[0] = aValue;
		for (int i = 1; i <= howMany; i++) {
			aValue = aValue.plus_(stepValue);
			aLine = new Jun3dLine(centerValue, aValue);
			aLine = aLine.normalizedLine();
			values[i] = aLine.atT_(aDistance);
		}
		return values;
	}

	/**
	 * Answer the array of the double, which is a collection between 'fromValue' and 'toValue' divided into 'numberOfValues'.
	 *
	 * @return double[]
	 * @param numberOfValues int
	 * @param fromValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param toValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected double[] getCollection_fromValue_toValue_(int numberOfValues, double fromValue, double toValue) {
		int howMany = numberOfValues - 1;
		double stepValue = (toValue - fromValue) / howMany;
		double[] values = new double[numberOfValues];
		double aValue = fromValue;
		values[0] = aValue;
		for (int i = 1; i < howMany; i++) {
			aValue = aValue + stepValue;
			values[i] = aValue;
		}
		values[howMany] = toValue;
		return values;
	}

	/**
	 * Answer the array of the Jun3dPoint, which is a collection between 'fromValue' and 'toValue' divided into 'numberOfValues'.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param numberOfValues int
	 * @param fromValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param toValue jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint[] getCollection_fromValue_toValue_(int numberOfValues, Jun3dPoint fromValue, Jun3dPoint toValue) {
		int howMany = numberOfValues - 1;
		Jun3dPoint stepValue = toValue.minus_(fromValue).dividedBy_(howMany);
		Jun3dPoint[] values = new Jun3dPoint[numberOfValues];
		Jun3dPoint aValue = fromValue;
		values[0] = aValue;
		for (int i = 1; i < howMany; i++) {
			aValue = aValue.plus_(stepValue);
			values[i] = aValue;
		}
		values[howMany] = toValue;
		return values;
	}

	/**
	 * Answer the spawning object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	public JunOpenGL3dObject spawningObject() {
		JunOpenGL3dObject spawningObject;

		if ((this.selectedObjects() == null) || (this.selectedObjects().isEmpty())) {
			if (this.displayObject() == null) {
				spawningObject = null;
			} else if (this.displayObject().isTransformedObject()) {
				spawningObject = ((JunOpenGL3dTransformedObject) this.displayObject()).transformedObject();
			} else {
				spawningObject = (JunOpenGL3dObject) this.displayObject().copy();
			}
		} else {
			Object[] objects = this.selectedObjects().toArray();
			if (objects.length > 1) {
				spawningObject = new JunOpenGL3dCompoundObject();
				for (int i = 0; i < objects.length; i++) {
					((JunOpenGL3dCompoundObject) spawningObject).add_((JunOpenGL3dObject) objects[i]);
				}
			} else {
				spawningObject = (JunOpenGL3dObject) ((JunOpenGL3dObject) objects[0]).copy();
			}
		}

		return spawningObject;
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunOpenGLDisplayView && ((JunOpenGLDisplayView) each).getOpenGLDisplayModel() == this) {
				return (StView) each;
			}
		}
		return null;
	}

	/**
	 * Answer the class to spawn.
	 * 
	 * @return java.lang.Class
	 * @category private
	 */
	protected Class classToSpawn() {
		return this.getClass();
	}

}
