package jp.co.sra.jun.opengl.display;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StRadioButtonGroup;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.surfaces.JunSphereSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.colors.JunColorChoiceDialog;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;
import jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraph;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.opengl.support.JunGLImage;
import jp.co.sra.jun.opengl.support.JunOpenGLDrawable;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderer;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.opengl.support.JunOpenGLResource;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSmallCompiler;

/**
 * JunOpenGL3dModel class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   2000/01/06 (by MATSUDA Ryouichi)
 *  @updated   2003/05/13 (by nisinaka)
 *  @updated   2004/03/26 (by nisinaka)
 *  @updated   2004/12/08 (by Mitsuhiro Asada)
 *  @updated   2005/03/01 (by nisinaka)
 *  @updated   2006/04/11 (by Mitsuhiro Asada)
 *  @updated   2006/10/16 (by Mitsuhiro Asada)
 *  @updated   2007/04/23 (by Mitsuhiro Asada)
 *  @updated   2007/08/23 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun696 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dModel.java,v 8.32 2008/02/20 06:32:18 nisinaka Exp $
 */
public class JunOpenGL3dModel extends JunApplicationModel implements JunOpenGLRenderer, ClipboardOwner {

	public static final Dimension DefaultImageExtent = new Dimension(256, 256);
	protected static final double ThetaAccuracy = 0.001;
	protected static Color DefaultBackgroundColor = Color.white;

	protected JunOpenGL3dObject openGL3dObject;
	protected JunOpenGLProjector openGLProjector;
	protected JunOpenGLDisplayLight[] displayLights;
	protected Vector selectedObjects;
	protected Map defaultProjectionTable;
	protected StValueHolder displayListHolder;
	protected StBlockClosure superimposeBlock;
	protected JunAttributeTable preferenceTable;
	protected Color _backgroundColor;
	protected JunOpenGLProjection[] _presetProjections;
	protected StMenuBar _menuBar;

	/**
	 * Create a new instance of JunOpenGL3dModel and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dModel() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dModel and initialize it with the JunOpenGL3dObject.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category Instance creation
	 */
	public JunOpenGL3dModel(JunOpenGL3dObject a3dObject) {
		super();
		this.openGL3dObject_(a3dObject);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		openGL3dObject = null;
		openGLProjector = null;
		displayLights = null;
		selectedObjects = null;
		defaultProjectionTable = null;
		displayListHolder = null; // new StValueHolder(); to make display list enable by default.
		superimposeBlock = null;
		preferenceTable = null;
		_menuBar = null;
		_presetProjections = null;
		_backgroundColor = null;
	}

	/**
	 * Answer my current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject openGL3dObject() {
		return openGL3dObject;
	}

	/**
	 * Set my new display object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public void openGL3dObject_(JunOpenGL3dObject a3dObject) {
		openGL3dObject = a3dObject;
		this.clearSelectedObjects();
		this.updateMenuIndication();
	}

	/**
	 * Answer my current projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	public JunOpenGLProjector openGLProjector() {
		if (openGLProjector == null) {
			JunOpenGLProjector aProjector = new JunOpenGLProjector();
			this.preCreateDisplayProjector_(aProjector);
			aProjector.setEyePoint_(this.defaultEyePoint());
			aProjector.setSightPoint_(this.defaultSightPoint());
			aProjector.setUpVector_(this.defaultUpVector());
			aProjector.setViewFactor_(this.defaultViewFactor());
			aProjector.setZoomHeight_(this.defaultZoomHeight());
			aProjector.setProjection();
			aProjector.presentation_(this.defaultPresentation());
			aProjector.shading_(this.defaultShading());
			if (this.defaultProjection() == JunOpenGLParallelProjection.Type) {
				aProjector.parallelProjection();
			} else {
				aProjector.perspectiveProjection();
			}
			this.postCreateDisplayProjector_(aProjector);
			openGLProjector = aProjector;
		}

		return openGLProjector;
	}

	/**
	 * Action hook for the pre-create display projector.
	 * 
	 * @param aProjector jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	protected JunOpenGLProjector preCreateDisplayProjector_(JunOpenGLProjector aProjector) {
		return aProjector;
	}

	/**
	 * Action hook for the post-create display projector.
	 * 
	 * @param aProjector jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @category accessing
	 */
	protected JunOpenGLProjector postCreateDisplayProjector_(JunOpenGLProjector aProjector) {
		return aProjector;
	}

	/**
	 * Answer my current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public JunOpenGLProjection openGLProjection() {
		return this.openGLProjector().projection();
	}

	/**
	 * Set my new projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public void openGLProjection_(JunOpenGLProjection aProjection) {
		this.openGLProjector().projection_(aProjection);
	}

	/**
	 * Answer my current background color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color backgroundColor() {
		if (_backgroundColor == null) {
			_backgroundColor = DefaultBackgroundColor;
		}
		return _backgroundColor;
	}

	/**
	 * Set my preset projections.
	 * 
	 * @param projections jp.co.sra.jun.opengl.projection.JunOpenGLProjection.Info[]
	 * @category accessing
	 */
	public void presetProjections_(JunOpenGLProjection[] projections) {
		_presetProjections = projections;
		this.updateMenuBar();
	}

	/**
	 * Set my new background color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	public void backgroundColor_(Color aColor) {
		_backgroundColor = aColor;
		this.changed_($("backgroundColor"));
	}

	/**
	 * Flush the object.
	 * 
	 * @category flushing
	 */
	public void flushObject() {
		openGL3dObject = null;
	}

	/**
	 * Flush the projector.
	 * 
	 * @category flushing
	 */
	public void flushProjector() {
		openGLProjector = null;
	}

	/**
	 * Flush the display object.
	 * 
	 * @category flushing
	 */
	protected void flushDisplayObject() {
		this.flushObject();
	}

	/**
	 * Flush the menus.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#flushMenus()
	 * @category flushing
	 */
	protected void flushMenus() {
		super.flushMenus();
		_menuBar = null;
	}

	/**
	 * Answer the bounding box of the display object.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (this.openGL3dObject() == null) {
			return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 0));
		}
		return this.openGL3dObject().boundingBox();
	}

	/**
	 * Answer whether the display object is empty or not.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		if (this.openGL3dObject() == null) {
			return true;
		}

		if (this.openGL3dObject().isCompound() == false) {
			return false;
		}

		JunOpenGL3dObject openGL3dObject = this.openGL3dObject();
		if (openGL3dObject instanceof JunOpenGL3dCompoundObject) {
			return ((JunOpenGL3dCompoundObject) openGL3dObject).components().length == 0;
		} else if (openGL3dObject instanceof JunOpenGL3dGraph) {
			return ((JunOpenGL3dGraph) openGL3dObject).components().length == 0;
		}
		return false;
	}

	/**
	 * Enumerate all points of the 3D object and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		if (this.openGL3dObject() == null) {
			return;
		}

		this.openGL3dObject().pointsDo_(aBlock);
	}

	/**
	 * Convert the receiver to an image as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		Image anImage = this.toImage();
		return (anImage == null) ? null : new StImage(anImage);
	}

	/**
	 * Convert the receiver to an image as Image.
	 *
	 * @return java.awt.Image
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#toImage()
	 * @category converting
	 */
	public Image toImage() {
		StView aView = this.getView();
		Dimension imageExtent = null;
		if (aView == null) {
			imageExtent = this.defaultImageExtent();
		} else {
			imageExtent = aView.toComponent().getSize();
		}
		return this.toImageExtent_(imageExtent);
	}

	/**
	 * Convert the view as an image with the specified extent..
	 *
	 * @param extent java.awt.Dimension
	 * @return java.awt.Image
	 * @category converting
	 */
	public Image toImageExtent_(Dimension extent) {
		JunGLImage aGLImage = new JunGLImage(extent.width, extent.height);
		aGLImage.setRenderer(this);
		return aGLImage.image();
	}

	/**
	 * Convert the model point to the view 3D point.
	 * 
	 * @param modelPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category converting
	 */
	public Jun3dPoint convertModelPointToView3dPoint_in_(Jun3dPoint modelPoint, Rectangle aRectangle) {
		double scaleFactor = aRectangle.height / 2.0d;
		Jun3dPoint deltaPoint = new Jun3dPoint(aRectangle.width / 2.0d, scaleFactor, 0);
		JunOpenGLProjection aProjection = this.openGLProjection();
		Jun3dPoint aPoint = aProjection.translateToPointFrom3dPoint_(modelPoint);
		Jun3dPoint view3dPoint = new Jun3dPoint(aPoint.x(), -aPoint.y(), -aPoint.z());
		view3dPoint = view3dPoint.scaledBy_(scaleFactor);
		view3dPoint = view3dPoint.translatedBy_(deltaPoint);
		return view3dPoint;
	}

	/**
	 * Convert the model points to the view 3D points.
	 * 
	 * @param modelPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category converting
	 */
	public Jun3dPoint[] convertModelPointsToView3dPoints_in_(Jun3dPoint[] modelPoints, Rectangle aRectangle) {
		Jun3dPoint[] view3dPoints = new Jun3dPoint[modelPoints.length];
		for (int i = 0; i < view3dPoints.length; i++) {
			view3dPoints[i] = this.convertModelPointToView3dPoint_in_(modelPoints[i], aRectangle);
		}
		return view3dPoints;
	}

	/**
	 * Convert the model points to the view 3D points sorted by Z.
	 * 
	 * @param modelPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aRectangle java.awt.Rectangle
	 * @return java.util.Map.Entry[]
	 * @category converting
	 */
	public Map.Entry[] convertModelPointsToView3dPointsSortedByZ_in_(Jun3dPoint[] modelPoints, Rectangle aRectangle) {
		Jun3dPoint[] viewPoints = this.convertModelPointsToView3dPoints_in_(modelPoints, aRectangle);

		HashMap aMap = new HashMap();
		for (int i = 0; i < viewPoints.length; i++) {
			aMap.put(new Double(viewPoints[i].z()), new Jun3dPoint[] { modelPoints[i], viewPoints[i] });
		}
		TreeSet aTreeSet = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				double z1 = ((Number) ((Map.Entry) o1).getKey()).doubleValue();
				double z2 = ((Number) ((Map.Entry) o2).getKey()).doubleValue();
				return (z1 < z2) ? -1 : (z2 < z1) ? 1 : 0;
			}
		});
		aTreeSet.addAll(aMap.entrySet());
		return (Map.Entry[]) aTreeSet.toArray(new Map.Entry[aTreeSet.size()]);
	}

	/**
	 * Answer the first display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight1() {
		return this.displayLights()[0];
	}

	/**
	 * Answer the second display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight2() {
		return this.displayLights()[1];
	}

	/**
	 * Answer the third display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight3() {
		return this.displayLights()[2];
	}

	/**
	 * Answer the fourth display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight4() {
		return this.displayLights()[3];
	}

	/**
	 * Answer the fifth display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight5() {
		return this.displayLights()[4];
	}

	/**
	 * Answer the array of lights which are on.
	 * 
	 * @return jp.co.sra.jun.opengl.lights.JunOpenGLLight[]
	 * @category lighting
	 */
	public JunOpenGLLight[] displayLightCollection() {
		ArrayList aList = new ArrayList();

		Jun3dTransformation t = (Jun3dTransformation) this.openGLProjection().asEyeTransformation().inverse();
		JunOpenGLDisplayLight[] displayLights = this.displayLights();
		for (int i = 0; i < displayLights.length; i++) {
			if (displayLights[i].isOn()) {
				aList.add(displayLights[i].light().transform_(t));
			}
		}

		return (JunOpenGLLight[]) aList.toArray(new JunOpenGLLight[aList.size()]);
	}

	/**
	 * Make the receiver's all light off.
	 * 
	 * @category lighting
	 */
	public void displayLightsAllOff() {
		JunOpenGLDisplayLight[] lights = this.displayLights();
		for (int i = 0; i < lights.length; i++) {
			lights[i].beOff();
		}
		this.updateLightMenuIndication();
		this.changed_($("light"));
	}

	/**
	 * Answer the array of all display lights.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @category lighting
	 */
	public JunOpenGLDisplayLight[] displayLights() {
		if (displayLights == null) {
			displayLights = new JunOpenGLDisplayLight[5];
			displayLights[0] = JunOpenGLDisplayLight.ParallelLight_color_position_(true, this.defaultLightColor(), this.defaultLightPoint());
			displayLights[1] = new JunOpenGLDisplayLight();
			displayLights[2] = new JunOpenGLDisplayLight();
			displayLights[3] = new JunOpenGLDisplayLight();
			displayLights[4] = JunOpenGLDisplayLight.AmbientLight_color_(true, this.defaultLightColor());

			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].compute_(new StBlockClosure() {
					public Object value() {
						JunOpenGL3dModel.this.updateLightMenuIndication();
						JunOpenGL3dModel.this.changed_($("light"));
						return null;
					}
				});
			}

			String[] name = new String[] { $String("Light") + "1", $String("Light") + "2", $String("Light") + "3", $String("Light") + "4", $String("Ambient Light", "Ambient") };
			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].lightName_(name[i]);
			}
		}

		return displayLights;
	}

	/**
	 * Set my new display lights.
	 * 
	 * @param anArrayOfDisplayLight jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @category lighting
	 */
	public void displayLights_(JunOpenGLDisplayLight[] anArrayOfDisplayLight) {
		if (anArrayOfDisplayLight == null || anArrayOfDisplayLight.length != 5) {
			return;
		}

		displayLights = anArrayOfDisplayLight;
	}

	/**
	 * Render the display objects on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.objects.JunOpenGLRenderingContext
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext renderingContext) {
		this.renderOn_withDisplayList_(renderingContext, displayListHolder);
	}

	/**
	 * Render the display objects on the rendering context with the display list.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.objects.JunOpenGLRenderingContext
	 * @param displayListHolder jp.co.sra.smalltalk.StValueHolder
	 * @category rendering
	 */
	public void renderOn_withDisplayList_(JunOpenGLRenderingContext renderingContext, StValueHolder displayListHolder) {
		renderingContext.clear();

		try {
			JunOpenGL3dObject renderingObject = this.openGL3dObject();
			if (renderingObject == null) {
				this.openGLProjector().projectOn_(renderingContext);
			} else {
				JunOpenGLLight[] lightCollection = this.displayLightCollection();
				if (this.selectedObjects().isEmpty()) {
					this.openGLProjector().project_withLights_on_withDisplayList_(renderingObject, lightCollection, renderingContext, displayListHolder);
				} else {
					JunOpenGLProjector aProjector = (JunOpenGLProjector) this.openGLProjector().copy();
					if (this.openGLProjector().presentation() == $("solidPresentation")) {
						aProjector.wireframePresentation();
					} else if (this.openGLProjector().presentation() == $("wireframePresentation")) {
						aProjector.wireframePresentation();
					} else if (this.openGLProjector().presentation() == $("hiddenlinePresentation")) {
						aProjector.hiddenlinePresentation();
					}
					aProjector.project_withLights_on_withDisplayList_(renderingObject, lightCollection, renderingContext, displayListHolder);
					if (this.openGLProjector().presentation() == $("solidPresentation")) {
						aProjector.solidPresentation();
					} else if (this.openGLProjector().presentation() == $("wireframePresentation")) {
						aProjector.solidPresentation();
					} else if (this.openGLProjector().presentation() == $("hiddenlinePresentation")) {
						aProjector.solidPresentation();
					}
					Object[] objects = this.selectedObjects().toArray();
					for (int i = 0; i < objects.length; i++) {
						aProjector.project_on_((JunOpenGL3dObject) objects[i], renderingContext);
					}
				}
			}
		} finally {
			renderingContext.flush();
		}
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param deltaPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void grab_(Jun2dPoint deltaPoint) {
		this.grab_(deltaPoint, false);
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param deltaPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param shiftDown boolean
	 * @category manipulating
	 */
	public void grab_(Jun2dPoint deltaPoint, boolean shiftDown) {
		if (this.openGL3dObject() == null) {
			return;
		}

		Jun2dPoint fromPoint = new Jun2dPoint(0, 0);
		Jun2dPoint toPoint;

		if (shiftDown) {
			toPoint = new Jun2dPoint(deltaPoint.x() / 10, deltaPoint.y() / 10);
		} else {
			toPoint = new Jun2dPoint(deltaPoint.x(), deltaPoint.y());
		}

		this.grab_xy_(fromPoint, toPoint);
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void grab_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.openGL3dObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.openGLProjection();
		JunSphereSurface grabSphere = this.grabSphere();
		Jun3dPoint centerPoint = grabSphere.center();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		JunAngle rotationAngle = (new Jun3dLine(centerPoint, toPoint)).angleWithLine_(new Jun3dLine(centerPoint, fromPoint));

		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(centerPoint, centerPoint.minus_((fromPoint.minus_(centerPoint)).product_(toPoint.minus_(centerPoint))));
			Jun3dTransformation transformation = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.negated(), rotationAxis);
			Jun3dPoint upPoint = (Jun3dPoint) aProjection.sightPoint().plus_(aProjection.unitUpVector());
			aProjection.eyePoint_(transformationInv.applyTo_(aProjection.eyePoint()));
			aProjection.upVector_((Jun3dPoint) transformation.applyTo_(upPoint).minus_(aProjection.sightPoint()));
			this.openGLProjection_(aProjection);
			this.changed_($("projection"));
		}
	}

	/**
	 * Change the eye point.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void look_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.openGL3dObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.openGLProjection();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		this.openGLProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Press the specified 2d point.
	 * 
	 * @param a2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param event java.awt.event.MouseEvent
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category manipulating
	 */
	public Jun3dPoint press_with_(Jun2dPoint a2dPoint, MouseEvent event) {
		JunOpenGLProjection aProjection = this.openGLProjection();
		Jun3dPoint aPoint = aProjection.translateTo3dPointFromPoint_(a2dPoint);
		if (event.isShiftDown()) {
			System.out.println(aPoint.toString());
		}
		return aPoint;
	}

	/**
	 * Slide the eye beam.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	public void slide_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.openGL3dObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.openGLProjection();
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		aProjection.sightPoint_((Jun3dPoint) aProjection.sightPoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
		this.openGLProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Add the JunOpenGL3dObject as one of the selected objects.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.Jun3dOpenGLObject
	 * @category selecting
	 */
	public void addSelectedObject_(JunOpenGL3dObject a3dObject) {
		if (this.openGL3dObject() == null) {
			return;
		}

		if (this.openGL3dObject().isPrimitive()) {
			return;
		}

		if (this.selectedObjects().contains(a3dObject) == false) {
			this.selectedObjects().add(a3dObject);
		}
	}

	/**
	 * Clear the variable which holds the currently selected objects.
	 * 
	 * @category selecting
	 */
	public void clearSelectedObjects() {
		selectedObjects = new Vector();
	}

	/**
	 * Remove the JunOpenGL3dObject from the selected objects.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category selecting
	 */
	public void removeSelectedObject_(JunOpenGL3dObject a3dObject) {
		if (this.openGL3dObject() == null) {
			return;
		}
		if (this.openGL3dObject().isPrimitive()) {
			return;
		}

		if (this.selectedObjects().contains(a3dObject)) {
			this.selectedObjects().remove(a3dObject);
		}
	}

	/**
	 * Answer my currently selected objects.
	 *
	 * @return java.util.Vector
	 * @category selecting
	 */
	public Vector selectedObjects() {
		if (selectedObjects == null) {
			this.clearSelectedObjects();
		}
		return selectedObjects;
	}

	/**
	 * Answer the receiver's preference table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category preferences
	 */
	public JunAttributeTable preferenceTable() {
		if (preferenceTable == null) {
			preferenceTable = new JunAttributeTable();
		}
		return preferenceTable;
	}

	/**
	 * Answer true if the display model is specified to use display list, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean useDisplayList() {
		Boolean aBoolean = (Boolean) this.preferenceTable().at_($("useDisplayList"));
		if (aBoolean == null) {
			aBoolean = Boolean.FALSE;
			this.preferenceTable().at_put_($("useDisplayList"), aBoolean);
		}
		return aBoolean.booleanValue();
	}

	/**
	 * Set whether the display model uses display list or not.
	 * 
	 * @param aBoolean boolean
	 * @category preferences
	 */
	public void useDisplayList_(boolean aBoolean) {
		this.preferenceTable().at_put_($("useDisplayList"), new Boolean(aBoolean));

		if (aBoolean == false) {
			displayListHolder = null;
		} else if (displayListHolder == null) {
			displayListHolder = new StValueHolder();
		}
	}

	/**
	 * Answer the window label in the preference table.
	 * 
	 * @return java.lang.String
	 * @category preferences
	 */
	public String windowLabel() {
		return (String) this.preferenceTable().at_ifAbsentPut_($("windowLabel"), new StBlockClosure() {
			public Object value() {
				return $String("Viewfinder");
			}
		});
	}

	/**
	 * Set the window label in the preference table.
	 * 
	 * @param aString java.lang.String
	 * @category preference
	 */
	public void windowLabel_(String aString) {
		this.preferenceTable().at_put_($("windowLabel"), aString);

		if (windowLabel() != null) {
			this.updateWindowTitle();
		}
	}

	/**
	 * Change the presentation mode to hidden-line.
	 * 
	 * @category presentation
	 */
	public void hiddenlinePresentation() {
		this.openGLProjector().hiddenlinePresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("hiddenlinePresentation"));
		}
	}

	/**
	 * Change the presentation mode to solid.
	 * 
	 * @category presentation
	 */
	public void solidPresentation() {
		this.openGLProjector().solidPresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("solidPresentation"));
		}
	}

	/**
	 * Change the presentation mode to wireframe.
	 * 
	 * @category presentation
	 */
	public void wireframePresentation() {
		this.openGLProjector().wireframePresentation();
		this.updateViewMenuIndication();
		this.changed_($("presentation"));

		if (this.defaultProjectionTable().containsKey($("presentation")) == false) {
			this.defaultPresentation_($("wireframePresentation"));
		}
	}

	/**
	 * Dolly the projector.
	 * 
	 * @param factor double
	 * @category projection
	 */
	public void dolly_(double factor) {
		double dollyFactor = factor;

		Jun3dLine aLine = new Jun3dLine(this.eyePoint(), this.sightPoint());
		Jun3dPoint eyePoint = (Jun3dPoint) aLine.atT_(0.0 + dollyFactor);
		Jun3dPoint sightPoint = (Jun3dPoint) aLine.atT_(1.0 + dollyFactor);
		this.openGLProjector.eyePoint_(eyePoint);
		this.openGLProjector.sightPoint_(sightPoint);
		this.changed_($("projection"));
	}

	/**
	 * Answer my current eye point.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public Jun3dPoint eyePoint() {
		return this.openGLProjector().eyePoint();
	}

	/**
	 * Set my new eye point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public void eyePoint_(Jun3dPoint aPoint) {
		this.openGLProjector().eyePoint_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("eyePoint")) == false) {
			this.defaultEyePoint_(aPoint);
		}
	}

	/**
	 * Fit the display object to the view.
	 * 
	 * @category projection
	 */
	public void fit() {
		if (this.openGL3dObject() == null) {
			return;
		}

		this.fitSilently();
		this.changed_($("projection"));
	}

	/**
	 * Fit the sight point to the current view.
	 * 
	 * @category projection
	 */
	public void fitSight() {
		if (this.openGL3dObject() == null) {
			return;
		}

		Jun3dPoint sightPoint = this.computeSightPoint();
		this.sightPoint_(sightPoint);
	}

	/**
	 * Fit the 3d object to the current view.
	 * 
	 * @category projection
	 */
	public void fitSilently() {
		if (this.openGL3dObject() == null) {
			return;
		}

		Jun3dPoint sightPoint = this.computeSightPoint();
		this.openGLProjector().sightPoint_(sightPoint);
		double zoomHeight = this.computeZoomHeight();
		this.openGLProjector().zoomHeight_(zoomHeight);
	}

	/**
	 * Menu message: Fit to the zoom factor.
	 * 
	 * @category projection
	 */
	public void fitZoom() {
		if (this.openGL3dObject() == null) {
			return;
		}

		double zoomHeight = this.computeZoomHeight();
		this.zoomHeight_(zoomHeight);
	}

	/**
	 * Pan.
	 * 
	 * @category projection
	 */
	public void pan() {
		this.zoom_(0.5);
	}

	/**
	 * Change the projection to parallel.
	 * 
	 * @category projection
	 */
	public void parallelProjection() {
		this.openGLProjector().parallelProjection();
		this.updateViewMenuIndication();
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("projection")) == false) {
			this.defaultProjection_(JunOpenGLParallelProjection.Type);
		}
	}

	/**
	 * Change the projection to perspective.
	 * 
	 * @category projection
	 */
	public void perspectiveProjection() {
		this.openGLProjector().perspectiveProjection();
		this.updateViewMenuIndication();
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("projection")) == false) {
			this.defaultProjection_(JunOpenGLPerspectiveProjection.Type);
		}
	}

	/**
	 * Answer my current sight point.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public Jun3dPoint sightPoint() {
		return this.openGLProjector().sightPoint();
	}

	/**
	 * Set my new sight point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public void sightPoint_(Jun3dPoint aPoint) {
		this.openGLProjector().sightPoint_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("sightPoint")) == false) {
			this.defaultSightPoint_(aPoint);
		}
	}

	/**
	 * Answer my current up vector.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public Jun3dPoint upVector() {
		return this.openGLProjector().upVector();
	}

	/**
	 * Set my new up vector.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public void upVector_(Jun3dPoint aPoint) {
		this.openGLProjector().upVector_(aPoint);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("upVector")) == false) {
			this.defaultUpVector_(aPoint);
		}
	}

	/**
	 * Answer my current view factor.
	 *
	 * @return double
	 * @category projection
	 */
	public double viewFactor() {
		return this.openGLProjector().viewFactor();
	}

	/**
	 * Set my new view factor.
	 * 
	 * @param factor double
	 * @category projection
	 */
	public void viewFactor_(double factor) {
		this.openGLProjector().viewFactor_(factor);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("viewFactor")) == false) {
			this.defaultViewFactor_(factor);
		}
	}

	/**
	 * Do the zoom.
	 * 
	 * @category projection
	 */
	public void zoom() {
		this.zoom_(2.0);
	}

	/**
	 * Do the zoom with the specified factor.
	 * 
	 * @param factor double
	 * @category projection
	 */
	public void zoom_(double factor) {
		if (StInputState.Default().shiftDown()) {
			if (factor >= 1.0) {
				factor = 1.0 + (factor - 1.0) * 0.1;
			} else {
				factor = 1.0 - (1.0 - factor) * 0.1;
			}
		}
		this.openGLProjector().zoom_(factor);
		this.changed_($("projection"));
	}

	/**
	 * Answer my current zoom height.
	 *
	 * @return double
	 * @category projection
	 */
	public double zoomHeight() {
		return this.openGLProjector().zoomHeight();
	}

	/**
	 * Post get projection table.
	 * 
	 * @param aDictionary java.util.Map
	 * @return java.util.Map
	 * @category projection
	 */
	protected Map postGetProjectionTable_(Map aDictionary) {
		return aDictionary;
	}

	/**
	 * Post put projection table.
	 * 
	 * @param aDictionary java.util.Map
	 * @param aProjector jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @return java.util.Map
	 * @category projection
	 */
	protected Map postPutProjectionTable_into_(Map aDictionary, JunOpenGLProjector aProjector) {
		return aDictionary;
	}

	/**
	 * Post get projection table.
	 * 
	 * @param aDictionary java.util.Map
	 * @return java.util.Map
	 * @category projection
	 */
	protected Map preGetProjectionTable_(Map aDictionary) {
		return aDictionary;
	}

	/**
	 * Pre put projection table.
	 * 
	 * @param aDictionary java.util.Map
	 * @param aProjector jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @return java.util.Map
	 * @category projection
	 */
	protected Map prePutProjectionTable_into_(Map aDictionary, JunOpenGLProjector aProjector) {
		return aDictionary;
	}

	/**
	 * Answer my current projection table.
	 * 
	 * @return java.util.Map
	 * @category projection
	 */
	public Map projectionTable() {
		HashMap projectionTable = new HashMap(8);
		this.preGetProjectionTable_(projectionTable);
		projectionTable.put($("eyePoint"), this.eyePoint());
		projectionTable.put($("sightPoint"), this.sightPoint());
		projectionTable.put($("upVector"), this.upVector());
		projectionTable.put($("viewFactor"), new Double(this.viewFactor()));
		projectionTable.put($("zoomHeight"), new Double(this.zoomHeight()));
		projectionTable.put($("presentation"), this.openGLProjector().presentation());
		projectionTable.put($("shading"), this.openGLProjector().shading());
		projectionTable.put($("projection"), ((this.openGLProjector().projection() instanceof JunOpenGLParallelProjection) ? $("parallelProjection") : $("perspectiveProjection")));
		this.postGetProjectionTable_(projectionTable);
		return projectionTable;
	}

	/**
	 * Set my new projection table.
	 * 
	 * @param aDictionary java.util.Map
	 * @category projection
	 */
	public void projectionTable_(Map aDictionary) {
		JunOpenGLProjector aProjector = this.openGLProjector();
		this.prePutProjectionTable_into_(aDictionary, aProjector);
		if (aDictionary.containsKey($("eyePoint"))) {
			aProjector.eyePoint_((Jun3dPoint) aDictionary.get($("eyePoint")));
		}
		if (aDictionary.containsKey($("sightPoint"))) {
			aProjector.sightPoint_((Jun3dPoint) aDictionary.get($("sightPoint")));
		}
		if (aDictionary.containsKey($("upVector"))) {
			aProjector.upVector_((Jun3dPoint) aDictionary.get($("upVector")));
		}
		if (aDictionary.containsKey($("viewFactor"))) {
			aProjector.viewFactor_(((Number) aDictionary.get($("viewFactor"))).doubleValue());
		}
		if (aDictionary.containsKey($("zoomHeight"))) {
			aProjector.zoomHeight_(((Number) aDictionary.get($("zoomHeight"))).doubleValue());
		}
		if (aDictionary.containsKey($("presentation"))) {
			aProjector.presentation_((StSymbol) aDictionary.get($("presentation")));
		}
		if (aDictionary.containsKey($("shading"))) {
			aProjector.shading_((StSymbol) aDictionary.get($("shading")));
		}
		if (aDictionary.containsKey($("projection"))) {
			if ((StSymbol) aDictionary.get($("projection")) == $("parallelProjection")) {
				aProjector.parallelProjection();
			} else {
				aProjector.perspectiveProjection();
			}
		}
		this.postPutProjectionTable_into_(aDictionary, aProjector);
		this.changed_($("projection"));
	}

	/**
	 * Read the projection table from the copy buffer..
	 * 
	 * @category projection
	 */
	protected void projectionTableFromCopyBuffer() {
		Transferable content = Toolkit.getDefaultToolkit().getSystemClipboard().getContents(this);
		String aString = null;
		try {
			aString = (String) content.getTransferData(DataFlavor.stringFlavor);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
		if (aString == null) {
			return;
		}

		String headerString = "((JunAttributeTable fromString: '";
		String footerString = "') attributeTable)";
		if (aString.startsWith(headerString) == false && aString.endsWith(footerString) == false) {
			headerString = "(JunAttributeTable fromString: '";
			footerString = "') attributeTable";
			if (aString.startsWith(headerString) == false && aString.endsWith(footerString) == false) {
				return;
			}
		}

		aString = aString.substring(headerString.length(), aString.length() - footerString.length());
		JunAttributeTable attributes = JunAttributeTable.FromString_(aString);
		if (attributes == null) {
			return;
		}

		this.projectionTable_(attributes.attributeTable());
	}

	/**
	 * Write the projection table to the copy buffer.
	 * 
	 * @category projection
	 */
	protected void projectionTableToCopyBuffer() {
		String footerString = ") yourself)";
		String aString = new JunAttributeTable(this.projectionTable()).storeString();
		if (aString.endsWith(footerString)) {
			aString = aString.substring(0, aString.length() - footerString.length()) + ") attributeTable)";
		}
		Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(aString), this);
	}

	/**
	 * Set my new zoom height.
	 * 
	 * @param aNumber double
	 * @category projection
	 */
	public void zoomHeight_(double aNumber) {
		this.openGLProjector().zoomHeight_(aNumber);
		this.changed_($("projection"));

		if (this.defaultProjectionTable().containsKey($("zoomHeight")) == false) {
			this.defaultZoomHeight_(aNumber);
		}
	}

	/**
	 * Change the shading mode to flat.
	 * 
	 * @category shading
	 */
	public void flatShading() {
		this.openGLProjector().flatShading();
		this.updateViewMenuIndication();
		this.changed_($("shading"));

		if (this.defaultProjectionTable().containsKey($("shading")) == false) {
			this.defaultShading_($("flatShading"));
		}
	}

	/**
	 * Change the shading mode to smooth.
	 * 
	 * @category shading
	 */
	public void smoothShading() {
		this.openGLProjector().smoothShading();
		this.updateViewMenuIndication();
		this.changed_($("shading"));

		if (this.defaultProjectionTable().containsKey($("shading")) == false) {
			this.defaultShading_($("smoothShading"));
		}
	}

	/**
	 * Answer true if the display object has a texture mapping.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean hasTextureMapping() {
		JunOpenGL3dObject displayObject = this.openGL3dObject();
		if (displayObject == null) {
			return false;
		}

		final StValueHolder aBoolean = new StValueHolder(false);
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture anotherTexture = each.texture();
				if (anotherTexture != null) {
					aBoolean.value_(true);
				}
				return null;
			}
		});

		return aBoolean._booleanValue();
	}

	/**
	 * Answer true if the display object need a clamp to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean clamp() {
		JunOpenGL3dObject displayObject = this.openGL3dObject();
		if (displayObject == null) {
			return false;
		}

		final StValueHolder aBoolean = new StValueHolder(false);
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture anotherTexture = each.texture();
				if (anotherTexture != null) {
					aBoolean.value_(anotherTexture.clamp());
				}
				return null;
			}
		});

		return aBoolean._booleanValue();
	}

	/**
	 * Answer true if the display object need linear to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean linear() {
		JunOpenGL3dObject displayObject = this.openGL3dObject();
		if (displayObject == null) {
			return false;
		}

		final StValueHolder aBoolean = new StValueHolder(false);
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					aBoolean.value_(texture.linear());
				}
				return null;
			}
		});

		return aBoolean._booleanValue();
	}

	/**
	 * Answer true if the display object need mipmap to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean mipmap() {
		JunOpenGL3dObject displayObject = this.openGL3dObject();
		if (displayObject == null) {
			return false;
		}

		final StValueHolder aBoolean = new StValueHolder(false);
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					aBoolean.value_(texture.mipmap());
				}
				return null;
			}
		});

		return aBoolean._booleanValue();
	}

	/**
	 * Answer true if the display object need modulate to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean modulate() {
		JunOpenGL3dObject displayObject = this.openGL3dObject();
		if (displayObject == null) {
			return false;
		}

		final StValueHolder aBoolean = new StValueHolder(false);
		displayObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				JunOpenGLTexture texture = each.texture();
				if (texture != null) {
					aBoolean.value_(texture.modulate());
				}
				return null;
			}
		});

		return aBoolean._booleanValue();
	}

	/**
	 * Answer true if the display object need nearest to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean nearest() {
		return !this.linear();
	}

	/**
	 * Answer true if the display object need repeat to be set.
	 * 
	 * @return boolean
	 * @category texture mapping
	 */
	public boolean repeat() {
		return !this.clamp();
	}

	/**
	 * Answer my current superimpose block.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category superimpose
	 */
	public StBlockClosure superimposeBlock() {
		return superimposeBlock;
	}

	/**
	 * Set my new superimpose block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category superimpose
	 */
	public void superimposeBlock_(StBlockClosure aBlock) {
		superimposeBlock = aBlock;
	}

	/**
	 * Superimpose on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aDrawable jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLRenderer#superimposeOn_(java.awt.Graphics)
	 * @category superimpose
	 */
	public void superimposeOn_(Graphics aGraphics, JunOpenGLDrawable aDrawable) {
		if (superimposeBlock == null) {
			return;
		}

		Shape clip = aGraphics.getClip();
		try {
			aGraphics.setClip(0, 0, aDrawable.toComponent().getWidth(), aDrawable.toComponent().getHeight());

			switch (superimposeBlock.numArgs()) {
				case 0:
					superimposeBlock.value();
					break;
				case 1:
					superimposeBlock.value_(aGraphics);
					break;
				case 2:
					superimposeBlock.value_value_(aGraphics, this.getOpenGLViewWith(aDrawable));
					break;
			}
		} finally {
			aGraphics.setClip(clip);
		}

	}

	/**
	 * Invoked when the clipboard ownership is lost.
	 * 
	 * @param clipboard java.awt.datatransfer.Clipboard
	 * @param contents java.awt.datatransfer.Transferable
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category events
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
		// System.out.println("Clipboard contents replaced");
	}

	/**
	 * Create a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGL3dViewAwt(this);
		} else {
			return new JunOpenGL3dViewSwing(this);
		}
	}

	/**
	 * Create a Frame for the application model.
	 * Do everything but open the frame.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationModel#allButOpenView_(jp.co.sra.smalltalk.StView)
	 * @category interface opening
	 */
	protected Frame allButOpenView_(StView aView) {
		this.flushProjector();
		this.projectionTable_(this.defaultProjectionTable());
		return super.allButOpenView_(aView);
	}

	/**
	 * This method will be sent by the view when its top component is opened.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StUIBulider#windowOpened(WindowEvent)
	 * @category interface opening
	 */
	public void postOpenWith_(StView aView) {
		super.postOpenWith_(aView);

		if (this.defaultProjectionTable().containsKey($("sightPoint")) == false || this.defaultProjectionTable().containsKey($("zoomHeight")) == false) {
			this.fitSilently();
		}
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return this.windowLabel();
	}

	/**
	 * Change the background color.
	 * 
	 * @category menu messages
	 */
	public void changeBackgroundColor() {
		Color aColor = JunColorChoiceDialog.Request();
		if (aColor == null) {
			return;
		}

		this.backgroundColor_(aColor);
	}

	/**
	 * Change the eye point.
	 * 
	 * @category menu messages
	 */
	public void changeEyePoint() {
		Jun3dPoint point = this.eyePoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new eye point.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.eyePoint_(point);
	}

	/**
	 * Menu message: Change the sight point.
	 * 
	 * @category menu messages
	 */
	public void changeSightPoint() {
		Jun3dPoint point = this.sightPoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new sight point.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.sightPoint_(point);
	}

	/**
	 * Change the smoothing angle.
	 * 
	 * @category menu messages
	 */
	public void changeSmoothingAngle() {
		JunOpenGL3dObject anObject = this.openGL3dObject();
		if (anObject == null) {
			return;
		}

		JunAngle smoothingAngle = anObject.smoothingAngle();
		if (smoothingAngle == null) {
			smoothingAngle = anObject.defaultSmoothingAngle();
		}
		double value = smoothingAngle.deg();
		String string = JunDialog.Request_($String("Input a smoothing angle."), String.valueOf((float) value));
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			value = Double.valueOf(string).doubleValue();
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid value."));
		}

		smoothingAngle = JunAngle.FromDeg_(value);
		this.openGL3dObject().establishAllNormalVectorsWithSmoothingAngle_(smoothingAngle);
		this.changed_($("presentation"));
	}

	/**
	 * Menu message: Change the up vector.
	 * 
	 * @category menu messages
	 */
	public void changeUpVector() {
		Jun3dPoint point = this.upVector();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String string = JunDialog.Request_($String("Input a new up vector.") + " (x, y, z)", "(" + x + "," + y + "," + z + ")");
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			point = (Jun3dPoint) JunSmallCompiler.Evaluate_(string);
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid point."));
		}

		this.upVector_(point);
	}

	/**
	 * Menu message: Change the view factor.
	 * 
	 * @category menu messages
	 */
	public void changeViewFactor() {
		double value = this.viewFactor();
		String string = JunDialog.Request_($String("Input a new view factor."), String.valueOf((float) value));
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			value = Double.valueOf(string).doubleValue();
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid value."));
		}

		this.viewFactor_(value);
	}

	/**
	 * Menu message: Change the zoom height.
	 * 
	 * @category menu messages
	 */
	public void changeZoomHeight() {
		double value = this.zoomHeight();
		String string = JunDialog.Request_($String("Input a new zoom height."), String.valueOf((float) value));
		if (string == null || string.length() == 0) {
			return;
		}

		try {
			value = Double.valueOf(string).doubleValue();
		} catch (Exception e) {
			JunDialog.Warn_(string + $String(" is invalid value."));
		}

		this.zoomHeight_(value);
	}

	/**
	 * Menu message: Close up.
	 * 
	 * @category menu messages
	 */
	public void closeUp() {
		JunOpenGLProjection aProjection = this.openGLProjection();
		double aRadius = aProjection.regularHeight() / 2;
		double aDistance = aProjection.sightPoint().distance_(aProjection.eyePoint());
		Jun3dLine aLine = new Jun3dLine(aProjection.sightPoint(), aProjection.eyePoint());
		Jun3dPoint aPoint = (Jun3dPoint) aLine.atT_(aRadius / aDistance);
		aProjection.eyePoint_(aPoint);
		aProjection.zoomHeight_(this.computeZoomHeight());
		this.openGLProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Menu message: Toggle a line smooth effect on a projector. 
	 * 
	 * @category menu messages
	 */
	public void lineSmooth() {
		this.openGLProjector().lineSmooth_(!this.openGLProjector().lineSmooth());
		this.updateViewMenuIndication();
		this.changed_($("smoothing"));
	}

	/**
	 * Menu message: Open the light #1.
	 * 
	 * @category menu messages
	 */
	public void openLight1() {
		JunOpenGLDisplayLight displayLight = this.displayLight1();
		if (displayLight.builder().windows().length > 0) {
			displayLight._windowExpandAndRaise();
		} else {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #2.
	 * 
	 * @category menu messages
	 */
	public void openLight2() {
		JunOpenGLDisplayLight displayLight = this.displayLight2();
		if (displayLight.builder().windows().length > 0) {
			displayLight._windowExpandAndRaise();
		} else {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #3.
	 * 
	 * @category menu messages
	 */
	public void openLight3() {
		JunOpenGLDisplayLight displayLight = this.displayLight3();
		if (displayLight.builder().windows().length > 0) {
			displayLight._windowExpandAndRaise();
		} else {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #4.
	 * 
	 * @category menu messages
	 */
	public void openLight4() {
		JunOpenGLDisplayLight displayLight = this.displayLight4();
		if (displayLight.builder().windows().length > 0) {
			displayLight._windowExpandAndRaise();
		} else {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #5.
	 * 
	 * @category menu messages
	 */
	public void openLight5() {
		JunOpenGLDisplayLight displayLight = this.displayLight5();
		if (displayLight.builder().windows().length > 0) {
			displayLight._windowExpandAndRaise();
		} else {
			displayLight.openAmbientLight();
		}
	}

	/**
	 * Menu message: Toggle a polygon smooth effect on a projector.
	 * 
	 * @category menu messages
	 */
	public void polygonSmooth() {
		this.openGLProjector().polygonSmooth_(!this.openGLProjector().polygonSmooth());
		this.updateViewMenuIndication();
		this.changed_($("smoothing"));
	}

	/**
	 * Menu message: Reset the view.
	 * 
	 * @category menu messages
	 */
	public void resetView() {
		this.flushProjector();
		this.fit();
		this.updateMenuIndication();
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "View" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createViewMenu() {
		StMenu viewMenu = new StMenu($String("View"), $("viewMenu"));

		// Background color
		viewMenu.add(new StMenuItem($String("Background color"), new MenuPerformer(this, "changeBackgroundColor")));

		viewMenu.addSeparator();

		// Zooming
		StMenu zoomingMenu = new StMenu($String("Zooming"), $("zoomingMenu"));
		zoomingMenu.add(new StMenuItem($String("Fit zoom"), new MenuPerformer(this, "fitZoom")));
		zoomingMenu.add(new StMenuItem($String("Close up"), new MenuPerformer(this, "closeUp")));
		zoomingMenu.addSeparator();
		zoomingMenu.add(new StMenuItem($String("Zoom"), new MenuPerformer(this, "zoom")));
		zoomingMenu.add(new StMenuItem($String("Pan"), new MenuPerformer(this, "pan")));
		viewMenu.add(zoomingMenu);

		// Pointing
		StMenu pointingMenu = new StMenu($String("Pointing"), $("pointingMenu"));
		pointingMenu.add(new StMenuItem($String("Fit sight"), new MenuPerformer(this, "fitSight")));
		pointingMenu.addSeparator();
		pointingMenu.add(new StMenuItem($String("Eye point") + "...", new MenuPerformer(this, "changeEyePoint")));
		pointingMenu.add(new StMenuItem($String("Sight point") + "...", new MenuPerformer(this, "changeSightPoint")));
		pointingMenu.add(new StMenuItem($String("Up vector") + "...", new MenuPerformer(this, "changeUpVector")));
		pointingMenu.addSeparator();
		pointingMenu.add(new StMenuItem($String("Zoom height") + "...", new MenuPerformer(this, "changeZoomHeight")));
		pointingMenu.add(new StMenuItem($String("View factor") + "...", new MenuPerformer(this, "changeViewFactor")));
		viewMenu.add(pointingMenu);

		viewMenu.addSeparator();

		// Projection
		StMenu projectionMenu = new StMenu($String("Projection"), $("projectionMenu"));
		StRadioButtonGroup projectionGroup = new StRadioButtonGroup();
		projectionMenu.add(new StRadioButtonMenuItem($String("Perspective Projection", "Perspective"), $("perspectiveProjectionMenu"), projectionGroup, new MenuPerformer(this, "perspectiveProjection")));
		projectionMenu.add(new StRadioButtonMenuItem($String("Parallel Projection", "Parallel"), $("parallelProjectionMenu"), projectionGroup, new MenuPerformer(this, "parallelProjection")));
		viewMenu.add(projectionMenu);

		// Presentation
		StMenu presentationMenu = new StMenu($String("Presentation"), $("presentationMenu"));
		StRadioButtonGroup presentationGroup = new StRadioButtonGroup();
		presentationMenu.add(new StRadioButtonMenuItem($String("Solid Presentation", "Solid"), $("solidPresentationMenu"), presentationGroup, new MenuPerformer(this, "solidPresentation")));
		presentationMenu.add(new StRadioButtonMenuItem($String("Wireframe Presentation", "Wireframe"), $("wireframePresentationMenu"), presentationGroup, new MenuPerformer(this, "wireframePresentation")));
		presentationMenu.add(new StRadioButtonMenuItem($String("Hidden-line Presentation", "Hidden-line"), $("hiddenlinePresentationMenu"), presentationGroup, new MenuPerformer(this, "hiddenlinePresentation")));
		viewMenu.add(presentationMenu);

		// Shading
		StMenu shadingMenu = new StMenu($String("Shading"), $("shadingMenu"));
		StRadioButtonGroup shadingGroup = new StRadioButtonGroup();
		shadingMenu.add(new StRadioButtonMenuItem($String("Flat Shading", "Flat"), $("flatShadingMenu"), shadingGroup, new MenuPerformer(this, "flatShading")));
		shadingMenu.add(new StRadioButtonMenuItem($String("Smooth Shading", "Smooth"), $("smoothShadingMenu"), shadingGroup, new MenuPerformer(this, "smoothShading")));
		viewMenu.add(shadingMenu);

		// Smoothing
		StMenu smoothingMenu = new StMenu($String("Smoothing"), $("smoothingMenu"));
		smoothingMenu.add(new StCheckBoxMenuItem($String("Line smooth"), $("lineSmoothMenu"), new MenuPerformer(this, "lineSmooth")));
		smoothingMenu.add(new StCheckBoxMenuItem($String("Polygon smooth"), $("polygonSmoothMenu"), new MenuPerformer(this, "polygonSmooth")));
		viewMenu.add(smoothingMenu);

		// Smoothing angle
		viewMenu.add(new StMenuItem($String("Smoothing angle"), new MenuPerformer(this, "changeSmoothingAngle")));

		viewMenu.addSeparator();

		// and others.
		viewMenu.add(new StMenuItem($String("Reset"), $("resetMenu"), new MenuPerformer(this, "resetView")));
		viewMenu.add(this._createPresetProjectionMenu());

		return viewMenu;
	}

	/**
	 * Create a "Light" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createLightMenu() {
		StMenu lightMenu = new StMenu($String("Light"), $("lightMenu"));
		lightMenu.add(new StCheckBoxMenuItem($String("Light") + "1", $("light1"), new MenuPerformer(this, "openLight1")));
		lightMenu.add(new StCheckBoxMenuItem($String("Light") + "2", $("light2"), new MenuPerformer(this, "openLight2")));
		lightMenu.add(new StCheckBoxMenuItem($String("Light") + "3", $("light3"), new MenuPerformer(this, "openLight3")));
		lightMenu.add(new StCheckBoxMenuItem($String("Light") + "4", $("light4"), new MenuPerformer(this, "openLight4")));
		lightMenu.addSeparator();
		lightMenu.add(new StCheckBoxMenuItem($String("Ambient Light", "Ambient"), $("light5"), new MenuPerformer(this, "openLight5")));
		return lightMenu;
	}

	/**
	 * Create my preset menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createPresetProjectionMenu() {
		StMenu presetProjectionMenu = new StMenu($String("Preset"), $("presetMenu"));
		if (_presetProjections != null) {
			for (int i = 0; i < _presetProjections.length; i++) {
				String label = _presetProjections[i].description();
				if (label == null || label.length() == 0) {
					label = "(no name)";
				}
				StMenuItem aMenuItem = new StMenuItem(label, new MenuPerformer(this, "defaultProjection_", _presetProjections[i]));
				presetProjectionMenu.add(aMenuItem);
			}
		}
		return presetProjectionMenu;
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		this.updateViewMenuIndication();
		this.updateLightMenuIndication();
	}

	/**
	 * Update the view menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateViewMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu viewMenu = (StMenu) this._menuBar().atNameKey_($("viewMenu"));
		if (viewMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Zooming
		menuItem = viewMenu.atNameKey_($("zoomingMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Pointing
		menuItem = viewMenu.atNameKey_($("pointingMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Reset
		menuItem = viewMenu.atNameKey_($("resetMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Preset
		menuItem = viewMenu.atNameKey_($("presetMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty && _presetProjections != null && _presetProjections.length > 0);
		}

		// Projection
		menuItem = viewMenu.atNameKey_($("projectionMenu"));
		if (menuItem != null) {
			StMenu projectionMenu = (StMenu) menuItem;
			if (this.openGLProjector().projection() instanceof JunOpenGLPerspectiveProjection) {
				menuItem = projectionMenu.atNameKey_($("perspectiveProjectionMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else {
				menuItem = projectionMenu.atNameKey_($("parallelProjectionMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Presentation
		menuItem = viewMenu.atNameKey_($("presentationMenu"));
		if (menuItem != null) {
			StMenu presentationMenu = (StMenu) menuItem;
			if (this.openGLProjector().presentation() == $("solidPresentation")) {
				menuItem = presentationMenu.atNameKey_($("solidPresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else if (this.openGLProjector().presentation() == $("wireframePresentation")) {
				menuItem = presentationMenu.atNameKey_($("wireframePresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else if (this.openGLProjector().presentation() == $("hiddenlinePresentation")) {
				menuItem = presentationMenu.atNameKey_($("hiddenlinePresentationMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Shading
		menuItem = viewMenu.atNameKey_($("shadingMenu"));
		if (menuItem != null) {
			StMenu shadingMenu = (StMenu) menuItem;
			if (this.openGLProjector().shading() == $("flatShading")) {
				menuItem = shadingMenu.atNameKey_($("flatShadingMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			} else {
				menuItem = shadingMenu.atNameKey_($("smoothShadingMenu"));
				if (menuItem != null) {
					((StRadioButtonMenuItem) menuItem).beOn();
				}
			}
		}

		// Smoothing
		menuItem = viewMenu.atNameKey_($("smoothingMenu"));
		if (menuItem != null) {
			StMenu smoothingMenu = (StMenu) menuItem;
			menuItem = smoothingMenu.atNameKey_($("lineSmoothMenu"));
			if (menuItem != null) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.openGLProjector().lineSmooth());
			}
			menuItem = smoothingMenu.atNameKey_($("polygonSmoothMenu"));
			if (menuItem != null) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.openGLProjector().polygonSmooth());
			}
		}
	}

	/**
	 * Update the light menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateLightMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu lightMenu = (StMenu) this._menuBar().atNameKey_($("lightMenu"));
		if (lightMenu == null) {
			return;
		}

		for (int i = 1; i <= 5; i++) {
			StCheckBoxMenuItem menuItem = (StCheckBoxMenuItem) lightMenu.atNameKey_($("light" + i));
			menuItem.beSelected(this.displayLights()[i - 1].isOn());
		}
	}

	/**
	 * Answer my current default eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultEyePoint() {
		if (this.openGL3dObject() == null) {
			return new Jun3dPoint(10000, 10000, 10000);
		}

		Jun3dPoint defaultEyePoint = (Jun3dPoint) this.defaultProjectionTable().get($("eyePoint"));
		if (defaultEyePoint != null) {
			return defaultEyePoint;
		} else {
			Jun3dBoundingBox box = this.boundingBox();
			double distance = box.origin().distance_(box.corner());
			distance = distance * 2;
			return new Jun3dPoint(distance, distance, distance);
		}
	}

	/**
	 * Set my new default eye point.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public void defaultEyePoint_(Jun3dPoint eyePoint) {
		this.defaultProjectionTable().put($("eyePoint"), eyePoint);
	}

	/**
	 * Answer the default extent for converting to an image.
	 *
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultImageExtent() {
		return DefaultImageExtent;
	}

	/**
	 * Answer the default light color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	public Color defaultLightColor() {
		return Color.gray;
	}

	/**
	 * Answer the default light point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultLightPoint() {
		return new Jun3dPoint(0, 0, 1);
	}

	/**
	 * Answer my current default presentation.
	 *
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public StSymbol defaultPresentation() {
		StSymbol defaultPresentation = (StSymbol) this.defaultProjectionTable().get($("presentation"));
		if (defaultPresentation != null) {
			return defaultPresentation;
		} else {
			return $("solidPresentation");
		}
	}

	/**
	 * Set my new default presentation.
	 *
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public void defaultPresentation_(StSymbol aSymbol) {
		if (aSymbol == $("solidPresentation") || aSymbol == $("wireframePresentation") || aSymbol == $("hiddenlinePresentation")) {
			this.defaultProjectionTable().put($("presentation"), aSymbol);
		}
	}

	/**
	 * Answer my current default projection.
	 *
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public StSymbol defaultProjection() {
		StSymbol defaultProjection = (StSymbol) this.defaultProjectionTable().get($("projection"));
		if (defaultProjection != null) {
			return defaultProjection;
		} else {
			return JunOpenGLPerspectiveProjection.Type;
		}
	}

	/**
	 * Set my new default projection.
	 *
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public void defaultProjection_(StSymbol aSymbol) {
		if (aSymbol == JunOpenGLParallelProjection.Type || aSymbol == JunOpenGLPerspectiveProjection.Type) {
			this.defaultProjectionTable().put($("projection"), aSymbol);
		}
	}

	/**
	 * Set my default projection information.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category defaults
	 */
	public void defaultProjection_(JunOpenGLProjection aProjection) {
		this.defaultProjection_(aProjection.type());
		this.defaultEyePoint_(aProjection.eyePoint());
		this.defaultSightPoint_(aProjection.sightPoint());
		this.defaultUpVector_(aProjection.upVector());
		this.defaultViewFactor_(aProjection.viewFactor());
		this.defaultZoomHeight_(aProjection.zoomHeight());
		this.flushProjector();
		this.changed_($("projection"));
	}

	/**
	 * Answer my default projection table which holds some default values of the projection.
	 *
	 * @return java.util.Hashtable
	 * @category defaults
	 */
	public Map defaultProjectionTable() {
		if (defaultProjectionTable == null) {
			defaultProjectionTable = new HashMap();
		}
		return defaultProjectionTable;
	}

	/**
	 * Set my new default projection table.
	 *
	 * @param newDefaultProjectionTable java.util.Map
	 * @category defaults
	 */
	public void defaultProjectionTable_(Map newDefaultProjectionTable) {
		defaultProjectionTable = newDefaultProjectionTable;
	}

	/**
	 * Answer my current default shading.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public StSymbol defaultShading() {
		StSymbol defaultShading = (StSymbol) this.defaultProjectionTable().get($("shading"));
		return (defaultShading != null) ? defaultShading : $("smoothShading");
	}

	/**
	 * Set my new default shading.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public void defaultShading_(StSymbol aSymbol) {
		if (aSymbol == $("flatShading") || aSymbol == $("smoothShading")) {
			this.defaultProjectionTable().put($("shading"), aSymbol);
		}
	}

	/**
	 * Answer my current default sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultSightPoint() {
		if (this.openGL3dObject() == null) {
			return new Jun3dPoint(0, 0, 0);
		}

		Jun3dPoint defaultSightPoint = (Jun3dPoint) this.defaultProjectionTable().get($("sightPoint"));
		if (defaultSightPoint != null) {
			return defaultSightPoint;
		} else {
			return this.boundingBox().center();
		}
	}

	/**
	 * Set my new default sight point.
	 * 
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public void defaultSightPoint_(Jun3dPoint sightPoint) {
		this.defaultProjectionTable().put($("sightPoint"), sightPoint);
	}

	/**
	 * Answer my current default up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		Jun3dPoint defaultUpVector = (Jun3dPoint) this.defaultProjectionTable().get($("upVector"));
		if (defaultUpVector != null) {
			return defaultUpVector;
		} else {
			return new Jun3dPoint(-1, -1, 1);
		}
	}

	/**
	 * Set my new default up vector.
	 * 
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public void defaultUpVector_(Jun3dPoint upVector) {
		this.defaultProjectionTable().put($("upVector"), upVector);
	}

	/**
	 * Answer my current default view factor.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultViewFactor() {
		Number defaultViewFactor = (Number) this.defaultProjectionTable().get($("viewFactor"));
		if (defaultViewFactor != null) {
			return defaultViewFactor.doubleValue();
		} else {
			return 10;
		}
	}

	/**
	 * Set my new default view factor.
	 * 
	 * @param viewFactor double
	 * @category defaults
	 */
	public void defaultViewFactor_(double viewFactor) {
		this.defaultProjectionTable().put($("viewFactor"), new Double(viewFactor));
	}

	/**
	 * Answer my current default zoom height.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultZoomHeight() {
		Number defaultZoomHeight = (Number) this.defaultProjectionTable().get($("zoomHeight"));
		if (defaultZoomHeight != null) {
			return defaultZoomHeight.doubleValue();
		} else {
			return this.defaultZoomHeightFactor();
		}
	}

	/**
	 * Set my new default zoom height.
	 * 
	 * @param zoomHeight double
	 * @category defaults
	 */
	public void defaultZoomHeight_(double zoomHeight) {
		this.defaultProjectionTable().put($("zoomHeight"), new Double(zoomHeight));
	}

	/**
	 * Answer the default zoom height factor.
	 *
	 * @return double
	 * @category defaults
	 */
	public double defaultZoomHeightFactor() {
		return 2.5;
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		return (Jun3dPoint) this.boundingBox().center();
	}

	/**
	 * Compute the current zoom height.
	 * 
	 * @return double
	 * @category private
	 */
	protected double computeZoomHeight() {
		double depth = this.openGLProjection().distance();
		Jun3dPoint up = this.openGLProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, -1), depth);
		Jun3dPoint down = this.openGLProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, 1), depth);
		Jun3dPoint right = this.openGLProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(1, 0), depth);
		Jun3dPoint left = this.openGLProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(-1, 0), depth);
		final JunPlane horizontal = new JunPlane(this.openGLProjection().eyePoint(), left, right);
		final JunPlane vertical = new JunPlane(this.openGLProjection().eyePoint(), up, down);
		final double maxOffset[] = new double[] { 0.0d, 0.0d };
		this.pointsDo_(new StBlockClosure() {
			public Object value_(Object p) {
				double offsetX = vertical.distanceFromPoint_((Jun3dPoint) p);
				double offsetY = horizontal.distanceFromPoint_((Jun3dPoint) p);
				if (offsetX > maxOffset[0]) {
					maxOffset[0] = offsetX;
				}
				if (offsetY > maxOffset[1]) {
					maxOffset[1] = offsetY;
				}
				return null;
			}
		});
		double maxOffsetX = maxOffset[0];
		double maxOffsetY = maxOffset[1];

		double maxHeight = 0;
		JunOpenGL3dView aView = this.getOpenGLView();
		Rectangle aBox = (aView == null) ? null : aView.toComponent().getBounds();
		if (aBox != null && aBox.height != 0 && maxOffsetX != 0 && maxOffsetY != 0) {
			if (maxOffsetY * aBox.width / aBox.height < maxOffsetX) {
				double tmpViewHeight = (aBox.width * maxOffsetY * maxOffsetY) / (aBox.height * maxOffsetX);
				maxHeight = maxOffsetY * maxOffsetY / tmpViewHeight;
			} else {
				maxHeight = maxOffsetY;
			}
		} else {
			maxHeight = Math.max(maxOffsetX, maxOffsetY);
		}

		maxHeight = maxHeight * this.defaultZoomHeightFactor();
		return maxHeight;
	}

	/**
	 * Grab at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint grab3dPoint_(Jun2dPoint aPoint) {
		JunOpenGLProjection projection = this.openGLProjection();
		JunSphereSurface handleSphere = this.grabSphere();
		JunOpenGLParallelProjection prj = (JunOpenGLParallelProjection) projection.asParallelProjection();
		Jun3dLine line = new Jun3dLine(prj.translateTo3dPointInNearBoundaryPlaneFromPoint_(aPoint), prj.translateTo3dPointInFarBoundaryPlaneFromPoint_(aPoint));
		Jun3dPoint[] candidates = handleSphere.crossPointsWithLine_(line);
		if (candidates.length == 0) {
			return prj.translateTo3dPointFromPoint_(aPoint);
		} else {
			return candidates[0];
		}
	}

	/**
	 * Create a sphere for grabbing.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunSphereSurface
	 * @category private
	 */
	protected JunSphereSurface grabSphere() {
		JunOpenGLProjection projection = this.openGLProjection();
		JunSphereSurface grabSphere = new JunSphereSurface(projection.sightPoint(), projection.regularHeight() / 2);
		return grabSphere;
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		return this.getOpenGLView();
	}

	/**
	 * Answer my OpenGL view.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGL3dView
	 * @category private
	 */
	protected JunOpenGL3dView getOpenGLView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunOpenGL3dView && ((JunOpenGL3dView) each).getOpenGL3dModel() == this) {
				return (JunOpenGL3dView) each;
			}
		}
		return null;
	}

	/**
	 * Answer my OpenGL view with the drawable.
	 * 
	 * @param aDrawable jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @return jp.co.sra.jun.opengl.display.JunOpenGL3dView
	 * @category private
	 */
	protected JunOpenGL3dView getOpenGLViewWith(JunOpenGLDrawable aDrawable) {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunOpenGL3dView) {
				JunOpenGL3dView view = (JunOpenGL3dView) each;
				if (view.getOpenGLDrawable() == aDrawable) {
					return view;
				}
			}
		}
		return null;
	}

	/**
	 * Create an OpenGL drawable.
	 * 
	 * @param aClass java.lang.Class
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @category private
	 */
	protected JunOpenGLDrawable createOpenGLDrawable(Class aClass) {
		if (aClass != null && JunOpenGLDrawable.class.isAssignableFrom(aClass) == false) {
			System.err.println("The specified class is not a kind of JunOpenGLDrawable : " + aClass);
			aClass = null;
		}
		if (aClass == null) {
			aClass = JunOpenGLResource.GetDefaultOpenGLDrawableClass();
		}

		JunOpenGLDrawable aDrawable = null;
		try {
			aDrawable = (JunOpenGLDrawable) aClass.newInstance();
			aDrawable.setRenderer(this);
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
		return aDrawable;
	}

}
