package jp.co.sra.jun.opengl.chart;

import java.awt.Color;
import java.awt.Font;
import java.io.PrintWriter;
import java.io.StringWriter;

import jp.co.sra.smalltalk.StComposedText;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.goodies.font.JunFontModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;

/**
 * JunChartPie class
 * 
 *  @author    nisinaka
 *  @created   1999/01/14 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun642 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunChartPie.java,v 8.13 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunChartPie extends JunChartWithRoundShape {

	protected boolean hasLabels;

	/**
	 * Create a new instance of JunChartPie and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunChartPie() {
		super();
	}

	/**
	 * Create a new instance of a Chart and initialize it with the collection of sample.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category Instance creation
	 */
	public JunChartPie(java.util.Vector aCollectionOfSample) {
		super(aCollectionOfSample);
	}

	/**
	 * Create a new instance of a Chart and initialize it with the collection of sample and a number of keys.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @param numberOfKeys int
	 * @category Instance creation
	 */
	public JunChartPie(java.util.Vector aCollectionOfSample, int numberOfKeys) {
		super(aCollectionOfSample, numberOfKeys);
	}

	/**
	 * Create a new instance of a Chart and initialize it with the JunChartData.
	 * 
	 * @param aChartData jp.co.sra.jun.opengl.chart.JunChartData
	 * @category Instance creation
	 */
	public JunChartPie(JunChartData aChartData) {
		super(aChartData);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.chart.JunChartAbstract#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		hasLabels = true;
	}

	/**
	 * Show the labels.
	 * 
	 * @category accessing
	 */
	public void showLabels() {
		if (this.hasLabels() == false) {
			hasLabels = true;
			this.flushDisplayObject();
			this.changed_($("object"));
		}
	}

	/**
	 * Hide the labels.
	 * 
	 * @category accessing
	 */
	public void hideLabels() {
		if (this.hasLabels() == true) {
			hasLabels = false;
			this.flushDisplayObject();
			this.changed_($("object"));
		}
	}

	/**
	 * Set my new label height.
	 * 
	 * @param height double
	 * @see jp.co.sra.jun.opengl.chart.JunChartAbstract#labelHeight_(double)
	 * @category axes accessing
	 */
	public void labelHeight_(double height) {
		super.labelHeight_(height);

		if (this.hasLabels()) {
			this.flushDisplayObject();
			this.changed_($("object"));
		}
	}

	/**
	 * Makes the receiver to use the scalable labels.
	 * 
	 * @see jp.co.sra.jun.opengl.chart.JunChartAbstract#useScalableLabels()
	 * @category axes accessing
	 */
	public void useScalableLabels() {
		super.useScalableLabels();

		if (this.hasLabels()) {
			this.flushDisplayObject();
			this.changed_($("object"));
		}
	}

	/**
	 * Makes the receiver to use the texture labels.
	 * 
	 * @see jp.co.sra.jun.opengl.chart.JunChartAbstract#useTextureLabels()
	 * @category axes accessing
	 */
	public void useTextureLabels() {
		super.useTextureLabels();

		if (this.hasLabels()) {
			this.flushDisplayObject();
			this.changed_($("object"));
		}
	}

	/**
	 * Initialize the receiver with the attributes of the chart.
	 * 
	 * @param aChart jp.co.sra.jun.opengl.chart.JunChartPie
	 * @category copying
	 */
	protected void _copyAttributes(JunChartPie aChart) {
		super._copyAttributes(aChart);

		if (this.hasLabels()) {
			aChart.showLabels();
		} else {
			aChart.hideLabels();
		}
	}

	/**
	 * Answer true if the receiver has labels, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasLabels() {
		return hasLabels;
	}

	/**
	 * Create a JunOpenGL3dObject which represents the JunChartBar.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category private
	 */
	protected JunOpenGL3dCompoundObject createChart3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();

		JunChartDataSheet[] sheets = this.data().sheets();
		if (sheets == null) {
			return compoundObject;
		}

		for (int n = 0; n < sheets.length; n++) {
			JunChartDataSheet sheet = sheets[n];
			double z = this.intervalBetweenCharts() * n;
			Object[] values = sheet.valuesAtColumn_(0);
			double totalValue = 0;
			for (int i = 0; i < values.length; i++) {
				totalValue += ((Number) values[i]).doubleValue();
			}
			double degree = 90;
			for (int i = 0; i < values.length; i++) {
				double value = ((Number) values[i]).doubleValue();
				double theta = (360 * value) / totalValue;
				JunOpenGL3dObject pie = JunOpenGL3dObject.PieFrom_to_by_radius_thickness_(degree - theta, degree, 10, this.radius(), this.thickness());
				pie = pie.translatedBy_(new Jun3dPoint(0, 0, z));
				pie.paint_(this.nextColor());
				compoundObject.add_(pie);
				this.registerVisualObject_withValue_(pie, new Object[] { sheet.keysAtRow_(0), new Double(value) });
				if (this.hasLabels()) {
					String label = this.labelStringFor_within_(value, totalValue);
					Jun3dPoint point = new Jun3dPoint(Jun2dPoint.Rho_theta_(this.radius(), JunAngle._DegreesToRadians(degree - (theta / 2))), z);
					compoundObject.add_(this.label_at_(label, point));
				}
				degree -= theta;
			}
		}

		return compoundObject;
	}

	/**
	 * Answer the JunOpenGL3dObject as for a label at the specified point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject label_at_(String aString, Jun3dPoint aPoint) {
		if (useScalableLabels) {
			return this.scalableLabel_at_(aString, aPoint);
		} else {
			return this.texturedLabel_at_(aString, aPoint);
		}
	}

	/**
	 * Answer the JunOpenGL3dObject as for a scalable label at the specified point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject scalableLabel_at_(String aString, Jun3dPoint aPoint) {
		char[] characters = aString.toCharArray();
		int numberOfLines = 0;
		for (int i = 0; i < characters.length; i++) {
			if (characters[i] == '\n') {
				numberOfLines++;
			}
		}

		Font font = JunFontModel.TextStyle(100);
		StComposedText text = new StComposedText(aString, font);
		text.centered();
		JunOpenGL3dObject object = JunOpenGL3dObject.Text_(text);
		Jun3dBoundingBox box = object.boundingBox();
		double scale = 1 / Math.max(box.height() / this.labelHeight(), 1);
		scale *= (numberOfLines + 1);
		object = object.scaledBy_(new Jun3dPoint(scale, scale, 1));
		box = box.scaledBy_(new Jun3dPoint(scale, scale, 1));
		double margin = this.labelHeight() / 10;

		Jun2dPoint point = aPoint.as2dPoint();
		point = Jun2dPoint.Rho_theta_(point.r() + box.origin().distance_(box.corner()) / 2 + margin, point.theta());
		Jun3dPoint offset = new Jun3dPoint(point.x(), point.y(), aPoint.z()).minus_(box.center());
		object = object.translatedBy_(offset);
		box = box.translatedBy_(offset);
		object.paint_(this.labelForegroundColor());
		return object;
	}

	/**
	 * Answer the JunOpenGL3dObject as for a textured label at the specified point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject texturedLabel_at_(String aString, Jun3dPoint aPoint) {
		JunOpenGLTexture texture = JunOpenGLTexture.TextureForString_foreColor_backColor_(aString, this.labelForegroundColor(), this.labelBackgroundColor());
		char[] characters = aString.toCharArray();
		int numberOfLines = 0;
		for (int i = 0; i < characters.length; i++) {
			if (characters[i] == '\n') {
				numberOfLines++;
			}
		}
		double height = this.defaultLabelHeight() * (numberOfLines + 1);
		double width = (height * texture.width()) / texture.height();
		Jun2dPoint a2dPoint = new Jun2dPoint(aPoint.x(), aPoint.y());
		a2dPoint = Jun2dPoint.Rho_theta_(a2dPoint.rho() + (Math.sqrt((width * width) + (height * height)) / 2), a2dPoint.theta());
		Jun2dPoint origin = new Jun2dPoint(a2dPoint.x() - (width / 2), a2dPoint.y() - (height / 2));
		Jun2dPoint corner = new Jun2dPoint(a2dPoint.x() + (width / 2), a2dPoint.y() + (height / 2));
		Jun3dPoint[] points = { new Jun3dPoint(origin.x(), origin.y(), aPoint.z()), new Jun3dPoint(corner.x(), origin.y(), aPoint.z()), new Jun3dPoint(corner.x(), corner.y(), aPoint.z()), new Jun3dPoint(origin.x(), corner.y(), aPoint.z()) };
		JunOpenGL3dPolygon label = new JunOpenGL3dPolygon(points);
		label.paint_(Color.white);
		label.texture_(texture);
		return label;
	}

	/**
	 * Answer the label string for the value.
	 * 
	 * @param value double
	 * @param totalValue double
	 * @return java.lang.String
	 * @category private
	 */
	protected String labelStringFor_within_(double value, double totalValue) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println(value);
		pw.print('(');
		pw.print(Math.floor((value * 100) / totalValue));
		pw.print("%)");
		pw.flush();
		return sw.toString();
	}

}
