package jp.co.sra.jun.opengl.chart;

import java.awt.Point;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunChartData class
 * 
 *  @author    nisinaka
 *  @created   1998/11/19 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunChartData.java,v 8.12 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunChartData extends JunAbstractObject {

	protected Object[][][] samples = null;
	protected int numberOfKeys = -1;
	protected JunChartDataSheet[] sheets = null;
	protected double[] valueRange = null;
	protected double minimumValue = Double.NaN;
	protected double maximumValue = Double.NaN;
	protected String[] keyLabels = null;

	/**
	 * Create a new instance of JunChartData and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunChartData() {
		super();
	}

	/**
	 * Create a new instance of a ChartData and initialize it with the
	 * collection of sample.. This method corresponds to the class method
	 * "samples:" of JunChartData in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category Instance creation
	 */
	public JunChartData(Vector aCollectionOfSample) {
		this();
		this.samples_(aCollectionOfSample);
	}

	/**
	 * Create a new JunChartData with a collection of sample and a number of
	 * keys. This method corresponds to the class method
	 * "samples:numberOfKeys:" of JunChartData in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @param numberOfKeys int
	 * @category Instance creation
	 */
	public JunChartData(Vector aCollectionOfSample, int numberOfKeys) {
		this();
		this.samples_(aCollectionOfSample);
		this.numberOfKeys_(numberOfKeys);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		samples = null;
		numberOfKeys = -1;
		this.flushCaches();
	}

	/**
	 * Answer the array of key labels.
	 * 
	 * @return java.lang.String[]
	 * @category accessing
	 */
	public String[] keyLabels() {
		if (keyLabels == null) {
			keyLabels = this.defaultKeyLabels();
		}
		return keyLabels;
	}

	/**
	 * Set the array of key labels.
	 * 
	 * @param anArray java.lang.String[]
	 * @category accessing
	 */
	public void keyLabels_(String[] anArray) {
		keyLabels = anArray;
	}

	/**
	 * Answer the number of keys of the receiver.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfKeys() {
		if (numberOfKeys < 0) {
			numberOfKeys = this.defaultNumberOfKeys();
		}

		return numberOfKeys;
	}

	/**
	 * Set the number of keys of the receiver.
	 * 
	 * @param anInteger int
	 * @category accessing
	 */
	public void numberOfKeys_(int anInteger) {
		numberOfKeys = anInteger;
		this.flushCaches();
	}

	/**
	 * Answer the samples of the receiver.
	 * 
	 * @return java.lang.Object[][][]
	 * @category accessing
	 */
	public Object[][][] samples() {
		return samples;
	}

	/**
	 * Set the samples of the receiver.
	 * 
	 * @param aCollectionOfSample java.lang.Object[][][]
	 * @category accessing
	 */
	public void samples_(Object[][][] aCollectionOfSample) {
		samples = aCollectionOfSample;
	}

	/**
	 * Set the samples of the receiver.
	 * 
	 * @param aCollectionOfSamples java.util.Vector
	 * @category accessing
	 */
	public void samples_(Vector aCollectionOfSamples) {
		this.setSamples_(aCollectionOfSamples);
		this.flushCaches();
	}

	/**
	 * Answer the first sheet of all JunChartDataSheet.
	 * 
	 * @return jp.co.sra.jun.opengl.chart.JunChartDataSheet
	 * @category accessing
	 */
	public JunChartDataSheet sheet() {
		if (this.sheets() == null) {
			return null;
		}

		return this.sheets()[0];
	}

	/**
	 * Answer the array of JunChartDataSheet.
	 * 
	 * @return jp.co.sra.jun.opengl.chart.JunChartDataSheet[]
	 * @category accessing
	 */
	public JunChartDataSheet[] sheets() {
		if (sheets == null) {
			if (samples == null) {
				return null;
			}
			if (!this.hasValidSamples()) {
				return null;
			}

			int size = samples.length;
			sheets = new JunChartDataSheet[size];
			for (int i = 0; i < size; i++) {
				sheets[i] = new JunChartDataSheet(samples[i], this.numberOfKeys());
			}
		}
		return sheets;
	}

	/**
	 * Answer the interval of the value range.
	 * 
	 * @return double
	 * @category value accessing
	 */
	public double intervalOfValueRange() {
		return this.valueRange()[2];
	}

	/**
	 * Answer the maximum of the value range.
	 * 
	 * @return double
	 * @category value accessing
	 */
	public double maximumOfValueRange() {
		return this.valueRange()[1];
	}

	/**
	 * Answer the maximum value of the chart data.
	 * 
	 * @return double
	 * @category value accessing
	 */
	public double maximumValue() {
		if (Double.isNaN(maximumValue)) {
			final StValueHolder maximumValueHolder = new StValueHolder(Double.NEGATIVE_INFINITY);
			JunChartDataSheet[] sheets = this.sheets();
			for (int i = 0; i < sheets.length; i++) {
				sheets[i].valuesDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						double aNumber = ((Number) anObject).doubleValue();
						if (aNumber > maximumValueHolder._doubleValue()) {
							maximumValueHolder.value_(aNumber);
						}
						return null;
					}
				});
			}
			maximumValue = maximumValueHolder._doubleValue();
		}
		return maximumValue;
	}

	/**
	 * Answer the minimum of the value range.
	 * 
	 * @return double
	 * @category value accessing
	 */
	public double minimumOfValueRange() {
		return this.valueRange()[0];
	}

	/**
	 * Answer the minimum value of the chart data.
	 * 
	 * @return double
	 * @category value accessing
	 */
	public double minimumValue() {
		if (Double.isNaN(minimumValue)) {
			final StValueHolder minimumValueHolder = new StValueHolder(Double.MAX_VALUE);
			JunChartDataSheet[] sheets = this.sheets();
			for (int i = 0; i < sheets.length; i++) {
				sheets[i].valuesDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						double aNumber = ((Number) anObject).doubleValue();
						if (aNumber < minimumValueHolder._doubleValue()) {
							minimumValueHolder.value_(anObject);
						}
						return null;
					}
				});
			}
			minimumValue = minimumValueHolder._doubleValue();
		}
		return minimumValue;
	}

	/**
	 * Answer the information about the value range.
	 * 
	 * @return double[]
	 * @category value accessing
	 */
	public double[] valueRange() {
		if (valueRange == null) {
			valueRange = this.defaultValueRange();
		}
		return valueRange;
	}

	/**
	 * Set the information about the value range.
	 * 
	 * @param anArray double[]
	 * @category value accessing
	 */
	public void valueRange_(double[] anArray) {
		valueRange = anArray;
	}

	/**
	 * Set the information about the value range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @category value accessing
	 */
	public void valueRangeFrom_to_(double minimumNumber, double maximumNumber) {
		this.valueRangeFrom_to_by_(minimumNumber, maximumNumber, this.defaultIntervalFrom_to_(minimumNumber, maximumNumber));
	}

	/**
	 * Set the information about the value range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @param interval double
	 * @category value accessing
	 */
	public void valueRangeFrom_to_by_(double minimumNumber, double maximumNumber, double interval) {
		if (minimumNumber >= maximumNumber) {
			return;
		}

		double[] anArray = { minimumNumber, maximumNumber, interval };
		this.valueRange_(anArray);
	}

	/**
	 * Answer the default inverval appropriate between two numbers.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @return double
	 * @category defaults
	 */
	protected double defaultIntervalFrom_to_(double minimumNumber, double maximumNumber) {
		double difference = maximumNumber - minimumNumber;
		double interval = difference / 10;
		interval = Math.pow(10, Math.floor((Math.log(interval) / Math.log(10)) / 1) + 1);
		if ((difference / interval) < 5) {
			interval = interval / 2;
		}
		return interval;
	}

	/**
	 * Answer the array of the default key labels.
	 * 
	 * @return java.lang.String[]
	 * @category defaults
	 */
	protected String[] defaultKeyLabels() {
		String[] labels = new String[this.sheet().rowSize()];
		JunChartDataSheet sheet = this.sheet();
		if (sheet.keySize() == 1) {
			for (int i = 0; i < sheet.rowSize(); i++) {
				Object key = sheet.keyAtPoint_(new Point(i, 0));
				labels[i] = key.toString();
			}
		} else {
			for (int i = 0; i < sheet.rowSize(); i++) {
				labels[i] = String.valueOf(i);
			}
		}
		return labels;
	}

	/**
	 * Answer the default number of keys.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultNumberOfKeys() {
		return 0;
	}

	/**
	 * Answer the default value range.
	 *      0 : minimum value
	 *      1 : maximum value
	 *      2 : interval
	 * 
	 * @return double[]
	 * @category defaults
	 */
	protected double[] defaultValueRange() {
		double max = this.maximumValue();
		double min = this.minimumValue();
		if (min > 0) {
			min = 0;
		}
		double[] range = { min, max, this.defaultIntervalFrom_to_(min, max) };
		return range;
	}

	/**
	 * Flush the cached values.
	 * 
	 * @category flushing
	 */
	protected void flushCaches() {
		sheets = null;
		minimumValue = Double.NaN;
		maximumValue = Double.NaN;
	}

	/**
	 * Normalize the number in the value range.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category normalizing
	 */
	public double normalizeValue_(double aNumber) {
		double[] range = { 0, 1 };

		return this.mapNumber_fromRange_toRange_(aNumber, this.valueRange(), range);
	}

	/**
	 * Denormalize the number in the value range.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category normalizing
	 */
	public double denormalizeValue_(double aNumber) {
		double[] range = { 0, 1 };
		return this.mapNumber_fromRange_toRange_(aNumber, range, this.valueRange());
	}

	/**
	 * Answer a number in the range2 which corresponds to the number in the range1.
	 * 
	 * @param aNumber double
	 * @param numberRange1 double[]
	 * @param numberRange2 double[]
	 * @return double
	 * @category normalizing
	 */
	protected double mapNumber_fromRange_toRange_(double aNumber, double[] numberRange1, double[] numberRange2) {
		double min1 = numberRange1[0];
		double max1 = numberRange1[1];
		double min2 = numberRange2[0];
		double max2 = numberRange2[1];
		return ((aNumber - min1) * max2 - (max1 - aNumber) * min2) / (max1 - min1);
	}

	/**
	 * Print my string representation on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);

		BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
		bw.write('(');
		bw.newLine();
		bw.write("     samples :  ");
		if (samples == null) {
			bw.write("null");
		} else {
			for (int i = 0; i < samples.length; i++) {
				bw.newLine();
				bw.write("        ");
				Object[][] sample = samples[i];
				for (int j = 0; j < sample.length; j++) {
					Object[] array = sample[j];
					bw.write(array[0].toString());
					for (int k = 1; k < array.length; k++) {
						bw.write(" ");
						bw.write(array[k].toString());
					}
					if (j < (sample.length - 1)) {
						bw.write(", ");
					}
				}
			}
		}

		bw.newLine();
		bw.write("     numberOfKeys :  ");
		bw.write(String.valueOf(numberOfKeys));
		bw.newLine();
		bw.write(')');
		bw.flush();
	}

	/**
	 * Answer true if the receiver has a valid samples which can be converted into JunChartDataSheet, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasValidSamples() {
		if (samples == null) {
			return false;
		}

		for (int i = 0; i < samples.length; i++) {
			Object[][] sample = samples[i];
			for (int j = 0; j < sample.length; j++) {
				Object[] array = sample[j];
				if (this.numberOfKeys > array.length) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Set the instance variable of samples.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category private
	 */
	private void setSamples_(Vector aCollectionOfSample) {
		int numberOfSamples = aCollectionOfSample.size();
		samples = new Object[numberOfSamples][][];
		for (int i = 0; i < numberOfSamples; i++) {
			Vector collectionOfArray = (Vector) aCollectionOfSample.elementAt(i);
			int numberOfArrays = collectionOfArray.size();
			Object[][] sample = new Object[numberOfArrays][];
			collectionOfArray.copyInto(sample);
			samples[i] = sample;
		}
	}
}
