package jp.co.sra.jun.graphics.navigator;

import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.display.JunDisplayView;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.movie.framework.JunMovieHandle;
import jp.co.sra.jun.goodies.multimedia.JunMultiMediaBodyModel;
import jp.co.sra.jun.goodies.multimedia.JunMultiMediaImageModel;
import jp.co.sra.jun.goodies.multimedia.JunMultiMediaMovieModel;
import jp.co.sra.jun.goodies.multimedia.JunMultiMediaTextModel;
import jp.co.sra.jun.goodies.utilities.JunFileUtility;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMultiMediaNavigator class
 * 
 *  @author    nisinaka
 *  @created   2004/02/09 (by nisinaka)
 *  @updated   2006/11/29 (by m-asada)
 *  @updated   2007/03/14 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun678 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMultiMediaNavigator.java,v 8.13 2008/02/20 06:32:16 nisinaka Exp $
 */
public class JunMultiMediaNavigator extends JunApplicationModel {
	protected JunFileNavigator fileModel;
	protected JunMultiMediaTextModel textModel;
	protected JunMultiMediaImageModel imageModel;
	protected JunMultiMediaMovieModel movieModel;
	protected JunMultiMediaBodyModel bodyModel;
	protected boolean isMoviePlaying;

	/**
	 * Create a new instance of <code>JunMultiMediaNavigator</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunMultiMediaNavigator() {
		this(JunFileModel.DefaultDirectory());
	}

	/**
	 * Create a new instance of <code>JunMultiMediaNavigator</code> and initialize it.
	 * 
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunMultiMediaNavigator(File aFile) {
		super();
		this.currentFile_(aFile);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release 
	 */
	protected void initialize() {
		super.initialize();
		fileModel = null;
		textModel = null;
		imageModel = null;
		movieModel = null;
		bodyModel = null;
		isMoviePlaying = false;
	}

	/**
	 * Release the resources of the <code>JunMultiMediaNavigator</code>.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.fileModel().release();
		this.movieModel().release();
		this.imageModel().release();
		this.textModel().release();
		this.bodyModel().release();
		super.release();
	}

	/**
	 * Answer my current file.
	 * 
	 * @category accessing
	 */
	public File currentFile() {
		return this.fileModel().currentFile();
	}

	/**
	 * Set my current file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void currentFile_(File aFile) {
		this.fileModel().currentFile_(aFile);
	}

	/**
	 * Answer the string representation of my current file. 
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	protected String currentFileAsString() {
		return this.fileModel().currentFileAsString();
	}

	/**
	 * Answer my file model.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunFileNavigator
	 * @category accessing
	 */
	public JunFileNavigator fileModel() {
		if (fileModel == null) {
			fileModel = new JunFileNavigator();
			fileModel.currentFileHolder().compute_(new StBlockClosure() {
				public Object value_(Object object) {
					File file = (File) object;
					JunMultiMediaNavigator.this.checkFile_(file);
					return null;
				}
			});
		}
		return fileModel;
	}

	/**
	 * Answer my text model.
	 * 
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaTextModel
	 * @category accessing
	 */
	public JunMultiMediaTextModel textModel() {
		if (textModel == null) {
			textModel = new JunMultiMediaTextModel();
		}
		return textModel;
	}

	/**
	 * Answer my image model.
	 * 
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaImageModel
	 * @category accessing
	 */
	public JunMultiMediaImageModel imageModel() {
		if (imageModel == null) {
			imageModel = new JunMultiMediaImageModel();
		}
		return imageModel;
	}

	/**
	 * Answer my movie model.
	 * 
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaMovieModel
	 * @category accessing
	 */
	public JunMultiMediaMovieModel movieModel() {
		if (movieModel == null) {
			movieModel = new JunMultiMediaMovieModel();
			movieModel.trackerModel().loopCondition_($("loop"));
			movieModel.trackerModel().enableMarkers();
		}
		return movieModel;
	}

	/**
	 * Answer my body model.
	 * 
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaBodyModel
	 * @category accessing
	 */
	public JunMultiMediaBodyModel bodyModel() {
		if (bodyModel == null) {
			bodyModel = new JunMultiMediaBodyModel();
		}
		return bodyModel;
	}

	/**
	 * Check the file.
	 * 
	 * @param aFile java.io.File
	 * @see jp.co.sra.jun.graphics.navigator.JunFileNavigator#checkFile_(java.io.File)
	 * @category actions
	 */
	protected void checkFile_(File aFile) {
		if (this.fileModel().centerList().getView() == null) {
			return;
		}
		this.fileModel().checkFile_(aFile);

		if (this.movieModel().getView() == null) {
			return;
		}

		if (this.movieModel().getView().toComponent().isVisible()) {
			isMoviePlaying = this.movieModel().isPlay();
			if (isMoviePlaying) {
				this.movieModel().stop();
			}
		}

		this.movieModel().beInvisible();
		this.imageModel().beInvisible();
		this.textModel().beInvisible();
		this.bodyModel().beInvisible();
		this.fileModel().updateCurrentFile();

		if (aFile == null || aFile.exists() == false || aFile.isDirectory()) {
			return;
		}

		String extensionString = JunFileUtility.ExtensionStringOf_(aFile);
		if (extensionString == null) {
			extensionString = new String();
		}

		if (Arrays.asList(JunSystem.DefaultMovieExtensions()).contains(extensionString)) {
			this.checkMovieFile(aFile);
		} else if (Arrays.asList(JunSystem.DefaultSoundExtensions()).contains(extensionString)) {
			this.checkMovieFile(aFile);
		} else if (Arrays.asList(JunSystem.DefaultImageExtensions()).contains(extensionString)) {
			this.checkImageFile(aFile);
		} else if (Arrays.asList(JunSystem.DefaultBodyExtensions()).contains(extensionString)) {
			this.checkBodyFile(aFile);
		} else if (Arrays.asList(JunSystem.DefaultTextExtensions()).contains(extensionString)) {
			this.checkTextFile(aFile);
		} else {
			String aString = null;
			int limitSize = 1024 * 1024 * 4;
			long fileSize = aFile.length();
			if (fileSize > limitSize) {
				String message = $String("The file is large: <1p> bytes.", "The file is large: " + fileSize + " bytes.", Long.toString(fileSize));
				message += "\n";
				message += $String("Would you like to use dump on the head?");
				if (JunDialog.Confirm_(message, true)) {
					aString = (String) (new JunMultiMediaTextModel(aFile, limitSize)).textHolder().value();
				}
			}
			if (aString == null) {
				aString = (String) (new JunMultiMediaTextModel(aFile)).textHolder().value();
			}
			this.textModel().textHolder().value_(aString);
			this.textModel().beVisible();
		}
	}

	/**
	 * Check for the body file.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void checkBodyFile(File aFile) {
		JunOpenGL3dObject aBody = null;
		try {
			aBody = new JunMultiMediaBodyModel(aFile).displayObject();
		} catch (IllegalArgumentException e) {
		}
		if (aBody == null) {
			this.bodyModel().beInvisible();
			return;
		}
		this.bodyModel().displayObject_(aBody);
		this.bodyModel().beVisible();
		this.bodyModel().flushMovementVector();
		this.bodyModel().resetView();
	}

	/**
	 * Check for the movie file.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void checkMovieFile(File aFile) {
		JunMovieHandle aMovieHandle = this.movieModel().movieHandle().Filename_(aFile);
		if (aMovieHandle == null) {
			return;
		}

		this.movieModel().movieHandle_(aMovieHandle);
		this.movieModel().normalSize();
		this.movieModel().beVisible();
		this.movieModel().goto_(this.movieModel().now());
		if (this.movieModel().isIntervalNotEmpty()) {
			this.movieModel().trackerModel().lastMarker_(this.movieModel().trackerModel().lastMarker());
			this.movieModel().trackerModel().firstMarker_(this.movieModel().trackerModel().firstMarker());
		}
		if (isMoviePlaying) {
			this.movieModel().start();
		}
	}

	/**
	 * Check for the image file.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void checkImageFile(File aFile) {
		StDisplayable anImage = (new JunMultiMediaImageModel(aFile)).visual();
		JunDisplayView aView = ((JunDisplayView) this.imageModel().getView());
		if (StInputState.Default().shiftDown()) {
			anImage = JunImageAdjuster.AdjustImage_extent_keepAspect_alignmentSymbol_(anImage.asImage(), aView._getViewportSize(), true, $("center"));
		}
		if (StInputState.Default().altDown()) {
			anImage = JunImageAdjuster.AdjustImage_extent_keepAspect_(anImage.asImage(), aView._getViewportSize(), true);
		}

		this.imageModel().visual_(anImage);
		this.imageModel().beVisible();
	}

	/**
	 * Check for the text file.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void checkTextFile(File aFile) {
		String aString = (String) (new JunMultiMediaTextModel(aFile)).textHolder().value();
		this.textModel().textHolder().value_(aString);
		this.textModel().beVisible();
	}

	/**
	 * Computes aBlock with the receiver's value as the argument.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public void compute_(StBlockClosure aBlock) {
		this.fileModel().currentFileHolder().compute_(aBlock);
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunMultiMediaNavigatorViewAwt(this);
		} else {
			return new JunMultiMediaNavigatorViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Multi-Media Navigator");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		this.fileModel().closeRequest();
		this.movieModel().closeRequest();
		this.imageModel().closeRequest();
		this.textModel().closeRequest();
		this.bodyModel().closeRequest();
		this.release();
		super.noticeOfWindowClose(e);
	}
}
