package jp.co.sra.jun.graphics.navigator;

import java.awt.Dialog;
import java.awt.event.WindowEvent;
import java.io.File;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSimpleDialog;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.files.JunFileModel.FileType;
import jp.co.sra.jun.system.framework.JunDialog;

/**
 * JunFileRequesterDialog class
 * 
 *  @author    nisinaka
 *  @created   2004/04/12 (by nisinaka)
 *  @updated   2006/11/29 (by m-asada)
 *  @updated   2007/03/19 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun656 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileRequesterDialog.java,v 8.11 2008/02/20 06:32:16 nisinaka Exp $
 */
public class JunFileRequesterDialog extends StSimpleDialog {
	protected static File PreviousPath = null;

	protected JunFileRequester fileModel;
	protected StBlockClosure acceptBlock;
	protected StBlockClosure cancelBlock;

	/**
	 * Create a new instance of <code>JunFileRequesterDialog</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFileRequesterDialog() {
		super();
	}

	/**
	 * Answer the current previous path.
	 *
	 * @return java.io.File
	 * @category Utilities
	 */
	protected static File PreviousPath() {
		return PreviousPath;
	}

	/**
	 * Set the new previous path.
	 *
	 * @param aFile java.io.File
	 * @category Utilities
	 */
	protected static void PreviousPath_(File aFile) {
		PreviousPath = aFile;
	}

	/**
	 * Flush the previos path.
	 * 
	 * @category Utilities
	 */
	protected static void FlushPreviousPath() {
		PreviousPath = null;
	}

	/**
	 * Open a file requester dialog to request a file.
	 * 
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request() {
		return Request(null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString) {
		return Request(messageString, null, null, null, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, FileType[] fileTypes, FileType initialFileType) {
		return Request(messageString, null, fileTypes, initialFileType, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @param acceptBlock jp.co.sra.smalltalk.StBlockClocure
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, File initialDirectory, File initialFile, FileType[] fileTypes, FileType initialFileType, StBlockClosure acceptBlock, StBlockClosure cancelBlock) {
		File aFile = null;
		if (initialDirectory != null) {
			aFile = initialDirectory;
		}
		if (aFile != null && initialFile != null) {
			aFile = new File(aFile, initialFile.getName());
		}
		return Request(messageString, aFile, fileTypes, initialFileType, acceptBlock, cancelBlock);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @param acceptBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, File initialFile, FileType[] fileTypes, FileType initialFileType, StBlockClosure acceptBlock, StBlockClosure cancelBlock) {
		JunFileRequesterDialog fileRequesterDialog = new JunFileRequesterDialog();
		fileRequesterDialog.fileModel().requestType_($("oldFile"));
		return fileRequesterDialog.request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(messageString, initialFile, fileTypes, initialFileType, acceptBlock, cancelBlock);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a directory.
	 * 
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory(String messageString) {
		return RequestDirectory(messageString, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a directory.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory(String messageString, File initialFile) {
		JunFileRequesterDialog fileRequesterDialog = new JunFileRequesterDialog();
		fileRequesterDialog.fileModel().requestType_($("oldDirectory"));
		return fileRequesterDialog.request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(messageString, initialFile, null, null, new StBlockClosure() {
			public Object value_(Object fileRequester) {
				File aFile = ((JunFileRequester) fileRequester).currentFile();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						return Boolean.TRUE;
					} else {
						message += aFile.getPath() + $String(" is not a directory.") + "\n";
						message += "(" + $String("It may be a file.") + ")\n";
						message += $String("Input a directory.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					}
				} else {
					String message = aFile.getPath() + $String(" does not exist.") + "\n";
					message += $String("Input a directory.");
					JunDialog.Warn_(message);
					return Boolean.FALSE;
				}
			}
		}, null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a file.
	 * 
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile() {
		return RequestFile($String("Select a file."), null, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString) {
		return RequestFile(messageString, null, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString, FileType[] fileTypes, FileType initialFileType) {
		return RequestFile(messageString, null, fileTypes, initialFileType);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString, File initialFile) {
		return RequestFile(messageString, initialFile, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString, File initialFile, FileType[] fileTypes, FileType initialFileType) {
		JunFileRequesterDialog fileRequesterDialog = new JunFileRequesterDialog();
		fileRequesterDialog.fileModel().requestType_($("oldFile"));
		return fileRequesterDialog.request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(messageString, initialFile, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object fileRequester) {
				File aFile = ((JunFileRequester) fileRequester).currentFile();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						message += aFile.getPath() + $String(" is not a file.") + "\n";
						message += "(" + $String("It may be a directory.") + ")\n";
						message += $String("Input a file.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					} else {
						return Boolean.TRUE;
					}
				} else {
					String message = aFile.getPath() + $String(" does not exist.") + "\n";
					message += $String("Input a file.");
					JunDialog.Warn_(message);
					return Boolean.FALSE;
				}
			}
		}, null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a new directory.
	 * 
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewDirectory(String messageString) {
		return RequestNewDirectory(messageString, null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a new directory.
	 * 
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewDirectory(String messageString, File initialDirectory) {
		JunFileRequesterDialog fileRequesterDialog = new JunFileRequesterDialog();
		fileRequesterDialog.fileModel().requestType_($("newDirectory"));
		File parentDirectory = null;
		if (initialDirectory != null) {
			fileRequesterDialog.fileModel().tailFileName_(initialDirectory.getName());
			parentDirectory = initialDirectory.getParentFile();
			while (parentDirectory != null && parentDirectory.exists() == false) {
				parentDirectory = parentDirectory.getParentFile();
			}

		}

		return fileRequesterDialog.request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(messageString, parentDirectory, null, null, new StBlockClosure() {
			public Object value_(Object fileRequester) {
				File aFile = ((JunFileRequester) fileRequester).currentFile();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						message += aFile.getPath() + $String(" already exists.") + "\n";
						message += $String("Do you want to go ahead?");
						return new Boolean(JunDialog.Confirm_(message, false));
					} else {
						message += aFile.getPath() + $String(" already exists as a file.") + "\n";
						message += $String("Input a directory.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					}
				} else {
					return Boolean.TRUE;
				}
			}
		}, null);
	}

	/**
	 * Open a file requester dialog with the specified message string to request a new file.
	 * 
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString) {
		return RequestNewFile(messageString, null, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a new file.
	 * 
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, FileType[] fileTypes, FileType initialFileType) {
		return RequestNewFile(messageString, null, fileTypes, initialFileType);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a new file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, File initialFile) {
		return RequestNewFile(messageString, initialFile, null, null);
	}

	/**
	 * Open a file requester dialog with the specified message string and others to request a new file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, File initialFile, FileType[] fileTypes, FileType initialFileType) {
		JunFileRequesterDialog fileRequesterDialog = new JunFileRequesterDialog();
		fileRequesterDialog.fileModel().requestType_($("newFile"));
		File parentDirectory = null;
		if (initialFile != null) {
			fileRequesterDialog.fileModel().tailFileName_(initialFile.getName());

			parentDirectory = initialFile.getParentFile();
			while (parentDirectory != null && parentDirectory.exists() == false) {
				parentDirectory = parentDirectory.getParentFile();
			}
		}
		return fileRequesterDialog.request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(messageString, parentDirectory, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object fileRequester) {
				File aFile = ((JunFileRequester) fileRequester).currentFile();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						message += aFile.getPath() + " is not a file.\n";
						message += "(" + $String("It may be a directory.") + ")\n";
						message += $String("Input a file.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					} else {
						message += aFile.getPath() + $String(" already exists.") + "\n";
						message += $String("Do you want to go ahead?");
						return new Boolean(JunDialog.Confirm_(message));
					}
				} else {
					return Boolean.TRUE;
				}
			}
		}, null);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		fileModel = null;
		acceptBlock = null;
		cancelBlock = null;
	}

	/**
	 * Answer my current file requester.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunFileRequester
	 * @category accessing
	 */
	public JunFileRequester fileModel() {
		if (fileModel == null) {
			fileModel = new JunFileRequester();
		}
		return fileModel;
	}

	/**
	 * Set my new block closure for accpeption.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	protected void acceptBlock_(StBlockClosure aBlock) {
		acceptBlock = aBlock;
	}

	/**
	 * Set my new block closure for cancellation.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	protected void cancelBlock_(StBlockClosure aBlock) {
		cancelBlock = aBlock;
	}

	/**
	 * Called when the accept button is pressed.
	 * 
	 * @return boolean
	 * @category actions
	 */
	protected boolean acceptButtonPressed() {
		if (acceptBlock != null) {
			return ((Boolean) acceptBlock.value_(this.fileModel())).booleanValue();
		}

		return true;

	}

	/**
	 * Called when the cancel button is pressed.
	 * 
	 * @return boolean
	 * @category actions
	 */
	protected boolean cancelButtonPressed() {
		if (cancelBlock != null) {
			return ((Boolean) cancelBlock.value_(this.fileModel())).booleanValue();
		}

		return true;
	}

	/**
	 * Answer my defualt view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFileRequesterDialogViewAwt(this);
		} else {
			return new JunFileRequesterDialogViewSwing(this);
		}
	}

	/**
	 * Answer my window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("File Dialog");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);
		this.fileModel().release();
	}

	/**
	 * A utility method to request a file.
	 * 
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @param acceptBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.io.File
	 * @category private
	 */
	protected File request_initialFile_fileTypes_initialFileType_onAccept_onCancel_(String messageString, File initialFile, FileType[] fileTypes, FileType initialFileType, StBlockClosure acceptBlock, StBlockClosure cancelBlock) {
		if (fileTypes != null) {
			this.fileModel().fileTypes_(fileTypes);
		}

		if (initialFileType == null) {
			FileType[] anArray = this.fileModel().fileTypes();
			if (anArray != null && anArray.length > 0) {
				initialFileType = anArray[0];
			}
		}
		this.fileModel().fileType().value_(initialFileType);

		if (initialFile == null) {
			if (PreviousPath != null && PreviousPath.exists()) {
				this.fileModel().currentFile_(PreviousPath);
			} else {
				FlushPreviousPath();
			}
		} else {
			if (initialFile.getParentFile() == null) {
				if (PreviousPath != null && PreviousPath.exists()) {
					initialFile = new File(PreviousPath, initialFile.getName());
					if (initialFile.exists() == false) {
						initialFile = PreviousPath;
					}
				} else {
					FlushPreviousPath();
				}
			}
			this.fileModel().currentFile_(initialFile);
		}

		Dialog aDialog = this.allButOpenDialog();
		if (messageString != null) {
			aDialog.setTitle(messageString);
		}
		if (acceptBlock != null) {
			this.acceptBlock_(acceptBlock);
		}
		if (cancelBlock != null) {
			this.cancelBlock_(cancelBlock);
		}

		_ShowAtMousePoint(aDialog);

		File theFile = null;
		if (this.accept()._booleanValue()) {
			theFile = this.fileModel().currentFile();
			if (theFile.exists() && theFile.isDirectory()) {
				PreviousPath = theFile;
			} else {
				PreviousPath = theFile.getParentFile();
			}
		}

		return theFile;
	}
}
