package jp.co.sra.jun.graphics.navigator;

import java.io.File;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.files.JunFileModel.FileType;
import jp.co.sra.jun.graphics.list.JunFileList;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileRequester class
 * 
 *  @author    nisinaka
 *  @created   2004/04/09 (by nisinaka)
 *  @updated   2006/11/29 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun686 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileRequester.java,v 8.14 2008/02/20 06:32:16 nisinaka Exp $
 */
public class JunFileRequester extends JunFileNavigator {
	protected StValueHolder fileType;
	protected FileType[] fileTypes;
	protected StSymbol requestType;
	protected StValueHolder tailFileNameHolder;

	/**
	 * Create a new instance of <code>JunFileRequester</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFileRequester() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		fileType = null;
		fileTypes = null;
		requestType = null;
		tailFileNameHolder = null;
		this.compute_(new StBlockClosure() {
			public Object value_(Object obj) {
				File aFile = (File) obj;
				JunFileRequester self = JunFileRequester.this;
				if (aFile != null && aFile.exists() && ((self.requestType() == $("newFile") && aFile.isDirectory() == false) || (self.requestType() == $("newDirectory") && aFile.isDirectory()))) {
					if (aFile.isDirectory()) {
						// do nothing. why???
					} else {
						self.tailFileName_(aFile.getName());
					}
				}
				return null;
			}
		});
	}

	/**
	 * Answer my current file.
	 * 
	 * @return java.io.File
	 * @see jp.co.sra.jun.graphics.navigator.JunFileNavigator#currentFile()
	 * @category accessing
	 */
	public File currentFile() {
		File aFile = super.currentFile();
		if (this.requestType() == $("oldFile") || this.requestType() == $("oldDirectory")) {
			if (aFile == null) {
				aFile = JunFileList._RootFile;
			}
		} else if (this.requestType() == $("newFile") || this.requestType() == $("newDirectory")) {
			if (this.tailFileName().length() > 0) {
				if (aFile != null && aFile.isDirectory() == false) {
					aFile = aFile.getParentFile();
				}
				aFile = new File(aFile, this.tailFileName());
			}
		}
		return aFile;
	}

	/**
	 * Answer my current file type.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder fileType() {
		if (fileType == null) {
			fileType = new StValueHolder(this.fileTypes()[0]);
			fileType.compute_(new StBlockClosure() {
				public Object value_(Object v) {
					JunFileRequester.this.filePatterns_(((FileType) v).extensions());
					return null;
				}
			});
		}
		return fileType;
	}

	/**
	 * Answer my current set of file types.
	 * 
	 * @return jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category accessing
	 */
	public FileType[] fileTypes() {
		if (fileTypes == null) {
			fileTypes = this.defaultFileTypes();
		}
		return fileTypes;
	}

	/**
	 * Set my new set of file types.
	 * 
	 * @param newFileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category accessing
	 */
	public void fileTypes_(FileType[] newFileTypes) {
		fileTypes = newFileTypes;
	}

	/**
	 * Answer my current request type.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol requestType() {
		if (requestType == null) {
			requestType = $("oldFile");
		}
		return requestType;
	}

	/**
	 * Set my new request type.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public void requestType_(StSymbol aSymbol) {
		if (Arrays.asList(new StSymbol[] { $("oldFile"), $("oldDirectory"), $("newFile"), $("newDirectory") }).contains(aSymbol) == false) {
			return;
		}

		if (aSymbol != this.requestType()) {
			requestType = aSymbol;
			if (aSymbol == $("oldDirectory") || aSymbol == $("newDirectory")) {
				this.filePatterns_(new String[0]);
			}
		}
	}

	/**
	 * Answer my current tail file name.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String tailFileName() {
		return (String) this.tailFileNameHolder().value();
	}

	/**
	 * Set my new tail file name.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void tailFileName_(String aString) {
		this.tailFileNameHolder().value_(aString);
	}

	/**
	 * Answer my tail file name holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	protected StValueHolder tailFileNameHolder() {
		if (tailFileNameHolder == null) {
			tailFileNameHolder = new StValueHolder(new String());
			tailFileNameHolder.compute_(new StBlockClosure() {
				public Object value_(Object v) {
					JunFileRequester.this.updateTailFileName();
					return null;
				}
			});
		}
		return tailFileNameHolder;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFileRequesterViewAwt(this);
		} else {
			return new JunFileRequesterViewSwing(this);
		}
	}

	/**
	 * Answer my default set of file types.
	 * 
	 * @return jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category defaults
	 */
	protected FileType[] defaultFileTypes() {
		FileType[] defaultFileTypes;
		if (this.requestType() == $("oldDirectory") || this.requestType() == $("newDirectory")) {
			defaultFileTypes = new FileType[] { new FileType($String("Directories"), new String[0]) };
		} else {
			defaultFileTypes = new FileType[] {
					FileType.All($String("All files")),
					new FileType($String("Text files"), JunSystem.DefaultTextExtensionPatterns()),
					new FileType($String("Image files"), JunSystem.DefaultImageExtensionPatterns()),
					new FileType($String("Movie files"), JunSystem.DefaultMovieExtensionPatterns()),
					new FileType($String("Sound files"), JunSystem.DefaultSoundExtensionPatterns()) };
		}

		return defaultFileTypes;
	}

	/**
	 * Change the file type to the one specified with the index. 
	 * 
	 * @param index int
	 * @category private
	 */
	protected void fileTypeChanged(int index) {
		this.fileType().value_(this.fileTypes()[index]);
	}

	/**
	 * Called when the tail file name is changed.
	 * 
	 * @category private
	 */
	protected void updateTailFileName() {
		// TODO: what to do with updateTailFileName()?
	}
}
