package jp.co.sra.jun.graphics.navigator;

import java.awt.Point;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StOpaqueImage;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.icon.JunOpaqueImageIcons;
import jp.co.sra.jun.goodies.utilities.JunFileUtility;
import jp.co.sra.jun.graphics.framework.JunGraphicModel;
import jp.co.sra.jun.graphics.framework.JunGraphicView;
import jp.co.sra.jun.graphics.list.JunFileList;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileNavigator class
 * 
 *  @author    nisinaka
 *  @created   2004/01/22 (by nisinaka)
 *  @updated   2006/11/29 (by m-asada)
 *  @updated   2007/03/19 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun656 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileNavigator.java,v 8.14 2008/02/20 06:32:16 nisinaka Exp $
 */
public class JunFileNavigator extends JunGraphicModel {
	protected JunEmbeddedCenterFileList centerList;
	protected JunEmbeddedLeftFileList leftList;
	protected JunEmbeddedRightFileList rightList;
	protected LinkedList accessHistory;
	protected StValueHolder currentFileHolder;
	protected File homeDirectory;
	protected JunButtonModel homeButton;
	protected StBlockClosure computeBlock;

	protected static Map Icon16x16Table;
	protected static Map Icon64x64Table;

	class State {
		File currentFile;
		File selectedFile;
		Point scrollOffset;

		/**
		 * Create a new instance of <code>State</code> and initialize it.
		 * 
		 * @param newCurrentFile java.io.File
		 * @param newSelectedFile java.io.File
		 * @param newScrollOffset java.awt.Point
		 * @category Instance creation
		 */
		protected State(File newCurrentFile, File newSelectedFile, Point newScrollOffset) {
			currentFile = newCurrentFile;
			selectedFile = newSelectedFile;
			scrollOffset = newScrollOffset;
		}

		/**
		 * Answer a string representation of the object. 
		 * 
		 * @return java.lang.String
		 * @see java.lang.Object#toString()
		 * @category printing
		 */
		public String toString() {
			return "current:[" + currentFile + "] selected: [" + selectedFile + "] offset: " + scrollOffset;
		}
	}

	/**
	 * Flush the icon tables.
	 * 
	 * @category Icons
	 */
	public static void FlushIconTables() {
		Icon16x16Table = null;
		Icon64x64Table = null;
	}

	/**
	 * Answer the table having icon 16x16.
	 * 
	 * @return java.uitl.Map
	 * @category Icons
	 */
	public static Map Icon16x16Table() {
		if (Icon16x16Table == null) {
			Icon16x16Table = new HashMap();
			String[] extensions = JunSystem.DefaultTextExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon16x16Table.put(extensions[i], JunOpaqueImageIcons.Text16x16());
			}
			extensions = JunSystem.DefaultImageExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon16x16Table.put(extensions[i], JunOpaqueImageIcons.Image16x16());
			}
			extensions = JunSystem.DefaultMovieExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon16x16Table.put(extensions[i], JunOpaqueImageIcons.Movie16x16());
			}
			extensions = JunSystem.DefaultSoundExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon16x16Table.put(extensions[i], JunOpaqueImageIcons.Sound16x16());
			}
			extensions = JunSystem.DefaultBodyExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon16x16Table.put(extensions[i], JunOpaqueImageIcons.Body16x16());
			}
		}
		return Icon16x16Table;
	}

	/**
	 * Answer the table having icon 64x64.
	 * 
	 * @return java.uitl.Map
	 * @category Icons
	 */
	public static Map Icon64x64Table() {
		if (Icon64x64Table == null) {
			Icon64x64Table = new HashMap();
			String[] extensions = JunSystem.DefaultTextExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon64x64Table.put(extensions[i], JunOpaqueImageIcons.Text64x64());
			}
			extensions = JunSystem.DefaultImageExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon64x64Table.put(extensions[i], JunOpaqueImageIcons.Image64x64());
			}
			extensions = JunSystem.DefaultMovieExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon64x64Table.put(extensions[i], JunOpaqueImageIcons.Movie64x64());
			}
			extensions = JunSystem.DefaultSoundExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon64x64Table.put(extensions[i], JunOpaqueImageIcons.Sound64x64());
			}
			extensions = JunSystem.DefaultBodyExtensions();
			for (int i = 0; i < extensions.length; i++) {
				Icon64x64Table.put(extensions[i], JunOpaqueImageIcons.Body64x64());
			}
		}
		return Icon64x64Table;
	}

	/**
	 * Answer an image of file icon from the specified file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.smalltalk.StOpaqueImage
	 * @category Icons
	 */
	public static StOpaqueImage ImageFileIconFromFile_(File aFile) {
		StOpaqueImage anImage = null;
		if (aFile.getParentFile() == null) {
			anImage = JunOpaqueImageIcons.Drive16x16();
		} else if (aFile.isDirectory()) {
			anImage = JunOpaqueImageIcons.FolderClose16x16();
		} else {
			String extensionString = JunFileUtility.ExtensionStringOf_(aFile);
			extensionString = (extensionString == null) ? "" : extensionString.toLowerCase();
			anImage = (StOpaqueImage) Icon16x16Table().get(extensionString);
			if (anImage == null) {
				anImage = JunOpaqueImageIcons.Other16x16();
			}
		}
		return anImage;
	}

	/**
	 * Answer an image of info icon from the specified file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.smalltalk.StOpaqueImage
	 * @category Icons
	 */
	public static StOpaqueImage ImageInfoIconFromFile_(File aFile) {
		StOpaqueImage anImage = null;
		if (aFile.isDirectory()) {
			anImage = JunOpaqueImageIcons.FolderClose64x64();
		} else {
			String extensionString = JunFileUtility.ExtensionStringOf_(aFile);
			extensionString = (extensionString == null) ? "" : extensionString.toLowerCase();
			anImage = (StOpaqueImage) Icon64x64Table().get(extensionString);
			if (anImage == null) {
				anImage = JunOpaqueImageIcons.Other64x64();
			}
		}
		return anImage;
	}

	/**
	 * Create a new instance of <code>JunFileNavigator</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFileNavigator() {
		super();
	}

	/**
	 * Create a new instance of <code>JunFileNavigator</code> and initialize it.
	 * 
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunFileNavigator(File aFile) {
		this();
		this.currentFile_(aFile);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		centerList = null;
		leftList = null;
		rightList = null;
		accessHistory = null;
		currentFileHolder = null;
		homeDirectory = null;
		homeButton = null;
		computeBlock = null;
	}

	/**
	 * Answer my center list.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunEmbeddedCenterFileList
	 * @category accessing
	 */
	public JunEmbeddedCenterFileList centerList() {
		if (centerList == null) {
			JunEmbeddedCenterFileList fileList = new JunEmbeddedCenterFileList();
			fileList.currentFile_(null);
			fileList.nullItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.centerNullItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.centerFileItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.directoryItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.centerDirectoryItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileNavigator_(this);
			centerList = fileList;
		}

		return centerList;
	}

	/**
	 * Answer my left list.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunEmbeddedLeftFileList
	 * @category accessing
	 */
	public JunEmbeddedLeftFileList leftList() {
		if (leftList == null) {
			JunEmbeddedLeftFileList fileList = new JunEmbeddedLeftFileList();
			fileList.nullItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.leftNullItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.leftFileItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.directoryItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.leftDirectoryItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileNavigator_(this);
			leftList = fileList;
		}

		return leftList;
	}

	/**
	 * Answer my right list.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunEmbeddedRightFileList
	 * @category accessing
	 */
	public JunEmbeddedRightFileList rightList() {
		if (rightList == null) {
			JunEmbeddedRightFileList fileList = new JunEmbeddedRightFileList();
			fileList.currentFile_(JunFileList._RootFile);
			fileList.nullItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.rightNullItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.rightFileItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.directoryItemBlock_(new StBlockClosure() {
				public Object value_(Object aFile) {
					JunFileNavigator.this.rightDirectoryItemPressed_((File) aFile);
					return null;
				}
			});
			fileList.fileNavigator_(this);
			rightList = fileList;
		}

		return rightList;
	}

	/**
	 * Answer my current file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File currentFile() {
		return (File) this.currentFileHolder().value();
	}

	/**
	 * Answer the string representation of my current file. 
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	protected String currentFileAsString() {
		if (this.currentFile() == null) {
			return "";
		}

		String aString = this.currentFile().getPath();
		if (this.currentFile().isDirectory() && aString.endsWith(File.separator) == false) {
			aString += File.separator;
		}
		return aString;
	}

	/**
	 * Set my current file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void currentFile_(File aFile) {
		if (aFile == null || aFile.exists() == false) {
			return;
		}

		File leftFile, leftSelectedFile, centerFile, centerSelectedFile, rightFile, rightSelectedFile;
		if (JunFileList._RootFile.equals(aFile)) {
			leftFile = null;
			leftSelectedFile = null;
			centerFile = null;
			centerSelectedFile = null;
			rightFile = JunFileList._RootFile;
			rightSelectedFile = null;
		} else if (Arrays.asList(JunFileList._RootFiles()).contains(aFile)) {
			leftFile = null;
			leftSelectedFile = null;
			centerFile = JunFileList._RootFile;
			centerSelectedFile = aFile;
			rightFile = aFile;
			rightSelectedFile = null;
		} else {
			leftFile = aFile.getParentFile().getParentFile();
			if (leftFile == null) {
				leftFile = JunFileList._RootFile;
			}
			leftSelectedFile = aFile.getParentFile();
			centerFile = aFile.getParentFile();
			centerSelectedFile = aFile;
			rightFile = aFile;
			rightSelectedFile = null;
		}
		this.leftListFile_selectedFile_scrollOffset_(leftFile, leftSelectedFile, new Point(0, 0));
		this.centerListFile_selectedFile_scrollOffset_(centerFile, centerSelectedFile, new Point(0, 0));
		this.rightListFile_selectedFile_scrollOffset_(rightFile, rightSelectedFile, new Point(0, 0));
		this.checkFile_(null);
		this.centerDirectoryItemPressed_(this.centerList().selectedFile());
		this.fixScrollingOffset();
	}

	/**
	 * Answer the string representation of the current status.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	protected String currentStatus() {
		if (this.currentFile() != null && this.currentFile().isFile()) {
			return "";
		} else if (this.isCurrentFileInaccessibleDirectory()) {
			return "access denied";
		}

		int numberOfDrives = 0;
		int numberOfDirectories = 0;
		int numberOfFiles = 0;
		File[] contents = this.rightList().currentContents();
		for (int i = 0; i < contents.length; i++) {
			if (contents[i].getParent() == null) {
				numberOfDrives++;
			} else if (contents[i].isDirectory()) {
				numberOfDirectories++;
			} else if (contents[i].isFile()) {
				numberOfFiles++;
			}
		}

		String aString = "";
		if (numberOfDrives > 0) {
			aString += numberOfDrives;
			aString += (numberOfDrives == 1) ? " drive" : " drives";
		}
		if (numberOfDirectories > 0) {
			if (aString.length() > 0) {
				aString += " and ";
			}
			aString += numberOfDirectories;
			aString += (numberOfDirectories == 1) ? " directory" : " directories";
		}
		if (numberOfFiles > 0) {
			if (aString.length() > 0) {
				aString += " and ";
			}
			aString += numberOfFiles;
			aString += (numberOfFiles == 1) ? " file" : " files";
		}
		return aString;
	}

	/**
	 * Answer true if the left list is not needed to be shown, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLeftEnd() {
		return this.centerList().isNull() && this.rightList().isRoot();
	}

	/**
	 * Answer true if the current file is an inaccessible directory, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCurrentFileInaccessibleDirectory() {
		File currentFile = this.currentFile();
		return (currentFile != null && currentFile.isDirectory() && currentFile.listFiles() == null);
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return this.currentFileHolder().compute_(aBlock);
	}

	/**
	 * Set whether to handle folders distinctively or not.
	 * 
	 * @param aBoolean boolean
	 * @category setting
	 */
	public void distinctFolder_(boolean aBoolean) {
		this.leftList().distinctFolder_(aBoolean);
		this.centerList().distinctFolder_(aBoolean);
		this.rightList().distinctFolder_(aBoolean);
	}

	/**
	 * Set the new file patterns.
	 * 
	 * @param stringPatternArray java.lang.String[]
	 * @category setting
	 */
	public void filePatterns_(String[] stringPatternArray) {
		this.leftList().filePatterns_(stringPatternArray);
		this.centerList().filePatterns_(stringPatternArray);
		this.rightList().filePatterns_(stringPatternArray);
	}

	/**
	 * Set the new sort key.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category setting
	 */
	public void sortKey_(StSymbol aSymbol) {
		this.leftList().sortKey_(aSymbol);
		this.centerList().sortKey_(aSymbol);
		this.rightList().sortKey_(aSymbol);
	}

	/**
	 * Action for the event when the center null item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void centerNullItemPressed_(File aFile) {
		return;
	}

	/**
	 * Action for the event when the center file item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void centerFileItemPressed_(File aFile) {
		return;
	}

	/**
	 * Action for the event when the center directory item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void centerDirectoryItemPressed_(File aFile) {
		if (aFile == null) {
			this.rightListFile_selectedFile_scrollOffset_(JunFileList._RootFile, null, new Point(0, 0));
			this.checkFile_(null);
			return;
		}

		this.memorizeIntoHistory();
		this.rightListFile_selectedFile_scrollOffset_(aFile, (aFile.isDirectory() ? this.rightList().selectedFile() : null), new Point(0, 0));
		this.checkFile_(null);
	}

	/**
	 * Update the center list.
	 * 
	 * @param aFile java.io.File
	 * @param selectedFile java.io.File
	 * @param scrollOffset java.awt.Point
	 * @category accessing
	 */
	protected void centerListFile_selectedFile_scrollOffset_(File aFile, File selectedFile, Point scrollOffset) {
		State state = this.recallFromHistory_(aFile, selectedFile, scrollOffset);
		this.centerList().currentFile_selectedFile_scrollOffset_(state.currentFile, state.selectedFile, state.scrollOffset);
	}

	/**
	 * Action for the event when the left null item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void leftNullItemPressed_(File aFile) {
		this.rightListFile_selectedFile_scrollOffset_(this.centerList().currentFile(), this.centerList().selectedFile(), this.centerList().scrollOffset());
		this.centerListFile_selectedFile_scrollOffset_(null, null, new Point(0, 0));
		this.checkFile_(null);
	}

	/**
	 * Action for the event when the left file item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void leftFileItemPressed_(File aFile) {
		return;
	}

	/**
	 * Action for the event when the left directory item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void leftDirectoryItemPressed_(File aFile) {
		if (aFile == null) {
			this.centerListFile_selectedFile_scrollOffset_(null, null, new Point(0, 0));
			this.rightListFile_selectedFile_scrollOffset_(null, null, new Point(0, 0));
			this.checkFile_(null);
			return;
		}

		this.memorizeIntoHistory();

		State leftState, centerState, rightState;
		if (aFile.isDirectory()) {
			if (this.leftList().isRoot()) {
				leftState = new State(null, null, new Point(0, 0));
				centerState = new State(this.leftList.currentFile(), aFile, this.leftList().scrollOffset());
				rightState = new State(aFile, this.centerList().selectedFile(), this.centerList().scrollOffset());
			} else {
				rightState = new State(aFile, this.centerList().selectedFile(), this.centerList().scrollOffset());
				centerState = new State(this.leftList().currentFile(), aFile, this.leftList().scrollOffset());
				leftState = new State(this.leftList().currentFile().getParentFile(), centerState.currentFile, new Point(0, 0));
				if (Arrays.asList(JunFileList._RootFiles()).contains(aFile.getParentFile())) {
					leftState.currentFile = JunFileList._RootFile;
				}
			}
		} else {
			rightState = new State(aFile, null, this.centerList().scrollOffset());
			centerState = new State(aFile.getParentFile(), aFile, this.leftList().scrollOffset());
			leftState = new State(centerState.currentFile.getParentFile(), centerState.currentFile, new Point(0, 0));
			if (Arrays.asList(JunFileList._RootFiles()).contains(aFile.getParentFile())) {
				leftState.currentFile = JunFileList._RootFile;
			}
		}

		this.leftListFile_selectedFile_scrollOffset_(leftState.currentFile, leftState.selectedFile, leftState.scrollOffset);
		this.centerListFile_selectedFile_scrollOffset_(centerState.currentFile, centerState.selectedFile, centerState.scrollOffset);
		this.rightListFile_selectedFile_scrollOffset_(rightState.currentFile, rightState.selectedFile, rightState.scrollOffset);
		this.checkFile_(null);
	}

	/**
	 * Update the left list.
	 * 
	 * @param aFile java.io.File
	 * @param selectedFile java.io.File
	 * @param scrollOffset java.awt.Point
	 * @category accessing
	 */
	protected void leftListFile_selectedFile_scrollOffset_(File aFile, File selectedFile, Point scrollOffset) {
		State state = this.recallFromHistory_(aFile, selectedFile, scrollOffset);
		this.leftList().currentFile_selectedFile_scrollOffset_(state.currentFile, state.selectedFile, state.scrollOffset);
	}

	/**
	 * Action for the event when the right null item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void rightNullItemPressed_(File aFile) {
		return;
	}

	/**
	 * Action for the event when the right file item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void rightFileItemPressed_(File aFile) {
		if (aFile == null) {
			return;
		}
		if (aFile.isDirectory()) {
			return;
		}

		this.rightList().currentFile_(aFile);
		this.rightList().changed_($("redisplay"));
		this.checkFile_(aFile);
		if (computeBlock != null) {
			computeBlock.value_(this.currentFile());
		}
	}

	/**
	 * Action for the event when the right directory item is pressed.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void rightDirectoryItemPressed_(File aFile) {
		if (aFile == null) {
			return;
		}

		this.memorizeIntoHistory();

		State leftState, centerState, rightState;
		if (aFile.isDirectory()) {
			rightState = new State(aFile, null, new Point(0, 0));
			centerState = new State(this.rightList().currentFile(), aFile, this.rightList.scrollOffset());
			leftState = new State(null, this.centerList().selectedFile(), this.centerList().scrollOffset());
			if (this.leftList() == null) {
				leftState.currentFile = this.centerList().currentFile();
			} else if (centerState.currentFile == null) {
				leftState.currentFile = null;
			} else if (Arrays.asList(JunFileList._RootFiles()).contains(centerState.currentFile)) {
				leftState.currentFile = JunFileList._RootFile;
			} else {
				leftState.currentFile = centerState.currentFile.getParentFile();
			}
		} else {
			rightState = new State(aFile, null, new Point(0, 0));
			centerState = new State(aFile.getParentFile(), rightState.currentFile, this.rightList().scrollOffset());
			leftState = new State(null, centerState.currentFile, this.centerList().scrollOffset());
			if (this.leftList() == null) {
				leftState.currentFile = this.centerList().currentFile();
			} else if (Arrays.asList(JunFileList._RootFiles()).contains(centerState.currentFile)) {
				leftState.currentFile = JunFileList._RootFile;
			} else {
				leftState.currentFile = centerState.currentFile.getParentFile();
			}
		}

		this.leftListFile_selectedFile_scrollOffset_(leftState.currentFile, leftState.selectedFile, leftState.scrollOffset);
		this.centerListFile_selectedFile_scrollOffset_(centerState.currentFile, centerState.selectedFile, centerState.scrollOffset);
		this.rightListFile_selectedFile_scrollOffset_(rightState.currentFile, rightState.selectedFile, rightState.scrollOffset);
		this.checkFile_(null);
	}

	/**
	 * Update the right list.
	 * 
	 * @param aFile java.io.File
	 * @param selectedFile java.io.File
	 * @param scrollOffset java.awt.Point
	 * @category accessing
	 */
	protected void rightListFile_selectedFile_scrollOffset_(File aFile, File selectedFile, Point scrollOffset) {
		State state = this.recallFromHistory_(aFile, selectedFile, scrollOffset);
		this.rightList().currentFile_selectedFile_scrollOffset_(state.currentFile, state.selectedFile, state.scrollOffset);
	}

	/**
	 * Answer the button model for the home button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel homeButton() {
		if (homeButton == null) {
			homeButton = new JunButtonModel(false, JunCursors.HomeCursorImage(), new StBlockClosure() {
				public Object value_(Object anObject) {
					JunFileNavigator.this.homeButtonAction();
					return null;
				}
			});
		}

		return homeButton;
	}

	/**
	 * Action for the event when the home button is pressed.
	 * 
	 * @category buttons
	 */
	protected void homeButtonAction() {
		this.currentFile_(JunFileModel.DefaultDirectory());
	}

	/**
	 * Answer the access history.
	 * 
	 * @return java.util.LinkedList
	 * @category history
	 */
	protected LinkedList accessHistory() {
		if (accessHistory == null) {
			accessHistory = new LinkedList();
		}
		return accessHistory;
	}

	/**
	 * Memorize the current state into the history.
	 * 
	 * @category history
	 */
	protected void memorizeIntoHistory() {
		this.memorizeIntoHistory_(new State(this.leftList().currentFile(), this.leftList().selectedFile(), this.leftList().scrollOffset()));
		this.memorizeIntoHistory_(new State(this.centerList().currentFile(), this.centerList().selectedFile(), this.centerList().scrollOffset()));
		this.memorizeIntoHistory_(new State(this.rightList().currentFile(), this.rightList().selectedFile(), this.rightList().scrollOffset()));
	}

	/**
	 * Memorize the specified state into the history.
	 * 
	 * @param aState jp.co.sra.jun.graphics.navigator.JunFileNavigator.State
	 * @category history
	 */
	protected void memorizeIntoHistory_(State aState) {
		if (aState == null) {
			return;
		}
		if (aState.currentFile == null) {
			return;
		}

		if (this.accessHistory().size() >= this.defaultHistorySize()) {
			this.accessHistory().removeLast();
		}
		this.accessHistory().addFirst(aState);
	}

	/**
	 * Recall the state from the history.
	 * 
	 * @param currentFile java.io.File
	 * @param selectedFile java.io.File
	 * @param scrollOffset java.awt.Point
	 * @return jp.co.sra.jun.graphics.navigator.JunFileNavigator.History
	 * @category history
	 */
	protected State recallFromHistory_(File currentFile, File selectedFile, Point scrollOffset) {
		if (selectedFile == null || new Point(0, 0).equals(scrollOffset)) {
			State theState = this.recallFromHistoryByKey_(currentFile);
			if (theState != null) {
				if (selectedFile == null) {
					selectedFile = theState.selectedFile;
				}
				if (new Point(0, 0).equals(scrollOffset)) {
					scrollOffset = theState.scrollOffset;
				}
			}
		}

		return new State(currentFile, selectedFile, scrollOffset);
	}

	/**
	 * Recall the specified history information by the key.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.navigator.JunFileNavigator.History
	 * @category history
	 */
	protected State recallFromHistoryByKey_(File aFile) {
		if (aFile == null || aFile.exists() == false) {
			return null;
		}

		State aState = null;
		State[] states = (State[]) this.accessHistory().toArray(new State[this.accessHistory().size()]);
		for (int i = 0; i < states.length; i++) {
			if (aFile.equals(states[i].currentFile)) {
				aState = states[i];
				break;
			}
		}

		if (aState == null || aState.selectedFile == null || aState.selectedFile.exists() == false) {
			return null;
		}

		return aState;
	}

	/**
	 * Answer the value holder of the current file.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder currentFileHolder() {
		if (currentFileHolder == null) {
			currentFileHolder = new StValueHolder();
		}
		return currentFileHolder;
	}

	/**
	 * Update for the current file.
	 * 
	 * @category aspects
	 */
	protected void updateCurrentFile() {
		File newFile = this.centerList().currentFile();
		if (newFile != null && this.centerList().selectedFile() != null) {
			newFile = this.centerList().selectedFile();
		}

		File oldFile = (File) this.currentFileHolder().value();
		if (oldFile == null) {
			if (newFile == null) {
				return;
			}
		} else {
			if (oldFile.equals(newFile)) {
				return;
			}
		}

		this.currentFileHolder().value_(newFile);
		this.changed_with_($("current file"), newFile);
	}

	/**
	 * Check the file.
	 * 
	 * @param aFile java.io.File
	 * @category actions
	 */
	protected void checkFile_(File aFile) {
		this.updateCurrentFile();
	}

	/**
	 * Fix the scrolling offset.
	 * 
	 * @category scrolling
	 */
	public void fixScrollingOffset() {
		this.centerList().fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.centerList().getView(), false);
		this.leftList().fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.leftList().getView(), false);
		this.rightList().fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.rightList().getView(), false);
	}

	/**
	 * Answer the default history size.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultHistorySize() {
		return 256;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFileNavigatorViewAwt(this);
		} else {
			return new JunFileNavigatorViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("File Navigator");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);
		this.release();
	}
}
