package jp.co.sra.jun.graphics.list;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Panel;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.ScrollPane;
import java.awt.Scrollbar;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseEvent;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StApplicationWindowAwt;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StViewCanvas;
import jp.co.sra.smalltalk.WindowSpecLayout;

import jp.co.sra.jun.graphics.framework.JunGraphicController;
import jp.co.sra.jun.graphics.framework.JunGraphicModel;
import jp.co.sra.jun.graphics.framework.JunGraphicViewAwt;

/**
 * JunSequenceListViewAwt class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2006/11/22 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSequenceListViewAwt.java,v 8.20 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunSequenceListViewAwt extends JunGraphicViewAwt implements JunSequenceListView {
	protected VerticalScrollPane _scrollPane;

	/**
	 * Create a new instance of <code>JunSequenceListViewAwt</code> and initialize it.
	 * 
	 * @param aModel jp.co.sra.jun.graphics.list.JunSequenceList
	 * @category Instance creation
	 */
	public JunSequenceListViewAwt(JunSequenceList aModel) {
		super(aModel);
	}

	/**
	 * Initialize the receiver. 
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release 
	 */
	protected void initialize() {
		super.initialize();
		_scrollPane = null;
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new BorderLayout());
		this.add(this._scrollPane(), BorderLayout.CENTER);
		this._scrollPane().setSize(200, 150);
	}

	/**
	 * Answer the receiver's model as <code>JunSequenceList</code>.
	 * 
	 * @return jp.co.sra.jun.graphics.list.JunSequenceList
	 * @see jp.co.sra.jun.graphics.list.JunSequenceListView#getSequenceList()
	 * @category accessing
	 */
	public JunSequenceList getSequenceList() {
		return (JunSequenceList) this.model();
	}

	/**
	 * Make the specified rectangle area visible.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#makeVisible(java.awt.Rectangle)
	 * @category scrolling
	 */
	public void makeVisible(Rectangle aRectangle) {
		Rectangle clipBounds = new Rectangle(this._scrollPane().getScrollPosition(), this._scrollPane().getViewportSize());
		if (clipBounds.contains(aRectangle)) {
			return;
		}

		Point position = this._scrollPane().getScrollPosition();
		if (aRectangle.y + aRectangle.height >= clipBounds.y + clipBounds.height) {
			position.y = aRectangle.y - (clipBounds.height - aRectangle.height);
		}
		if (aRectangle.y <= clipBounds.y || aRectangle.height >= clipBounds.height) {
			position.y = aRectangle.y;
		}

		this._scrollPane().setScrollPosition(position);
	}

	/**
	 * Answer the receiver's current scroll offset.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollOffset()
	 * @category scrolling
	 */
	public Point scrollOffset() {
		return this._scrollPane().getScrollPosition();
	}

	/**
	 * Position the view to a particular place, updating the display.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollTo_(java.awt.Point)
	 * @category scrolling
	 */
	public void scrollTo_(Point aPoint) {
		if (this.scrollOffset().equals(aPoint)) {
			return;
		}

		this._scrollPane().setScrollPosition(aPoint);
	}

	/**
	 * Update the canvas size.
	 * 
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicViewAwt#updateCanvasSize()
	 * @category updating
	 */
	protected void updateCanvasSize() {
		this.getGraphicModel().recomposeIn_(new Rectangle(new Point(0, 0), this._scrollPane().getViewportSize()));
		this._scrollPane().doLayout();
	}

	/**
	 * Answer the graphic view.
	 * 
	 * @return jp.co.sra.smalltalk.StViewCanvas
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicViewAwt#canvas()
	 * @category user interface
	 */
	public StViewCanvas canvas() {
		if (canvas == null) {
			canvas = new StViewCanvas(this.getGraphicModel()) {
				/**
				 * Answer the receiver's default controller
				 * 
				 * @return jp.co.sra.smalltalk.StController
				 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
				 * @category controller accessing
				 */
				protected StController defaultController() {
					return new JunGraphicController() {
						public void mousePressed(MouseEvent event) {
							if (event.getButton() == MouseEvent.BUTTON1) {
								Point mousePoint = event.getPoint();
								Point scrollOffset = JunSequenceListViewAwt.this.scrollOffset();
								mousePoint.translate(scrollOffset.x, scrollOffset.y);
								((JunGraphicModel) this.model()).redButtonPressedAt_(mousePoint);
							} else {
								super.mousePressed(event);
							}
						}
					};
				}

				/**
				 * Display the receiver on the graphics.
				 * 
				 * @param aGraphics java.awt.Graphics
				 * @see jp.co.sra.smalltalk.StViewJPanel#displayOn_(java.awt.Graphics)
				 * @category displaying
				 */
				public void displayOn_(Graphics aGraphics) {
					if (this.isShowing()) {
						aGraphics.translate(0, -JunSequenceListViewAwt.this.scrollOffset().y);
						JunSequenceListViewAwt.this.displayCanvasOn_(aGraphics);
					}
				}

				/**
				 * Update the receiver according to the change notification from the model.
				 * 
				 * @param evt jp.co.sra.smalltalk.DependentEvent
				 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
				 * @category updating
				 */
				public void update_(DependentEvent evt) {
					if (evt.getAspect() == $("redisplay")) {
						this.popupMenu_(null);
					}

					super.update_(evt);
				}
			};
			canvas.setBackground(Color.white);
		}

		return canvas;
	}

	/**
	 * Answer the scroll pane.
	 * 
	 * @return java.awt.ScrollPane
	 * @category user interface
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicViewAwt#scrollPane()
	 * @deprecated useing _scrollPane()
	 * @category user interface
	 */
	protected ScrollPane scrollPane() {
		return null;
	}

	/**
	 * Answer the scroll pane.
	 * 
	 * @return jp.co.sra.jun.graphics.list.JunSequenceListViewAwt.VerticalScrollPane
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicViewAwt#scrollPane()
	 * @category user interface
	 */
	protected VerticalScrollPane _scrollPane() {
		if (_scrollPane == null) {
			_scrollPane = new VerticalScrollPane(this.canvas());
			_scrollPane.addComponentListener(new ComponentAdapter() {
				public void componentResized(ComponentEvent e) {
					JunSequenceListViewAwt.this.updateCanvasSize();
				}
			});
			this.updateCanvasSize();
		}
		return _scrollPane;
	}

	/**
	 * VerticalScrollPane class
	 */
	protected class VerticalScrollPane extends Panel implements AdjustmentListener {
		protected Panel borderPanel;
		protected Scrollbar verticalScrollBar;
		protected StViewCanvas viewport;

		/**
		 * Create a new instance of <code>VerticalScrollPane</code> and initialize it.
		 * 
		 * @param aCanvas jp.co.sra.smalltalk.StViewCanvas
		 * @category Instance creation
		 */
		public VerticalScrollPane(StViewCanvas aCanvas) {
			super();
			this.initialize();
			this.setViewport(aCanvas);

			int scrollBarWidth = this.getVerticalScrollBar().getWidth();
			if (scrollBarWidth == 0) {
				scrollBarWidth = this.defaultScrollBarWidth();
			}
			this.setLayout(new BorderLayout());
			this.add(this.getBorderPanel(), BorderLayout.CENTER);
			this.getBorderPanel().setLayout(new WindowSpecLayout());
			this.getBorderPanel().add(this.getVerticalScrollBar(), WindowSpecLayout.Position(1f, -scrollBarWidth, 0f, 0, 1f, 0, 1f, 0));
			this.getBorderPanel().add(this.getViewport(), WindowSpecLayout.Position(0f, 0, 0f, 0, 1f, -scrollBarWidth, 1f, 0));
		}

		/**
		 * Initialize the receiver.
		 * 
		 * @category initialize-release
		 */
		protected void initialize() {
			borderPanel = null;
			verticalScrollBar = null;
			viewport = null;
		}

		/**
		 * Do the layout on the scroll pane.
		 * 
		 * @see java.awt.Container#addNotify()
		 * @category actions
		 */
		public void addNotify() {
			super.addNotify();
			this.doLayout();
		}

		/**
		 * Determines which scrollbars to enable/disable.
		 * 
		 * @see java.awt.Container#doLayout()
		 * @category actions
		 */
		public void doLayout() {
			super.doLayout();

			Dimension viewportSize = this.getViewportSize();
			Dimension scrollableSize = this.getScrollableSize();
			int itemSize = JunSequenceListViewAwt.this.getSequenceList().sequenceMap().componentItems().size();
			int unitHeight = (itemSize > 0 && scrollableSize.height > viewportSize.height) ? Math.max(1, scrollableSize.height / itemSize) : 1;
			Scrollbar verticalScrollBar = this.getVerticalScrollBar();
			verticalScrollBar.setValues(verticalScrollBar.getValue(), viewportSize.height, 0, scrollableSize.height);
			verticalScrollBar.setBlockIncrement(Math.min(viewportSize.height, scrollableSize.height));
			verticalScrollBar.setUnitIncrement(unitHeight);
			verticalScrollBar.setEnabled(scrollableSize.height > viewportSize.height);
		}

		/**
		 * Answer the receiver's scrollable size.
		 * 
		 * @return java.awt.Dimension
		 * @category bounds accessing
		 */
		public Dimension getScrollableSize() {
			return JunSequenceListViewAwt.this.getSequenceList().sequenceMap().bounds().getSize();
		}

		/**
		 * Answer the receiver's viewport size.
		 * 
		 * @return java.awt.Dimension
		 * @category bounds accessing
		 */
		public Dimension getViewportSize() {
			Dimension size = this.getViewport().getSize();
			if (size.width > 0 || size.height > 0) {
				return size;
			}

			Rectangle bounds = this.getBounds();
			int scrollBarWidth = this.getVerticalScrollBar().getWidth();
			if (scrollBarWidth == 0) {
				scrollBarWidth = this.defaultScrollBarWidth();
			}
			return new Dimension(Math.max(bounds.width - scrollBarWidth, 0), bounds.height);
		}

		/**
		 * Answer the default vertical scroll bar width.
		 * 
		 * @return int
		 * @category defaults
		 */
		public int defaultScrollBarWidth() {
			return 16;
		}

		/**
		 * Invoked when the value of the adjustable has changed.
		 * 
		 * @param evt java.awt.event.AdjustmentEvent
		 * @see java.awt.event.AdjustmentListener#adjustmentValueChanged(java.awt.event.AdjustmentEvent)
		 * @category events
		 */
		public void adjustmentValueChanged(AdjustmentEvent evt) {
			this.getViewport().repaint();
		}

		/**
		 * Answer the current x,y position within the child which is displayed at the 0,0 location
		 * of the scrolled panel's view port.
		 * 
		 * @return java.awt.Point
		 * @category scrolling
		 */
		public Point getScrollPosition() {
			return new Point(0, this.getVerticalScrollBar().getValue());
		}

		/**
		 * Scrolls to the specified position within the child component.
		 * 
		 * @param aPoint java.awt.Point
		 * @category scrolling
		 */
		public void setScrollPosition(Point aPoint) {
			this.getVerticalScrollBar().setValue(aPoint.y);
			this.getViewport().repaint();
		}

		/**
		 * Answer the receiver's border panel.
		 * 
		 * @return java.awt.Panel
		 * @category user interface
		 */
		protected Panel getBorderPanel() {
			if (borderPanel == null) {
				borderPanel = StApplicationWindowAwt._PanelWithBorder();
			}
			return borderPanel;
		}

		/**
		 * Answer the receiver's vertical scroll bar.
		 * 
		 * @return java.awt.Scrollbar
		 * @category user interface
		 */
		public Scrollbar getVerticalScrollBar() {
			if (verticalScrollBar == null) {
				verticalScrollBar = new Scrollbar(Scrollbar.VERTICAL);
				verticalScrollBar.addAdjustmentListener(this);
				verticalScrollBar.setFocusable(false);
				verticalScrollBar.setEnabled(true);
			}
			return verticalScrollBar;
		}

		/**
		 * Answer the receiver's viewport.
		 * 
		 * @return jp.co.sra.smalltalk.StViewCanvas
		 * @category user interface
		 */
		public StViewCanvas getViewport() {
			return viewport;
		}

		/**
		 * Set the receiver's viewport.
		 * 
		 * @param aCanvas jp.co.sra.smalltalk.StViewCanvas
		 * @category user interface
		 */
		protected void setViewport(StViewCanvas aCanvas) {
			viewport = aCanvas;
		}
	}
}
