package jp.co.sra.jun.graphics.item;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.List;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.graphics.abstracts.JunAbstractItem;

/**
 * JunCompositeItem class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2003/12/17 (by Ryouichi Matsuda)
 *  @updated   2006/11/22 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCompositeItem.java,v 8.9 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunCompositeItem extends JunAbstractItem {
	// instanceVariableNames:
	protected List componentItems;
	protected StSymbol drawMode;

	/**
	 * Create a new instance of <code>JunCompositeItem</code> and initialize it.
	 * 
	 * @param originPoint java.awt.Point
	 * @param extentPoint java.awt.Point
	 * @category Instance creation
	 */
	public JunCompositeItem(Point originPoint, Point extentPoint) {
		super(originPoint, extentPoint);
	}

	/**
	 * Create a new instance of <code>JunCompositeItem</code> and initialize it.
	 * Only for the subclasses.
	 * 
	 * @category Instance creation
	 */
	protected JunCompositeItem() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		componentItems = null;
	}

	/**
	 * Answer the component items.
	 * 
	 * @return java.util.List
	 * @category accessing
	 */
	public List componentItems() {
		if (componentItems == null) {
			componentItems = new ArrayList();
		}
		return componentItems;
	}

	/**
	 * Set the component items.
	 * 
	 * @param anArray java.util.List
	 * @category accessing
	 */
	public void componentItems_(List anArray) {
		componentItems = anArray;
	}

	/**
	 * Answer the first item.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category accessing
	 */
	public JunAbstractItem firstItem() {
		if (this.componentItems().isEmpty()) {
			return null;
		}
		JunAbstractItem anItem = (JunAbstractItem) this.componentItems().get(0);
		return anItem;
	}

	/**
	 * Answer the item at the specified position in this components.
	 * 
	 * @param anIndex int
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category accessing
	 */
	public JunAbstractItem itemAt_(int anIndex) {
		if (this.componentItems().isEmpty()) {
			return null;
		}
		if (!(0 <= anIndex && anIndex < this.componentItems().size())) {
			return null;
		}
		JunAbstractItem anItem = (JunAbstractItem) this.componentItems().get(anIndex);
		return anItem;
	}

	/**
	 * Answer the last item.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category accessing
	 */
	public JunAbstractItem lastItem() {
		if (this.componentItems().isEmpty()) {
			return null;
		}
		JunAbstractItem anItem = (JunAbstractItem) this.componentItems().get(this.componentItems().size() - 1);
		return anItem;
	}

	/**
	 * Add the specified item.
	 * 
	 * @see jp.co.sra.jun.graphics.Abstract.JunAbstractMap#add_(jp.co.sra.jun.graphics.Abstract.JunAbstractItem)
	 * @category adding
	 */
	public void add_(JunAbstractItem anItem) {
		if (this.componentItems().contains(anItem)) {
			return;
		}
		this.componentItems().add(anItem);
	}

	/**
	 * Add the specified item with alignment symbol.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @category adding
	 */
	public void add_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem) {
		this.add_alignment_alignment_(anItem, alignmentSymbolOfItem, alignmentSymbolOfItem);
	}

	/**
	 * Add the specified item with alignment symbol and another item's alignment symbol.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @category adding
	 */
	public void add_alignment_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, StSymbol alignmentSymbolOfAnotherItem) {
		this.add_alignment_alignment_offset_(anItem, alignmentSymbolOfItem, alignmentSymbolOfAnotherItem, new Point(0, 0));
	}

	/**
	 * Add the specified item with alignment symbol and another item's alignment symbol and delta point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @category adding
	 */
	public void add_alignment_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, StSymbol alignmentSymbolOfAnotherItem, Point deltaPoint) {
		Rectangle aBox = this.arrange_alignment_alignment_offset_(anItem, alignmentSymbolOfItem, alignmentSymbolOfAnotherItem, deltaPoint);
		this.add_in_(anItem, aBox);
	}

	/**
	 * Add the specified item with alignment symbol and offset point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @category adding
	 */
	public void add_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, Point deltaPoint) {
		this.add_alignment_alignment_offset_(anItem, alignmentSymbolOfItem, alignmentSymbolOfItem, deltaPoint);
	}

	/**
	 * Add the specified item with alignment symbol and another item with another item's alignment symbol.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param anotherItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @category adding
	 */
	public void add_alignment_with_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, JunAbstractItem anotherItem, StSymbol alignmentSymbolOfAnotherItem) {
		this.add_alignment_with_alignment_offset_(anItem, alignmentSymbolOfItem, anotherItem, alignmentSymbolOfAnotherItem, new Point(0, 0));
	}

	/**
	 * Add the specified item with alignment symbol and another item with another item's alignment symbol and delta point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param anotherItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @category adding
	 */
	public void add_alignment_with_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, JunAbstractItem anotherItem, StSymbol alignmentSymbolOfAnotherItem, Point deltaPoint) {
		Rectangle aBox = this.arrange_alignment_with_alignment_offset_(anItem, alignmentSymbolOfItem, anotherItem, alignmentSymbolOfAnotherItem, deltaPoint);
		this.add_in_(anItem, aBox);
	}

	/**
	 * Add the specified item at point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param aPoint java.awt.Point
	 * @category adding
	 */
	public void add_at_(JunAbstractItem anItem, Point aPoint) {
		this.arrange_at_(anItem, aPoint);
		this.add_(anItem);
	}

	/**
	 * Add the specified item in specified rectangle.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param aRectangle java.awt.Rectangle
	 * @category adding
	 */
	public void add_in_(JunAbstractItem anItem, Rectangle aRectangle) {
		this.arrange_in_(anItem, aRectangle);
		this.add_(anItem);
	}

	/**
	 * Arrange item with specified alignment.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem) {
		return this.arrange_alignment_alignment_(anItem, alignmentSymbolOfItem, alignmentSymbolOfItem);
	}

	/**
	 * Arrange item with alignment symbol and alignment symbol of another item.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, StSymbol alignmentSymbolOfAnotherItem) {
		return this.arrange_alignment_alignment_offset_(anItem, alignmentSymbolOfItem, alignmentSymbolOfAnotherItem, new Point(0, 0));
	}

	/**
	 * Arrange item with alignment symbol and alignment symbol of another item and offset point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, StSymbol alignmentSymbolOfAnotherItem, Point deltaPoint) {
		try {
			StRectangle aBox = new StRectangle(anItem.bounds().getSize());
			Point _align = (Point) aBox.perform_(alignmentSymbolOfItem.toString());
			Point _with = (Point) new StRectangle(this.insideBounds().getSize()).perform_(alignmentSymbolOfAnotherItem.toString());
			aBox = aBox.align_with_(_align, _with);
			aBox = aBox.translatedBy_(deltaPoint.x, deltaPoint.y);
			return this.arrange_in_(anItem, aBox.toRectangle());
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Arrange item with alignment symbol and offset point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, Point deltaPoint) {
		return this.arrange_alignment_alignment_offset_(anItem, alignmentSymbolOfItem, alignmentSymbolOfItem, deltaPoint);
	}

	/**
	 * Arrange item with alignment symbol and another item with another item's alignment symbol.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param anotherItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_with_alignment_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, JunAbstractItem anotherItem, StSymbol alignmentSymbolOfAnotherItem) {
		return this.arrange_alignment_with_alignment_offset_(anItem, alignmentSymbolOfItem, anotherItem, alignmentSymbolOfAnotherItem, new Point(0, 0));
	}

	/**
	 * Arrange item with alignment symbol and another item with another item's alignment symbol and offset point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfItem jp.co.sra.smalltalk.StSymbol
	 * @param anotherItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param alignmentSymbolOfAnotherItem jp.co.sra.smalltalk.StSymbol
	 * @param deltaPoint java.awt.Point
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_alignment_with_alignment_offset_(JunAbstractItem anItem, StSymbol alignmentSymbolOfItem, JunAbstractItem anotherItem, StSymbol alignmentSymbolOfAnotherItem, Point deltaPoint) {
		try {
			StRectangle aBox = new StRectangle(anItem.bounds().getSize());
			Point _align = (Point) aBox.perform_(alignmentSymbolOfItem.toString());
			Point _with = (Point) new StRectangle(anotherItem.insideBounds()).perform_(alignmentSymbolOfAnotherItem.toString());
			aBox = aBox.align_with_(_align, _with);
			aBox = aBox.translatedBy_(deltaPoint.x, deltaPoint.y);
			return this.arrange_in_(anItem, aBox.toRectangle());
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Arrange item at point.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param aPoint java.awt.Point
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_at_(JunAbstractItem anItem, Point aPoint) {
		anItem.originPoint_(aPoint);
		return anItem.bounds();
	}

	/**
	 * Arrange item in rectangle.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @param aRectangle java.awt.Rectangle
	 * @return java.awt.Rectangle
	 * @category arranging
	 */
	public Rectangle arrange_in_(JunAbstractItem anItem, Rectangle aRectangle) {
		anItem.originPoint_(aRectangle.getLocation());
		anItem.extentPoint_(new Point(aRectangle.width, aRectangle.height));
		return anItem.bounds();
	}

	/**
	 * Display the receiver's background color on a graphics context inset the specified rectangle.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Rectangle
	 * @category displaying
	 */
	public void displayBackgroundOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		Graphics gc = graphicsContext.create();
		try {
			if (this.isEmphasis()) {
				gc.setColor(this.selectionBackgroundColor());
			} else {
				gc.setColor(this.backgroundColor());
			}
			gc.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}
	}

	/**
	 * Display the receiver's border on a graphics context inset the specified rectangle.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Rectangle
	 * @category displaying
	 */
	public void displayBorderOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		Graphics gc = graphicsContext.create();
		try {
			gc.setColor(this.borderColor());
			StRectangle[] areas = new StRectangle(this.bounds()).areasOutside_(new StRectangle(insetBox));
			for (int i = 0; i < areas.length; i++) {
				gc.drawLine(areas[i].originX(), areas[i].originY(), areas[i].cornerX() - 1, areas[i].cornerY() - 1);
			}
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}
	}

	/**
	 * Display the receiver's items on a graphics context inset the specified rectangle.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Graphics
	 * @category displaying
	 */
	public void displayComponentItemsOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		if (this.componentItems().isEmpty()) {
			return;
		}
		for (int i = 0; i < this.componentItems().size(); i++) {
			JunAbstractItem item = (JunAbstractItem) this.componentItems().get(i);
			item.displayOn_at_(graphicsContext, new Point(insetBox.x, insetBox.y));
		}
	}

	/**
	 * Display the receiver's items on a graphics context.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphicsContext) {
		Rectangle clipBounds = aGraphicsContext.getClipBounds();
		if (clipBounds == null || clipBounds.intersects(this.bounds())) {
			Rectangle insetBox = this.insideBounds();
			this.displayBackgroundOn_insetBox_(aGraphicsContext, insetBox);
			this.displayComponentItemsOn_insetBox_(aGraphicsContext, insetBox);
			this.displayBorderOn_insetBox_(aGraphicsContext, insetBox);
		}
	}

	/**
	 * Enumerate all items and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.graphics.Abstract.JunAbstractVisual#itemsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object itemsDo_(StBlockClosure aBlock) {
		for (int i = 0; i < this.componentItems().size(); i++) {
			aBlock.value_(this.componentItems().get(i));
		}
		return this;
	}

	/**
	 * Answer true if the receiver is composite object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#isComposite()
	 * @category testing
	 */
	public boolean isComposite() {
		return true;
	}
}
