package jp.co.sra.jun.graphics.framework;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StViewJPanel;

import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunGraphicViewSwing class
 * 
 *  @author    nisinaka
 *  @created   2004/01/19 (by nisinaka)
 *  @updated   2006/11/22 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun640 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGraphicViewSwing.java,v 8.16 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunGraphicViewSwing extends JunAbstractViewJPanel implements JunGraphicView {
	protected StViewJPanel canvas;
	protected JScrollPane scrollPane;

	/**
	 * Create a new instance of <code>JunGraphicViewSwing</code> and initialize it.
	 * 
	 * @param aModel jp.co.sra.jun.graphics.framework.JunGraphicModel
	 * @category Instance creation
	 */
	public JunGraphicViewSwing(JunGraphicModel aModel) {
		super(aModel);
	}

	/**
	 * Initialize the receiver. 
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release 
	 */
	protected void initialize() {
		super.initialize();
		canvas = null;
		scrollPane = null;
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new BorderLayout());
		this.add(this.scrollPane(), BorderLayout.CENTER);
		this.setPreferredSize(new Dimension(200, 150));
	}

	/**
	 * Answer the receiver's model as <code>JunGraphicModel</code>.
	 * 
	 * @return jp.co.sra.jun.graphics.framework.JunGraphModel
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#getGraphicModel()
	 * @category accessing
	 */
	public JunGraphicModel getGraphicModel() {
		return (JunGraphicModel) this.model();
	}

	/**
	 * Display the canvas part of the receiver on the graphics.
	 * The subclasses may override this.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayCanvasOn_(Graphics aGraphics) {
		if (this.canvas().isShowing() == false) {
			return;
		}

		Rectangle aRectangle = this.canvas().getBounds();
		aGraphics.setColor(this.canvas().getBackground());
		aGraphics.fillRect(aRectangle.x, aRectangle.y, aRectangle.width, aRectangle.height);

		StDisplayable displayObject = this.getGraphicModel().displayObject();
		if (displayObject != null) {
			displayObject.displayOn_(aGraphics);
		}
	}

	/**
	 * Set up the keyboard for the view on the window.
	 * 
	 * @param aWindow java.awt.Window
	 * @see jp.co.sra.smalltalk.StView#_setupKeyboard(java.awt.Window)
	 * @category keyboard
	 */
	public void _setupKeyboard(final Window aWindow) {
		KeyListener keyListener = this.getGraphicModel()._keyListener();
		if (keyListener == null) {
			return;
		}

		this.addKeyListener(keyListener);

		aWindow.addWindowListener(new WindowAdapter() {
			public void windowActivated(WindowEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						Component focusOwner = aWindow.getFocusOwner();
						if (focusOwner == null || focusOwner != JunGraphicViewSwing.this) {
							JunGraphicViewSwing.this.requestFocus();
						}
					}
				});
			}
		});
	}

	/**
	 * Make the specified rectangle area visible.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#makeVisible(java.awt.Rectangle)
	 * @category scrolling
	 */
	public void makeVisible(Rectangle aRectangle) {
		this.canvas().scrollRectToVisible(aRectangle);
	}

	/**
	 * Answer the receiver's current scroll offset.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollOffset()
	 * @category scrolling
	 */
	public Point scrollOffset() {
		return this.scrollPane().getViewport().getViewPosition();
	}

	/**
	 * Position the view to a particular place, updating the display.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollTo_(java.awt.Point)
	 * @category scrolling
	 */
	public void scrollTo_(Point aPoint) {
		if (this.scrollOffset().equals(aPoint)) {
			return;
		}

		this.scrollPane().getViewport().setViewPosition(aPoint);
	}

	/**
	 * Update the receiver according to the change notification from the model.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("redisplay")) {
			this.updateCanvasSize();
			return;
		}
		if (aSymbol == $("selection") && evt.getParameter() instanceof Rectangle) {
			this.getGraphicModel().recomposeIn_((Rectangle) evt.getParameter());
			this.canvas().repaint();
			return;
		}

		this.getGraphicModel().recomposeIn_(new Rectangle(new Point(0, 0), this.canvas().getSize()));
		// this.fixScrollingOffset();
		super.update_(evt);
	}

	/**
	 * Update the canvas size.
	 * 
	 * @category updating
	 */
	protected void updateCanvasSize() {
		this.getGraphicModel().recomposeIn_(this.scrollPane().getViewport().getViewRect());
		if (this.getGraphicModel().displayObject() != null) {
			this.canvas().setPreferredSize(this.getGraphicModel().displayObject().bounds().getSize());
		}
		this.scrollPane().getViewport().doLayout();
		this.canvas().repaint();
	}

	/**
	 * Answer the graphic view.
	 * 
	 * @return jp.co.sra.smalltalk.StViewJPanel
	 * @category user interface
	 */
	public StViewJPanel canvas() {
		if (canvas == null) {
			canvas = new StViewJPanel(this.getGraphicModel()) {
				/**
				 * Answer the receiver's default controller
				 * 
				 * @return jp.co.sra.smalltalk.StController
				 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
				 * @category controller accessing
				 */
				protected StController defaultController() {
					return new JunGraphicController();
				}

				/**
				 * Display the receiver on the graphics.
				 * 
				 * @param aGraphics java.awt.Graphics
				 * @see jp.co.sra.smalltalk.StViewJPanel#displayOn_(java.awt.Graphics)
				 * @category displaying
				 */
				public void displayOn_(Graphics aGraphics) {
					JunGraphicViewSwing.this.displayCanvasOn_(aGraphics);
				}
			};
			canvas.setBackground(Color.white);
		}

		return canvas;
	}

	/**
	 * Answer the scroll pane.
	 * 
	 * @return javax.swing.JScrollPane
	 * @category user interface
	 */
	protected JScrollPane scrollPane() {
		if (scrollPane == null) {
			scrollPane = new JScrollPane(this.canvas(), JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
			scrollPane.getHorizontalScrollBar().setFocusable(false);
			scrollPane.getVerticalScrollBar().setFocusable(false);
			scrollPane.addComponentListener(new ComponentAdapter() {
				public void componentResized(ComponentEvent e) {
					JunGraphicViewSwing.this.updateCanvasSize();
				}
			});
		}

		return scrollPane;
	}
}
