package jp.co.sra.jun.goodies.wheels;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunThumbWheelViewAwt class
 * 
 *  @author    He Weijie
 *  @created   1998/10/23 (by He Weijie)
 *  @updated   1999/12/17 (by MATSUDA Ryouichi)
 *  @updated   2000/11/16 (by Mitsuhiro Asada)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2007/03/13 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun651 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunThumbWheelViewAwt.java,v 8.12 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunThumbWheelViewAwt extends JunAbstractViewCanvas implements JunThumbWheelView {
	protected Image[] thumbWheelImages;
	protected int currentIndex;
	protected boolean isVertical;

	/**
	 * Create a new instance of <code>JunThumbWheelViewAwt</code> and initialize it.
	 * 
	 * @param newThumbWheel jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category Instance creation
	 */
	public JunThumbWheelViewAwt(JunThumbWheel newThumbWheel) {
		super(newThumbWheel);
		isVertical = newThumbWheel.verticalFlag();
	}

	/**
	 * Create a new instance of <code>JunThumbWheelViewAwt</code> and initialize it.
	 * 
	 * @param newThumbWheel jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @param isVertical boolean
	 * @deprecated since Jun643
	 * @category Instance creation
	 */
	public JunThumbWheelViewAwt(JunThumbWheel newThumbWheel, boolean isVertical) {
		super(newThumbWheel);
		this.setVertical(isVertical);
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		Rectangle box = this.thumbWheelArea();
		if (this.getModel().verticalFlag()) {
			box = new Rectangle(box.x - 3, box.y - 6, box.width + 6, box.height + 12);
		} else {
			box = new Rectangle(box.x - 6, box.y - 3, box.width + 12, box.height + 6);
		}
		this.setSize(box.width, box.height);
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunThumbWheelController();
	}

	/**
	 * Answer a default model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultModel()
	 * @category model accessing
	 */
	protected StModel defaultModel() {
		return new JunThumbWheel();
	}

	/**
	 * Answer the receiver's model as JunThumbWheel.
	 * 
	 * @return jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#getModel()
	 * @category model accessing
	 */
	public JunThumbWheel getModel() {
		return (JunThumbWheel) model();
	}

	/**
	 * Set the receiver's model.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category model accessing
	 */
	public void setModel(JunThumbWheel aModel) {
		this.model_(aModel);
	}

	/**
	 * Return the current Image.
	 * 
	 * @return java.awt.Image
	 * @category accessing
	 */
	protected Image currentImage() {
		return thumbWheelImages()[currentIndex()];
	}

	/**
	 * Return the current Index.
	 * 
	 * @return int
	 * @category accessing
	 */
	protected int currentIndex() {
		return currentIndex;
	}

	/**
	 * Decrease the current Index.
	 * 
	 * @return int
	 * @category accessing
	 */
	protected int decreaseIndex() {
		if (currentIndex <= 0) {
			currentIndex = this.thumbWheelImages().length - 1;
		} else {
			currentIndex--;
		}
		return currentIndex;
	}

	/**
	 * Increase the current Index.
	 * 
	 * @return int
	 * @category accessing
	 */
	protected int increaseIndex() {
		if (currentIndex >= (this.thumbWheelImages().length - 1)) {
			currentIndex = 0;
		} else {
			currentIndex++;
		}
		return currentIndex;
	}

	/**
	 * Answer the receiver's wheel area.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#thumbWheelArea()
	 * @category accessing
	 */
	public Rectangle thumbWheelArea() {
		Image image = this.currentImage();
		int imageWidth = image.getWidth(null);
		int imageHeight = image.getHeight(null);
		Dimension size = this.getSize();
		Rectangle bounds = new Rectangle(0, 0, size.width, size.height);
		int x = bounds.x + ((bounds.width - imageWidth) / 2);
		int y = bounds.y + ((bounds.height - imageHeight) / 2);
		return new Rectangle(x, y, imageWidth, imageHeight);
	}

	/**
	 * Answer the Vector which is stored the wheel images.
	 * 
	 * @return java.awt.Image[]
	 * @category accessing
	 */
	protected Image[] thumbWheelImages() {
		if (thumbWheelImages == null) {
			if (this.isVertical()) {
				thumbWheelImages = VerticalThumbWheelImages;
			} else {
				thumbWheelImages = HorizontalThumbWheelImages;
			}
		}
		return thumbWheelImages;
	}

	/**
	 * Set the receivers vertical property.
	 * 
	 * @param aBoolean boolean
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#setVertical(boolean)
	 * @deprecated since Jun643
	 * @category accessing
	 */
	public void setVertical(boolean aBoolean) {
		this.getModel().verticalFlag_(aBoolean);
		thumbWheelImages = null;
		isVertical = aBoolean;
		this.buildComponent();
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewCanvas#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		Rectangle area = this.thumbWheelArea();
		if (area.width == 0 || area.height == 0) {
			return;
		}
		Rectangle box;
		if (this.isVertical()) {
			box = new Rectangle(area.x - 3, area.y - 6, area.width + 6, area.height + 12);
		} else {
			box = new Rectangle(area.x - 6, area.y - 3, area.width + 12, area.height + 6);
		}

		aGraphics.clipRect(box.x, box.y, box.width, box.height);
		aGraphics.setColor(new Color(76, 76, 76));
		aGraphics.fillRect(box.x, box.y, box.width, box.height);
		aGraphics.setColor(Color.black);
		aGraphics.drawRect(box.x + 2, box.y + 2, box.width - 4, box.height - 4);
		aGraphics.setColor(new Color(178, 178, 178));
		aGraphics.drawRect(box.x + 1, box.y + 1, box.width - 2, box.height - 2);
		this.displayCurrentImageOn_(aGraphics);
	}

	/**
	 * Display an current image on a graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayCurrentImageOn_(Graphics aGraphics) {
		Rectangle area = this.thumbWheelArea();
		aGraphics.clipRect(area.x, area.y, area.width, area.height);
		aGraphics.drawImage(this.currentImage(), area.x, area.y, null);
	}

	/**
	 * Rotate with the specified delta point.
	 * 
	 * @param deltaPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#rotateDelta_(java.awt.Point)
	 * @category rotating
	 */
	public void rotateDelta_(Point deltaPoint) {
		double radius;
		int diameter;
		double angle;

		if (this.isVertical()) {
			radius = diameter = this.currentImage().getHeight(null) / 2;
			if (deltaPoint.y > diameter) {
				deltaPoint.y = diameter;
			}
			angle = (double) (Math.atan(deltaPoint.y / radius) / Math.PI * 180);
		} else {
			radius = diameter = this.currentImage().getWidth(null) / 2;
			if (deltaPoint.x > diameter) {
				deltaPoint.x = diameter;
			}
			angle = (double) (Math.atan(deltaPoint.x / radius) / Math.PI * 180);
		}

		if (angle == 0) {
			return;
		}

		if (this.model().respondsTo_("angularValue")) {
			try {
				this.model().perform_with_("angularValue_", new Double(angle));
			} catch (Exception e) {
				throw new SmalltalkException(e);
			}
		} else {
			this.update_(new DependentEvent(this, $("value"), new Double(angle)));
		}
	}

	/**
	 * Answer true if the receiver's property is horizontal, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#isHorizontal()
	 * @category testing
	 */
	public boolean isHorizontal() {
		return !this.isVertical();
	}

	/**
	 * Answer true if the receiver's property is vertical, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.wheels.JunThumbWheelView#isVertical()
	 * @category testing
	 */
	public boolean isVertical() {
		if (this.getModel() != null) {
			boolean aBoolean = this.getModel().verticalFlag();
			if (aBoolean != isVertical) {
				thumbWheelImages = null;
			}
			isVertical = aBoolean;
		} else {
			isVertical = false;
		}
		return isVertical;
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a dependent.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.StViewCanvas#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		if (evt.getAspect() == $("value")) {
			this.updateValue_(((Number) evt.getParameter()).doubleValue());
		}
	}

	/**
	 * Update the receiver's value.
	 * 
	 * @param anglevalue double
	 * @category updating
	 */
	protected void updateValue_(double anglevalue) {
		Graphics gc = this.getGraphics();
		try {
			int times = ((int) Math.abs(anglevalue) / 10) + 1;
			if (anglevalue > 0) {
				for (int i = 0; i < times; i++) {
					this.increaseIndex();
					this.displayCurrentImageOn_(gc);
				}
			} else {
				for (int i = 0; i < times; i++) {
					this.decreaseIndex();
					this.displayCurrentImageOn_(gc);
				}
			}
		} finally {
			if (gc != null) {
				gc.dispose();
			}
		}
	}
}
