package jp.co.sra.jun.goodies.track;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunTrackSliderModel class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/19 (by MATSUDA Ryouichi)
 *  @updated   1999/12/10 (by MATSUDA Ryouichi)
 *  @updated   2002/10/28 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2004/09/22 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun651 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTrackSliderModel.java,v 8.11 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTrackSliderModel extends JunApplicationModel implements StBlockValued {
	protected StValueHolder valueHolder;
	protected StValueHolder intervalHolder;
	protected double[] intervalBounds;
	protected boolean fixFirstMarker;
	protected boolean fixLastMarker;
	protected boolean useMarkers;
	protected StPopupMenu enabledPopupMenu;
	protected StPopupMenu disabledPopupMenu;
	protected JunTrackerModel parentTracker;

	/**
	 * Create a new instance of <code>JunTrackSliderModel</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTrackSliderModel() {
		this.initialize();
	}

	/**
	 * Initialize the JunTrackSliderModel.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		valueHolder = null;
		intervalHolder = null;
		intervalBounds = null;
		fixFirstMarker = false;
		fixLastMarker = false;
		useMarkers = false;
		enabledPopupMenu = null;
		disabledPopupMenu = null;
		parentTracker = null;
		this.valueHolder();
		this.intervalHolder();
	}

	/**
	 * Answer a child tracker if exist.
	 * 
	 * @return JunTrackSliderModel or null
	 * @category accessing
	 */
	public JunTrackSliderModel childTracker() {
		return null;
	}

	/**
	 * Disable markers.
	 * 
	 * @category accessing
	 */
	public void disableMarkers() {
		useMarkers = false;
		if (this.parentTracker() == null) {
			this.changed_($("useMarkers"));
		}
	}

	/**
	 * Answer my value as double.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double doubleValue() {
		return ((Number) this.value()).doubleValue();
	}

	/**
	 * Enable markers.
	 * 
	 * @category accessing
	 */
	public void enableMarkers() {
		useMarkers = true;
		if (this.parentTracker() == null) {
			this.changed_($("useMarkers"));
		}
	}

	/**
	 * Answer my current first marker.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double firstMarker() {
		return this.interval()[0];
	}

	/**
	 * Set the first marker.
	 * 
	 * @param normalizedValue double
	 * @category accessing
	 */
	public void firstMarker_(double normalizedValue) {
		if (this.fixFirstMarker() == false) {
			double first = Math.min(normalizedValue, this.lastMarker());
			first = Math.max(first, this.intervalBounds()[0]);
			first = Math.min(first, this.intervalBounds()[1]);
			this.interval_(new double[] { first, this.lastMarker() });
		}
	}

	/**
	 * Answer true if the first marker is needed to be fixed, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean fixFirstMarker() {
		return fixFirstMarker;
	}

	/**
	 * Set whether to fix the first marker or not.
	 *
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void fixFirstMarker_(boolean aBoolean) {
		fixFirstMarker = aBoolean;
	}

	/**
	 * Answer true if the last marker is needed to be fixed, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean fixLastMarker() {
		return fixLastMarker;
	}

	/**
	 * Set whether to fix the last marker or not.
	 *
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void fixLastMarker_(boolean aBoolean) {
		fixLastMarker = aBoolean;
	}

	/**
	 * Answer my current interval as an array of double.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] interval() {
		return (double[]) this.intervalHolder().value();
	}

	/**
	 * Set the new interval.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void interval_(double[] anArray) {
		this.intervalHolder().value_(this.adjustInterval_(anArray));
		this.changed_($("interval"));
	}

	/**
	 * Answer my current last marker.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double lastMarker() {
		return this.interval()[1];
	}

	/**
	 * Set the last marker.
	 * 
	 * @param normalizedValue double
	 * @category accessing
	 */
	public void lastMarker_(double normalizedValue) {
		if (this.fixLastMarker() == false) {
			double last = Math.max(normalizedValue, this.firstMarker());
			last = Math.max(last, this.intervalBounds()[0]);
			last = Math.min(last, this.intervalBounds()[1]);
			this.interval_(new double[] { this.firstMarker(), last });
		}
	}

	/**
	 * Answer my current parent tracker.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	public JunTrackerModel parentTracker() {
		return parentTracker;
	}

	/**
	 * Set my new parent tracker.
	 * 
	 * @param aTracker jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	public void parentTracker_(JunTrackerModel aTracker) {
		parentTracker = aTracker;
		aTracker.childTracker_(this);
	}

	/**
	 * Get the value of valueHolder.
	 * 
	 * @return java.lang.Object
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return valueHolder.value();
	}

	/**
	 * Set the value of valueHolder.
	 * 
	 * @param normalizedNumber double
	 * @category accessing
	 */
	public void value_(double normalizedNumber) {
		valueHolder.value_(Math.max(0, Math.min(normalizedNumber, 1)));
		this.changed_($("value"));
	}

	/**
	 * Get the valueHolder with double value.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder valueHolder() {
		if (valueHolder == null) {
			valueHolder = new StValueHolder(0.0d);
		}
		return valueHolder;
	}

	/**
	 * Answer my current interval holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder intervalHolder() {
		if (intervalHolder == null) {
			intervalHolder = new StValueHolder(new double[] { 0, 0 });
		}
		return intervalHolder;
	}

	/**
	 * Answer a BlockValue.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunTrackSliderViewAwt(this);
		} else {
			return new JunTrackSliderViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Track Slider");
	}

	/**
	 * Menu message: Flush my markers.
	 * 
	 * @category menu messages
	 */
	public void flushMarkers() {
		this.lastMarker_(0);
	}

	/**
	 * Menu message: Select entire movie with markers.
	 * 
	 * @category menu messages
	 */
	public void selectAll() {
		this.lastMarker_(1);
		this.firstMarker_(0);
	}

	/**
	 * Menu message: Set the first marker.
	 * 
	 * @category menu messages
	 */
	public void setFirstMarker() {
		this.firstMarker_(this.doubleValue());
	}

	/**
	 * Menu message: Set the last marker.
	 * 
	 * @category menu messages
	 */
	public void setLastMarker() {
		this.lastMarker_(this.doubleValue());
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);
		aWriter.write(" (");
		aWriter.write(String.valueOf(this.value()));
		aWriter.write(')');
	}

	/**
	 * Answer true if the markers are in active, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean areMarkersActive() {
		return useMarkers;
	}

	/**
	 * Answer true if the current interval is not empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isIntervalNotEmpty() {
		return this.areMarkersActive() && (this.firstMarker() < this.lastMarker());
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (this.parentTracker() != null) {
			return this.parentTracker()._popupMenu();
		}
		if (this.areMarkersActive()) {
			if (enabledPopupMenu == null) {
				enabledPopupMenu = this._createEnabledPopupMenu();
			}
			StMenuItem menuItem = enabledPopupMenu.atNameKey_($("flushMarkers"));
			if (menuItem != null) {
				menuItem.beEnabled(this.isIntervalNotEmpty());
			}
			return enabledPopupMenu;
		} else {
			if (disabledPopupMenu == null) {
				disabledPopupMenu = this._createDisabledPopupMenu();
			}
			return disabledPopupMenu;
		}
	}

	/**
	 * Create a popup menu for the disabled status.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu _createDisabledPopupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Enable markers"), new MenuPerformer(this, "enableMarkers")));
		return aPopupMenu;
	}

	/**
	 * Create a popup menu for the enabled status.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu _createEnabledPopupMenu() {
		StPopupMenu aPopupMenu = new StPopupMenu();
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Select all"), new MenuPerformer(this, "selectAll")));
		aPopupMenu.addSeparator();
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Set first marker"), new MenuPerformer(this, "setFirstMarker")));
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Set last marker"), new MenuPerformer(this, "setLastMarker")));
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Flush markers"), $("flushMarkers"), new MenuPerformer(this, "flushMarkers")));
		aPopupMenu.addSeparator();
		aPopupMenu.add(new StMenuItem(JunSystem.$String("Disable markers"), new MenuPerformer(this, "disableMarkers")));
		return aPopupMenu;
	}

	/**
	 * Adjust the interval.
	 * 
	 * @param anArray double[]
	 * @return double[]
	 * @category private
	 */
	protected double[] adjustInterval_(double[] anArray) {
		double first = Math.min(anArray[0], anArray[1]);
		double last = Math.max(anArray[0], anArray[1]);
		first = Math.max(0, Math.min(first, 1));
		last = Math.max(first, Math.min(last, 1));
		return new double[] { first, last };
	}

	/**
	 * Answer my current interval bounds.
	 * 
	 * @return double[]
	 * @category private
	 */
	protected double[] intervalBounds() {
		if (intervalBounds == null) {
			intervalBounds = new double[] { 0, 1 };
		}
		return intervalBounds;
	}

	/**
	 * Set my new interval bounds.
	 *
	 * @param anArray double[]
	 * @category private
	 */
	protected void intervalBounds_(double[] anArray) {
		intervalBounds = this.adjustInterval_(anArray);
		this.changed_($("intervalBounds"));
	}
}
