package jp.co.sra.jun.goodies.tools;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;

import jp.co.sra.smalltalk.StObject;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;

/**
 * JunTextFileToHtmlFile class
 * 
 *  @author    nisinaka
 *  @created   2006/03/31 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextFileToHtmlFile.java,v 8.11 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTextFileToHtmlFile extends StObject {

	protected File textFile;

	private char startingChar;

	/**
	 * Request a test file and create a new instance of JunTextFileToHtmlFile with it.
	 * 
	 * @return jp.co.sra.jun.goodies.tools.JunTextFileToHtmlFile
	 * @category Utilities 
	 */
	public static JunTextFileToHtmlFile Request() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Text files"), new String[] { "*.txt", "*.TXT" }), JunFileModel.FileType.All($String("All files")) };
		File aFile = JunFileRequesterDialog.Request($String("Select a file"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return null;
		}
		if (aFile.exists() == false) {
			return null;
		}

		return new JunTextFileToHtmlFile(aFile);
	}

	/**
	 * Create a new instance of JunTextFileToHtmlFile and initialize it.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunTextFileToHtmlFile(File aFile) {
		this.initialize();
		this.textFile_(aFile);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		textFile = null;
	}

	/**
	 * Answer my current text file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File textFile() {
		return textFile;
	}

	/**
	 * Set my new text file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void textFile_(File aFile) {
		textFile = aFile;
	}

	/**
	 * Answer my current html file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File htmlFile() {
		if (this.textFile() == null) {
			return null;
		}

		File parentFile = this.textFile().getParentFile();
		String filename = this.textFile().getName();
		int index = filename.lastIndexOf('.');
		if (index >= 0) {
			filename = filename.substring(0, index);
		}

		return new File(parentFile, filename + ".html");
	}

	/**
	 * Answer my current title string.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	protected String titleString() {
		if (this.textFile() == null) {
			return "";
		}

		String filename = this.textFile().getName();
		int index = filename.lastIndexOf('.');
		if (index >= 0) {
			filename = filename.substring(0, index);
		}
		return filename;
	}

	/**
	 * Convert the text file to the html file.
	 * 
	 * @category converting
	 */
	public void convert() {
		File htmlFile = this.htmlFile();
		if (htmlFile == null) {
			return;
		}

		PrintWriter aWriter = null;
		try {
			aWriter = new PrintWriter(new BufferedWriter(new FileWriter(htmlFile)));

			JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
			try {
				cursor._show();

				this.beginConversion_(aWriter);
				this.doConversion_(aWriter);
				this.endConversion_(aWriter);

			} finally {
				cursor._restore();
			}
		} catch (IOException e) {
			System.err.println(e.getMessage());
		} finally {
			if (aWriter != null) {
				aWriter.flush();
				aWriter.close();
			}
		}
	}

	/**
	 * Write the beginning part of the html to the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @category private
	 */
	protected void beginConversion_(PrintWriter aWriter) {
		aWriter.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">");
		aWriter.println("<HTML lang=\"ja\">");
		aWriter.println("<HEAD>");
		aWriter.println("<META http-equiv=\"Content-Type\" content=\"text/html; charset=Shift_JIS\">");
		aWriter.println("<TITLE>" + this.titleString() + "</TITLE>");
		aWriter.println("</HEAD>");
		aWriter.println("<BODY bgcolor=\"#ffffff\">");
	}

	/**
	 * Write the ending part of the html to the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @category private
	 */
	protected void endConversion_(PrintWriter aWriter) {
		aWriter.println("</BODY>");
		aWriter.println("</HTML>");
	}

	/**
	 * Write the main part of the html to the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @category private
	 */
	protected void doConversion_(PrintWriter aWriter) {
		BufferedReader aReader = null;
		try {
			aReader = new BufferedReader(new FileReader(this.textFile()));

			aWriter.println("<PRE>");

			String aString;
			while ((aString = aReader.readLine()) != null) {
				int index = aString.indexOf("http://");
				if (index >= 0) {
					aWriter.println(this.htmlAnchorTagString_startingAt_(aString, index));
				} else {
					aWriter.println(JunStringUtility.HtmlCanonicalString_(aString));
				}
			}
		} catch (FileNotFoundException e) {
			System.err.println(e.getMessage());
		} catch (IOException e) {
			System.err.println(e.getMessage());
		} finally {
			aWriter.println("</PRE>");

			if (aReader != null) {
				try {
					aReader.close();
				} catch (IOException e) {
				}
			}
		}
	}

	/**
	 * Convert the string which contains an HTML anchor tag.
	 * 
	 * @param aString java.lang.String
	 * @param startingIndex int
	 * @return java.lang.String
	 * @category private
	 */
	private String htmlAnchorTagString_startingAt_(String aString, int startingIndex) {
		startingChar = (startingIndex > 0) ? aString.charAt(startingIndex - 1) : Character.MAX_VALUE;

		int endingIndex = startingIndex;
		while (endingIndex < aString.length()) {
			if (this.isSeparatorCharacter(aString.charAt(endingIndex))) {
				break;
			}
			endingIndex++;
		}

		StringBuffer aStringBuffer = new StringBuffer();
		aStringBuffer.append(aString.substring(0, startingIndex));
		aStringBuffer.append("<A HREF=\"");
		aStringBuffer.append(aString.substring(startingIndex, endingIndex));
		aStringBuffer.append("\">");
		aStringBuffer.append(aString.substring(startingIndex, endingIndex));
		aStringBuffer.append("</A>");
		aStringBuffer.append(aString.substring(endingIndex));
		return aStringBuffer.toString();
	}

	/**
	 * Create a collection of separator characters.
	 * 
	 * @param ch char
	 * @return java.util.Collection
	 * @category private
	 */
	private boolean isSeparatorCharacter(char ch) {
		if (Character.isWhitespace(ch)) {
			return true;
		}

		switch (ch) {
			case '\u3000':
			case '\u3001':
			case '\u3002':
			case '\uFF0C':
			case '\uFF0E':
				return true;
		}

		if (startingChar != Character.MAX_VALUE) {
			if (startingChar == '(' && ch == ')') {
				return true;
			}
			if (startingChar == '[' && ch == ']') {
				return true;
			}
			if (startingChar == '<' && ch == '>') {
				return true;
			}
			if (startingChar == '{' && ch == '}') {
				return true;
			}
			if (startingChar == '\u3008' && ch == '\u3009') {
				return true;
			}
			if (startingChar == '\u300A' && ch == '\u300B') {
				return true;
			}
			if (startingChar == '\u300C' && ch == '\u300D') {
				return true;
			}
			if (startingChar == '\u300E' && ch == '\u300F') {
				return true;
			}
			if (startingChar == '\u3010' && ch == '\u3011') {
				return true;
			}
			if (startingChar == '\u3014' && ch == '\u3015') {
				return true;
			}
			if (startingChar == '\u2018' && ch == '\u2019') {
				return true;
			}
			if (startingChar == '\u201C' && ch == '\u201D') {
				return true;
			}
			if (startingChar == '\u226A' && ch == '\u226B') {
				return true;
			}
			if (startingChar == '\uFF08' && ch == '\uFF09') {
				return true;
			}
			if (startingChar == '\uFF1C' && ch == '\uFF1E') {
				return true;
			}
			if (startingChar == '\uFF3B' && ch == '\uFF3D') {
				return true;
			}
			if (startingChar == '\uFF5B' && ch == '\uFF5D') {
				return true;
			}
		}

		return false;
	}

}
