package jp.co.sra.jun.goodies.texteditor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.WindowSpecLayout;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunTextEditorViewSwing class
 * 
 *  @author    m-asada
 *  @created   2005/05/11 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextEditorViewSwing.java,v 8.12 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTextEditorViewSwing extends JunAbstractViewJPanel implements JunTextEditorView {
	protected JTextArea textArea;

	protected UndoManager undoManager;
	private UndoAction undoAction;
	private RedoAction redoAction;

	class UndoAction extends AbstractAction {
		/**
		 * Create a new instance of <code>UndoAction</code> and initialize it.
		 * 
		 * @category Instance creation
		 */
		public UndoAction() {
			super("Undo");
			this.setEnabled(false);
		}

		/**
		 * Invoked when an action event.
		 * 
		 * @param e java.awt.event.ActionEvent
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 * @category event listeners
		 */
		public void actionPerformed(ActionEvent e) {
			try {
				JunTextEditorViewSwing.this.undoManager().undo();
			} catch (CannotUndoException ex) {
				System.out.println("Unable to undo: " + ex);
				ex.printStackTrace();
			}
			this.update();
			JunTextEditorViewSwing.this.redoAction().update();
		}

		/**
		 * Update the receiver.
		 * 
		 * @category updating
		 */
		protected void update() {
			if (JunTextEditorViewSwing.this.undoManager().canUndo()) {
				this.setEnabled(true);
				putValue(Action.NAME, JunTextEditorViewSwing.this.undoManager().getUndoPresentationName());
			} else {
				this.setEnabled(false);
				this.putValue(Action.NAME, "Undo");
			}
		}
	}

	class RedoAction extends AbstractAction {
		/**
		 * Create a new instance of <code>RedoAction</code> and initialize it.
		 * 
		 * @category Instance creation
		 */
		public RedoAction() {
			super("Redo");
			this.setEnabled(false);
		}

		/**
		 * Invoked when an action event.
		 * 
		 * @param e java.awt.event.ActionEvent
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 * @category event listeners
		 */
		public void actionPerformed(ActionEvent e) {
			try {
				JunTextEditorViewSwing.this.undoManager().redo();
			} catch (CannotRedoException ex) {
				System.out.println("Unable to redo: " + ex);
				ex.printStackTrace();
			}
			this.update();
			JunTextEditorViewSwing.this.undoAction().update();
		}

		/**
		 * Update the receiver.
		 * 
		 * @category updating
		 */
		protected void update() {
			if (JunTextEditorViewSwing.this.undoManager().canRedo()) {
				this.setEnabled(true);
				this.putValue(Action.NAME, JunTextEditorViewSwing.this.undoManager().getRedoPresentationName());
			} else {
				this.setEnabled(false);
				this.putValue(Action.NAME, "Redo");
			}
		}
	}

	/**
	 * Create a new instance of JunTextEditorViewSwing and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTextEditorViewSwing() {
		super();
	}

	/**
	 * Create a new instance of JunTextEditorViewSwing and initialize it.
	 * 
	 * @param newModel jp.co.sra.jun.goodies.texteditor.JunTextEditor
	 * @category Instance creation
	 */
	public JunTextEditorViewSwing(JunTextEditor newModel) {
		super(newModel);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		textArea = null;
		undoManager = null;
		undoAction = null;
		redoAction = null;
	}

	/**
	 * Answer a default model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewPanel#defaultModel()
	 * @category model accessing
	 */
	protected StModel defaultModel() {
		return new JunTextEditor();
	}

	/**
	 * Answer the model as JunTextEditor.
	 * 
	 * @return jp.co.sra.jun.goodies.texteditor.JunTextEditor
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getModel()
	 * @category model accessing
	 */
	public JunTextEditor getModel() {
		return (JunTextEditor) this.model();
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewPanel#buildComponent()
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.getTextArea().setText(this.getModel().text());
		this.undoManager().discardAllEdits();
		JScrollPane scrollPane = new JScrollPane(this.getTextArea());
		scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

		this.setLayout(new WindowSpecLayout());
		this.add(scrollPane, WindowSpecLayout.Position(0f, 0, 0f, 0, 1f, 0, 1f, 0));
		this.setPreferredSize(new Dimension(480, 300));
	}

	/**
	 * Rebuild this component.
	 * 
	 * @param oldModel jp.co.sra.smalltalk.StModel
	 * @param newModel jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewPanel#rebuildComponent(jp.co.sra.smalltalk.StModel, jp.co.sra.smalltalk.StModel)
	 * @category interface opening
	 */
	protected void rebuildComponent(StModel oldModel, StModel newModel) {
		JunTextEditor textEditor = (JunTextEditor) newModel;
		this.getTextArea().setText(textEditor.text());
	}

	/**
	 * Answer the text area.
	 * 
	 * @return javax.swing.JTextArea
	 * @category interface opening
	 */
	public JTextArea getTextArea() {
		if (textArea == null) {
			textArea = new JTextArea();
			textArea.setFont(JunTextEditor.DefaultTextFont());
			textArea.setBackground(Color.white);
			textArea.setEditable(this.getModel().isEditable());
			textArea.setLineWrap(true);
			textArea.setWrapStyleWord(true);
			textArea.addMouseListener(new MouseAdapter() {
				public void mousePressed(MouseEvent event) {
					if (event.isMetaDown() || event.isPopupTrigger()) {
						_showPopupMenu(event.getX(), event.getY());
					}
				}
			});
			textArea.getDocument().putProperty("name", "textProperties");
			textArea.getDocument().addDocumentListener(new DocumentListener() {
				public void changedUpdate(DocumentEvent event) {
				}

				public void insertUpdate(DocumentEvent event) {
					JunTextEditorViewSwing self = JunTextEditorViewSwing.this;
					String key = (String) event.getDocument().getProperty("name");
					if (key.equals("textProperties")) {
						if (self.getText().equals(self.getModel().text()) == false) {
							self.getModel().text_(self.getText());
						}
					}
				}

				public void removeUpdate(DocumentEvent event) {
					JunTextEditorViewSwing self = JunTextEditorViewSwing.this;
					String key = (String) event.getDocument().getProperty("name");
					if (key.equals("textProperties")) {
						if (self.getText().equals(self.getModel().text()) == false) {
							self.getModel().text_(self.getText());
						}
					}
				}
			});
			textArea.getKeymap().addActionForKeyStroke(KeyStroke.getKeyStroke(KeyEvent.VK_Z, KeyEvent.CTRL_MASK), new AbstractAction() {
				public void actionPerformed(ActionEvent e) {
					JunTextEditorViewSwing.this.getModel().undoText();
				}
			});
			textArea.getKeymap().addActionForKeyStroke(KeyStroke.getKeyStroke(KeyEvent.VK_Y, KeyEvent.CTRL_MASK), new AbstractAction() {
				public void actionPerformed(ActionEvent e) {
					JunTextEditorViewSwing.this.getModel().redoText();
				}
			});
			textArea.getDocument().addUndoableEditListener(new UndoableEditListener() {
				public void undoableEditHappened(UndoableEditEvent e) {
					JunTextEditorViewSwing.this.undoManager().addEdit(e.getEdit());
					JunTextEditorViewSwing.this.undoAction().update();
					JunTextEditorViewSwing.this.redoAction().update();
				}

			});
		}
		return textArea;
	}

	/**
	 * Answer the text in text area.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getText()
	 * @category text accessing
	 */
	public String getText() {
		return this.getTextArea().getText();
	}

	/**
	 * Answer the selected text in text area.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#getSelectedText()
	 * @category text accessing
	 */
	public String getSelectedText() {
		String selectedText = null;
		try {
			selectedText = this.getTextArea().getSelectedText();
		} catch (IllegalArgumentException e) {
		}
		return selectedText;
	}

	/**
	 * Copy selected text to clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#copyText()
	 * @category text accessing
	 */
	public void copyText() {
		this.getTextArea().copy();
	}

	/**
	 * Cut selected text to clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#cutText()
	 * @category text accessing
	 */
	public void cutText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		this.getTextArea().cut();
	}

	/**
	 * Paste selected text from clipboard.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#pasteText()
	 * @category text accessing
	 */
	public void pasteText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		this.getTextArea().paste();
	}

	/**
	 * Clear text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditoreditor.JunTextEditorView#clearText()
	 * @category text accessing
	 */
	public void clearText() {
		if (this.getModel().isEditable() == false) {
			return;
		}

		String aText = this.getText();

		if (aText != null && aText.length() > 0) {
			this.getTextArea().setText("");
		}
	}

	/**
	 * Undo text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditor.JunTextEditorView#undoText()
	 * @category text accessing
	 */
	public void undoText() {
		if (this.undoManager().canUndo()) {
			this.undoManager().undo();
		}
	}

	/**
	 * Redo text.
	 * 
	 * @see jp.co.sra.jun.goodies.texteditor.JunTextEditorView#redoText()
	 * @category text accessing
	 */
	public void redoText() {
		if (this.undoManager().canRedo()) {
			this.undoManager().redo();
		}
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		this.getModel().updateMenuIndication();
		StSymbol aspect = evt.getAspect();
		if (aspect == $("text")) {
			if (this.getModel().text().equals(this.getTextArea().getText()) == false) {
				this.getTextArea().setText(this.getModel().text());
				this.getTextArea().setCaretPosition(0);
			}
			return;
		}
		if (aspect == $("editable")) {
			this.getTextArea().setEditable(this.getModel().isEditable());
			return;
		}

		super.update_(evt);
	}

	/**
	 * Answer the receiver's undo manager.
	 * 
	 * @return javax.swing.undo.UndoManager
	 * @category private
	 */
	protected UndoManager undoManager() {
		if (undoManager == null) {
			undoManager = new UndoManager();
		}
		return undoManager;
	}

	/**
	 * Answer the receiver's undo action.
	 * 
	 * @return jp.co.sra.jun.goodies.texteditor.JunTextEditorViewSwing#UndoAction
	 * @category private
	 */
	protected UndoAction undoAction() {
		if (undoAction == null) {
			undoAction = new UndoAction();
		}
		return undoAction;
	}

	/**
	 * Answer the receiver's redo action.
	 * 
	 * @return jp.co.sra.jun.goodies.texteditor.JunTextEditorViewSwing#RedoAction
	 * @category private
	 */
	protected RedoAction redoAction() {
		if (redoAction == null) {
			redoAction = new RedoAction();
		}
		return redoAction;
	}
}
