package jp.co.sra.jun.goodies.tables;

import java.io.IOException;
import java.io.Writer;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunAttributeTable class
 * 
 *  @author    nisinaka
 *  @created   2006/04/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAttributeTable.java,v 8.13 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunAttributeTable extends JunAbstractObject {
	protected HashMap attributeTable;

	/**
	 * Create a new instance of JunAttributeTable from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category Lisp support
	 */
	public static JunAttributeTable FromLispList_(JunLispList aList) {
		StSymbol kindName = (StSymbol) aList.head();
		if (kindName == $("JunAttributeTable")) {
			return new JunAttributeTable(aList);
		}

		throw new IllegalArgumentException("Expected a kind name of JunAttributeTable, but got " + kindName);
	}

	/**
	 * Create a new instance of JunAttributeTable from the string.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category Instance creation
	 */
	public static JunAttributeTable FromString_(String aString) {
		JunLispList aList = (JunLispList) JunLispParser.Parse_(aString);
		return FromLispList_(aList);
	}

	/**
	 * Create a new instance of JunAttributeTable and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunAttributeTable() {
		this(0);
	}

	/**
	 * Create a new instance of JunAttributeTable and initialize it.
	 *
	 * @param size int
	 * @category Instance creation
	 */
	public JunAttributeTable(int size) {
		super();
		this.initialize(size);
	}

	/**
	 * Create a new instance of JunAttributeTable and initialize it.
	 *
	 * @param aMap java.util.Map
	 * @category Instance creation
	 */
	public JunAttributeTable(Map aMap) {
		this(aMap.size());

		Iterator i = aMap.entrySet().iterator();
		while (i.hasNext()) {
			Map.Entry e = (Map.Entry) i.next();
			this.at_put_(e.getKey(), e.getValue());
		}
	}

	/**
	 * Create a new instance of JunAttributeTable and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunAttributeTable(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @param size int
	 * @category initialize-release
	 */
	protected void initialize(int size) {
		attributeTable = new HashMap(Math.max(size, 0));
	}

	/**
	 * Answer the current attribute table.
	 * 
	 * @return java.util.Map
	 * @category accessing
	 */
	public Map attributeTable() {
		return attributeTable;
	}

	/**
	 * Answer the size of the receiver.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return attributeTable.size();
	}

	/**
	 * Answer the keys of the attribute table.
	 * 
	 * @return java.util.Set
	 * @category accessing
	 */
	public Set keys() {
		return attributeTable.keySet();
	}

	/**
	 * Answer the entries of the attribute table.
	 * This method corresponds to 'associations' in Smalltalk version.
	 * 
	 * @return java.util.Set
	 * @category accessing
	 */
	public Set entries() {
		return attributeTable.entrySet();
	}

	/**
	 * Answer my current bindings.
	 * 
	 * @return java.util.Map.Entry[]
	 * @category accessing
	 */
	public Map.Entry[] bindings() {
		TreeSet bindings = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				Object key1 = ((Map.Entry) o1).getKey();
				Object key2 = ((Map.Entry) o2).getKey();
				if (key1 instanceof Comparable) {
					return ((Comparable) key1).compareTo(key2);
				} else if (key2 instanceof Comparable) {
					return ((Comparable) key2).compareTo(key1) * -1;
				} else {
					return key1.toString().compareTo(key2.toString());
				}
			}
		});
		bindings.addAll(this.entries());
		return (Map.Entry[]) bindings.toArray(new Map.Entry[bindings.size()]);
	}

	/**
	 * Answer the current value for the key.
	 * 
	 * @param key java.lang.Object
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object at_(Object key) {
		return attributeTable.get(key);
	}

	/**
	 * Answer the value at key.  If key is not found, answer the result of
	 * evaluation valueBlock.
	 * 
	 * @param key java.lang.Object
	 * @param valueBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object at_ifAbsent_(Object key, StBlockClosure valueBlock) {
		Object obj = attributeTable.get(key);
		if (obj == null) {
			return valueBlock.value();
		} else {
			return obj;
		}
	}

	/**
	 * Answer the value at key.  If key is not found, create a new entry
	 * for key, set its value to be the result of evaluating valueBlock,
	 * and answer the value.
	 * 
	 * @param key java.lang.Object
	 * @param valueBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object at_ifAbsentPut_(final Object key, final StBlockClosure valueBlock) {
		return this.at_ifAbsent_(key, new StBlockClosure() {
			public Object value() {
				Object value = valueBlock.value();
				JunAttributeTable.this.at_put_(key, value);
				return value;
			}
		});
	}

	/**
	 * Set the new value for the key.
	 * 
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 * @category accessing
	 */
	public void at_put_(Object key, Object value) {
		attributeTable.put(key, value);
	}

	/**
	 * Merge with the specified attributes.
	 * 
	 * @param anotherAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category accessing
	 */
	public JunAttributeTable merge_(JunAttributeTable anotherAttributes) {
		JunAttributeTable mergedAttributes = (JunAttributeTable) this.copy();

		Iterator i = anotherAttributes.entries().iterator();
		while (i.hasNext()) {
			Map.Entry entry = (Map.Entry) i.next();
			if (mergedAttributes.includesKey_(entry.getKey()) == false) {
				mergedAttributes.at_put_(entry.getKey(), entry.getValue());
			}
		}

		return mergedAttributes;
	}

	/**
	 * Add the entry.
	 * 
	 * @param anEntry java.util.Map.Entry
	 * @category adding
	 */
	public void add_(Map.Entry anEntry) {
		this.at_put_(anEntry.getKey(), anEntry.getValue());
	}

	/**
	 * Remove the entry with the specified key.
	 * 
	 * @param key java.lang.Object
	 * @return java.lang.Object
	 * @category removing 
	 */
	public Object removeKey_(Object key) {
		return attributeTable.remove(key);
	}

	/**
	 * Convert to a map.
	 * 
	 * @return java.util.Map
	 * @category converting
	 */
	public Map asDictionary() {
		return this.toMap();
	}

	/**
	 * Convert the receiver to the string.
	 * 
	 * @return java.lang.String
	 * @category converting
	 */
	public String asString() {
		return this.saveString();
	}

	/**
	 * Convert to a map.
	 * 
	 * @return java.util.Map
	 * @category converting
	 */
	public Map toMap() {
		return new HashMap(this.attributeTable());
	}

	/**
	 * Answer true if the receiver has no entities, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.size() == 0;
	}

	/**
	 * Answer true if the receiver has an entry with the specified key, otherwise false.
	 * 
	 * @param key java.lang.Object
	 * @return boolean
	 * @category testing
	 */
	public boolean includesKey_(Object key) {
		return attributeTable.containsKey(key);
	}

	/**
	 * Answer a copy of receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		try {
			JunAttributeTable anAttributeTable = (JunAttributeTable) this.getClass().newInstance();
			anAttributeTable.fromLispList(this.toLispList());
			return anAttributeTable.postCopy();
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		if (this.isEmpty() == false) {
			aWriter.write(" (bindings");

			Map.Entry[] bindings = this.bindings();
			int size = bindings.length;
			if (size <= this.printMax()) {
				for (int i = 0; i < size; i++) {
					this.printEntryOn_(bindings[i], aWriter);
				}
			} else {
				for (int i = 0; i < this.printMax() / 2; i++) {
					this.printEntryOn_(bindings[i], aWriter);
				}
				aWriter.write(" ...");
				for (int i = size - this.printMax() / 2; i < size; i++) {
					this.printEntryOn_(bindings[i], aWriter);
				}
			}

			aWriter.write(')');
		}
		aWriter.write(')');
	}

	/**
	 * Print the entry on the writer.
	 * 
	 * @param anEntry java.util.Map.Entry
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category printing
	 */
	protected void printEntryOn_(Map.Entry anEntry, Writer aWriter) throws IOException {
		aWriter.write(" (");
		aWriter.write(anEntry.getKey().toString());
		aWriter.write("->");
		aWriter.write(anEntry.getValue().toString());
		aWriter.write(')');
	}

	/**
	 * Answer the maximum number for printing elements.
	 * 
	 * @return int
	 * @category printing
	 */
	protected int printMax() {
		return 30;
	}

	/**
	 * Answer the save string.
	 * 
	 * @return java.lang.String
	 * @category printing
	 */
	protected String saveString() {
		JunLispCons list = JunLispCons.Cell();
		list.head_(this._className());
		list.add_(this.attributeTableToLispList());
		return list.saveString();
	}

	/**
	 * Print my storable string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write("((");
		aWriter.write(this._className().toString());
		aWriter.write(" fromString: '");
		aWriter.write(this.saveString());
		aWriter.write("') yourself)");
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return this._className();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.sizeFromLispList(aList);
		this.attributeTableFromLispList(aList);
	}

	/**
	 * Convert my attributes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons aList = new JunLispCons(this.kindName());
		aList.add_(this.sizeToLispList());
		aList.add_(this.attributeTableToLispList());
		return aList;
	}

	/**
	 * Get my size from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void sizeFromLispList(JunLispList aList) {
		JunLispList sizeList = aList._findSublistWithHead($("size"));
		if (sizeList == null) {
			return;
		}

		this.initialize(((Number) sizeList.tail()).intValue());
	}

	/**
	 * Convert my size to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons sizeToLispList() {
		return new JunLispCons($("size"), new Integer(this.size()));
	}

	/**
	 * Get my attribute table from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void attributeTableFromLispList(JunLispList aList) {
		JunLispList attributeTableList = aList._findSublistWithHead($("attributeTable"));
		if (attributeTableList == null) {
			return;
		}

		Object[] entities = ((JunLispList) attributeTableList.tail()).asArray();
		for (int i = 0; i < entities.length; i++) {
			JunLispCons cons = (JunLispCons) entities[i];
			Object value = cons.tail();
			if (value instanceof JunLispCons) {
				// TODO: respondsTo: #fromLispList:
			}
			this.at_put_(cons.head(), value);
		}
	}

	/**
	 * Convert my attribute table to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons attributeTableToLispList() {
		JunLispCons attributeTableList = new JunLispCons($("attributeTable"));

		Map.Entry[] bindings = this.bindings();
		for (int i = 0; i < bindings.length; i++) {
			Object value = bindings[i].getValue();
			if (value instanceof StObject && ((StObject) value).respondsTo_("toLispList")) {
				try {
					value = ((StObject) value).perform_("toLispList");
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			attributeTableList.add_(new JunLispCons(bindings[i].getKey(), value));
		}

		return attributeTableList;
	}
}
